--- lang.lua
--
-- @module lang
local lang = {}

local util = require "util"
local peg = require "peg"
local xxhash -- = require "xxhash"
local dprf, json -- = require "dprf"

local langCurrent = "en" -- "fi"
local langPrf
local format = string.format
local unpack = table.unpack

local function answerError(err)
	local rest = require "rest"
	rest.answerError(err)
end

local function loadLang()
	if not langPrf then
		langPrf = {} -- recursive calls, we must set langPrf to dome value before dprf.prf() -call, dprf.prf() will do db connections and call lang -functions many times when printing connections
		dprf = require "dprf"
		langPrf = dprf.prf("language/lang_" .. langCurrent .. "_message.json", "no-error no-cache")
	end
end

function lang.langFormat(txt, ...)
	if ... then
		local arg = {...}
		arg = util.setArrayNullValue(arg, "nil")
		for i, val in ipairs(arg) do
			if type(val) == "table" then
				json = require "json"
				val = json.toJsonRaw(val)
				val = peg.replace(val, '"', "'") -- change json " -> '
				arg[i] = val
			elseif type(val) ~= "number" and type(val) ~= "string" then
				arg[i] = tostring(val)
			end
		end
		local ok, ret = pcall(format, txt, unpack(arg))
		return ok and ret or txt
	else
		return txt
	end
end

local execute, langCustom
function lang.tag(txt) -- l == lang()
	loadLang()
	if not execute then
		execute = require "execute"
	end
	if not langCustom then
		langCustom = dprf.prf("language/lang_" .. langCurrent .. "_custom.json")
	end
	return execute.getTagValue(txt, langCustom)
end

function lang.l(txt, ...) -- l == lang()
	loadLang()
	-- count usage
	if not lang[txt] then
		lang[txt] = {}
		lang[txt].callCount = 1
	else
		lang[txt].callCount = lang[txt].callCount + 1
	end
	if not lang[txt][langCurrent] and langPrf[txt] then
		lang[txt][langCurrent] = langPrf[txt]
	end
	if lang[txt][langCurrent] then
		return lang.langFormat(lang[txt][langCurrent], ...)
	end
	return lang.langFormat(txt, ...)
end

function lang.ll(langTbl, txt, ...) -- ll = "local lang"
	if langTbl[txt] ~= nil then
		return lang.langFormat(langTbl[txt], ...)
	else
		return lang.l(txt, ...)
	end
end

function lang.setLangCurrent(language) -- l == lang()
	langCurrent = language
end

function lang.printLangTranslations() -- l == lang()
	for k, v in pairs(lang) do
		print(k, v[langCurrent])
	end
end

local langCache = {}
local function loadLanguage(fileName)
	fileName = "language/" .. fileName
	if peg.find(fileName, "_custom") < 1 and langCache[fileName] then
		return langCache[fileName]
	end
	loadLang()
	local language, err = dprf.prf(fileName, "no-cache hash")
	if not language then
		return answerError(tostring(err))
	end
	if peg.find(fileName, "_custom") < 1 then
		langCache[fileName] = language
	end
	return language
end

local function langJoin(langDefault, customLanguage)
	for key, val in pairs(customLanguage) do
		langDefault[key] = val
	end
	return langDefault
end

function lang.getLanguageTbl(param)
	-- print("getLanguage")
	if not param.language then
		return answerError("param.language does not exist")
	elseif type(param.language) ~= "string" then
		return answerError("param.language is not a string")
	elseif param.language == "" then
		return answerError("param.language is empty")
	end
	--[[
	if not param.table then
		return answerError(l"param.table does not exist")
	end
	]]
	local language = peg.parseBefore(param.language, "_") -- old en_us -> en
	language = language:lower()
	local langDefault = loadLanguage("lang_" .. language .. ".json").language
	local customLanguage = loadLanguage("lang_" .. language .. "_custom.json").language
	if not customLanguage and not langDefault then -- or not langDefault[1]
		util.printWarning("default language '%s' and '%s' custom were not found", "lang_" .. language .. ".json", "lang_" .. language .. "_custom.json")
		language = {}
	elseif not customLanguage then -- or not customLanguage[1]
		language = langDefault
	elseif not langDefault then -- or not langDefault[1]
		util.printWarning("default language '%s' was not found", "lang_" .. language .. ".json")
		language = customLanguage
	else
		local hash
		if langDefault.hash and customLanguage.hash then
			if not xxhash then
				xxhash = require "xxhash"
			end
			hash = xxhash.hash64string(langDefault.hash .. customLanguage.hash) -- has is combination of two tables and must be changed, so calculate a new hash from previous ones
		end
		language = langJoin(langDefault, customLanguage)
		if hash then
			language.hash = hash
		end
	end

	language.language = param.language
	return {hdr = language}
end

return lang
