--- lib/sodium/sodium.lua
-- coped from: https://github.com/daurnimator/luasodium
-- @module crypto
local ffi = require "mffi"
local sodium = require 'luasodium/pureffi'
local sodium_lib

ffi.cdef [[
void escrypt_PBKDF2_SHA256(const uint8_t * passwd, size_t passwdlen, const uint8_t * salt, size_t saltlen, uint64_t c, uint8_t * buf, size_t outLength);
/*
typedef struct crypto_hash_sha256_state {
  uint32_t state[8];
  uint64_t count;
  uint8_t buf[64];
} crypto_hash_sha256_state;

typedef struct crypto_auth_hmacsha256_state {
    crypto_hash_sha256_state ictx;
    crypto_hash_sha256_state octx;
} crypto_auth_hmacsha256_state;

int crypto_auth_hmacsha256_init(crypto_auth_hmacsha256_state *state, const unsigned char *key, size_t keylen);
int crypto_auth_hmacsha256_update(crypto_auth_hmacsha256_state *state, const unsigned char *in, unsigned long long inlen);
int crypto_auth_hmacsha256_final(crypto_auth_hmacsha256_state *state, unsigned char *out);
*/
]]

local function loadLib()
	if sodium_lib == nil then
		local lib_loader = require 'luasodium._ffi.lib_loader'
		sodium_lib = lib_loader({})
	end
end

sodium.pbkdf2 = function(passwd, salt, c, outLength)
	-- escrypt_PBKDF2_SHA256(passwd, passwdlen, salt, saltlen, c, buf, outLength):
	-- Compute PBKDF2(passwd, salt, c, outLength) using HMAC-SHA256 as the PRF, and
	-- write the output to buf.  The value outLength must be at most 32 * (2^32 - 1).
	loadLib()
	local buf = ffi.new("uint8_t[?]", outLength)
	sodium_lib.escrypt_PBKDF2_SHA256(passwd, #passwd, salt, #salt, c, buf, outLength)
	local ret = ffi.string(buf, outLength)
	sodium_lib.sodium_memzero(buf, outLength)
	return ret
end

return sodium

--[=[
local rshift = bit.rshift
local bxor = bit.bxor
local function STORE32_BE(dst, w)
	--[[ if NATIVE_BIG_ENDIAN then
		memcpy(dst, &w, sizeof w)
	else ]]
	dst[3] = ffi.cast("uint8_t", w)
	w = rshift(w, 8)
	dst[2] = ffi.cast("uint8_t", w)
	w = rshift(w, 8)
	dst[1] = ffi.cast("uint8_t", w)
	w = rshift(w, 8)
	dst[0] = ffi.cast("uint8_t", w)
end

sodium.pbkdf2lua = function(passwd, salt, c, outLength)
	if outLength ~= 32 then
		return nil, "pbkdf2Lua outLength must be 32"
	end
	-- copied from: ~/installed/C/tls/libsodium/src/libsodium/crypto_pwhash/scryptsalsa208sha256/pbkdf2-sha256.c
	loadLib()
	local PShctx = ffi.new("crypto_auth_hmacsha256_state")
	local hctx = ffi.new("crypto_auth_hmacsha256_state")
	local T = ffi.new("uint8_t[32]")
	local U = ffi.new("uint8_t[32]")
	local ivec = ffi.new("uint8_t[4]")
	local buf = ffi.new("uint8_t[?]", outLength)
	sodium_lib.crypto_auth_hmacsha256_init(PShctx, passwd, #passwd)
	sodium_lib.crypto_auth_hmacsha256_update(PShctx, salt, #salt)
	local stateSize = ffi.sizeof(PShctx)
	local i = 0
	local clen
	-- while i * 32 < outLength do
	-- STORE32_BE(ivec, (uint32_t)(i + 1))
	STORE32_BE(ivec, i + 1)
	ffi.copy(hctx, PShctx, stateSize)
	sodium_lib.crypto_auth_hmacsha256_update(hctx, ivec, 4)
	sodium_lib.crypto_auth_hmacsha256_final(hctx, U)
	ffi.copy(T, U, 32)
	-- LCOV_EXCL_START
	for _ = 2, c do
		sodium_lib.crypto_auth_hmacsha256_init(hctx, passwd, #passwd)
		sodium_lib.crypto_auth_hmacsha256_update(hctx, U, 32)
		sodium_lib.crypto_auth_hmacsha256_final(hctx, U)
		for k = 0, 31 do
			-- T[k] ^= U[k] -- C ^= 2 is same as C = C ^ 2
			T[k] = bxor(T[k], U[k])
		end
	end
	-- LCOV_EXCL_STOP
	clen = outLength - i * 32
	if clen > 32 then
		clen = 32
	end
	ffi.copy(buf, T, clen) -- ffi.copy(buf[i * 32], T, clen)
	-- i = i + 1
	-- end
	sodium_lib.sodium_memzero(ffi.cast("void *", PShctx), stateSize)
	return ffi.string(buf, outLength)
end
--]=]
