--- mffi.lua
-- trace all ffi.new library calls and optionally anchor cdata allocations to prevent collectgarbage chrashes
-- @module mffi
local ffi = require "ffi" -- MUST be "ffi" here and only here, elsewhere use: require "mffi"
local C = ffi.C
local from4d, C2
if ffi.os == "Windows" then
	C = ffi.load("kernel32")
end

ffi.cdef [[
	void* malloc(size_t size);
	void free(void* ptr);
]]

if not jit then
	ffi.null = C.NULL -- https://github.com/jmckaskill/luaffi
else
	ffi.null = nil -- Luajit
end
local color

local prefix = "    "
local use = true -- true false
local anchorAll = false

local useFinalizer = true
local saveFunctionInfo = true

local tracePlainCall = true -- how to deal with cairo.lua?
local traceAnchorAlloc = false
local traceAnchorFinalize = false

local traceAllAlloc = false
local traceAllFinalize = false

local traceCollectgarbage = false
local noTraceName = {}
-- local useMalloc = false -- future option

ffi.cdef [[
	typedef void *HANDLE;
]]
local INVALID_HANDLE_VALUE = ffi.cast('HANDLE', -1)
function ffi.invalidWinHandle(handle)
	return handle == INVALID_HANDLE_VALUE
end

local function write(str)
	if not color then
		local ok
		ok, color = pcall(require, "ansicolors")
		if not ok then
			color = nil
		end
	end
	if str:find("ERROR", 1, true) then
		if color then
			if ffi.os == "Windows" then
				io.write(color("%{bright redbg}" .. str))
			else
				io.write(color("%{bright red}" .. str))
			end
			return
		end
	end
	if color then
		io.write(color("%{bright cyan}" .. str))
	else
		io.write(str)
	end
	-- io.flush()
end

local function setOption(opt)
	-- write("setOption")
	if opt.trace_all == true then
		useFinalizer = true
		saveFunctionInfo = true
		tracePlainCall = true
		traceAnchorAlloc = true
		traceAnchorFinalize = true
		traceAllAlloc = false
		traceAllFinalize = true
	elseif opt.trace_all == false then
		useFinalizer = false
		saveFunctionInfo = false
		tracePlainCall = false
		traceAnchorAlloc = false
		traceAnchorFinalize = false
		traceAllAlloc = false
		traceAllFinalize = false
	end
end
ffi.setOption = setOption

if not use or ffi.allocation then
	ffi.newAnchor = ffi.new
	ffi.newAnchorTrace = ffi.new
	ffi.newNoAnchor = ffi.new
	ffi.newNoAnchorTrace = ffi.new
	ffi.newNoAnchorNoTrace = ffi.new
	ffi.newNoAnchorBasic = ffi.new
	ffi.free = function()
		return
	end
	ffi.allocationIdTable = function()
		return
	end
	ffi.printAllocation = function()
		return
	end
else

	if ffi.allocation then -- prevent problems with possible the future ffi
		write(prefix .. "XXXXXXXXXXXXXXXXXXXXXXXXXX ffi ERROR: ffi.allocation already exists: " .. tostring(ffi.allocation) .. "\n")
		return
	end
	local allocation = {}
	local countAll = 0
	local count = 0
	local ffiNew = ffi.new -- or as option use malloc?

	ffi.allocation = allocation -- anchor anchor-table to ffi that will be returned

	local upperLevelInfo = { -- do not accpet containing these string -files as debug files, get upper level call
		"/mffi." -- ourself, fileName end may be .lua or something else
	}
	local getinfo = debug.getinfo

	local function finalizer(ptr)
		countAll = countAll - 1
		local ptrId = tostring(ptr)
		if allocation[ptrId] then
			if allocation[ptrId].ptr ~= nil and traceAnchorFinalize then
				write(prefix .. "ffi finalize anchor " .. count .. "/" .. countAll .. ": " .. allocation[ptrId].id .. "\n")
			elseif traceAllFinalize then
				write(prefix .. "ffi finalize " .. count .. "/" .. countAll .. ": " .. allocation[ptrId].id .. "\n")
			end
			allocation[ptrId].id = nil
			allocation[ptrId].ptr = nil
			allocation[ptrId] = nil
		end
	end

	local function free(ptr)
		local ptrId = tostring(ptr)
		if allocation[ptrId] then
			if traceAnchorFinalize or traceAllFinalize then
				write(prefix .. "ffi anchor free: " .. allocation[ptrId].id .. "\n")
			end
			-- finalizer(ptr)
			allocation[ptrId].id = nil
			allocation[ptrId].ptr = nil
			allocation[ptrId] = nil
		else
			write(prefix .. "ffi anchor free ERROR: failed to find from anchor: " .. ptrId .. "\n")
		end
	end

	local function findUpperLevel(src)
		for _, path in ipairs(upperLevelInfo) do
			if src:find(path, 1, true) then
				return true
			end
		end
		return false
	end

	local function setFinalize(ptr, id)
		local ok = pcall(ffi.gc, ptr, finalizer) -- we don't need second return, it is the same ptr
		if not ok then
			id = id or tostring(ptr)
			write(prefix .. "ffi.new: ERROR, trying to anchor basic type: " .. id .. "\n")
		end
	end

	local function newObject(anchor, trace, ct, nElem, init)
		-- ffi.new(ct [,nElem] [,init...]), http://luajit.org/ext_ffi_api.html
		countAll = countAll + 1
		local ptr
		if init ~= nil then
			ptr = ffiNew(ct, nElem, init)
		elseif nElem ~= nil then
			ptr = ffiNew(ct, nElem)
		else
			ptr = ffiNew(ct)
		end
		if anchor == false and trace == false and traceAllFinalize == false then
			if useFinalizer and ptr ~= nil then
				setFinalize(ptr)
			end
			return ptr
		end
		if count == 0 and (traceAllAlloc or traceAnchorAlloc) then
			write(prefix .. count .. ". ffi.new: first call, options: useFinalizer = " .. tostring(useFinalizer) .. ", anchorAll = " .. tostring(anchorAll) .. ", traceAllAlloc = " .. tostring(traceAllAlloc) .. ", traceAllFinalize = " .. tostring(traceAllFinalize) .. "\n")
		end
		count = count + 1
		local ptrId = tostring(ptr)
		local id
		if saveFunctionInfo == false then
			id = count .. ". " .. ptrId
		else
			local callDepth = 2
			local debugInfo = getinfo(callDepth)
			while findUpperLevel(debugInfo.short_src) do
				callDepth = callDepth + 1
				debugInfo = getinfo(callDepth)
			end
			local funcName = debugInfo.short_src .. " #" .. debugInfo.currentline
			if init ~= nil then
				id = count .. ". " .. funcName .. " " .. tostring(ct) .. " " .. tostring(nElem) .. " " .. tostring(init) .. " " .. tostring(ptr)
			elseif nElem ~= nil then
				id = count .. ". " .. funcName .. " " .. tostring(ct) .. " " .. tostring(nElem) .. " " .. tostring(ptr)
			else
				id = count .. ". " .. funcName .. " " .. tostring(ct) .. " " .. tostring(ptr)
			end
		end
		if useFinalizer and ptr ~= nil then
			setFinalize(ptr, id)
		end
		if allocation[ptrId] then
			write(prefix .. "ffi.new: ERROR, replacing: " .. ptrId .. " " .. id .. "\n")
		end
		if anchor then
			allocation[ptrId] = {id = id, ptr = ptr} -- ptr is real anchoring to ffi.allocation table
			if traceAnchorAlloc then
				write(prefix .. "ffi.new anchor: " .. id .. "\n")
			end
		else
			allocation[ptrId] = {id = id}
			if trace or traceAllAlloc then
				local doWrite = true
				for _, name in ipairs(noTraceName) do
					print(prefix, name)
					if prefix:find(name, 1, true) then
						doWrite = false
						break
					end
				end
				if doWrite then
					-- write(prefix.."ffi.new: "..id.."\n")
				end
			end
		end
		return ptr
	end

	local function newAnchorTrace(ct, nElem, init)
		return newObject(true, true, ct, nElem, init)
	end

	local function newAnchor(ct, nElem, init)
		return newObject(true, traceAllAlloc, ct, nElem, init)
	end

	local function newNoAnchorTrace(ct, nElem, init)
		return newObject(false, true, ct, nElem, init)
	end

	local newNoAnchor
	if traceAllAlloc then
		newNoAnchor = function(ct, nElem, init)
			return newObject(false, traceAllAlloc, ct, nElem, init)
		end
	else
		newNoAnchor = ffiNew
	end

	local function allocationIdTable()
		local ret = {} -- return sorted table
		for _, rec in pairs(allocation) do
			ret[#ret + 1] = {id = rec.id, ptr = rec.ptr}
		end
		table.sort(ret, function(a, b) -- sort by id, it starts with count number
			return a.id < b.id
		end)
		for i, rec in ipairs(ret) do -- convert to string table
			if rec.ptr ~= nil then
				ret[i] = "anchored: " .. rec.id
			else
				ret[i] = "not anchored: " .. rec.id
			end
		end
		return ret
	end

	local function printAllocation()
		write("\n" .. prefix .. "ffi print allocation, total: " .. countAll .. ", in use: " .. count .. "\n")
		local tbl = allocationIdTable()
		for i, id in ipairs(tbl) do
			write(prefix .. i .. ". " .. id .. "\n")
		end
		write(prefix .. "ffi print allocation end, total: " .. countAll .. ", in use: " .. count .. "\n\n")
	end

	--[[
	local function newNoAnchorNoTrace(...)
		return ffiNew(...)
	end

	local function newNoAnchorBasic(...)
		return ffiNew(...) -- could trace this too?
	end
	]]

	ffi.newAnchor = newAnchor
	ffi.newAnchorTrace = newAnchorTrace
	ffi.newNoAnchor = newNoAnchor
	ffi.newNoAnchorTrace = newNoAnchorTrace
	-- ffi.newNoAnchorNoTrace = newNoAnchorNoTrace
	-- ffi.newNoAnchorBasic = newNoAnchorBasic
	ffi.newNoAnchorNoTrace = ffiNew
	ffi.newNoAnchorBasic = ffiNew

	if anchorAll and tracePlainCall then
		ffi.new = newAnchorTrace
	elseif anchorAll then
		ffi.new = newAnchor
	elseif tracePlainCall then
		ffi.new = newNoAnchorTrace
	elseif useFinalizer then
		ffi.new = newNoAnchor
	else
		ffi.new = ffiNew -- ffiNew is original ffi.new
	end
	if ffi.free then -- prevent problems with possible the future ffi
		write(prefix .. "XXXXXXXXXXXXXXXXXXXXXXXXXX ffi ERROR: ffi.free already exists: " .. tostring(ffi.free) .. "\n")
	else
		ffi.free = free
	end
	ffi.allocationIdTable = allocationIdTable
	ffi.printAllocation = printAllocation
end

--[[if ffi.os == "Windows" then
	function ffi.errorText()
		if C == ffi.C and not util then
			util = require "util"
			if not util.from4d()  then
				C = ffi.load("kernel32")
			end
		end
		return ffi.string(C.strerror(ffi.errno()))
	end
else ]]
local util
if ffi.os == "Windows" then
	function ffi.loadMsvcr(loadVersion120)
		if from4d == nil then
			util = require "util"
			from4d = util.from4d()
		end
		if from4d and not loadVersion120 then
			return ffi.load("msvcrt")
		end
		local ok, dll
		ok, dll = pcall(ffi.load, "msvcr120_clr0400.dll")
		if not ok then
			ok, dll = pcall(ffi.load, "msvcr120.dll")
		end
		if not ok then
			util = require "util"
			dll = util.loadDll("msvcr120.dll")
			if dll == nil then
				dll = ffi.load("msvcr100.dll")
			end
			if dll == nil then
				util.printError("msvcr120.dll and msvcr100.dll load failed")
			end
		end
		return dll
	end

	function ffi.loadMsvcp(loadVersion120)
		if from4d == nil then
			util = require "util"
			from4d = util.from4d()
		end
		if from4d and not loadVersion120 then
			return ffi.load("msvcp90.dll")
		end
		local ok, dll
		ok, dll = pcall(ffi.load, "msvcp120_clr0400.dll")
		if not ok then
			ok, dll = pcall(ffi.load, "msvcp120.dll")
		end
		if not ok then
			dll = ffi.load("msvcp100.dll")
			if not ok then
				local util = require "util"
				dll = util.loadDll("msvcp120.dll")
				if dll == nil then
					util.printError("msvcp120.dll and msvcp100.dll load failed")
				end
			end
		end
		return dll
	end
end

function ffi.errorText()
	return ffi.string(C.strerror(ffi.errno()))
end

function ffi.doNotReport() -- (str)
	if not noTraceName[noTraceName] then
		noTraceName[noTraceName] = true
	end
end

if ffi.cstr then -- prevent problems with possible the future ffi
	write(prefix .. "XXXXXXXXXXXXXXXXXXXXXXXXXX ffi ERROR: ffi.cstr already exists: " .. tostring(ffi.cstr) .. "\n")
else
	function ffi.cstr(str)
		do
			--[[
			if util == nil then
				util = require "util"
			end
			util.printError("*** WARNING: using ffi.cstr()") --]]
			print("*** WARNING: using ffi.cstr()" .. debug.traceback())
			return str
		end
		local buf
		if traceAllAlloc then
			buf = ffi.newNoAnchorTrace("uint8_t[?]", #str + 1)
		else
			buf = ffi.newNoAnchor("uint8_t[?]", #str + 1)
		end
		ffi.copy(buf, str, #str)
		return buf
	end
end

function ffi.cstrAnchor(str)
	local buf
	if traceAnchorAlloc then
		buf = ffi.newAnchorTrace("uint8_t[?]", #str + 1)
	else
		buf = ffi.newAnchor("uint8_t[?]", #str + 1)
	end
	ffi.copy(buf, str, #str)
	return buf
end

function ffi.createBuffer(datalen)
	if datalen < 1 then
		write(prefix .. "ffi createBuffer ERROR: datalen < 1, datalen: " .. tostring(datalen) .. "\n")
		-- util.printError("datalen < 1")
		return nil
	end
	if traceAnchorAlloc then
		return ffi.newAnchorTrace("uint8_t[?]", datalen) -- newNoAnchor?
	end
	return ffi.newAnchor("uint8_t[?]", datalen) -- newNoAnchor?
end

function ffi.mallocChar(str)
	local strLen = #str
	if strLen < 1 then
		write(prefix .. "ffi mallocChar ERROR: string length" .. tostring(strLen) .. " < 1\n" .. debug.traceback())
		return nil
	end
	if C2 == nil then
		if ffi.os == "Windows" then
			C2 = ffi.loadMsvcr()
			if C2 == nil then
				util.printError("C is nil")
				return
			end
		else
			C2 = C
		end
	end
	local buf = ffi.cast("char *", C2.malloc(strLen + 1)) -- (strLen + 1) * ffi.sizeof("char"), but sizeof("char") is always 1: , https://stackoverflow.com/questions/2215445/are-there-machines-where-sizeofchar-1-or-at-least-char-bit-8
	ffi.copy(buf, str, strLen)
	buf[strLen] = 0 -- add C-string end "\0", buf is 0-based index
	return buf
end

function ffi.freeChar(cStr, option)
	if cStr == ffi.null then
		if option == "no-error" then
			return
		end
		write(prefix .. "ffi freeChar ERROR: parameter is null\n" .. debug.traceback())
		return
	end
	C2.free(ffi.cast("void *", cStr))
end

function ffi.copyStringToBuffer(buf, str, strlen)
	if strlen < 1 then
		write(prefix .. "ffi copyStringToBuffer ERROR: strlen < 1, strlen: " .. tostring(strlen) .. "\n" .. debug.traceback())
		-- util.printError("datalen < 1")
		return nil
	end
	ffi.copy(buf, str, strlen)
	buf[strlen] = 0 -- add C-string end "\0"
	return buf
end

function ffi.tableToCharArray(arg)
	local argv = ffi.newNoAnchor("char*[?]", #arg)
	for i = 1, #arg do
		argv[i - 1] = ffi.cast("char *", arg[i])
	end
	return argv
end

--[[
function ffi.pointerAddress(typeName, pointer)
	-- check calling function if we need anchored pointer address?
	-- create new empty pointer (usually structure)
	local emptyPtr
	if traceAllAlloc then
		emptyPtr = ffi.newNoAnchorTrace(typeName .. "[1]")
	else
		emptyPtr = ffi.newNoAnchor(typeName .. "[1]")
	end
	local ptrAddress = ffi.cast(typeName .. "**", emptyPtr) -- cast it to &typeName == typeName** == pointer-pointer
	ptrAddress[0] = pointer -- must set original pointer to ptrAddress[0]
	return ptrAddress
end
]]

if jit then -- Luajit
	function ffi.isNull(pointer)
		return pointer == nil
	end
	function ffi.isNotNull(pointer)
		return pointer ~= nil
	end
else -- plain Lua with https://github.com/jmckaskill/luaffi
	function ffi.isNull(pointer)
		return pointer == C.NULL
	end
	function ffi.isNotNull(pointer)
		return pointer ~= C.NULL
	end
end

--[[
function ffi.ptr(p) -- https://luapower.com/luajit-notes
   return p ~= ffi.newNoAnchor("void*") and p or nil
end
]]

if traceCollectgarbage then
	local collectgarbageOrig = collectgarbage
	local function newCollectgarbage(...)
		local arg = {...}
		write(prefix .. "ffi collectgarbage(" .. table.concat(arg, ", ") .. ")\n") -- does not print " or ' quotes correctly in params
		return collectgarbageOrig(...)
	end
	collectgarbage = newCollectgarbage
end

local loadDllAnchor = {}
ffi.loadDllAnchor = loadDllAnchor

local function fileExists(filePath)
	local f = io.open(filePath, "r")
	if f then
		f:close()
	end
	return f ~= nil
end

local function loadDll(fileNameOrig, path, option) -- path is path to forder where binaries exist
	local fileName = fileNameOrig:lower() -- prevents loading same module many times with different upper/lowercase name
	if loadDllAnchor[fileName] then
		return loadDllAnchor[fileName]
	end
	local suffix
	-- print("* ffi load: "..tostring(fileNameOrig))
	if ffi.os == "Windows" then
		if fileName:lower():sub(-4) == ".dll" then
			suffix = ""
		else
			suffix = ".dll"
		end
	elseif ffi.os == "OSX" then
		suffix = ".dylib"
	else
		suffix = ".so"
	end
	-- end
	local err, ok, ret
	local filePath = path .. fileName
	local exists = fileExists(filePath)
	if exists == false then
		filePath = path .. fileName .. suffix
		exists = fileExists(filePath)
		if exists == false then
			if fileName:find("/", 1, true) then
				filePath = path .. fileName:gsub("/", "/lib") .. suffix
			else
				filePath = path .. "lib" .. fileName .. suffix
			end
			exists = fileExists(filePath)
		end
	end

	local function loadLib(filePath2)
		if option and option:find("no-init", 1, true) then -- example: util.loadDll("libharfbuzz-0.dll", "no-init") -- do not call dll main()
			ok, ret = pcall(ffi.C.LoadLibraryExA, filePath2, nil, ffi.C.DONT_RESOLVE_DLL_REFERENCES)
		else
			ok, ret = pcall(ffi.load, filePath2)
		end
		return ok, ret
	end

	if exists then
		ok, ret = loadLib(filePath)
	end
	-- ret = ffi.load(filePath)
	if exists and not ok and (ret:find("undefined symbol", 1, true) or ret:find("Referenced from", 1, true)) then
		err = "shared library contains undefined symbol" -- go to error
	elseif exists == false then
		local pos = fileName:find("lib", 1, true)
		if not ok and ffi.os ~= "Windows" and (not pos or pos < #fileName - 3) then -- not ".dylib"
			filePath = path .. "lib" .. fileName .. suffix
			ok, ret = loadLib(filePath)
		end
		if not ok and ffi.os == "OSX" and not fileName:find(".dylib", 1, true) then
			suffix = ".so"
			filePath = path .. fileName .. suffix
			ok, ret = loadLib(filePath)
			if not ok then
				filePath = path .. "lib" .. fileName .. suffix
				ok, ret = loadLib(filePath)
			end
		end
		if not ok then -- plain system fileName
			filePath = fileName
			err = ret -- save old error
			ok, ret = loadLib(filePath)
		end
	end
	if not ok then
		if err then
			err = ret .. "/n (" .. err .. ")"
		else
			err = ret
		end
		err = string.format("loading dll fileName '%s' failed, path '%s', error '%s'", fileNameOrig, path, err)
		return nil, err
	end
	-- print("loadDll: "..filePath)
	loadDllAnchor[fileName] = ret -- anchor dll
	return ret
end
ffi.loadDll = loadDll

ffi.cdef [[
// void _mm_lfence(void);
// void _mm_sfence(void);
// void _os_atomic_full_barrier(void);
// void _mm_mfence(void) asm("mfence");
// void mm_mfence(void); //  asm("mfence")
// void __sync_synchronize(void);
void memfence();
]]
local memfence
function ffi.ReadWriteMemoryBarrierMfenceFunc()
	if not memfence then
		local util = require "util"
		memfence = util.loadDll("memfence").memfence
		-- memlib = loadDll("libSystem.B.dylib", "/usr/lib")
	end
	-- C.mm_mfence()
	-- C._mm_mfence()
	-- C.__sync_synchronize()
	-- C._os_atomic_full_barrier()
	return memfence
end

if ffi.os == "Windows" then
	--[=[ffi.cdef[[void 	MemoryBarrier(); // win]]
  function ffi.ReadWriteMemoryBarrier()
		C.MemoryBarrier()
  end
	]=]
	ffi.ReadWriteMemoryBarrier = ffi.ReadWriteMemoryBarrierMfence
elseif ffi.os == "OSX" then
	-- warning: 'OSMemoryBarrier' is deprecated: first deprecated in macOS 10.12 - Use std::atomic_thread_fence() from <atomic> instead
	ffi.cdef [[void OSMemoryBarrier();]] -- // mac TODO: 'OSMemoryBarrier' has been explicitly marked deprecated
	ffi.ReadWriteMemoryBarrier = C.OSMemoryBarrier
else
	ffi.ReadWriteMemoryBarrier = ffi.ReadWriteMemoryBarrierMfence
end

local peg
function ffi.fixCHeader(str)
	-- #define BARCODE_CODE11		1 --> static const int BARCODE_CODE11		= 1;
	if peg == nil then
		peg = require "peg"
	end
	local def = peg.pattern("#define")
	local notDef = peg.other(def, 0)
	local space, notSpace = peg.patternAntiPattern(peg.define.whiteSpace, 1)
	local _, notEol = peg.patternAntiPattern(peg.define.endOfLine, 1)
	local patt = notDef * def * space * peg.capture(notSpace * space) * peg.capture(notEol)
	local ret = {}
	peg.iterateLines(str, function(line)
		local var, val = patt:match(line)
		if var and val then
			if tostring(tonumber(val)) == val or peg.startsWith(val, "0x") then
				ret[#ret + 1] = "static const int " .. var .. " = " .. val .. ";"
			else
				ret[#ret + 1] = "// nada: " .. line -- "static const char*  "..var.." = "..val.."; // check this"
				print(" - warning: unknown define: " .. ret[#ret])
			end
		else
			ret[#ret + 1] = line
		end
	end)
	return table.concat(ret, "\n")
end

return ffi
