--- lib/net/fd-client.lua
-- see: lib/net/fd-server.lua
local util = require "util"
local json = require "json"
local fdSend = require "net/fd-send"
local fdServer = require "net/fd-server"
local ffi = require "mffi"
local l = require"lang".l
local tcpSend -- = require "system/socket"
local C = ffi.C
local skip = util.skip
local sendFd = fdSend.sendFd

local callCount = 0
local log = true
local useSocket = fdServer.useSocket
local fdBufferSize = fdServer.fdBufferSize

local sockFd, fdWrite, pipeFd, fdBuffer, fdSocket, fdSocketLen
local printFd = 0
local fdBufferSent = 0
local sockFdSent = {}

local function initPipe()
	tcpSend = require"system/socket".send
	fdSocketLen = ffi.sizeof("uintptr_t")
	sockFd = fdSend.connectToUnixServer()
	if sockFd < 0 then
		sockFd = nil
		local err = "Failed to connect to fd server"
		util.printWarning(err)
		return {echo2 = err} -- rest_answerError(err)
	end
	if useSocket then
		util.print("initPipe fdWrite %d", sockFd)
		local netsocket = require "system/socket"
		fdWrite = netsocket.new({socket = sockFd, ai_family = C.AF_UNIX, ai_socktype = C.SOCK_STREAM, ai_protocol = 0})
	else
		pipeFd = ffi.newAnchor("int[2]")
		C.pipe(pipeFd)
		printFd = printFd + 1
		sendFd(sockFd, pipeFd[0], printFd)
		C.close(pipeFd[0]) -- close the reading end
		util.print("closed sent pipe: %d", pipeFd[0])
		fdWrite = pipeFd[1]
	end
	fdBuffer = ffi.newAnchor("char[?]", fdBufferSize)
	fdSocket = ffi.newAnchor("uint64_t[2]")
	fdSocketLen = ffi.sizeof(fdSocket)
	fdBufferSize = fdBufferSize - fdSocketLen
end

local function writeData(data, dataLen)
	if useSocket then
		return tcpSend(fdWrite, data, dataLen)
	end
	return C.write(fdWrite, data, dataLen)
end

local function sendMessage(socket, data)
	local dataLen = #data
	if dataLen > fdBufferSize then
		local err = l("Socket data size %d is bigger than fdBufferSize %d", dataLen, fdBufferSize)
		util.printError(err)
		return {echo2 = err}
	end
	fdSocket[0] = ffi.cast("uint64_t", socket)
	fdSocket[1] = ffi.cast("uint64_t", dataLen)
	ffi.copy(fdBuffer, fdSocket, fdSocketLen)
	ffi.copy(fdBuffer + fdSocketLen, data, dataLen) -- All bytes of the Lua string plus a zero-terminator are copied to dst (i.e. #src+1 bytes).
	local ret = writeData(fdBuffer, fdSocketLen + dataLen)
	if ret < 0 then
		local err = util.printError("Failed to write data to fd server, error '%s'", ffi.string(C.strerror(ffi.errno())))
		return nil, {echo2 = err} -- rest_answerError(err)
	end
	return dataLen
end

local function echo2(sock)
	local ret
	if sockFd == nil then
		ret = initPipe()
		if ret then
			return ret
		end
	end
	if #sock.uri > fdBufferSize then
		local err = l("Call uri size %d is bigger than fdBufferSize %d", #sock.uri, fdBufferSize)
		util.printError(err)
		return {echo2 = err}
	end
	local dataLen, err
	if not sockFdSent[sock.socket] then
		local data = json.toJsonRaw(sock)
		dataLen, err = sendMessage(sock.socket, data)
		if err then
			return err
		end
		printFd = printFd + 1
		sendFd(sockFd, sock.socket, printFd)
		sockFdSent[sock.socket] = true
	else
		dataLen, err = sendMessage(sock.socket, sock.uri)
		if err then
			return err
		end
	end
	fdBufferSent = fdBufferSent + dataLen
	if log then
		callCount = callCount + 1
		if skip(callCount, 50000) == 0 then
			util.printOk("call fd server count: %d", callCount)
		end
	end
	return {skip_answer = true}
end

local function closeWorkerSocket(socketNum)
	if sockFd and sockFdSent[socketNum] then
		echo2({socket = socketNum, uri = "close-socket"})
		sockFdSent[socketNum] = nil
	end
end

return {echo2 = echo2, closeWorkerSocket = closeWorkerSocket}
