--- lib/net/fd-server.lua
-- see: lib/net/fd-client.lua
local util = require "util"
local json = require "json"
local fdSend = require "net/fd-send"
local ffi = require "mffi"
local rest -- = require "rest"
local socket -- = require "system/socket"
local C = ffi.C
local receiveFd = fdSend.receiveFd
local skip = util.skip

local callCount = 0
local log = true
local useSocket = util.isWin() or false
local fdBufferSize = 4096

local fdRead, fdBuffer, fdSocket, fdSocketLen
local sockFdSent = {}
local bytesSentTotal = 0
local printFd = 0

local function initPipe(cSockFd)
	if fdRead == nil then
		printFd = printFd + 1
		fdRead = receiveFd(cSockFd, printFd)
		printFd = printFd + 1
		local fd = type(fdRead) == "number" and fdRead or fdRead.socket -- initPipe() is called once, it's ok to have locals
		util.print("pipeFd received: read %d", fd)
		fdBuffer = ffi.newAnchor("char[?]", fdBufferSize)
		fdSocket = ffi.newAnchor("uint64_t[2]")
		fdSocketLen = ffi.sizeof(fdSocket)
	end
end

local function closePipe()
	if fdRead then
		if type(fdRead) == "table" then
			fdRead:close()
		else
			C.close(fdRead)
			-- C.close(fdWrite)
		end
		fdRead = nil
	end
end

local ret, err, len, retLen, sentSockNum
local function readData(data, dataLen)
	if useSocket then
		ret, err, retLen = fdRead:receive(dataLen) -- , true true == raw data
		if retLen == dataLen then
			if data == fdSocket then
				sentSockNum = tonumber(ffi.cast("uint64_t *", ret)[0])
				len = tonumber(ffi.cast("uint64_t *", ret)[1])
				return sentSockNum, len
			end
			return ffi.string(ret, retLen)
		end
		util.printError("readData length %d is not equal to %d", retLen, dataLen)
	else
		retLen = C.read(fdRead, data, dataLen)
	end
	if retLen < 0 then
		util.printError("readData failed")
	end
	if data == fdSocket then
		sentSockNum = tonumber(fdSocket[0])
		dataLen = tonumber(fdSocket[1])
		return sentSockNum, dataLen
	end
	return ffi.string(fdBuffer, dataLen)
end

local uri, dataLen
local function readPipe()
	sentSockNum, dataLen = readData(fdSocket, fdSocketLen)
	if sentSockNum < 1 then
		util.printWarning("error, sentSockNum: '%s'", sentSockNum)
		return
	end
	uri = readData(fdBuffer, dataLen)
	return uri, sentSockNum
end

local sock
local function readMessage(cSockFd)
	uri, sentSockNum = readPipe()
	sock = sockFdSent[sentSockNum]
	-- util.print("received socket %d, dataLen %d, uri '%s'", sentSockNum, dataLen, uri)
	if sentSockNum == nil then
		util.printError("readPipe failed, sent socket number is nil")
		return
	end
	if uri == "close-socket" then
		if sock then
			util.print("closing local socket %d, remote socket %d", sock.socket, sentSockNum)
			sock:close()
			sockFdSent[sentSockNum] = nil
		else
			uri = nil
		end
	elseif sock == nil and sentSockNum then
		sock, err = json.fromJson(uri)
		if err then
			util.printWarning("fromJson failed, uri: '%s'", uri)
			return
		end
		printFd = printFd + 1
		sock.socket = receiveFd(cSockFd, printFd) -- returned socket number is not usually same as sent socket number
		sock = socket.new(sock) -- we need to set metatable to sock() and keep info about open sockets
		-- util.printOk("created new local socket %d, remote socket %d", sock.socket, sentSockNum)
		sockFdSent[sentSockNum] = sock
		uri = sock.uri
	end
	return sock, uri
end

local function server() -- (arg)
	rest = require "rest"
	socket = require "system/socket"
	--[[ if true or util.isWin() then
		return serverTcp(arg)
	end ]]
	local sockFd = fdSend.setupFdServer()
	if sockFd == nil then
		util.print("failed to set up fd-server")
		return "quit"
	end
	local pid = fdSend.processId() -- server() is called once, it's ok to have locals
	local cSockFd = fdSend.acceptFd(sockFd)
	if cSockFd == nil then
		return "quit"
	end
	if fdRead == nil then
		ret = initPipe(cSockFd)
	end
	if ret then
		return ret
	end
	local data, answerData
	repeat
		sock, uri = readMessage(cSockFd)
		if uri == nil then
			util.print("closing fd-server")
			break
			-- elseif uri == "close-socket" then
			-- skip this call
		elseif uri ~= "close-socket" then
			callCount = callCount + 1
			if uri == "/rest/nc/echo2" then
				data = {call = callCount} -- echo = uri,
			else
				data = {hello = "world", pid = pid, call = callCount, bytesSentTotal = bytesSentTotal}
			end
			answerData = rest.createAnswerData(sock, data)
			ret = sock:send(answerData)
			if ret < 0 then
				util.printError("socketSend() failed with error: " .. ret)
				break
			end
			if log then
				if skip(callCount, 50000) == 0 then
					util.print("nc-calc-server call count: %d", callCount)
				end
			end
			bytesSentTotal = bytesSentTotal + ret
		end
	until false
	-- cleanup:
	closePipe()
	-- callCount = 0
	ret = fdSend.close(cSockFd) -- close accepted socket after sending data
	ret = fdSend.close(sockFd) --  close listening socket
	return uri
end

return {server = server, useSocket = useSocket, fdBufferSize = fdBufferSize}
