--- lib/net/http3-lsquic-prog.lua
-- https://lsquic.readthedocs.io/en/latest/tutorial.html
-- /Users/pasi/installed/C/tls/lsquic/bin/prog.c
package.path = "lib/?.lua;lib/?.lx;" .. package.path
package.path = "../lib/?.lua;../lib/?.lx;" .. package.path
package.path = "../../lib/?.lua;../../lib/?.lx;" .. package.path

local program = {}

local ffi = require "mffi"
local C = ffi.C
local util = require "util"
local peg = require "peg"
local bit = require "bit"
local band = bit.band
local bor = bit.bor
-- local l = require "lang".l
-- local net = require "system/net"
local socket = require "system/socket"
local lsquic = require "net/lsquic"
lsquic = lsquic.lib

local progId = 0
local programs = {}
local prog_stopped
local connectTimeout, blocking = 3, 0

local function progToSports(prog)
	return programs[prog.id[0]].sports
end

local function idToProgramRec(id)
	return programs[id[0]]
end

function program.event_add(prog)
end

function program.proc_new()
	-- ffi.newNoAnchor("struct prog")
	progId = progId + 1
	local prog = {
		id = ffi.newNoAnchor("uint32_t[1]", progId),
		prog_settings = ffi.newNoAnchor("struct lsquic_engine_settings"),
		prog_api = ffi.newNoAnchor("struct lsquic_engine_api")
		-- prog_ssl_ctx = ffi.newNoAnchor("struct ssl_ctx_st*"),
		-- prog_certs = ffi.newNoAnchor("struct lsquic_hash*"),
		-- prog_engine = ffi.newNoAnchor("struct lsquic_engine*"),
	}
	programs[progId] = {prog = prog}
	return prog
end

-- /Users/pasi/installed/C/tls/lsquic/bin/test_cert.c
local function delete_certs(certs)
	-- struct lsquic_hash_elem *el;
	-- struct server_cert *cert;
	local el = lsquic.lsquic_hash_first(certs)
	while ffi.isNotNull(el) do
		local cert = lsquic.lsquic_hashelem_getdata(el)
		C.SSL_CTX_free(cert.ce_ssl_ctx)
		C.free(cert.ce_sni)
		C.free(cert)
		el = lsquic.lsquic_hash_next(certs)
	end
	lsquic.lsquic_hash_destroy(certs)
end

local function prog_is_stopped()
	return prog_stopped
end

local function logger(logger_ctx, buf, len)
	local id = ffi.cast("uint32_t *", logger_ctx) -- proc.id
	util.print("* logger: %d. '%s'", id[0], ffi.string(buf, len))
	return len -- must return int
end

local function send_packets_one_by_one(specs, count)
	-- /Users/pasi/installed/C/tls/lsquic/bin/test_common.c
	-- todo: copy this
	return count -- must return int
end

local function sport_packets_out(ctx, specs, count)
	--[[ #if HAVE_SENDMMSG
			const struct prog *prog = ctx;
			if (prog->prog_use_sendmmsg)
					return send_packets_using_sendmmsg(specs, count);
			else
	#endif ]]
	return send_packets_one_by_one(specs, count)
end

-- /Users/pasi/installed/C/tls/lsquic/bin/prog.c

-- typedef void SSL_CTX; static SSL_CTX * get_ssl_ctx (void *); == static void* get_ssl_ctx (void *)
local function get_ssl_ctx(peer_ctx)
	-- const struct service_port *const sport = peer_ctx
	local sport = ffi.cast("const struct service_port *", peer_ctx)
	return sport.sp_prog.prog_ssl_ctx
end

local function sport_new(arg, prog)
	-- see: struct service_port
	local host, port = peg.split(arg, ":")
	port = port or 443
	local sas = socket.addressInfo(host, port, "http3")
	if sas == nil then
		return
	end
	local sport = {
		-- fd = nil, -- SOCKET
		addr = arg,
		host = host,
		port = port,
		sas = sas,
		sp_local_addr = ffi.newNoAnchor("struct sockaddr_storage"),
		sp_prog = prog
	}
	return sport
end

local function prog_add_sport(prog, arg)
	local sport = sport_new(arg, prog)
	if sport == nil then
		return -1
	end
	--[[ Default settings: --]]
	sport.sp_flags = 0 -- prog.prog_dummy_sport.sp_flags
	sport.sp_sndbuf = nil -- prog.prog_dummy_sport.sp_sndbuf
	sport.sp_rcvbuf = nil -- prog.prog_dummy_sport.sp_rcvbuf
	local sports = progToSports(prog)
	sports[#sports + 1] = sport
	return 0
end

function program.prog_eb(prog)
	return prog.prog_eb
end

local function event_add(prog_timer, timeout)
	return -- todo
end

local function prog_process_conns(prog)
	local diff = ffi.newNoAnchor("int[1]")
	local timeout = ffi.newNoAnchor("struct timeval")
	lsquic.lsquic_engine_process_conns(prog.prog_engine)
	local ret = lsquic.lsquic_engine_earliest_adv_tick(prog.prog_engine, diff)
	if ret ~= 0 then
		if diff[0] < 0 or diff[0] < prog.prog_settings.es_clock_granularity then
			timeout.tv_sec = 0
			timeout.tv_usec = prog.prog_settings.es_clock_granularity
		else
			timeout.tv_sec = diff[0] / 1000000
			timeout.tv_usec = diff[0] % 1000000
		end
		if not prog_is_stopped() then
			event_add(prog.prog_timer, timeout)
		end
	end
end

function program.prog_connect(prog, zero_rtt, zero_rtt_len)
	-- struct service_port *sport
	local sport = progToSports(prog)[1]
	local ret = lsquic.lsquic_engine_connect(prog.prog_engine, C.N_LSQVER, ffi.cast("struct sockaddr *", sport.sp_local_addr), ffi.cast("struct sockaddr *", sport.sas), prog.id, nil, prog.prog_hostname or sport.host, --[[prog.prog_hostname ? prog.prog_hostname
					--[=[ SNI is required for HTTP --]=]
					: prog.prog_engine_flags & LSENG_HTTP ? sport.host
					: nil,--]] prog.prog_max_packet_size, zero_rtt, zero_rtt_len, sport.sp_token_buf, sport.sp_token_sz)
	if ffi.isNull(ret) then
		return -1
	end
	prog_process_conns(prog)
	return 0
end

local function sport_init_client(sport, engine, eb)
	-- see: lib/net/http3-lsquic-test_common.lua
	local sock = socket.connect(sport.host, sport.port, "http3", connectTimeout, blocking)
	if sock == nil then
		return -1
	end
	sport.engine = engine
	sport.sock = sock
	sport.sp_token_buf = nil
	sport.sp_token_sz = 0
	-- sport.sp_flags |= SPORT_SERVER
	-- return add_to_event_loop(sport, eb)
	return 0
end

local function prog_init_client(prog)
	-- struct service_port *sport
	local sport = progToSports(prog)[1]
	return sport_init_client(sport, prog.prog_engine, prog.prog_eb)
end

local function prog_init_server(prog)
	-- struct service_port *sport
	-- unsigned char ticket_keys[48]
	prog.prog_ssl_ctx = SSL_CTX_new(TLS_method())
	if prog.prog_ssl_ctx then
		SSL_CTX_set_min_proto_version(prog.prog_ssl_ctx, TLS1_3_VERSION)
		SSL_CTX_set_max_proto_version(prog.prog_ssl_ctx, TLS1_3_VERSION)
		SSL_CTX_set_default_verify_paths(prog.prog_ssl_ctx)
		--[[ This is obviously test code: the key is just an array of NUL bytes --]]
		memset(ticket_keys, 0, ffi.sizeof(ticket_keys))
		if (1 ~= SSL_CTX_set_tlsext_ticket_keys(prog.prog_ssl_ctx, ticket_keys, ffi.sizeof(ticket_keys))) then
			util.printError("SSL_CTX_set_tlsext_ticket_keys failed")
			return -1
		end
	else
		util.printWarning("cannot create SSL context")
	end
	TAILQ_FOREACH(sport, prog.sports, next_sport)
	if sport == nil_init_server(sport, prog.prog_engine, prog.prog_eb) then
		return -1
	end
	return 0
end

local function rog_timer_handler(fd, what, prog)
	if not prog_is_stopped() then
		prog_process_conns(prog)
	end
end

local function prog_usr1_handler(fd, what, prog)
	util.printWarn("Got SIGUSR1, stopping engine")
	program.prog_stop(prog) -- prog.id[0]
end

local function prog_usr2_handler(fd, what, prog)
	util.printWarn("Got SIGUSR2, cool down engine")
	prog.prog_flags = bor(prog.prog_flags, C.PROG_FLAG_COOLDOWN)
	lsquic.lsquic_engine_cooldown(prog.prog_engine)
	prog_process_conns(prog)
end

function program.prog_run(prog)
	-- #ifndef WIN32
	--   prog.prog_usr1 = evsignal_new(prog.prog_eb, SIGUSR1,
	--                           prog_usr1_handler, prog)
	--   evsignal_add(prog.prog_usr1, nil)
	--   prog.prog_usr2 = evsignal_new(prog.prog_eb, SIGUSR2,
	--                           prog_usr2_handler, prog)
	--   evsignal_add(prog.prog_usr2, nil)
	-- #endif
	-- event_base_loop(prog.prog_eb, 0)
	return 0
end

function program.prog_cleanup(prog)
	lsquic.lsquic_engine_destroy(prog.prog_engine)
	-- event_base_free(prog.prog_eb)
	if prog.prog_use_stock_pmi then
		pba_cleanup(prog.prog_pba)
	end
	if prog.prog_ssl_ctx then
		SSL_CTX_free(prog.prog_ssl_ctx)
	end
	if prog.prog_certs then
		delete_certs(prog.prog_certs)
	end
	lsquic.lsquic_global_cleanup()
end

local function sport_destroy(sport)
	if sport.ev then
		event_del(sport.ev)
		event_free(sport.ev)
	end
	if sport.sock then
		sport.sock:close()
	end
	if sport.packs_in then
		free_packets_in(sport.packs_in)
	end
	if sport.sp_local_addr then
		sport.sp_local_addr = nil
	end
	if sport.sp_token_buf then
		C.free(sport.sp_token_buf)
	end
	-- free(sport)
end

function program.prog_stop(progId)
	-- struct service_port *sport
	prog_stopped = true
	local rec = idToProgramRec(progId)
	for idx = #rec.sports, 1 do
		local sport = rec.sports[idx]
		sport_destroy(sport)
	end
	local prog = rec.prog
	if prog.prog_timer then
		event_del(prog.prog_timer)
		event_free(prog.prog_timer)
		prog.prog_timer = nil
	end
	if prog.prog_usr1 then
		event_del(prog.prog_usr1)
		event_free(prog.prog_usr1)
		prog.prog_usr1 = nil
	end
	if prog.prog_usr2 then
		event_del(prog.prog_usr2)
		event_free(prog.prog_usr2)
		prog.prog_usr2 = nil
	end
end

local function keylog_open(ctx, conn)
	--[[
	const struct prog *const prog = ctx
	const lsquic_cid_t *cid
	FILE *fh
	int sz
	char id_str[MAX_CID_LEN * 2 + 1]
	char path[PATH_MAX] ]]
	local prog = programs[ctx.id]
	local id_str = ffi.newNoAnchor("char[?]", C.MAX_CID_LEN * 2 + 1)
	local path = ffi.newNoAnchor("char[?]", C.PATH_MAX)
	local cid = lsquic.lsquic_conn_id(conn)
	lsquic.lsquic_hexstr(cid.idbuf, cid.len, id_str, ffi.sizeof(id_str))
	local sz = C.snprintf(path, ffi.sizeof(path), "%s/%s.keys", prog.prog_keylog_dir, id_str)
	if sz >= ffi.sizeof(path) then
		util.printWarning("%s: file too long", "keylog_open")
		return nil
	end
	local fh = fopen(path, "w")
	if 0 ~= fh then
		util.printWarning("could not open %s for writing: %s", path, strerror(errno))
	end
	return fh
end

local function keylog_log_line(handle, line)
	local len = C.strlen(line)
	if len < ffi.sizeof("QUIC_") - 1 or C.strncmp(line, "QUIC_", 5) then
		C.fputs("QUIC_", handle)
	end
	C.fputs(line, handle)
	C.fputs("\n", handle)
	C.fflush(handle)
end

local function keylog_close(handle)
	C.fclose(handle)
end

local keylog_if = ffi.newAnchor("const struct lsquic_keylog_if", {
	kli_open = ffi.cast("void * (*)(void *keylog_ctx, lsquic_conn_t *)", keylog_open),
	kli_log_line = ffi.cast("void (*)(void *handle, const char *line)", keylog_log_line),
	kli_close = ffi.cast("void (*)(void *handle)", keylog_close)
})

local function no_cert(cert_lu_ctx, sa_UNUSED, sni)
	return nil
end

local function prog_timer_handler(fd, what, arg)
	local prog = arg
	if not prog_is_stopped(prog) then
		prog_process_conns(prog)
	end
end

function program.prog_prep(prog)
	-- int s
	-- char err_buf[100]
	local err_buf = ffi.newNoAnchor("char[?]", 100)
	if prog.prog_keylog_dir then
		prog.prog_api.ea_keylog_if = keylog_if
		prog.prog_api.ea_keylog_ctx = prog
	end
	if 0 ~= lsquic.lsquic_engine_check_settings(prog.prog_api.ea_settings, prog.prog_engine_flags, err_buf, ffi.sizeof(err_buf)) then
		util.printError("Error in settings: %s", err_buf)
		return -1
	end
	if prog.prog_use_stock_pmi then
		pba_init(prog.prog_pba, prog.prog_packout_max)
	else
		prog.prog_api.ea_pmi = nil
		prog.prog_api.ea_pmi_ctx = nil
	end
	local sports = progToSports(prog)
	if #sports == 0 then
		local s
		if prog.prog_hostname then
			s = prog_add_sport(prog, prog.prog_hostname)
		else
			s = prog_add_sport(prog, "0.0.0.0:4433")
		end
		if s == -1 then
			return -1
		end
	end

	if prog.prog_certs then
		prog.prog_api.ea_lookup_cert = lookup_cert
		prog.prog_api.ea_cert_lu_ctx = prog.prog_certs
	elseif band(prog.prog_engine_flags, C.LSENG_SERVER) ~= 0 then
		util.printWarning("Not a single service specified.  Use -c option.")
		prog.prog_api.ea_lookup_cert = no_cert
	end

	-- prog.prog_eb = -- event_base_new()
	prog.prog_engine = lsquic.lsquic_engine_new(prog.prog_engine_flags, prog.prog_api)
	if ffi.isNull(prog.prog_engine) then
		return -1
	end
	-- prog.prog_timer = event_new(prog.prog_eb, -1, 0, prog_timer_handler, prog)
	local s
	if band(prog.prog_engine_flags, C.LSENG_SERVER) ~= 0 then
		return prog_init_server(prog)
	end
	return prog_init_client(prog)
end

local function send_unsent(fd, what, arg)
	-- struct prog *const prog = arg
	-- assert(prog.prog_send)
	event_del(prog.prog_send)
	event_free(prog.prog_send)
	prog.prog_send = nil
	util.printInfo("on_write event fires")
	lsquic.lsquic_engine_send_unsent_packets(prog.prog_engine)
end

local function prog_sport_cant_send(prog, fd)
	-- assert(0 ~= prog.prog_send)
	util.printInfo("cannot send: register on_write event")
	prog.prog_send = event_new(prog.prog_eb, fd, EV_WRITE, send_unsent, prog)
	-- event_add(prog.prog_send, nil)
end

local pmi = nil --[[ ffi.newAnchor("static const struct lsquic_packout_mem_if", {
	pmi_allocate = pba_allocate,
	pmi_release  = pba_release,
	pmi_return   = pba_release,
}) ]]

function program.prog_set_opt(prog, opt, arg)
	prog.prog_packout_max = 0
	prog.prog_max_packet_size = 0
	prog.prog_set_opt = 0
	-- prog.prog_use_stock_pmi = nil
	--[[ prog.prog_hostname = arg
	if not prog.prog_certs then
     prog.prog_certs = lsquic_hash_create();
     load_cert(prog.prog_certs, arg);
	end
	]]
	return 0
end

function program.prog_init(prog, flags, sports, stream_if, stream_if_ctx)
	--[[ prog-specific initialization: --]]
	prog.prog_engine_flags = flags
	-- prog.sports     = sports
	programs[prog.id[0]].sports = sports
	lsquic.lsquic_engine_init_settings(prog.prog_settings, flags)
	--[[ #if ECN_SUPPORTED
	prog.prog_settings.es_ecn    = LSQUIC_DF_ECN
#else ]]
	prog.prog_settings.es_ecn = 0
	-- #endif
	prog.prog_api.ea_settings = prog.prog_settings
	prog.prog_api.ea_stream_if = stream_if
	prog.prog_api.ea_stream_if_ctx = stream_if_ctx
	prog.prog_api.ea_packets_out = ffi.cast("int (*)(void *ctx, const struct lsquic_out_spec *specs, unsigned count)", sport_packets_out)
	prog.prog_api.ea_packets_out_ctx = prog.id
	-- prog.prog_api.ea_pmi       = &pmi
	-- prog.prog_api.ea_pmi_ctx     = prog.prog_pba
	prog.prog_api.ea_get_ssl_ctx = ffi.cast("void* (*)(void *peer_ctx)", get_ssl_ctx)
	--[[ #if LSQUIC_PREFERRED_ADDR
	if getenv("LSQUIC_PREFERRED_ADDR4") || getenv("LSQUIC_PREFERRED_ADDR6")
		prog.prog_flags |= PROG_SEARCH_ADDRS
#endif ]]
	--[[ Non prog-specific initialization: --]]
	lsquic.lsquic_global_init(band(flags, C.LSENG_SERVER) ~= 0 and C.LSQUIC_GLOBAL_SERVER or C.LSQUIC_GLOBAL_CLIENT)
	-- lsquic.lsquic_logger_init(ffi.cast("int (*)(void *logger_ctx, const char *buf, size_t len)", logger), prog.id, C.LLTS_HHMMSSMS)
	local logger_if = ffi.newNoAnchor("struct lsquic_logger_if", {ffi.cast("int (*)(void *logger_ctx, const char *buf, size_t len)", logger)})
	lsquic.lsquic_logger_init(logger_if, prog.id, C.LLTS_HHMMSSMS)
	lsquic.lsquic_logger_lopt("=notice")
end

return program

-- /Users/pasi/installed/C/tls/lsquic/bin/prog.h
--[[
struct service_port {
    TAILQ_ENTRY(service_port)  next_sport;
#ifndef WIN32
    int                        fd;
#else
    SOCKET                        fd;
#endif
#if __linux__
    uint32_t                   n_dropped;
    int                        drop_init;
    char                       if_name[IFNAMSIZ];
#endif
    struct event              *ev;
    struct lsquic_engine      *engine;
    void                      *conn_ctx;
    char                       host[80];
    struct sockaddr_storage    sas;
    struct sockaddr_storage    sp_local_addr;
    struct packets_in         *packs_in;
    enum sport_flags           sp_flags;
    SOCKOPT_VAL                sp_sndbuf;   /* If SPORT_SET_SNDBUF is set */
    SOCKOPT_VAL                sp_rcvbuf;   /* If SPORT_SET_RCVBUF is set */
    struct prog               *sp_prog;
    unsigned char             *sp_token_buf;
    size_t                     sp_token_sz;
};
]]

--[[
struct prog {
    // struct packout_buf_allocator    prog_pba;
    struct lsquic_engine_settings   prog_settings;
    struct lsquic_engine_api        prog_api;
    unsigned                        prog_engine_flags;
    // struct service_port             prog_dummy_sport;   /* Use for options */
    unsigned                        prog_packout_max;
    unsigned short                  prog_max_packet_size;
    int                             prog_version_cleared;
    unsigned long                   prog_read_count;
		/*
#if HAVE_SENDMMSG
    int                             prog_use_sendmmsg;
#endif
#if HAVE_RECVMMSG
    int                             prog_use_recvmmsg;
#endif
		*/
    int                             prog_use_stock_pmi;
    struct event_base              *prog_eb;
    struct event                   *prog_timer,
                                   *prog_send,
                                   *prog_usr1;
    struct event                   *prog_usr2;
    struct ssl_ctx_st              *prog_ssl_ctx;
    struct lsquic_hash             *prog_certs;
    struct event                   *prog_event_sni;
    char                           *prog_susp_sni;
    struct sport_head              *prog.sports;
    struct lsquic_engine           *prog_engine;
    const char                     *prog_hostname;
    int                             prog_ipver;     /* 0, 4, or 6 */
    const char                     *prog_keylog_dir;
    enum {
        PROG_FLAG_COOLDOWN  = 1 << 0,
		/*
#if LSQUIC_PREFERRED_ADDR
        PROG_SEARCH_ADDRS   = 1 << 1,
#endif
		*/
    }                               prog_flags;
};
]]
