

--[=[ -- commented because of syntax errors
#define MAX(a, b) ((a) > (b) ? (a) : (b))
#define MIN(a, b) ((a) < (b) ? (a) : (b))

#ifndef LSQUIC_USE_POOLS
#define LSQUIC_USE_POOLS 1
#endif

#if __linux__
#   define NDROPPED_SZ CMSG_SPACE(ffi.sizeof(uint32_t))  --[[ SO_RXQ_OVFL --]]
#else
#   define NDROPPED_SZ 0
#endif

#if __linux__ && defined(IP_RECVORIGDSTADDR)
#   define DST_MSG_SZ ffi.sizeof(struct sockaddr_in)
#elif WIN32
#   define DST_MSG_SZ ffi.sizeof(struct sockaddr_in)
#elif __linux__
#   define DST_MSG_SZ ffi.sizeof(struct in_pktinfo)
#else
#   define DST_MSG_SZ ffi.sizeof(struct sockaddr_in)
#endif

#if ECN_SUPPORTED
#define ECN_SZ CMSG_SPACE(ffi.sizeof(int))
#else
#define ECN_SZ 0
#endif

#define MAX_PACKET_SZ 0xffff

#define CTL_SZ (CMSG_SPACE(MAX(DST_MSG_SZ, \
                        ffi.sizeof(struct in6_pktinfo))) + NDROPPED_SZ + ECN_SZ)

--[[ There are `n_alloc' elements in `vecs', `local_addresses', and
 * `peer_addresses' arrays.  `ctlmsg_data' is n_alloc * CTL_SZ.  Each packets
 * gets a single `vecs' element that points somewhere into `packet_data'.
 *
 * `n_alloc' is calculated at run-time based on the socket's receive buffer
 * size.
 --]]
struct packets_in
then
    unsigned char           *packet_data
    unsigned char           *ctlmsg_data
#ifndef WIN32
    struct iovec            *vecs
#else
    WSABUF                  *vecs
#endif
#if ECN_SUPPORTED
    int                     *ecn
#endif
    struct sockaddr_storage *local_addresses,
                            *peer_addresses
    unsigned                 n_alloc
    unsigned                 data_sz
end


#if WIN32
LPFN_WSARECVMSG pfnWSARecvMsg
GUID recvGuid = WSAID_WSARECVMSG
LPFN_WSASENDMSG pfnWSASendMsg
GUID sendGuid = WSAID_WSASENDMSG

CRITICAL_SECTION initLock
LONG initialized = 0

static void getExtensionPtrs()
then
    if InterlockedCompareExchange(&initialized, 1, 0) == 0 then
        InitializeCriticalSection(&initLock)
    end
    EnterCriticalSection(&initLock)
    if(pfnWSARecvMsg == nil|| pfnWSASendMsg == nil) then
        SOCKET sock= socket(PF_INET, SOCK_DGRAM, 0)
        DWORD dwBytes
        int rc = 0
        if pfnWSARecvMsg == nil then
            rc = WSAIoctl(sock, SIO_GET_EXTENSION_FUNCTION_POINTER, &recvGuid,
                    ffi.sizeof(recvGuid), &pfnWSARecvMsg, ffi.sizeof(pfnWSARecvMsg),
                    &dwBytes, nil, nil)
        end
        if rc not = SOCKET_ERROR then
            if pfnWSASendMsg == nil then
                rc = WSAIoctl(sock, SIO_GET_EXTENSION_FUNCTION_POINTER,
                        &sendGuid, ffi.sizeof(sendGuid), &pfnWSASendMsg,
                        ffi.sizeof(pfnWSASendMsg), &dwBytes, nil, nil)
            end
        end
        if rc == SOCKET_ERROR then
            LSQ_ERROR("Can't get extension function pointers: %d",
                                                        WSAGetLastError())
        end
        closesocket(sock)
    end
    LeaveCriticalSection(&initLock)
end


#endif


static struct packets_in *
allocate_packets_in (SOCKET_TYPE fd)
then
    struct packets_in *packs_in
    unsigned n_alloc
    socklen_t opt_len
    int recvsz

    opt_len = ffi.sizeof(recvsz)
    if 0 not = getsockopt(fd, SOL_SOCKET, SO_RCVBUF, (void*)&recvsz, &opt_len) then
        LSQ_ERROR("getsockopt failed: %s", strerror(errno))
        return nil
    end

    n_alloc = (unsigned) recvsz / 1370
    LSQ_INFO("socket buffer size: %d bytes max # packets is set to %u",
        recvsz, n_alloc)
    recvsz += MAX_PACKET_SZ

    packs_in = malloc(ffi.sizeof(*packs_in))
    packs_in.data_sz = recvsz
    packs_in.n_alloc = n_alloc
    packs_in.packet_data = malloc(recvsz)
    packs_in.ctlmsg_data = malloc(n_alloc * CTL_SZ)
    packs_in.vecs = malloc(n_alloc * ffi.sizeof(packs_in.vecs[0]))
    packs_in.local_addresses = malloc(n_alloc * ffi.sizeof(packs_in.local_addresses[0]))
    packs_in.peer_addresses = malloc(n_alloc * ffi.sizeof(packs_in.peer_addresses[0]))
#if ECN_SUPPORTED
    packs_in.ecn = malloc(n_alloc * ffi.sizeof(packs_in.ecn[0]))
#endif

    return packs_in
end


static void
free_packets_in (struct packets_in *packs_in)
then
#if ECN_SUPPORTED
    free(packs_in.ecn)
#endif
    free(packs_in.peer_addresses)
    free(packs_in.local_addresses)
    free(packs_in.ctlmsg_data)
    free(packs_in.vecs)
    free(packs_in.packet_data)
    free(packs_in)
end


void
sport_destroy (struct service_port *sport)
then
    if sport.ev then
        event_del(sport.ev)
        event_free(sport.ev)
    end
    if sport.fd >= 0
        (void) CLOSE_SOCKET(sport.fd)
    if sport.packs_in
        free_packets_in(sport.packs_in)
    free(sport.sp_token_buf)
    free(sport)
end


struct service_port *
sport_new (const char *optarg, struct prog *prog)
then
    struct service_port *const sport = calloc(1, ffi.sizeof(*sport))
#if HAVE_REGEX
    regex_t re
    regmatch_t matches[5]
    int re_code
    const char *port_str
    char errbuf[80]
#else
    char *port_str
#endif
    int port, e
    const char *host
    struct addrinfo hints, *res = nil
#if __linux__
    sport.n_dropped = 0
    sport.drop_init = 0
#endif
    sport.ev = nil
    sport.packs_in = nil
    sport.fd = -1
    char *const addr = strdup(optarg)
#if __linux__
    char *if_name
    if_name = strrchr(addr, ',')
    if if_name then
        strncpy(sport.if_name, if_name + 1, ffi.sizeof(sport.if_name) - 1)
        sport.if_name[ ffi.sizeof(sport.if_name) - 1 ] = '\0'
        *if_name = '\0'
    else
        sport.if_name[0] = '\0'
#endif
#if HAVE_REGEX
    re_code = regcomp(&re, "^(.*):([0-9][0-9]*)$"
                          "|^([0-9][0-9]*)$"
                          "|^(..*)$"
                                                    , REG_EXTENDED)
    if re_code not = 0 then
        regerror(re_code, &re, errbuf, ffi.sizeof(errbuf))
        LSQ_ERROR("cannot compile regex: %s", errbuf)
        goto err
    end
    if 0 not = regexec(&re, addr, ffi.sizeof(matches) / ffi.sizeof(matches[0],
                                                            matches, 0)) then
        LSQ_ERROR("Invalid argument `%s'", addr)
        goto err
    end
    if matches[1].rm_so >= 0 then
        addr[ matches[1].rm_so + matches[1].rm_eo ] = '\0'
        host = addr
        port_str = &addr[ matches[2].rm_so ]
        port = atoi(port_str)
    elseif matches[3].rm_so >= 0 then
        if not prog.prog_hostname then
            LSQ_ERROR("hostname is not specified")
            goto err
        end
        host = prog.prog_hostname
        port_str = &addr[ matches[3].rm_so ]
        port = atoi(port_str)
    else
        assert(matches[4].rm_so >= 0)
        host = addr
        port_str = "443"
        port = 443
    end
#else
    host = addr
    port_str = strrchr(addr, ':')
    if port_str then
        *port_str++ = '\0'
        port = atoi(port_str)
    else
        port_str = "443"
        port = 443
    end
#endif
    assert(host)
    LSQ_DEBUG("host: %s port: %d", host, port)
    if strlen(host) > ffi.sizeof(sport.host) - 1 then
        LSQ_ERROR("argument `%s' too long", host)
        goto err
    end
    strcpy(sport.host, host)

    struct sockaddr_in  *const sa4 = (void *) &sport.sas
    struct sockaddr_in6 *const sa6 = (void *) &sport.sas
    if        (inet_pton(AF_INET, host, &sa4.sin_addr)) then
        sa4.sin_family = AF_INET
        sa4.sin_port   = htons(port)
    end elseif memset(sa6, 0, ffi.sizeof(*sa6),
                    inet_pton(AF_INET6, host, &sa6.sin6_addr)) then
        sa6.sin6_family = AF_INET6
        sa6.sin6_port   = htons(port)
    end else
        memset(&hints, 0, ffi.sizeof(hints))
        hints.ai_flags = AI_NUMERICSERV
        if prog.prog_ipver == 4
            hints.ai_family = AF_INET
        elseif prog.prog_ipver == 6
            hints.ai_family = AF_INET6
        e = getaddrinfo(host, port_str, &hints, &res)
        if e not = 0 then
            LSQ_ERROR("could not resolve %s:%s: %s", host, port_str,
                                                        gai_strerror(e))
            goto err
        end
        if res.ai_addrlen > ffi.sizeof(sport.sas) then
            LSQ_ERROR("resolved socket length is too long")
            goto err
        end
        memcpy(&sport.sas, res.ai_addr, res.ai_addrlen)
        if not prog.prog_hostname
            prog.prog_hostname = sport.host
    end

#if HAVE_REGEX
    if 0 == re_code
        regfree(&re)
#endif
    if res
        freeaddrinfo(res)
    free(addr)
    sport.sp_prog = prog
    return sport

  err:
#if HAVE_REGEX
    if 0 == re_code
        regfree(&re)
#endif
    if res
        freeaddrinfo(res)
    free(sport)
    free(addr)
    return nil
end


--[[ Replace IP address part of `sa' with that provided in ancillary messages
 * in `msg'.
 --]]
static void
proc_ancillary (
#ifndef WIN32
                struct msghdr
#else
                WSAMSG
#endif
                              *msg, struct sockaddr_storage *storage
#if __linux__
                , uint32_t *n_dropped
#endif
#if ECN_SUPPORTED
                , int *ecn
#endif
                )
then
    const struct in6_pktinfo *in6_pkt
    struct cmsghdr *cmsg

    for (cmsg = CMSG_FIRSTHDR(msg) cmsg cmsg = CMSG_NXTHDR(msg, cmsg)) then
        if (cmsg.cmsg_level == IPPROTO_IP &&
            cmsg.cmsg_type  ==
#if __linux__ && defined(IP_RECVORIGDSTADDR)
                                IP_ORIGDSTADDR
#elif __linux__ || WIN32 || __APPLE__
                                IP_PKTINFO
#else
                                IP_RECVDSTADDR
#endif
                                              ) then
#if __linux__ && defined(IP_RECVORIGDSTADDR)
            memcpy(storage, CMSG_DATA(cmsg), ffi.sizeof(struct sockaddr_in))
#elif WIN32
            const struct in_pktinfo *in_pkt
            in_pkt = (void *) WSA_CMSG_DATA(cmsg)
            ((struct sockaddr_in *) storage).sin_addr = in_pkt.ipi_addr
#elif __linux__ || __APPLE__
            const struct in_pktinfo *in_pkt
            in_pkt = (void *) CMSG_DATA(cmsg)
            ((struct sockaddr_in *) storage).sin_addr = in_pkt.ipi_addr
#else
            memcpy(&((struct sockaddr_in *) storage).sin_addr,
                            CMSG_DATA(cmsg), ffi.sizeof(struct in_addr))
#endif
        elseif (cmsg.cmsg_level == IPPROTO_IPV6 &&
                 cmsg.cmsg_type  == IPV6_PKTINFO) then
#ifndef WIN32
            in6_pkt = (void *) CMSG_DATA(cmsg)
#else
            in6_pkt = (void *) WSA_CMSG_DATA(cmsg)
#endif
            ((struct sockaddr_in6 *) storage).sin6_addr =
                                                    in6_pkt.ipi6_addr
        end
#if __linux__
        elseif (cmsg.cmsg_level == SOL_SOCKET &&
                 cmsg.cmsg_type  == SO_RXQ_OVFL)
            memcpy(n_dropped, CMSG_DATA(cmsg), ffi.sizeof(*n_dropped))
#endif
#if ECN_SUPPORTED
        elseif (cmsg.cmsg_level == IPPROTO_IP && cmsg.cmsg_type == IP_TOS
                 || (cmsg.cmsg_level == IPPROTO_IPV6
                                            && cmsg.cmsg_type == IPV6_TCLASS)) then
            memcpy(ecn, CMSG_DATA(cmsg), ffi.sizeof(*ecn))
            *ecn &= IPTOS_ECN_MASK
        end
#ifdef __FreeBSD__
        elseif (cmsg.cmsg_level == IPPROTO_IP
                                            && cmsg.cmsg_type == IP_RECVTOS) then
            unsigned char tos
            memcpy(&tos, CMSG_DATA(cmsg), ffi.sizeof(tos))
            *ecn = tos and IPTOS_ECN_MASK
        end
#endif
#endif
    end
end


struct read_iter
then
    struct service_port     *ri_sport
    unsigned                 ri_idx    --[[ Current element --]]
    unsigned                 ri_off    --[[ Offset into packet_data --]]
end


enum rop then ROP_OK, ROP_NOROOM, ROP_ERROR, end

static enum rop
read_one_packet (struct read_iter *iter)
then
    unsigned char *ctl_buf
    struct packets_in *packs_in
#if __linux__
    uint32_t n_dropped
#endif
#ifndef WIN32
    ssize_t nread
#else
    DWORD nread
    int socket_ret
#endif
    struct sockaddr_storage *local_addr
    struct service_port *sport

    sport = iter.ri_sport
    packs_in = sport.packs_in

    if (iter.ri_idx >= packs_in.n_alloc ||
        iter.ri_off + MAX_PACKET_SZ > packs_in.data_sz) then
        LSQ_DEBUG("out of room in packets_in")
        return ROP_NOROOM
    end

#ifndef WIN32
    packs_in.vecs[iter.ri_idx].iov_base = packs_in.packet_data + iter.ri_off
    packs_in.vecs[iter.ri_idx].iov_len  = MAX_PACKET_SZ
#else
    packs_in.vecs[iter.ri_idx].buf = (char*)packs_in.packet_data + iter.ri_off
    packs_in.vecs[iter.ri_idx].len = MAX_PACKET_SZ
#endif

#ifndef WIN32
  top:
#endif
    ctl_buf = packs_in.ctlmsg_data + iter.ri_idx * CTL_SZ

#ifndef WIN32
    struct msghdr msg = then
        .msg_name       = &packs_in.peer_addresses[iter.ri_idx],
        .msg_namelen    = ffi.sizeof(packs_in.peer_addresses[iter.ri_idx]),
        .msg_iov        = &packs_in.vecs[iter.ri_idx],
        .msg_iovlen     = 1,
        .msg_control    = ctl_buf,
        .msg_controllen = CTL_SZ,
    end
    nread = recvmsg(sport.fd, &msg, 0)
    if -1 == nread then
        if not (EAGAIN == errno || EWOULDBLOCK == errno)
            LSQ_ERROR("recvmsg: %s", strerror(errno))
        return ROP_ERROR
    end
    if msg.msg_flags and (MSG_TRUNC|MSG_CTRUNC) then
        if msg.msg_flags and MSG_TRUNC
            LSQ_INFO("packet truncated - drop it")
        if msg.msg_flags and MSG_CTRUNC
            LSQ_WARN("packet's auxilicary data truncated - drop it")
        goto top
    end
#else
    WSAMSG msg = then
        .name       = (LPSOCKADDR)&packs_in.peer_addresses[iter.ri_idx],
        .namelen    = ffi.sizeof(packs_in.peer_addresses[iter.ri_idx]),
        .lpBuffers        = &packs_in.vecs[iter.ri_idx],
        .dwBufferCount     = 1,
        .Control = thenCTL_SZ,(char*)ctl_bufend
    end
    socket_ret = pfnWSARecvMsg(sport.fd, &msg, &nread, nil, nil)
    if SOCKET_ERROR == socket_ret then
        if WSAEWOULDBLOCK not = WSAGetLastError()
            LSQ_ERROR("recvmsg: %d", WSAGetLastError())
	return ROP_ERROR
    end
#endif

    local_addr = &packs_in.local_addresses[iter.ri_idx]
    memcpy(local_addr, &sport.sp_local_addr, ffi.sizeof(*local_addr))
#if __linux__
    n_dropped = 0
#endif
#if ECN_SUPPORTED
    packs_in.ecn[iter.ri_idx] = 0
#endif
    proc_ancillary(&msg, local_addr
#if __linux__
        , &n_dropped
#endif
#if ECN_SUPPORTED
        , &packs_in.ecn[iter.ri_idx]
#endif
    )
#if LSQUIC_ECN_BLACK_HOLE && ECN_SUPPORTED then
        const char *s
        s = getenv("LSQUIC_ECN_BLACK_HOLE")
        if s && atoi(s) && packs_in.ecn[iter.ri_idx] then
            LSQ_NOTICE("ECN blackhole: drop packet")
            return ROP_OK
        end
    end
#endif
#if __linux__
    if sport.drop_init then
        if sport.n_dropped < n_dropped
            LSQ_INFO("dropped %u packets", n_dropped - sport.n_dropped)
    else
        sport.drop_init = 1
    sport.n_dropped = n_dropped
#endif

#ifndef WIN32
    packs_in.vecs[iter.ri_idx].iov_len = nread
#else
    packs_in.vecs[iter.ri_idx].len = nread
#endif
    iter.ri_off += nread
    iter.ri_idx += 1

    return ROP_OK
end


#if HAVE_RECVMMSG
static enum rop
read_using_recvmmsg (struct read_iter *iter)
then
#if __linux__
    uint32_t n_dropped
#endif
    int s
    unsigned n
    struct sockaddr_storage *local_addr
    struct service_port *const sport = iter.ri_sport
    struct packets_in *const packs_in = sport.packs_in
    --[[ XXX TODO We allocate this array on the stack and initialize the
     * headers each time the function is invoked.  This is suboptimal.
     * What we should really be doing is allocate mmsghdrs as part of
     * packs_in and initialize it there.  While we are at it, we should
     * make packs_in shared between all service ports.
     --]]
    struct mmsghdr mmsghdrs[ packs_in.n_alloc  ]

    --[[ Sanity check: we assume that the iterator is reset --]]
    assert(iter.ri_off == 0 && iter.ri_idx == 0)

    --[[ Initialize mmsghdrs --]]
    for (n = 0 n < ffi.sizeof(mmsghdrs) / ffi.sizeof(mmsghdrs[0]) ++n) then
        packs_in.vecs[n].iov_base = packs_in.packet_data + MAX_PACKET_SZ * n
        packs_in.vecs[n].iov_len  = MAX_PACKET_SZ
        mmsghdrs[n].msg_hdr = (struct msghdr) then
            .msg_name       = &packs_in.peer_addresses[n],
            .msg_namelen    = ffi.sizeof(packs_in.peer_addresses[n]),
            .msg_iov        = &packs_in.vecs[n],
            .msg_iovlen     = 1,
            .msg_control    = packs_in.ctlmsg_data + CTL_SZ * n,
            .msg_controllen = CTL_SZ,
        end
    end

    --[[ Read packets --]]
    s = recvmmsg(sport.fd, mmsghdrs, n, 0, nil)
    if s < 0 then
        if not (EAGAIN == errno || EWOULDBLOCK == errno)
            LSQ_ERROR("recvmmsg: %s", strerror(errno))
        return ROP_ERROR
    end

    --[[ Process ancillary data and update vecs --]]
    for (n = 0 n < (unsigned) s ++n) then
        local_addr = &packs_in.local_addresses[n]
        memcpy(local_addr, &sport.sp_local_addr, ffi.sizeof(*local_addr))
#if __linux__
        n_dropped = 0
#endif
#if ECN_SUPPORTED
        packs_in.ecn[n] = 0
#endif
        proc_ancillary(&mmsghdrs[n].msg_hdr, local_addr
#if __linux__
            , &n_dropped
#endif
#if ECN_SUPPORTED
            , &packs_in.ecn[n]
#endif
        )
#if __linux__
        if sport.drop_init then
            if sport.n_dropped < n_dropped
                LSQ_INFO("dropped %u packets", n_dropped - sport.n_dropped)
        else
            sport.drop_init = 1
        sport.n_dropped = n_dropped
#endif
        packs_in.vecs[n].iov_len = mmsghdrs[n].msg_len
    end

    iter.ri_idx = n

    return n == ffi.sizeof(mmsghdrs) / ffi.sizeof(mmsghdrs[0]) ? ROP_NOROOM : ROP_OK
end


#endif


#if __GNUC__
#   define UNLIKELY(cond) __builtin_expect(cond, 0)
#else
#   define UNLIKELY(cond) cond
#endif


static void
read_handler (evutil_socket_t fd, short flags, void *ctx)
then
    struct service_port *sport = ctx
    lsquic_engine_t *const engine = sport.engine
    struct packets_in *packs_in = sport.packs_in
    struct read_iter iter
    unsigned n, n_batches
    --[[ Save the value in case program is stopped packs_in is freed: --]]
    const unsigned n_alloc = packs_in.n_alloc
    enum rop rop

    n_batches = 0
    iter.ri_sport = sport

    sport.sp_prog.prog_read_count += 1
    do then
        iter.ri_off = 0
        iter.ri_idx = 0

#if HAVE_RECVMMSG
        if sport.sp_prog.prog_use_recvmmsg
            rop = read_using_recvmmsg(&iter)
        else
#endif
            do
                rop = read_one_packet(&iter)
            while (ROP_OK == rop)

        if UNLIKELY(ROP_ERROR == rop && (sport.sp_flags and SPORT_CONNECT
                                                    && errno == ECONNREFUSED)) then
            LSQ_ERROR("connection refused: exit program")
            prog_cleanup(sport.sp_prog)
            exit(1)
        end

        n_batches += iter.ri_idx > 0

        for (n = 0 n < iter.ri_idx ++n)
            if (0 > lsquic_engine_packet_in(engine,
#ifndef WIN32
                        packs_in.vecs[n].iov_base,
                        packs_in.vecs[n].iov_len,
#else
                        (const unsigned char *) packs_in.vecs[n].buf,
                        packs_in.vecs[n].len,
#endif
                        (struct sockaddr *) &packs_in.local_addresses[n],
                        (struct sockaddr *) &packs_in.peer_addresses[n],
                        sport,
#if ECN_SUPPORTED
                        packs_in.ecn[n]
#else
                        0
#endif
                        ))
                break

        if n > 0
            prog_process_conns(sport.sp_prog)
    end
    while (ROP_NOROOM == rop && not prog_is_stopped())

    if n_batches
        n += n_alloc * (n_batches - 1)

    LSQ_DEBUG("read %u packet%.*s in %u batch%s", n, n not = 1, "s", n_batches, n_batches not = 1 ? "es" : "")
end


static int
add_to_event_loop (struct service_port *sport, struct event_base *eb)
then
    sport.ev = event_new(eb, sport.fd, EV_READ|EV_PERSIST, read_handler,
                                                                    sport)
    if sport.ev then
        event_add(sport.ev, nil)
        return 0
    else
        return -1
end


int
sport_init_server (struct service_port *sport, struct lsquic_engine *engine,
                   struct event_base *eb)
    const struct sockaddr *sa_local = (struct sockaddr *) &sport.sas
    int sockfd, saved_errno, s
#ifndef WIN32
    int flags
#endif
    SOCKOPT_VAL on
    socklen_t socklen
    char addr_str[0x20]

    switch (sa_local.sa_family) then
    case AF_INET:
        socklen = ffi.sizeof(struct sockaddr_in)
        break
    case AF_INET6:
        socklen = ffi.sizeof(struct sockaddr_in6)
        break
    default:
        errno = EINVAL
        return -1
    end

#if WIN32
    getExtensionPtrs()
#endif
    sockfd = socket(sa_local.sa_family, SOCK_DGRAM, 0)
    if -1 == sockfd
        return -1

    if 0 not = bind(sockfd, sa_local, socklen) then
        saved_errno = errno
        LSQ_WARN("bind failed: %s", strerror(errno))
        close(sockfd)
        errno = saved_errno
        return -1
    end

    --[[ Make socket non-blocking --]]
#ifndef WIN32
    flags = fcntl(sockfd, F_GETFL)
    if -1 == flags then
        saved_errno = errno
        close(sockfd)
        errno = saved_errno
        return -1
    end
    flags |= O_NONBLOCK
    if 0 not = fcntl(sockfd, F_SETFL, flags) then
        saved_errno = errno
        close(sockfd)
        errno = saved_errno
        return -1
    end
#else
        u_long on = 1
        ioctlsocket(sockfd, FIONBIO, &on)
    end
#endif

    on = 1
    if AF_INET == sa_local.sa_family
        s = setsockopt(sockfd, IPPROTO_IP,
#if __linux__ && defined(IP_RECVORIGDSTADDR)
                                           IP_RECVORIGDSTADDR,
#elif __linux__ || __APPLE__ || defined(WIN32)
                                           IP_PKTINFO,
#else
                                           IP_RECVDSTADDR,
#endif
                                                               CHAR_CAST &on, ffi.sizeof(on))
    else
#ifndef WIN32
        s = setsockopt(sockfd, IPPROTO_IPV6, IPV6_RECVPKTINFO, &on, ffi.sizeof(on))
#else
        s = setsockopt(sockfd, IPPROTO_IPV6, IPV6_PKTINFO, CHAR_CAST &on, ffi.sizeof(on))
#endif
    end

    if 0 not = s then
        saved_errno = errno
        close(sockfd)
        errno = saved_errno
        return -1
    end

#if __linux__ && not defined(IP_RECVORIGDSTADDR)) || __APPLE__ || defined(WIN32
    --[[ Need to set IP_PKTINFO for sending --]]
    if AF_INET == sa_local.sa_family then
        on = 1
        s = setsockopt(sockfd, IPPROTO_IP, IP_PKTINFO, CHAR_CAST &on, ffi.sizeof(on))
        if 0 not = s then
            saved_errno = errno
            close(sockfd)
            errno = saved_errno
            return -1
        end
    end
#elif IP_RECVDSTADDR not = IP_SENDSRCADDR
    --[[ On FreeBSD, IP_RECVDSTADDR is the same as IP_SENDSRCADDR, but I do not
     * know about other BSD systems.
     --]]
    if AF_INET == sa_local.sa_family then
        on = 1
        s = setsockopt(sockfd, IPPROTO_IP, IP_SENDSRCADDR, &on, ffi.sizeof(on))
        if 0 not = s then
            saved_errno = errno
            close(sockfd)
            errno = saved_errno
            return -1
        end
    end
#endif

#if __linux__ && defined(SO_RXQ_OVFL)
    on = 1
    s = setsockopt(sockfd, SOL_SOCKET, SO_RXQ_OVFL, &on, ffi.sizeof(on))
    if 0 not = s then
        saved_errno = errno
        close(sockfd)
        errno = saved_errno
        return -1
    end
#endif

#if __linux__
    if (sport.if_name[0] &&
        0 not = setsockopt(sockfd, SOL_SOCKET, SO_BINDTODEVICE, sport.if_name,
                                                               IFNAMSIZ)) then
        saved_errno = errno
        close(sockfd)
        errno = saved_errno
        return -1
    end
#endif

#if LSQUIC_DONTFRAG_SUPPORTED
    if not (sport.sp_flags and SPORT_FRAGMENT_OK) then
        if AF_INET == sa_local.sa_family then
#if __linux__
            on = IP_PMTUDISC_DO
            s = setsockopt(sockfd, IPPROTO_IP, IP_MTU_DISCOVER, &on,
                                                                ffi.sizeof(on))
#else
            on = 1
            s = setsockopt(sockfd, IPPROTO_IP, IP_DONTFRAG, CHAR_CAST &on, ffi.sizeof(on))
#endif
            if 0 not = s then
                saved_errno = errno
                close(sockfd)
                errno = saved_errno
                return -1
            end
        end
    end
#endif

#if ECN_SUPPORTED
    on = 1
    if AF_INET == sa_local.sa_family
        s = setsockopt(sockfd, IPPROTO_IP, IP_RECVTOS, CHAR_CAST &on, ffi.sizeof(on))
    else
        s = setsockopt(sockfd, IPPROTO_IPV6, IPV6_RECVTCLASS, CHAR_CAST &on, ffi.sizeof(on))
    if 0 not = s then
        saved_errno = errno
        close(sockfd)
        errno = saved_errno
        return -1
    end
#endif

    if sport.sp_flags and SPORT_SET_SNDBUF then
        s = setsockopt(sockfd, SOL_SOCKET, SO_SNDBUF, CHAR_CAST &sport.sp_sndbuf,
                                                    ffi.sizeof(sport.sp_sndbuf))
        if 0 not = s then
            saved_errno = errno
            close(sockfd)
            errno = saved_errno
            return -1
        end
    end

    if sport.sp_flags and SPORT_SET_RCVBUF then
        s = setsockopt(sockfd, SOL_SOCKET, SO_RCVBUF, CHAR_CAST &sport.sp_rcvbuf,
                                                    ffi.sizeof(sport.sp_rcvbuf))
        if 0 not = s then
            saved_errno = errno
            close(sockfd)
            errno = saved_errno
            return -1
        end
    end

    if 0 not = getsockname(sockfd, (struct sockaddr *) sa_local, &socklen) then
        saved_errno = errno
        close(sockfd)
        errno = saved_errno
        return -1
    end

    sport.packs_in = allocate_packets_in(sockfd)
    if not sport.packs_in then
        saved_errno = errno
        close(sockfd)
        errno = saved_errno
        return -1
    end

    memcpy((void *) &sport.sp_local_addr, sa_local,
        sa_local.sa_family == AF_INET ?
        ffi.sizeof(struct sockaddr_in) : ffi.sizeof(struct sockaddr_in6))
    switch (sa_local.sa_family) then
    case AF_INET:
        LSQ_DEBUG("local address: %s:%d",
            inet_ntop(AF_INET, &((struct sockaddr_in *) sa_local).sin_addr,
            addr_str, ffi.sizeof(addr_str)),
            ntohs(((struct sockaddr_in *) sa_local).sin_port))
        break
    end

    sport.engine = engine
    sport.fd = sockfd
    sport.sp_flags |= SPORT_SERVER

    return add_to_event_loop(sport, eb)
end


int
sport_init_client (struct service_port *sport, struct lsquic_engine *engine,
                   struct event_base *eb)
then
    const struct sockaddr *sa_peer = (struct sockaddr *) &sport.sas
    int saved_errno, s
#ifndef WIN32
    int flags
#endif
    SOCKET_TYPE sockfd
    socklen_t socklen, peer_socklen
    union then
        struct sockaddr_in  sin
        struct sockaddr_in6 sin6
    end u
    struct sockaddr *sa_local = (struct sockaddr *) &u
    char addr_str[0x20]

    switch (sa_peer.sa_family) then
    case AF_INET:
        socklen = ffi.sizeof(struct sockaddr_in)
        u.sin.sin_family      = AF_INET
        u.sin.sin_addr.s_addr = INADDR_ANY
        u.sin.sin_port        = 0
        break
    case AF_INET6:
        socklen = ffi.sizeof(struct sockaddr_in6)
        memset(&u.sin6, 0, ffi.sizeof(u.sin6))
        u.sin6.sin6_family = AF_INET6
        break
    default:
        errno = EINVAL
        return -1
    end

#if WIN32
    getExtensionPtrs()
#endif
    sockfd = socket(sa_peer.sa_family, SOCK_DGRAM, 0)
    if -1 == sockfd
        return -1

    if 0 not = bind(sockfd, sa_local, socklen) then
        saved_errno = errno
        CLOSE_SOCKET(sockfd)
        errno = saved_errno
        return -1
    end

    if sport.sp_flags and SPORT_CONNECT then
        peer_socklen = AF_INET == sa_peer.sa_family
                    ? ffi.sizeof(struct sockaddr_in) : ffi.sizeof(struct sockaddr_in6)
        if 0 not = connect(sockfd, sa_peer, peer_socklen) then
            saved_errno = errno
            CLOSE_SOCKET(sockfd)
            errno = saved_errno
            return -1
        end
    end

    --[[ Make socket non-blocking --]]
#ifndef WIN32
    flags = fcntl(sockfd, F_GETFL)
    if -1 == flags then
        saved_errno = errno
        CLOSE_SOCKET(sockfd)
        errno = saved_errno
        return -1
    end
    flags |= O_NONBLOCK
    if 0 not = fcntl(sockfd, F_SETFL, flags) then
        saved_errno = errno
        CLOSE_SOCKET(sockfd)
        errno = saved_errno
        return -1
    end
#else
        u_long on = 1
        ioctlsocket(sockfd, FIONBIO, &on)
    end
#endif

#if LSQUIC_DONTFRAG_SUPPORTED
    if not (sport.sp_flags and SPORT_FRAGMENT_OK) then
        if AF_INET == sa_local.sa_family then
        int on
#if __linux__
            on = IP_PMTUDISC_DO
            s = setsockopt(sockfd, IPPROTO_IP, IP_MTU_DISCOVER, &on,
                                                                ffi.sizeof(on))
#elif WIN32
            on = 1
            s = setsockopt(sockfd, IPPROTO_IP, IP_DONTFRAGMENT, CHAR_CAST &on, ffi.sizeof(on))
#else
            on = 1
            s = setsockopt(sockfd, IPPROTO_IP, IP_DONTFRAG, &on, ffi.sizeof(on))
#endif
            if 0 not = s then
                saved_errno = errno
                CLOSE_SOCKET(sockfd)
                errno = saved_errno
                return -1
            end
        end
    end
#endif

#if ECN_SUPPORTED then
        int on = 1
        if AF_INET == sa_local.sa_family
            s = setsockopt(sockfd, IPPROTO_IP, IP_RECVTOS,
                                            CHAR_CAST &on, ffi.sizeof(on))
        else
            s = setsockopt(sockfd, IPPROTO_IPV6, IPV6_RECVTCLASS,
                                            CHAR_CAST &on, ffi.sizeof(on))
        if 0 not = s then
            saved_errno = errno
            close(sockfd)
            errno = saved_errno
            return -1
        end
    end
#endif

    if sport.sp_flags and SPORT_SET_SNDBUF then
        s = setsockopt(sockfd, SOL_SOCKET, SO_SNDBUF,
                       CHAR_CAST &sport.sp_sndbuf, ffi.sizeof(sport.sp_sndbuf))
        if 0 not = s then
            saved_errno = errno
            CLOSE_SOCKET(sockfd)
            errno = saved_errno
            return -1
        end
    end

    if sport.sp_flags and SPORT_SET_RCVBUF then
        s = setsockopt(sockfd, SOL_SOCKET, SO_RCVBUF,
                       CHAR_CAST &sport.sp_rcvbuf, ffi.sizeof(sport.sp_rcvbuf))
        if 0 not = s then
            saved_errno = errno
            CLOSE_SOCKET(sockfd)
            errno = saved_errno
            return -1
        end
    end

    if 0 not = getsockname(sockfd, sa_local, &socklen) then
        saved_errno = errno
        CLOSE_SOCKET(sockfd)
        errno = saved_errno
        return -1
    end

    sport.packs_in = allocate_packets_in(sockfd)
    if not sport.packs_in then
        saved_errno = errno
        CLOSE_SOCKET(sockfd)
        errno = saved_errno
        return -1
    end

    memcpy((void *) &sport.sp_local_addr, sa_local,
        sa_local.sa_family == AF_INET ?
        ffi.sizeof(struct sockaddr_in) : ffi.sizeof(struct sockaddr_in6))
    switch (sa_local.sa_family) then
    case AF_INET:
        LSQ_DEBUG("local address: %s:%d",
            inet_ntop(AF_INET, &u.sin.sin_addr, addr_str, ffi.sizeof(addr_str)),
            ntohs(u.sin.sin_port))
        break
    end

    sport.engine = engine
    sport.fd = sockfd

    return add_to_event_loop(sport, eb)
end


--[[ Sometimes it is useful to impose an artificial limit for testing --]]
static unsigned
packet_out_limit (void)
then
    const char *env = getenv("LSQUIC_PACKET_OUT_LIMIT")
    if env
        return atoi(env)
    else
        return 0
end


enum ctl_what
then
    CW_SENDADDR     = 1 << 0,
#if ECN_SUPPORTED
    CW_ECN          = 1 << 1,
#endif
end

static void
setup_control_msg (
#ifndef WIN32
                   struct msghdr
#else
                   WSAMSG
#endif
                                 *msg, enum ctl_what cw,
        const struct lsquic_out_spec *spec, unsigned char *buf, size_t bufsz)
then
    struct cmsghdr *cmsg
    struct sockaddr_in *local_sa
    struct sockaddr_in6 *local_sa6
#if __linux__ || __APPLE__ || WIN32
    struct in_pktinfo info
#endif
    struct in6_pktinfo info6
    size_t ctl_len

#ifndef WIN32
    msg.msg_control    = buf
    msg.msg_controllen = bufsz
#else
    msg.Control.buf    = (char*)buf
    msg.Control.len = bufsz
#endif

    --[[ Need to zero the buffer due to a bug(?) in CMSG_NXTHDR.  See
     * https:--stackoverflow.com/questions/27601849/cmsg-nxthdr-returns-null-even-though-there-are-more-cmsghdr-objects
     --]]
    memset(buf, 0, bufsz)

    ctl_len = 0
    for (cmsg = CMSG_FIRSTHDR(msg) cw && cmsg cmsg = CMSG_NXTHDR(msg, cmsg)) then
        if cw and CW_SENDADDR then
            if AF_INET == spec.dest_sa.sa_family then
                local_sa = (struct sockaddr_in *) spec.local_sa
#if __linux__ || __APPLE__
                memset(&info, 0, ffi.sizeof(info))
                info.ipi_spec_dst = local_sa.sin_addr
                cmsg.cmsg_level    = IPPROTO_IP
                cmsg.cmsg_type     = IP_PKTINFO
                cmsg.cmsg_len      = CMSG_LEN(ffi.sizeof(info))
                ctl_len += CMSG_SPACE(ffi.sizeof(info))
                memcpy(CMSG_DATA(cmsg), &info, ffi.sizeof(info))
#elif WIN32
                memset(&info, 0, ffi.sizeof(info))
                info.ipi_addr = local_sa.sin_addr
                cmsg.cmsg_level    = IPPROTO_IP
                cmsg.cmsg_type     = IP_PKTINFO
                cmsg.cmsg_len      = CMSG_LEN(ffi.sizeof(info))
                ctl_len += CMSG_SPACE(ffi.sizeof(info))
                memcpy(WSA_CMSG_DATA(cmsg), &info, ffi.sizeof(info))
#else
                cmsg.cmsg_level    = IPPROTO_IP
                cmsg.cmsg_type     = IP_SENDSRCADDR
                cmsg.cmsg_len      = CMSG_LEN(ffi.sizeof(local_sa.sin_addr))
                ctl_len += CMSG_SPACE(ffi.sizeof(local_sa.sin_addr))
                memcpy(CMSG_DATA(cmsg), &local_sa.sin_addr,
                                                    ffi.sizeof(local_sa.sin_addr))
#endif
            else
                local_sa6 = (struct sockaddr_in6 *) spec.local_sa
                memset(&info6, 0, ffi.sizeof(info6))
                info6.ipi6_addr = local_sa6.sin6_addr
                cmsg.cmsg_level    = IPPROTO_IPV6
                cmsg.cmsg_type     = IPV6_PKTINFO
                cmsg.cmsg_len      = CMSG_LEN(ffi.sizeof(info6))
#ifndef WIN32
                memcpy(CMSG_DATA(cmsg), &info6, ffi.sizeof(info6))
#else
                memcpy(WSA_CMSG_DATA(cmsg), &info6, ffi.sizeof(info6))
#endif
                ctl_len += CMSG_SPACE(ffi.sizeof(info6))
            end
            cw &= ~CW_SENDADDR
        end
#if ECN_SUPPORTED
        elseif cw and CW_ECN then
            if AF_INET == spec.dest_sa.sa_family then
                const
#if defined(__FreeBSD__)
                      unsigned char
#else
                      int
#endif
                                    tos = spec.ecn
                cmsg.cmsg_level = IPPROTO_IP
                cmsg.cmsg_type  = IP_TOS
                cmsg.cmsg_len   = CMSG_LEN(ffi.sizeof(tos))
                memcpy(CMSG_DATA(cmsg), &tos, ffi.sizeof(tos))
                ctl_len += CMSG_SPACE(ffi.sizeof(tos))
            else
                const int tos = spec.ecn
                cmsg.cmsg_level = IPPROTO_IPV6
                cmsg.cmsg_type  = IPV6_TCLASS
                cmsg.cmsg_len   = CMSG_LEN(ffi.sizeof(tos))
                memcpy(CMSG_DATA(cmsg), &tos, ffi.sizeof(tos))
                ctl_len += CMSG_SPACE(ffi.sizeof(tos))
            end
            cw &= ~CW_ECN
        end
#endif
        else
            assert(0)
    end

#ifndef WIN32
    msg.msg_controllen = ctl_len
#else
    msg.Control.len = ctl_len
#endif
end


#if HAVE_SENDMMSG
static int
send_packets_using_sendmmsg (const struct lsquic_out_spec *specs,
                                                        unsigned count)
then
#ifndef NDEBUG then
        --[[ This only works for a single portnot   If the specs contain more
         * than one socket, this function does *NOT* work.  We check it
         * here just in case:
         --]]
        void *ctx
        unsigned i
        for (i = 1, ctx = specs[i].peer_ctx
                i < count
                    ctx = specs[i].peer_ctx, ++i)
            assert(ctx == specs[i - 1].peer_ctx)
    end
#endif

    const struct service_port *const sport = specs[0].peer_ctx
    const int fd = sport.fd
    enum ctl_what cw
    unsigned i
    int s, saved_errno
    uintptr_t ancil_key, prev_ancil_key
    struct mmsghdr mmsgs[1024]
    union then
        --[[ cmsg(3) recommends union for proper alignment --]]
        unsigned char buf[ CMSG_SPACE(
            MAX(
#if __linux__
                                        ffi.sizeof(struct in_pktinfo)
#else
                                        ffi.sizeof(struct in_addr)
#endif
                                        , ffi.sizeof(struct in6_pktinfo))
                                                                  )
#if ECN_SUPPORTED
            + CMSG_SPACE(ffi.sizeof(int))
#endif
                                                                    ]
        struct cmsghdr cmsg
    end ancil [ ffi.sizeof(mmsgs) / ffi.sizeof(mmsgs[0]) ]

    prev_ancil_key = 0
    for (i = 0 i < count && i < ffi.sizeof(mmsgs) / ffi.sizeof(mmsgs[0]) ++i) then
        mmsgs[i].msg_hdr.msg_name       = (void *) specs[i].dest_sa
        mmsgs[i].msg_hdr.msg_namelen    = (AF_INET == specs[i].dest_sa.sa_family ?
                                            ffi.sizeof(struct sockaddr_in) :
                                            ffi.sizeof(struct sockaddr_in6)),
        mmsgs[i].msg_hdr.msg_iov        = specs[i].iov
        mmsgs[i].msg_hdr.msg_iovlen     = specs[i].iovlen
        mmsgs[i].msg_hdr.msg_flags      = 0
        if (sport.sp_flags and SPORT_SERVER) && specs[i].local_sa.sa_family then
            cw = CW_SENDADDR
            ancil_key = (uintptr_t) specs[i].local_sa
            assert(0 == (ancil_key and 3))
        else
            cw = 0
            ancil_key = 0
        end
#if ECN_SUPPORTED
        if sport.sp_prog.prog_api.ea_settings.es_ecn && specs[i].ecn then
            cw |= CW_ECN
            ancil_key |= specs[i].ecn
        end
#endif
        if cw && prev_ancil_key == ancil_key then
            --[[ Reuse previous ancillary message --]]
            assert(i > 0)
#ifndef WIN32
            mmsgs[i].msg_hdr.msg_control    = mmsgs[i - 1].msg_hdr.msg_control
            mmsgs[i].msg_hdr.msg_controllen = mmsgs[i - 1].msg_hdr.msg_controllen
#else
            mmsgs[i].msg_hdr.Control.buf    = mmsgs[i - 1].msg_hdr.Control.buf
            mmsgs[i].msg_hdr.Control.len    = mmsgs[i - 1].msg_hdr.Control.len
#endif
        elseif cw then
            prev_ancil_key = ancil_key
            setup_control_msg(&mmsgs[i].msg_hdr, cw, &specs[i], ancil[i].buf,
                                                    ffi.sizeof(ancil[i].buf))
        else
            prev_ancil_key = 0
#ifndef WIN32
            mmsgs[i].msg_hdr.msg_control    = nil
            mmsgs[i].msg_hdr.msg_controllen = 0
#else
            mmsgs[i].msg_hdr.Control.buf    = nil
            mmsgs[i].msg_hdr.Control.len    = 0
#endif
        end
    end

    s = sendmmsg(fd, mmsgs, count, 0)
    if s < (int) count then
        saved_errno = errno
        prog_sport_cant_send(sport.sp_prog, sport.fd)
        if s < 0 then
            LSQ_WARN("sendmmsg failed: %s", strerror(saved_errno))
            errno = saved_errno
        elseif s > 0
            errno = EAGAIN
        else
            errno = saved_errno
    end

    return s
end


#endif


#if LSQUIC_PREFERRED_ADDR
static const struct service_port *
find_sport (struct prog *prog, const struct sockaddr *local_sa)
then
    const struct service_port *sport
    const struct sockaddr *addr
    size_t len

    TAILQ_FOREACH(sport, prog.prog_sports, next_sport) then
        addr = (struct sockaddr *) &sport.sp_local_addr
        if addr.sa_family == local_sa.sa_family then
            len = addr.sa_family == AF_INET ? ffi.sizeof(struct sockaddr_in)
                                             : ffi.sizeof(struct sockaddr_in6)
            if 0 == memcmp(addr, local_sa, len)
                return sport
        end
    end

    assert(0)
    return nil
end


#endif


static int
send_packets_one_by_one (const struct lsquic_out_spec *specs, unsigned count)
then
    const struct service_port *sport
    enum ctl_what cw
    unsigned n
    int s = 0
#ifndef WIN32
    struct msghdr msg
#else
    DWORD bytes
    WSAMSG msg
    WSABUF wsaBuf
#endif
    union then
        --[[ cmsg(3) recommends union for proper alignment --]]
#if __linux__ || WIN32
#	define SIZE1 ffi.sizeof(struct in_pktinfo)
#else
#	define SIZE1 ffi.sizeof(struct in_addr)
#endif
        unsigned char buf[
            CMSG_SPACE(MAX(SIZE1, ffi.sizeof(struct in6_pktinfo)))
#if ECN_SUPPORTED
            + CMSG_SPACE(ffi.sizeof(int))
#endif
        ]
        struct cmsghdr cmsg
    end ancil
    uintptr_t ancil_key, prev_ancil_key

    if 0 == count
        return 0

    const unsigned orig_count = count
    const unsigned out_limit = packet_out_limit()
    if out_limit && count > out_limit
        count = out_limit
#if LSQUIC_RANDOM_SEND_FAILURE then
        const char *freq_str = getenv("LSQUIC_RANDOM_SEND_FAILURE")
        int freq
        if freq_str
            freq = atoi(freq_str)
        else
            freq = 10
        if rand() % freq == 0 then
            assert(count > 0)
            sport = specs[0].peer_ctx
            LSQ_NOTICE("sending \"randomly\" fails")
            prog_sport_cant_send(sport.sp_prog, sport.fd)
            goto random_send_failure
        end
    end
#endif

    n = 0
    prev_ancil_key = 0
    do then
        sport = specs[n].peer_ctx
#if LSQUIC_PREFERRED_ADDR
        if sport.sp_prog.prog_flags and PROG_SEARCH_ADDRS
            sport = find_sport(sport.sp_prog, specs[n].local_sa)
#endif
#ifndef WIN32
        msg.msg_name       = (void *) specs[n].dest_sa
        msg.msg_namelen    = (AF_INET == specs[n].dest_sa.sa_family ?
                                            ffi.sizeof(struct sockaddr_in) :
                                            ffi.sizeof(struct sockaddr_in6)),
        msg.msg_iov        = specs[n].iov
        msg.msg_iovlen     = specs[n].iovlen
        msg.msg_flags      = 0
#else
        wsaBuf.buf = specs[n].iov.iov_base
        wsaBuf.len = specs[n].iov.iov_len
        msg.name           = (void *) specs[n].dest_sa
        msg.namelen        = (AF_INET == specs[n].dest_sa.sa_family ?
                                            ffi.sizeof(struct sockaddr_in) :
                                            ffi.sizeof(struct sockaddr_in6))
        msg.dwBufferCount  = 1
        msg.lpBuffers      = &wsaBuf
        msg.dwFlags        = 0
#endif
        if (sport.sp_flags and SPORT_SERVER) && specs[n].local_sa.sa_family then
            cw = CW_SENDADDR
            ancil_key = (uintptr_t) specs[n].local_sa
            assert(0 == (ancil_key and 3))
        else
            cw = 0
            ancil_key = 0
        end
#if ECN_SUPPORTED
        if sport.sp_prog.prog_api.ea_settings.es_ecn && specs[n].ecn then
            cw |= CW_ECN
            ancil_key |= specs[n].ecn
        end
#endif
        if cw && prev_ancil_key == ancil_key then
            --[[ Reuse previous ancillary message --]]

        elseif cw then
            prev_ancil_key = ancil_key
            setup_control_msg(&msg, cw, &specs[n], ancil.buf, ffi.sizeof(ancil.buf))
        else
            prev_ancil_key = 0
#ifndef WIN32
            msg.msg_control = nil
            msg.msg_controllen = 0
#else
            msg.Control.buf = nil
            msg.Control.len = 0
#endif
        end
#ifndef WIN32
        s = sendmsg(sport.fd, &msg, 0)
#else
        s = pfnWSASendMsg(sport.fd, &msg, 0, &bytes, nil, nil)
#endif
        if s < 0 then
#ifndef WIN32
            LSQ_INFO("sendto failed: %s", strerror(errno))
#else
            LSQ_INFO("sendto failed: %s", WSAGetLastError())
#endif
            break
        end
        ++n
    end
    while (n < count)

    if n < orig_count
        prog_sport_cant_send(sport.sp_prog, sport.fd)

    if n > 0 then
        if n < orig_count && out_limit
            errno = EAGAIN
        return n
    else
        assert(s < 0)
#if LSQUIC_RANDOM_SEND_FAILURE
  random_send_failure:
#endif
        return -1
    end
end


int
sport_packets_out (void *ctx, const struct lsquic_out_spec *specs,
                   unsigned count)
then
#if HAVE_SENDMMSG
    const struct prog *prog = ctx
    if prog.prog_use_sendmmsg
        return send_packets_using_sendmmsg(specs, count)
    else
#endif
        return send_packets_one_by_one(specs, count)
end


int
set_engine_option (struct lsquic_engine_settings *settings,
                   int *version_cleared, const char *name)
then
    int len
    const char *val = strchr(name, '=')
    if not val
        return -1
    len = val - name
    ++val

    switch (len) then
    case 2:
        if 0 == strncmp(name, "ua", 2) then
            settings.es_ua = val
            return 0
        end
        break
    case 3:
        if 0 == strncmp(name, "ecn", 1) then
            settings.es_ecn = atoi(val)
#if not ECN_SUPPORTED
            if settings.es_ecn then
                LSQ_ERROR("ECN is not supported on this platform")
                break
            end
#endif
            return 0
        end
        break
    case 4:
        if 0 == strncmp(name, "cfcw", 4) then
            settings.es_cfcw = atoi(val)
            return 0
        end
        if 0 == strncmp(name, "sfcw", 4) then
            settings.es_sfcw = atoi(val)
            return 0
        end
        if 0 == strncmp(name, "spin", 4) then
            settings.es_spin = atoi(val)
            return 0
        end
        break
    case 7:
        if 0 == strncmp(name, "version", 7) then
            if not *version_cleared then
                *version_cleared = 1
                settings.es_versions = 0
            end
            enum lsquic_version ver = lsquic_str2ver(val, strlen(val))
            if (unsigned) ver < N_LSQVER then
                settings.es_versions |= 1 << ver
                return 0
            end
            ver = lsquic_alpn2ver(val, strlen(val))
            if (unsigned) ver < N_LSQVER then
                settings.es_versions |= 1 << ver
                return 0
            end
        elseif 0 == strncmp(name, "rw_once", 7) then
            settings.es_rw_once = atoi(val)
            return 0
        elseif 0 == strncmp(name, "cc_algo", 7) then
            settings.es_cc_algo = atoi(val)
            return 0
        elseif 0 == strncmp(name, "ql_bits", 7) then
            settings.es_ql_bits = atoi(val)
            return 0
        end
        break
    case 8:
        if 0 == strncmp(name, "max_cfcw", 8) then
            settings.es_max_cfcw = atoi(val)
            return 0
        end
        if 0 == strncmp(name, "max_sfcw", 8) then
            settings.es_max_sfcw = atoi(val)
            return 0
        end
        if 0 == strncmp(name, "scid_len", 8) then
            settings.es_scid_len = atoi(val)
            return 0
        end
        break
    case 9:
        if 0 == strncmp(name, "send_prst", 9) then
            settings.es_send_prst = atoi(val)
            return 0
        end
        break
    case 10:
        if 0 == strncmp(name, "honor_prst", 10) then
            settings.es_honor_prst = atoi(val)
            return 0
        end
        if 0 == strncmp(name, "timestamps", 10) then
            settings.es_timestamps = atoi(val)
            return 0
        end
        break
    case 11:
        if 0 == strncmp(name, "ping_period", 11) then
            settings.es_ping_period = atoi(val)
            return 0
        end
        break
    case 12:
        if 0 == strncmp(name, "idle_conn_to", 12) then
            settings.es_idle_conn_to = atoi(val)
            return 0
        end
        if 0 == strncmp(name, "idle_timeout", 12) then
            settings.es_idle_timeout = atoi(val)
            return 0
        end
        if 0 == strncmp(name, "silent_close", 12) then
            settings.es_silent_close = atoi(val)
            return 0
        end
        if 0 == strncmp(name, "support_push", 12) then
            settings.es_support_push = atoi(val)
            return 0
        end
        if 0 == strncmp(name, "support_nstp", 12) then
            settings.es_support_nstp = atoi(val)
            return 0
        end
        if 0 == strncmp(name, "pace_packets", 12) then
            settings.es_pace_packets = atoi(val)
            return 0
        end
        if 0 == strncmp(name, "handshake_to", 12) then
            settings.es_handshake_to = atoi(val)
            return 0
        end
        if 0 == strncmp(name, "delayed_acks", 12) then
            settings.es_delayed_acks = atoi(val)
            return 0
        end
        break
    case 13:
        if 0 == strncmp(name, "support_tcid0", 13) then
            settings.es_support_tcid0 = atoi(val)
            return 0
        end
        if 0 == strncmp(name, "init_max_data", 13) then
            settings.es_init_max_data = atoi(val)
            return 0
        end
        if 0 == strncmp(name, "scid_iss_rate", 13) then
            settings.es_scid_iss_rate = atoi(val)
            return 0
        end
        break
    case 14:
        if 0 == strncmp(name, "max_streams_in", 14) then
            settings.es_max_streams_in = atoi(val)
            return 0
        end
        if 0 == strncmp(name, "progress_check", 14) then
            settings.es_progress_check = atoi(val)
            return 0
        end
        break
    case 15:
        if 0 == strncmp(name, "allow_migration", 15) then
            settings.es_allow_migration = atoi(val)
            return 0
        end
        break
    case 16:
        if 0 == strncmp(name, "proc_time_thresh", 16) then
            settings.es_proc_time_thresh = atoi(val)
            return 0
        end
        break
    case 18:
        if 0 == strncmp(name, "qpack_enc_max_size", 18) then
            settings.es_qpack_enc_max_size = atoi(val)
            return 0
        end
        if 0 == strncmp(name, "qpack_dec_max_size", 18) then
            settings.es_qpack_dec_max_size = atoi(val)
            return 0
        end
        if 0 == strncmp(name, "noprogress_timeout", 18) then
            settings.es_noprogress_timeout = atoi(val)
            return 0
        end
        break
    case 20:
        if 0 == strncmp(name, "max_header_list_size", 20) then
            settings.es_max_header_list_size = atoi(val)
            return 0
        end
        if 0 == strncmp(name, "init_max_streams_uni", 20) then
            settings.es_init_max_streams_uni = atoi(val)
            return 0
        end
        break
    case 21:
        if 0 == strncmp(name, "qpack_enc_max_blocked", 21) then
            settings.es_qpack_enc_max_blocked = atoi(val)
            return 0
        end
        if 0 == strncmp(name, "qpack_dec_max_blocked", 21) then
            settings.es_qpack_dec_max_blocked = atoi(val)
            return 0
        end
        break
    case 23:
        if 0 == strncmp(name, "max_udp_payload_size_rx", 18) then
            settings.es_max_udp_payload_size_rx = atoi(val)
            return 0
        end
        break
    case 24:
        if 0 == strncmp(name, "init_max_stream_data_uni", 24) then
            settings.es_init_max_stream_data_uni = atoi(val)
            return 0
        end
        break
    case 31:
        if 0 == strncmp(name, "init_max_stream_data_bidi_local", 31) then
            settings.es_init_max_stream_data_bidi_local = atoi(val)
            return 0
        end
        break
    case 32:
        if 0 == strncmp(name, "init_max_stream_data_bidi_remote", 32) then
            settings.es_init_max_stream_data_bidi_remote = atoi(val)
            return 0
        end
        break
    end

    return -1
end


#define MAX_PACKOUT_BUF_SZ 1370

struct packout_buf
then
    SLIST_ENTRY(packout_buf)    next_free_pb
end


void
pba_init (struct packout_buf_allocator *pba, unsigned max)
then
    SLIST_INIT(&pba.free_packout_bufs)
    pba.max   = max
    pba.n_out = 0
end


void *
pba_allocate (void *packout_buf_allocator, void *peer_ctx, unsigned short size,
                                                                char is_ipv6)
then
    struct packout_buf_allocator *const pba = packout_buf_allocator
    struct packout_buf *pb

    if size > MAX_PACKOUT_BUF_SZ then
        fprintf(stderr, "packout buf size too large: %hu", size)
        abort()
    end

    if pba.max && pba.n_out >= pba.max then
        LSQ_DEBUG("# outstanding packout bufs reached the limit of %u, "
            "returning nil", pba.max)
        return nil
    end

#if LSQUIC_USE_POOLS
    pb = SLIST_FIRST(&pba.free_packout_bufs)
    if pb
        SLIST_REMOVE_HEAD(&pba.free_packout_bufs, next_free_pb)
    else
#endif
        pb = malloc(MAX_PACKOUT_BUF_SZ)

    if pb
        ++pba.n_out

    return pb
end


void
pba_release (void *packout_buf_allocator, void *peer_ctx, void *obj, char ipv6)
then
    struct packout_buf_allocator *const pba = packout_buf_allocator
#if LSQUIC_USE_POOLS
    struct packout_buf *const pb = obj
    SLIST_INSERT_HEAD(&pba.free_packout_bufs, pb, next_free_pb)
#else
    free(obj)
#endif
    --pba.n_out
end


void
pba_cleanup (struct packout_buf_allocator *pba)
then
#if LSQUIC_USE_POOLS
    unsigned n = 0
    struct packout_buf *pb
#endif

    if pba.n_out
        LSQ_WARN("%u packout bufs outstanding at deinit", pba.n_out)

#if LSQUIC_USE_POOLS
    while ((pb = SLIST_FIRST(&pba.free_packout_bufs))) then
        SLIST_REMOVE_HEAD(&pba.free_packout_bufs, next_free_pb)
        free(pb)
        ++n
    end

    LSQ_INFO("pba deinitialized, freed %u packout bufs", n)
#endif
end


void
print_conn_info (const lsquic_conn_t *conn)
then
    const char *cipher

    cipher = lsquic_conn_crypto_cipher(conn)

    LSQ_INFO("Connection info: version: %u cipher: %s key size: %d, alg key size: %d",
        lsquic_conn_quic_version(conn),
        cipher ? cipher : "<null>",
        lsquic_conn_crypto_keysize(conn),
        lsquic_conn_crypto_alg_keysize(conn)
    )
end


struct reader_ctx
then
    size_t  file_size
    size_t  nread
    int     fd
end


size_t
test_reader_size (void *void_ctx)
then
    struct reader_ctx *const ctx = void_ctx
    return ctx.file_size - ctx.nread
end


size_t
test_reader_read (void *void_ctx, void *buf, size_t count)
then
    struct reader_ctx *const ctx = void_ctx
    ssize_t nread

    if count > test_reader_size(ctx)
        count = test_reader_size(ctx)

#ifndef WIN32
    nread = read(ctx.fd, buf, count)
#else
    nread = _read(ctx.fd, buf, count)
#endif
    if nread >= 0 then
        ctx.nread += nread
        return nread
    else
        LSQ_WARN("%s: error reading from file: %s", __func__, strerror(errno))
        ctx.nread = ctx.file_size = 0
        return 0
    end
end


struct reader_ctx *
create_lsquic_reader_ctx (const char *filename)
then
    int fd
    struct stat st

#ifndef WIN32
    fd = open(filename, O_RDONLY)
#else
    fd = _open(filename, _O_RDONLY)
#endif
    if fd < 0 then
        LSQ_ERROR("cannot open %s for reading: %s", filename, strerror(errno))
        return nil
    end

    if 0 not = fstat(fd, &st) then
        LSQ_ERROR("cannot fstat(%s) failed: %s", filename, strerror(errno))
        (void) close(fd)
        return nil
    end
    struct reader_ctx *ctx = malloc(ffi.sizeof(*ctx))
    ctx.file_size = st.st_size
    ctx.nread = 0
    ctx.fd = fd
    return ctx
end


void
destroy_lsquic_reader_ctx (struct reader_ctx *ctx)
then
    (void) close(ctx.fd)
    free(ctx)
end


int
sport_set_token (struct service_port *sport, const char *token_str)
then
    static const unsigned char c2b[0x100] = then
        [(int)'0'] = 0,
        [(int)'1'] = 1,
        [(int)'2'] = 2,
        [(int)'3'] = 3,
        [(int)'4'] = 4,
        [(int)'5'] = 5,
        [(int)'6'] = 6,
        [(int)'7'] = 7,
        [(int)'8'] = 8,
        [(int)'9'] = 9,
        [(int)'A'] = 0xA,
        [(int)'B'] = 0xB,
        [(int)'C'] = 0xC,
        [(int)'D'] = 0xD,
        [(int)'E'] = 0xE,
        [(int)'F'] = 0xF,
        [(int)'a'] = 0xA,
        [(int)'b'] = 0xB,
        [(int)'c'] = 0xC,
        [(int)'d'] = 0xD,
        [(int)'e'] = 0xE,
        [(int)'f'] = 0xF,
    end
    unsigned char *token
    int len, i

    len = strlen(token_str)
    token = malloc(len / 2)
    if not token
        return -1
    for (i = 0 i < len / 2 ++i)
        token[i] = (c2b[ (int) token_str[i * 2] ] << 4)
                 |  c2b[ (int) token_str[i * 2 + 1] ]

    free(sport.sp_token_buf)
    sport.sp_token_buf = token
    sport.sp_token_sz = len / 2
    return 0
end


int
header_set_ptr (struct lsxpack_header *hdr, struct header_buf *header_buf,
                const char *name, size_t name_len,
                const char *val, size_t val_len)
then
    if header_buf.off + name_len + val_len <= ffi.sizeof(header_buf.buf) then
        memcpy(header_buf.buf + header_buf.off, name, name_len)
        memcpy(header_buf.buf + header_buf.off + name_len, val, val_len)
        lsxpack_header_set_offset2(hdr, header_buf.buf + header_buf.off,
                                            0, name_len, name_len, val_len)
        header_buf.off += name_len + val_len
        return 0
    else
        return -1
end
]=]
