--- lib/net/http3-quiche-client.lua
-- /Users/pasi/installed/C/tls/quiche/examples/http3-client.c
package.path = "lib/?.lua;lib/?.lx;" .. package.path
package.path = "../lib/?.lua;../lib/?.lx;" .. package.path
package.path = "../../lib/?.lua;../../lib/?.lx;" .. package.path
require "start"

local ffi = require "mffi"
local C = ffi.C
local util = require "util"
local l = require"lang".l
local scrypt = require "system/scrypt"
local socket = require "system/socket"
local quiche = require "net/quiche"
local print = util.print
local QUICHE_H3_APPLICATION_PROTOCOL = quiche.QUICHE_H3_APPLICATION_PROTOCOL
quiche = quiche.lib

local conn_io = {
	timer = {delay = 0, data = ""}
	-- const char *host;
	-- int sock;
	-- quiche_conn *conn;
	-- quiche_h3_conn *http3;
}
-- local conn_io = ffi.newAnchor("struct conn_io")
local LOCAL_CONN_ID_LEN = C.QUICHE_MAX_CONN_ID_LEN -- 16
local MAX_DATAGRAM_SIZE = 1350
local recvBufLen = 65535
local recvBuf = C.malloc(recvBufLen)
local sendFlags = 0
local receiveFlags = 0
if util.isLinux() then
	sendFlags = C.MSG_NOSIGNAL
	receiveFlags = C.MSG_NOSIGNAL
end
local callTimeout = 0.8 -- seconds
local sleepMillisec = 100
local loopCount = 0
local req_sent = false
local out = ffi.newAnchor("uint8_t[?]", MAX_DATAGRAM_SIZE)

local function debugLog(line, argp)
	util.printInfo("%s", ffi.string(line))
end

local function flush_egress()
	while true do
		local written = tonumber(quiche.quiche_conn_send(conn_io.conn, out, MAX_DATAGRAM_SIZE))
		if written == C.QUICHE_ERR_DONE then
			util.print("done writing")
			break
		end
		if written < 0 then
			return l("failed to create packet: %d", written)
		end
		local sent = conn_io.sock:send(out, written, 0)
		if sent ~= written then
			return l("failed to send")
		end
		util.print("sent %d bytes", sent)
	end
	conn_io.timer.delay = quiche.quiche_conn_timeout_as_nanos(conn_io.conn) / 1e9
end

local function for_each_header(name, name_len, value, value_len, argp)
	util.print("got HTTP header: %s=%s", ffi.string(name, name_len), ffi.string(value, value_len))
end

local function setHeader(headers, i, tag, value)
	headers[i].name = ffi.cast("const uint8_t *", tag)
	headers[i].name_len = #tag
	headers[i].value = ffi.cast("const uint8_t *", value)
	headers[i].value_len = #value
end

local function timeout_cb()
	quiche.quiche_conn_on_timeout(conn_io.conn)
	print("timeout")
	flush_egress()
	if quiche.quiche_conn_is_closed(conn_io.conn) then
		local stats = ffi.newNoAnchor("quiche_stats[1]")
		quiche.quiche_conn_stats(conn_io.conn, stats)
		util.print("connection closed, recv=%d sent=%d lost=%d rtt=%d ns", tonumber(stats[0].recv), tonumber(stats[0].sent), tonumber(stats[0].lost), tonumber(stats[0].rtt))
		return "connection closed"
	end
end

local function recv_cb()
	local startTime = util.seconds()
	local readBytes, done
	while true do
		readBytes = conn_io.sock:recv(recvBuf, recvBufLen, receiveFlags) -- socket.receive(sock)
		if readBytes < 0 then
			local errno = socket.lastError()
			if errno == C.EWOULDBLOCK or errno == C.EAGAIN then
				loopCount = loopCount + 1
				util.print(loopCount .. ". recv would block")
				break
			end
			util.print("failed to read")
			return "failed to read"
		end
		local done = tonumber(quiche.quiche_conn_recv(conn_io.conn, recvBuf, readBytes))
		if done < 0 then
			util.print("failed to process packet: %d", done)
			goto continue
		end
		util.print("recv %d bytes", done)
		::continue::
	end
	util.sleep(sleepMillisec)

	--[[
		if readBytes < 1 then
			local tcpErr = socket.lastError()
			if socket.eagain(tcpErr) then
				print("recv would block")
				util.sleep(sleepMillisec)
			else
				local err
				if socket.econnreset(tcpErr) then
					if util.isLinux() and tcpErr == 0 then
						err = l("receive error: %d, socket error: %d, %s", err, tcpErr, net.errorText(C.ECONNRESET))
					else
						err = l("receive error: %d, socket error: %d, %s", err, tcpErr, net.errorText(tcpErr))
					end
					return err
				end
				util.printWarning("receive error: %d, socket error: %d, %s", err, tcpErr, net.errorText(tcpErr))
			end
			util.sleep(sleepMillisec)
			if util.seconds(startTime) >= callTimeout then
				readBytes = -2 -- end loop
				-- socket.disconnect(conn_io.sock)
				-- timeout_cb()
				util.print("call worker readBytes content socket.receive() timeout")
			end
		else
			-- result.data = result.data..recv
			readBytes = tonumber(quiche.quiche_conn_recv(conn_io.conn, recvBuf, readBytes))
      if readBytes < 0 then
        util.print("failed to process packet: %d", readBytes)
      else
        util.print("recv %d bytes", readBytes)
			end
		end
	until readBytes > 0 or readBytes <= -2
	]]

	util.print("done reading")

	if quiche.quiche_conn_is_closed(conn_io.conn) then
		util.print("connection closed")
		return -1
	end

	local established = quiche.quiche_conn_is_established(conn_io.conn)
	if established and not req_sent then
		local app_proto = ffi.newNoAnchor("const uint8_t *[1]")
		local app_proto_len = ffi.newNoAnchor("size_t[1]")
		quiche.quiche_conn_application_proto(conn_io.conn, app_proto, app_proto_len)
		util.print("connection established: %.*s", app_proto_len[0], app_proto[0])
		local config = quiche.quiche_h3_config_new()
		if config == nil then
			util.print("failed to create HTTP/3 config")
			return -1
		end
		conn_io.http3 = quiche.quiche_h3_conn_new_with_transport(conn_io.conn, config)
		if conn_io.http3 == nil then
			util.print("failed to create HTTP/3 connection")
			return -1
		end
		quiche.quiche_h3_config_free(config)
		local headerCount = 5
		local headers = ffi.newAnchor("quiche_h3_header[?]", headerCount)
		setHeader(headers, 1, ":method", "GET")
		setHeader(headers, 2, ":scheme", "https")
		setHeader(headers, 3, ":authority", conn_io.host)
		setHeader(headers, 4, ":path", "/")
		setHeader(headers, 5, ":user-agent", "quiche")
		local stream_id = quiche.quiche_h3_send_request(conn_io.http3, conn_io.conn, headers, 5, true)
		util.print("sent HTTP request %s", stream_id)
		req_sent = true
	end

	if established then
		local ev = ffi.newNoAnchor("quiche_h3_event[1]")
		while true do
			local s = quiche.quiche_h3_conn_poll(conn_io.http3, conn_io.conn, ev)
			if s < 0 then
				break
			end
			local evType = quiche.quiche_h3_event_type(ev)
			if evType == C.quiche_H3_EVENT_HEADERS then
				local rc = quiche.quiche_h3_event_for_each_header(ev, ffi.cast("int (*)(uint8_t *name, size_t name_len, uint8_t *value, size_t value_len, void *argp))", for_each_header), nil)
				if rc ~= 0 then
					print("failed to process headers")
				end
			elseif evType == C.quiche_H3_EVENT_DATA then
				local len = quiche.quiche_h3_recv_body(conn_io.http3, conn_io.conn, s, buf, sizeof(buf))
				if len <= 0 then
					break
				end
				print("%.*s", len, buf)
			elseif evType == C.quiche_H3_EVENT_FINISHED then
				if quiche.quiche_conn_close(conn_io.conn, true, 0, nil, 0) < 0 then
					print("failed to close connection")
				end
			end
			quiche.quiche_h3_event_free(ev)
		end
	end
	return flush_egress()
end

local function main(arg)
	local host = arg and arg[1] or "quic.tech"
	local port = tostring(arg and arg[2] or 8443)
	util.print("connecting to %s:%s", host, port)
	local blocking = 0
	local connectTimeout = 4
	local sock = socket.connect(host, port, "http3", connectTimeout, blocking)
	if sock == nil then
		return
	end
	quiche.quiche_enable_debug_logging(ffi.cast("void (*)(const char *line, void *argp)", debugLog), nil)
	local config = quiche.quiche_config_new(C.QUICHE_PROTOCOL_VERSION) -- 0xbabababa -- C.QUICHE_PROTOCOL_VERSION
	if ffi.isNull(config) then
		util.printError("failed to create quiche config")
		return
	end
	quiche.quiche_config_verify_peer(config, false)
	print("QUICHE_H3_APPLICATION_PROTOCOL: " .. QUICHE_H3_APPLICATION_PROTOCOL)
	print(#QUICHE_H3_APPLICATION_PROTOCOL)
	local err = quiche.quiche_config_set_application_protos(config, ffi.cast("uint8_t *", QUICHE_H3_APPLICATION_PROTOCOL), #QUICHE_H3_APPLICATION_PROTOCOL)
	if err ~= 0 then
		util.printError("quiche_config_set_application_protos error %d", err)
		return
	end

	quiche.quiche_config_set_max_idle_timeout(config, 5000)
	quiche.quiche_config_set_max_udp_payload_size(config, MAX_DATAGRAM_SIZE)
	quiche.quiche_config_set_initial_max_data(config, 10000000)
	quiche.quiche_config_set_initial_max_stream_data_bidi_local(config, 1000000)
	quiche.quiche_config_set_initial_max_stream_data_bidi_remote(config, 1000000)
	quiche.quiche_config_set_initial_max_stream_data_uni(config, 1000000)
	quiche.quiche_config_set_initial_max_streams_bidi(config, 100)
	quiche.quiche_config_set_initial_max_streams_uni(config, 100)
	quiche.quiche_config_set_disable_active_migration(config, true)
	if os.getenv("SSLKEYLOGFILE") then
		quiche.quiche_config_log_keys(config)
	end
	local scid = ffi.newNoAnchor("uint8_t[?]", LOCAL_CONN_ID_LEN)
	scrypt.randomBytes(scid, LOCAL_CONN_ID_LEN)
	-- local scid = ffi.cast("const uint8_t *", scidStr)
	local conn = quiche.quiche_connect(host, scid, ffi.sizeof(scid), config)
	if ffi.isNull(conn) then
		util.printError("http3 connect failed")
		return
	end
	conn_io.sock = sock
	conn_io.conn = conn
	conn_io.host = host
	local ret = flush_egress()
	repeat
		ret = recv_cb()
		if not ret then
			-- ret = timeout_cb()
		end
	until ret
	if conn_io.http3 then
		quiche.quiche_h3_conn_free(conn_io.http3)
	end
	quiche.quiche_conn_free(conn)
	quiche.quiche_config_free(config)
end

util.printInfo("* quiche version: %s", ffi.string(quiche.quiche_version()))
main({...})
