--- peg.lua
-- Parsing Expression Grammar (PEG)
-- http://www.inf.puc-rio.br/~roberto/lpeg/
-- http://leafo.net/guides/parsing-expression-grammars.html
--[[
* compile lpeg and change it to lpeg2
- https://github.com/luvit/lpeg
- change lptree.c : lpeg -> lpeg2
- rename lpeg.so -> lpeg2.so

int luaopen_lpeg2 (lua_State *L);
int luaopen_lpeg2 (lua_State *L) {
  luaL_newmetatable(L, PATTERN_T);
  lua_pushnumber(L, MAXBACK);  /* initialize maximum backtracking * /
  lua_setfield(L, LUA_REGISTRYINDEX, MAXSTACKIDX);
  luaL_register(L, NULL, metareg);
  luaL_register(L, "lpeg2", pattreg);
  lua_pushvalue(L, -1);
  lua_setfield(L, -3, "__index");
  return 1;
}
--]] -- comment here to fix luaformat bug
local peg = {}
local lpeg
if false and jit and jit.version and jit.version:find("LuaJIT", 1, true) then
	lpeg = require "lpeglj/lpeglj" -- https://github.com/sacek/LPegLJ
else -- puc lua5.1
	lpeg = require "lpeg2" -- C-version - does not work?
end
-- local utf8 = require "utf"
local utf8 = require "lua-utf8"
local l --  = require "lang".l -- load later
local util --  = require "util" -- load later
local match, P, R, S, V, C, Cc, Cf, Cg, Cp, Cs, Ct = lpeg.match, lpeg.P, lpeg.R, lpeg.S, lpeg.V, lpeg.C, lpeg.Cc, lpeg.Cf, lpeg.Cg, lpeg.Cp, lpeg.Cs, lpeg.Ct

local pattTbl = {} -- collection of useful patterns

local function toPattern(patt)
	if type(patt) ~= "userdata" then
		return lpeg.P(patt)
	end
	return patt
end
peg.toPattern = toPattern

function peg.other(patt, leastCount)
	if leastCount then
		return peg.least(1 - toPattern(patt), leastCount)
	end
	return 1 - toPattern(patt)
end

function peg.least(patt, count)
	return toPattern(patt) ^ count
end

function peg.most(patt, count)
	return toPattern(patt) ^ -count
end

pattTbl.locale = lpeg.locale()
local digit = R "09" -- S"0123456789"
pattTbl.digit = digit
pattTbl.space = P " "
pattTbl.whiteSpace = S " \t"
pattTbl.anyWhiteSpace = S " \t\r\n\f" -- space, tab, return, newline, formfeed
pattTbl.notSpace = 1 - pattTbl.space
pattTbl.endOfLine = P "\n\r" + P "\r\n" + P "\n" + P "\r"
pattTbl.separator = (1 - pattTbl.locale.alnum - P "_")
-- pattTbl.wordEnd = pattTbl.whiteSpace + pattTbl.endOfLine
pattTbl.emptyLineEnd = P(-1)
pattTbl.anychar = 1 - S "" -- P(string.char(0))
pattTbl.anytext = peg.least(pattTbl.anychar, 0)
-- patt.anychar = (1 - lpeg.S(""))^0
-- comments

peg.define = pattTbl
peg.match = match
peg.set = S
peg.range = R
peg.pattern = P
peg.V = V
peg.capture = C
peg.Cc = Cc
peg.Cf = Cf
peg.Cg = Cg
peg.Cp = Cp
peg.Cs = Cs
peg.Ct = Ct

--[[
local function token(id, patt) -- callback for [=[ long strings ]=]
	return Ct(Cc(id) * C(patt))
end
]]

pattTbl.singlequoted = P "'" * ((1 - S "'\r\n\f\\") + (P '\\' * 1)) ^ 0 * P "'"
pattTbl.doublequoted = P '"' * ((1 - S '"\r\n\f\\') + (P '\\' * 1)) ^ 0 * P '"'
pattTbl.quoted = pattTbl.singlequoted + pattTbl.doublequoted
--[=[
local longstring = #(P '[[' + (P '[' * P '=' ^ 0 * P '['))
pattTbl.longstring = longstring * P(function(input, index)
   local level = input:match('^%[(=*)%[', index)
   if level then
      local _, stop = input:peg.find(']' .. level .. ']', index, true)
      if stop then return stop + 1 end
   end
end)
pattTbl.anystring = pattTbl.quoted + pattTbl.longstring
]=]
pattTbl.longStringStart = P '[[' + (P '[' * P '=' ^ 0 * P '[')
pattTbl.longCommentStart = P '--' * pattTbl.longStringStart
pattTbl.longStringEnd = P ']]' + (P ']' * P '=' ^ 0 * P ']')
--[=[
peg.longstring = #(peg.longStringStart)
peg.longstring = peg.longstring * peg.pattern(function(input, index)
	local level = input:match('^%[(=*)%[', index)
	if level then
		local _, stop = input:peg.find(']' .. level .. ']', index, true)
		if stop then
			return stop + 1
		end
	end
end)
peg.singleline_comment = peg.pattern '--' * (1 -  peg.set '\r\n\f') ^ 0
peg.multiline_comment = peg.pattern '--' * peg.longstring
peg.comment = token('comment', peg.multiline_comment + peg.singleline_comment)
-- comments end
]=]

local function loadRequire()
	if not l then
		l = require"lang".l
	end
	if not util then
		util = require "util"
	end
end

local function printError(err, ...)
	loadRequire()
	util.printError(err, ...)
end

function peg.leastMost(patt, countLeast, countMost)
	if countMost < countLeast then
		printError("peg.least count is bigger than peg.most count")
		return nil
	end
	if countLeast < 0 then
		printError("peg.least count is smaller than 0")
		return nil
	end
	if countMost < 1 then
		printError("peg.most count is smaller than 1")
		return nil
	end
	if countLeast == 0 then
		return peg.most(patt, countMost)
	end

	-- if countLeast == countMost then
	-- return peg.most(peg.least(patt, countLeast), countMost)
	-- end
	local ret = patt
	for _ = 2, countLeast do
		ret = ret * patt
	end
	if countMost > countLeast then
		return ret * peg.most(patt, countMost - countLeast)
	end
	return ret
end

function peg.patternAntiPattern(patt, countLeast, countMost)
	local p = toPattern(patt)
	local notP
	if countLeast and countMost then
		-- notP = peg.leastMost(peg.other(p), countLeast, countMost)
		p = peg.leastMost(p, countLeast, countMost)
	elseif countLeast then
		-- notP = peg.least(peg.other(p), countLeast)
		p = peg.least(p, countLeast)
	elseif countMost then
		-- notP = peg.most(peg.other(p), countMost)
		p = peg.most(p, countMost)
		-- else
		-- notP = peg.other(p)
	end
	if countLeast and countLeast > 0 then
		notP = peg.least(peg.other(p), countLeast)
	else
		notP = peg.least(peg.other(p), 1)
	end
	return p, notP
end

local iterateLines_line = (1 - pattTbl.endOfLine) ^ 0
local endOfLine = pattTbl.endOfLine
function peg.iterateLines(str, func)
	local lines = ((iterateLines_line / func) * endOfLine) ^ 0 * (iterateLines_line / func)
	-- local lines = ((iterateLines_line / func) * patt.endOfLine)^0 * (iterateLines_line / func) -- aga: patt.endOfLine is nil here, WHY?
	return match(lines, str)
end

function peg.iterateSeparators(str, sep, func)
	local iterateSeparator_sep = toPattern(sep)
	local iterateSeparator_part = (1 - iterateSeparator_sep) ^ 0
	local parts = ((iterateSeparator_part / func) * iterateSeparator_sep) ^ 0 * (iterateSeparator_part / func)
	return match(parts, str)
end

function peg.iterateSeparatorsCapture(str, sep, func)
	local iterateSeparator_sep = toPattern(sep)
	local iterateSeparator_part = (1 - iterateSeparator_sep) ^ 0
	local parts = (C(iterateSeparator_part) * C(iterateSeparator_sep) / func) ^ 0 * (iterateSeparator_part / func)
	return match(parts, str)
end

function peg.iterateCapture(str, patt, func)
	local iterateSeparator_sep = toPattern(patt)
	local iterateSeparator_part = (1 - iterateSeparator_sep) ^ 0
	local parts = (C(iterateSeparator_part) * C(iterateSeparator_sep) / func) ^ 0 * (iterateSeparator_part / func)
	return match(parts, str)
end

function peg.countSeparators(str, sep)
	if str == nil then
		printError("peg count separators string to find is empty")
		return 0
	end
	local sepCount = 0
	local function func(txt)
		if txt ~= "" then
			sepCount = sepCount + 1
		end
	end
	peg.iterateSeparators(str, sep, func)
	return sepCount - 1
end

function peg.splitToArray(s, sep)
	local sepC = P(sep)
	-- if not match(sepC, s) then
	-- 	return {s}
	-- end
	local elem = C(peg.least(peg.other(sepC), 0)) -- C((1 - sepC)^0) -- C(peg.least(peg.other(sepC), 0))
	local p = Ct(elem * peg.least(sepC * elem, 0)) -- Ct(elem * (sepC * elem)^0)   -- make a table capture
	return match(p, s)
end

function peg.stringToTable(str)
	-- TODO: do this witg lpeg, is much faster
	if not str then
		return nil
	end
	-- local split = Ct(C(patt.anychar^1))
	-- return match(split, str)
	local ret = {}
	for i = 1, #str do
		ret[i] = str:sub(i, i)
	end
	return ret
end

--[[
local function times(p, n)
	return n == 1 and p or p * times(p, n - 1)
end
]]

function peg.trimFunction(patt)
	-- see: http://lua-users.org/wiki/StringTrim
	if patt == nil then
		patt = S(" \t\n\v\f\r")
	elseif type(patt) ~= "userdata" then
		patt = S(patt)
	end
	local nospace = 1 - patt
	local ptrim = patt ^ 0 * C((patt ^ 0 * nospace ^ 1) ^ 0)
	return function(str)
		return match(ptrim, str)
	end
end

function peg.replace(str, from, repl)
	local patt
	if type(from) == "string" then
		patt = P(from)
	elseif type(from) == "userdata" then
		patt = from
	else
		printError("peg.replace from value '%s' type is not a string or uservalue", tostring(from))
		return str
	end
	patt = Cs((patt / repl + 1) ^ 0)
	return match(patt, str)
end

function peg.replaceOnce(s, from, repl)
	local patt
	if type(from) ~= "userdata" then
		patt = P(from)
	else
		patt = from
	end
	patt = Cs(patt / repl + 1)
	return match(patt, s)
end

function peg.replaceOnce(str, from, repl)
	if peg.found(str, from) == false then
		return str
	end
	local part1 = peg.parseBeforeWithDivider(str, from)
	local part2 = peg.parseAfter(str, from)
	part1 = peg.replace(part1, from, repl)
	return part1 .. part2
end

function peg.removePattern(s, patt)
	-- patt = P(patt)
	patt = Cs((patt / "" + 1) ^ 0)
	return match(patt, s)
end

function peg.find(str, pattern)
	if str == "" or not pattern or pattern == "" then
		return 0
	end
	if type(str) ~= "string" then
		printError("peg.find value '%s' type '%s' is not a string", tostring(str), type(str))
		-- str = tostring(str)
		return 0
	end
	local find
	if type(pattern) ~= "userdata" then
		find = P(pattern)
	else
		find = pattern
	end
	local notFind = (1 - find)
	local patt = notFind ^ 0
	local pos = patt:match(str)
	if pos > #str then
		return 0
	end
	return pos
end

function peg.found(str, pattern)
	return peg.find(str, pattern) > 0
end

function peg.foundNumber(str)
	return peg.find(str, digit) > 0
end

function peg.findFromEnd(str, pattern)
	if type(pattern) == "string" then
		pattern = pattern:reverse()
	end
	if type(str) ~= "string" then
		printError("peg.findFromEnd parameter 1 is not a string")
		return 0
	end
	local pos = peg.find(str:reverse(), pattern)
	if pos > 0 then
		return #str - pos + 1
	end
	return pos
end

function peg.startsWith(str, pattern)
	local pos = peg.find(str, pattern)
	if pos == 1 then
		return true
	end
	return false
end

function peg.endsWith(str, pattern)
	if type(pattern) == "string" then
		pattern = pattern:reverse()
	end
	local pos = peg.find(str:reverse(), pattern)
	if pos == 1 then
		return true
	end
	return false
end

function peg.addToEnd(str, addStr)
	if not peg.endsWith(str, addStr) then
		return str .. addStr
	end
	return str
end

function peg.splitLast(str, divider)
	local pos = peg.findFromEnd(str, divider)
	if pos > 0 then
		return str:sub(1, pos - #divider), str:sub(pos + 1)
	end
	return str, ""
end

function peg.parseLast(str, divider, returnWithoutDivider)
	local a, b = peg.splitLast(str, divider)
	if b == "" and returnWithoutDivider then
		return a
	end
	return b
end

function peg.parseLastWithDivider(str, divider)
	local a, b = peg.splitLast(str, divider)
	if b == "" then
		return a
	end
	return divider .. b
end

function peg.removeFromStart(str, remove)
	local pos = 1
	local pos2 = #remove
	while str:sub(pos, pos2) == remove do
		pos = pos + #remove
		pos2 = pos2 + #remove
	end
	return str:sub(pos)
end

function peg.removeFromEnd(str, remove)
	while str:sub(-(#remove)) == remove do
		str = str:sub(1, -(#remove + 1))
	end
	return str
end

function peg.removeFromStartEnd(str, remove)
	str = peg.removeFromStart(str, remove)
	return peg.removeFromEnd(str, remove)
end

local space2
function peg.cleanBeforeAfterDoubleSpace(str)
	if space2 == nil then
		space2 = toPattern("  ")
	end
	str = peg.removeFromStartEnd(str, " ")
	while peg.found(str, space2) do
		str = peg.replace(str, space2, " ")
	end
	return str
end

function peg.parseBefore(str, divider)
	local pos = peg.find(str, divider)
	if pos > 0 then
		return str:sub(1, pos - 1)
	end
	return str
end

function peg.parseBeforeLast(str, divider)
	local pos = peg.findFromEnd(str, divider)
	if pos > 0 then
		return str:sub(1, pos - 1)
	end
	return str
end

function peg.parseBeforeWithDivider(str, divider)
	local pos = peg.find(str, divider)
	if pos > 0 then
		return str:sub(1, pos + #divider - 1)
	end
	return str
end

function peg.parseAfterLast(str, divider)
	local pos = peg.findFromEnd(str, divider)
	if pos > 0 then
		return str:sub(pos + #divider)
	end
	return str
end

function peg.parseBeforeLast(str, divider)
	local pos = peg.findFromEnd(str, divider)
	if pos > 0 then
		return str:sub(1, pos - 1)
	end
	return str
end

function peg.parseAfter(str, divider)
	local pos = peg.find(str, divider)
	if pos > 0 then
		return str:sub(pos + #divider)
	end
	return str
end

function peg.parseAfterStart(str, divider)
	if str:sub(1, #divider) == divider then
		return str:sub(#divider + 1)
	end
	return str
end

function peg.parseAfterWithDivider(str, divider)
	local pos = peg.find(str, divider)
	if pos > 0 then
		return str:sub(pos)
	end
	return str
end

function peg.parse(str, divider, num)
	if num == 0 then
		return nil
	end
	if num < 0 then
		str = str:reverse()
		divider = divider:reverse()
	end
	if type(str) ~= "string" then
		printError("peg.parse first parameter type '%s' is not a string", type(str))
		return nil
	end
	local ret = peg.splitToArray(str, divider)
	if num < 0 then
		num = -num
		if #ret >= num and type(ret[num]) == "string" then
			return ret[num]:reverse()
		end
	else
		if #ret >= num then
			return ret[num]
		end
	end
	return nil
	--[[
	local count = 1
	local pos = peg.find(str, divider)
	local prevStart = 1
	while pos > prevStart and count < num do
		count = count + 1
		prevStart = pos + 1
		pos = peg.find(str:sub(prevStart), divider)
		if pos > 0 then
			pos = pos + prevStart - 1
		end
	end
	if count == num and pos then
		return str:sub(prevStart, pos - 1)
	end
	return nil
	]]
	--[[
	local mark = lpeg.P(divider)
	local notMark = (1 - mark)

	if num == 1 then
		local patt = lpeg.C(notMark^0)
		local matched = patt:match(str)
		return matched
	end

	local patt = ((notMark^0 * mark)^-(num-1)) * lpeg.C(notMark^0)
	--local patt = lpeg.C((notMark^0 * mark)^-(num-1)) --* lpeg.C(notMark^0)
	local matched = patt:match(str)
	return matched
	]]
end

function peg.countOccurrence(str, charToFind)
	local count = 0
	if type(str) ~= "string" then
		printError("text to peg.countOccurrence parameter 1 type '%s' is not a string", type(str))
	elseif type(charToFind) ~= "string" then
		printError("character to peg.countOccurrence find type '%s' is not a string", type(charToFind))
	elseif #charToFind < 1 then
		printError("character to peg.countOccurrence find length is less than 1", charToFind:sub(1, 10))
	else
		local forward = #charToFind - 1
		for i = 1, #str do
			if str:sub(i, i + forward) == charToFind then
				count = count + 1
			end
		end
	end
	return count
end

function peg.countOccurrenceFromTable(tbl, key, charToFind)
	local count = 0
	if type(tbl) ~= "table" then
		printError("text to countOccurrenceFromTable parameter 1 type '%s' is not a table", type(tbl))
	elseif type(tbl[key]) ~= "string" then
		printError("text to countOccurrenceFromTable table key '%s' type '%s' is not a string", tostring(key), type(tbl[key]))
	elseif type(charToFind) ~= "string" then
		printError("character to countOccurrenceFromTable find type '%s' is not a string", type(charToFind))
	elseif #charToFind < 1 then
		printError("character to countOccurrenceFromTable find length is less than 1", charToFind:sub(1, 10))
	else
		local forward = #charToFind - 1
		for i = 1, #tbl[key] do
			if tbl[key]:sub(i, i + forward) == charToFind then
				count = count + 1
			end
		end
	end
	return count
end

function peg.replaceBetweenWithDelimiter(source, startPatt, endPatt, newText)
	local notEndPatt
	endPatt, notEndPatt = peg.patternAntiPattern(endPatt)
	local patt = toPattern(startPatt) * peg.least(notEndPatt, 0) * endPatt
	return peg.replace(source, patt, newText)
end

function peg.parseBetweenWithDelimiter(source, startText, endText)
	local posStart = peg.find(source, startText)
	if posStart > 0 then
		local str = source:sub(posStart + #startText)
		local posEnd = peg.find(str, endText)
		if posEnd > 0 then
			str = startText .. str:sub(1, posEnd + #endText - 1)
			return str, posStart, posStart + #startText + posEnd + #endText - 1
		end
	end
	return nil
	--[[
		local notEndPatt
		endPatt, notEndPatt = peg.patternAntiPattern(endPatt)
		local patt = toPattern(startPatt) * peg.least(notEndPatt, 0) * endPatt
		return lpeg.match(Cs(patt), source)
	]]
end

function peg.parseBetweenWithoutDelimiter(source, startText, endText, returnIfNoEndWasFound)
	local posStart = peg.find(source, startText)
	if posStart > 0 then
		local str = source:sub(posStart + string.len(startText))
		local posEnd = peg.find(str, endText)
		if posEnd > 0 then
			str = str:sub(1, posEnd - 1)
			return str
		end
		if returnIfNoEndWasFound then
			return str
		end
	end
	return nil
end

function peg.matchString(str, patt)
	local patt2 = toPattern(patt)
	return match(patt2, str)
end

function peg.substitute(openp, repl, endp)
	-- http://lua-users.org/wiki/LpegTutorial
	openp = P(openp)
	endp = endp and P(endp) or openp
	local upto_endp = (1 - endp) ^ 1
	return openp * C(upto_endp) / repl * endp
end

function peg.replaceFirst(str, from, repl)
	local pos = peg.find(str, from)
	if pos > 0 then
		return str:sub(1, pos - 1) .. repl .. str:sub(pos + #from)
	end
	return str
end

--[[
function peg.replaceUppercase(str, from, repl) --, action)
	local arr = peg.splitToArray(str, from)
	if pos > 0 then
		return str:sub(1, pos - 1)..repl..str:sub(pos + #from)
	end
	return str
end
]]

function peg.split(str, divider)
	local pos = peg.find(str, divider)
	if pos > 0 then
		return str:sub(1, pos - 1), str:sub(pos + #divider)
	end
	return str, ""
end

function peg.splitWithDivider(str, divider)
	local pos = peg.find(str, divider)
	if pos > 0 then
		return str:sub(1, pos - 1), str:sub(pos)
	end
	return str, ""
end

function peg.toNumber(source)
	if type(source) == "number" then
		return source
	elseif type(source) == "string" then
		return tonumber(peg.replace(source, ",", "."))
	end
	return nil
end

function peg.parseFirstNumber(source)
	if type(source) == "string" then
		local newStr = ""
		local prevChar = 0
		local startPos = 0
		local char
		for i = 1, #source do
			char = source:sub(i, i)
			if char >= "0" and char <= "9" and i == prevChar + 1 then
				if startPos == 0 then
					startPos = i
				end
				newStr = newStr .. char
				prevChar = i
			elseif newStr == "" then
				prevChar = i
			end
		end
		return peg.toNumber(newStr), startPos
	end
	return peg.toNumber(source)
end

function peg.concat(...)
	local arg = {...}
	return table.concat(arg)
end

function peg.concatWith(separator, ...)
	local param = {...}
	if type(param) ~= "table" then
		loadRequire()
		return nil, l("concatWith parameter 2 '%s' is not a table", tostring(param))
	end
	if #param < 1 then
		return "" -- no data to concat
		-- loadRequire()
		-- return nil, l("concatWith needs at least 2 parameters", param)
	end
	local ret = ""
	for i = 1, #param do
		if param[i] ~= "" and param[i] ~= nil then
			if ret == "" then
				ret = ret .. tostring(param[i])
			else
				ret = ret .. tostring(separator) .. tostring(param[i])
			end
		end
	end
	return ret
end

function peg.upper(txt)
	if type(txt) ~= "string" then
		printError("peg.upper value '%s' type '%s' is not a string", tostring(txt), type(txt))
		txt = tostring(txt)
	end
	return utf8.upper(txt)
end

function peg.lower(txt)
	if type(txt) ~= "string" then
		printError("peg.lower value '%s' type '%s' is not a string", tostring(txt), type(txt))
		txt = tostring(txt)
	end
	return utf8.lower(txt)
	--[[
	local ret = string.lower(txt) -- txt:lower()
	ret = peg.replace(ret, "Å", "å")
	ret = peg.replace(ret, "Ä", "ä")
	return peg.replace(ret, "Ö", "ö")
	]]
	--[[
	local ffi = require "mffi"
	ffi.cdef[=[
		char *setlocale(int category, const char *locale);
	]=]
	local C = ffi.C
	local LC_ALL = 0

	local locale = os.setlocale()
	if util.isWin() then
		os.setlocale("Finnish_Finland.1252") -- TODO: set current locale
	else
		os.setlocale("fi_FI")
	end
  local ret = C.setlocale(LC_ALL, "fi_FI")
	os.setlocale("fi_FI")
	]]
end

function peg.replaceFromStart(str, replaceFrom, replaceTo)
	if peg.startsWith(str, replaceFrom) then
		return replaceTo .. peg.removeFromStart(str, replaceFrom)
	end
	return str
end

local pattUpperSeparator = peg.set("-_ ")
function peg.toFirstUpperCase(str)
	local ret
	peg.iterateSeparators(str, pattUpperSeparator, function(start)
		if ret == nil then
			ret = start:sub(1, 1):upper() .. start:sub(2)
		else
			ret = ret .. " " .. start
		end
	end)
	return ret
end

local pattDash = peg.pattern("-")
function peg.toPascalCase(str) -- UserAccount is a PascalCase
	local ret
	peg.iterateSeparators(str, pattDash, function(start)
		ret = (ret or "") .. start:sub(1, 1):upper() .. start:sub(2)
	end)
	return ret
end

function peg.toCamelCase(str) -- userAccount is a camelCase
	local ret = peg.toPascalCase(str)
	return ret:sub(1, 1):lower() .. ret:sub(2)
end

local pattUpper = peg.range("AZ")
function peg.toKebabCase(str) -- user-account is a kebab-case
	local ret
	peg.iterateSeparatorsCapture(str, pattUpper, function(start, separator)
		ret = (ret or "") .. start .. (separator and "-" .. separator:lower() or "")
	end)
	return ret
end

return peg
