--- printer.lua
--
--

-- https://social.msdn.microsoft.com/Forums/en-US/7ed3dd99-6677-4d43-9ce2-6f119b835e93/printing-directly-to-a-printer-without-a-printDataUnix-dialog?forum=netfxbcl
local printer = {}

local util = require "util"
local l = require "lang".l
local ffi = require "mffi"
local C = ffi.C
local spool

local cups
local function cstr(str)
	return ffi.cast("char*", str) -- "const char*" is better but does not work in windows
end

local function loadCups()
	if not cups then
		if util.isMac() then
			cups = util.loadDll("libcups.dylib")
		else
			cups = util.loadDll("/usr/lib/x86_64-linux-gnu/libcups.so.2")
		end
	end
end

local function loadDll()
  if not spool then
    spool = util.loadDll("winspool.drv")
  end
end

local function getPrinterListWin(option, findName)
	loadDll()
	local ret = {}
	local level = ffi.newNoAnchor("DWORD", 2)
	local size = ffi.newNoAnchor("DWORD[1]", 0)
	local count = ffi.newNoAnchor("DWORD[1]", 0)
	spool.EnumPrintersA(bit.bor(C.PRINTER_ENUM_LOCAL, C.PRINTER_ENUM_CONNECTIONS), nil, level, nil, 0, size, count)
	local list = ffi.newNoAnchor("uint8_t[?]", size[0]) --size[0] C.malloc(size))
	if list then
		list = ffi.cast("PRINTER_INFO_2A*", list)
		if spool.EnumPrintersA(bit.bor(C.PRINTER_ENUM_LOCAL, C.PRINTER_ENUM_CONNECTIONS), nil, level, ffi.cast("LPBYTE", list), size[0], size, count) then
			-- print("Number of installed printers on this pc: ", count[0])
			for i = 0, count[0] - 1 do
				local printerName = ffi.string(list[i].pPrinterName)
				if option == "exists" and printerName == findName then
					return {printerName}
				end
				--if (list[i].Attributes  & PRINTER_ATTRIBUTE_NETWORK
				--	print(" network printer: ", printerName)
				ret[i + 1] = printerName
			end
		end
		-- C.free(list)
	end
	return ret
end

local function getPrinterListUnix(option, findName)
	loadCups()
	local ret = {}
  local dests = ffi.newNoAnchor("cups_dest_t*[1]") -- *dests, *dest;
  local numDests = cups.cupsGetDests(dests)
  for i = 0, numDests - 1 do
		local dest = dests[0][i]
		local printerName = ffi.string(dest.name)
		if option == "exists" and printerName == findName then
			return {printerName}
		end
		if dest.is_default == 1 then
			table.insert(ret, 1, printerName)
			if option == "only-default" then
				break
			end
    elseif option ~= "only-default" then
			ret[i + 1] = printerName
		end
  end
	cups.cupsFreeDests(numDests, dests[0])
	return ret
end

function printer.getPrinterList()
	if util.isWin() then
		return getPrinterListWin()
	end
	return getPrinterListUnix()
end

function printer.exists(name)
	local ret
	if util.isWin() then
		ret = getPrinterListWin("exists", name)
	else
		ret = getPrinterListUnix("exists", name)
	end
	return ret[1] == name
end

function printer.getDefaultPrinter()
	if util.isWin() then
		loadDll()
		local size = ffi.newNoAnchor("DWORD[1]", 0)
		spool.GetDefaultPrinterA(nil, size)
		local buffer = ffi.newNoAnchor("uint8_t[?]", size[0])
		local ok = spool.GetDefaultPrinterA(buffer, size)
		if ok == 0 then
			return ""
		end
		return ffi.string(buffer, size[0] - 1)
	else
		local dests = getPrinterListUnix("only-default")
		return dests[1] or ""
	end
end

function printer.setDefaultPrinter(name)
	-- returns true if set is ok (in unix printer was found), false if printer was not found or set failed
	if util.isWin() then
		loadDll()
		local ok = spool.SetDefaultPrinterA(cstr(name))
		if ok == 0 then
			return false
		end
		return true
	else
		loadCups()
		local dests = ffi.newNoAnchor("cups_dest_t*[1]")
		local numDests = cups.cupsGetDests(dests)
		local currentDefault
		local newDest
		for i = 0, numDests - 1 do
			local dest = dests[0][i]
			if dest.is_default == 1 then
				currentDefault = ffi.string(dest.name)
			end
			if ffi.string(dest.name) == name then
				if dest.is_default == 1 then
					break -- already default, do nothing
				end
				newDest = dest
				dests[0][i].is_default = 1
			else
				dests[0][i].is_default = 0
			end
		end
		if newDest and name ~= currentDefault then
			-- cups.cupsSetDefaultDest(cstr(name), nil, 1, newDest) -- this does not work in osx
			cups.cupsSetDests(numDests, dests[0])
		end
		cups.cupsFreeDests(numDests, dests[0])
		return newDest ~= nil and name -- false or new printer name
		--[[
		I got it working. The way I did is, i set isDefault property to 1 of the printer and then calls cupsSetDest
function again to set the printer with new changes.
		]]
	end
end

local function printDataWin(printerName, lpData, docName, copies)
  -- exampe code from: https://support.microsoft.com/en-us/kb/138594
  --[[ http://www.screenio.com/gui_screenio/gs_htmlhelp_subweb/advanced/windows-apis_printapis.htm
    You can do many WritePrinter calls on any given page.
    In fact, you can send the entire document (in multiple WritePrinter calls) within a single
    set of StartPagePrinter - EndPagePrinter calls, because you can insert your own
    pagination (hex 0C) codes in the print.  EndPagePrinter serves mostly to allow windows
    to start printing the page if the user set the printer to "Start Printing Immediately"
    in the printer settings.
  ]]
  if not spool then
    spool = util.loadDll("winspool.drv")
  end
  local err
  local hPrinter = ffi.newNoAnchor("HANDLE[1]")
  local docInfo = ffi.newNoAnchor("DOC_INFO_1[1]")
  local dwJob -- is return value, dwJob = ffi.newNoAnchor("DWORD")
  local dwBytesWritten = ffi.newNoAnchor("DWORD[1]")

  local ok = false
  -- Need a handle to the printer.
  printerName = cstr(printerName)
  local ret = spool.OpenPrinterA(printerName, hPrinter, nil) -- ffi.cast("CHAR*", printerName)
  if ret ~= 0 then
    hPrinter = hPrinter[0]
    -- Fill in the structure with info about this "document."
    docInfo[0].pDocName = cstr(docName or "Document")
    docInfo[0].pOutputFile = nil
    docInfo[0].pDatatype = cstr("RAW")
    -- Inform the spooler the document is beginning.
    dwJob = spool.StartDocPrinterA(hPrinter, 1, ffi.cast("LPSTR", docInfo)) -- job number
    if dwJob ~= 0 then
      local dwCount = #lpData
      local lpDataC = cstr(lpData)
      for _ = 1, copies do
        -- we want to loop StartPagePrinter because we want printer to start printin as soon as possible
        -- EndPagePrinter will start printing if "Start Printing Immediately"  setting is on
        -- print(docName..", copy: "..i.."/"..copies)
        -- Start a page.
        if spool.StartPagePrinter(hPrinter) ~= 0 then
          ok = false -- many loops, reset for every loop to false
          -- Send the data to the printer.
          if spool.WritePrinter(hPrinter, lpDataC, dwCount, dwBytesWritten) ~= 0 then
            if dwBytesWritten[0] == dwCount then
              ok = true
            end
          end
          -- End the page.
          spool.EndPagePrinter(hPrinter)
          if ok ~= true then
            -- print(docName..", copy - "..i.." print failed")
            break
          end
        end
      end
      -- Inform the spooler that the document is ending.
      spool.EndDocPrinter(hPrinter)
    end
    -- Tidy up the printer handle.
    spool.ClosePrinter(hPrinter)
    -- Check to see if correct number of bytes were written.
  end
  if ok == false then
    local errNum = C.GetLastError()-- GetLastWin32Error();
    err = util.winErrorText(errNum)
    return nil, err
  end
  return ok
end


local function printDataUnix(printerName, lpData, docName, copies)
	loadCups()
  -- http://www.maclife.com/article/columns/terminal_101_printing_command_line
  -- https://developer.apple.com/library/mac/documentation/Darwin/Reference/ManPages/man1/lp.1.html

	--[[
	local ok, err, tmpDoc
	tmpDoc = util.tempDirPath()
  tmpDoc = tmpDoc.."temp_print_"..peg.replace(docName, " ", "_") -- ..".pdf"
  tmpDoc = util.filePathFix(tmpDoc)
  util.writeFile(tmpDoc, lpData)
  local cmd = 'lp -d "'..printerName..'" -n '..copies.." "..tmpDoc
  ok,err = os.execute(cmd)
	os.remove(tmpDoc) -- we assume that it is safe to delete the temp doc now - job should be in print queue
  -- local ok2,err2 = os.remove(tmpDoc)
  if ok ~= true then
    return nil, err
  end
  return ok
	--]]
	if printerName == "default" then
		local default = cups.cupsGetDefault()
		if default ~= nil then
			printerName = ffi.string(default)
		else
			local arr = printer.getPrinterList()
			if #arr > 0 then
				printerName = arr[1]
			else
				util.printError("there are no printers installed")
			end
		end
	end
	local err
	local numOptions = 0
	local options = ffi.newNoAnchor("cups_option_t*[1]")
	local optionsC = ffi.cast("cups_option_t**", options)
	numOptions = cups.cupsAddOption(cstr("copies"), cstr(tostring(copies)), numOptions, optionsC)
	-- ok = cups.cupsPrintFile(printerName, cstr(tmpDoc), cstr("nc"), numOptions, options[0])
	local CUPS_HTTP_DEFAULT = nil -- ffi.cast("void*", 0)
	local CUPS_FORMAT_AUTO	= cstr("application/octet-stream")
	local printerNameC = cstr(printerName)
	local jobId = cups.cupsCreateJob(CUPS_HTTP_DEFAULT, printerNameC, cstr(docName), numOptions, options[0])
	-- HTTP_STATUS_CONTINUE = 100,		/* Everything OK, keep going... */
	-- HTTP_STATUS_NONE = 0
	if jobId == 0 then
		err = l("creating a print job for document '%s' to printer '%s' failed", docName, printerName)
	else
		err = cups.cupsStartDocument(CUPS_HTTP_DEFAULT, printerNameC, jobId, cstr(docName), CUPS_FORMAT_AUTO, 1) -- 1 == last document
	end
	if err == 100 or err == 0 then
		err = cups.cupsWriteRequestData(CUPS_HTTP_DEFAULT, cstr(lpData), #lpData)
	else
		err = l("starting print document '%s' to printer '%s' failed", docName, printerName)
	end
	if err == 100 or err == 0 then
		err = cups.cupsFinishDocument(CUPS_HTTP_DEFAULT, printerNameC)
	else
		err = l("writing document '%s' data to printer '%s' failed", docName, printerName)
	end
	if not(err == 100 or err == 0) then
		err = l("finishing document '%s' to printer '%s' failed", docName, printerName)
	end
	if numOptions > 0 then
		cups.cupsFreeOptions(numOptions, options[0])
	end
	if err == 100 or err == 0 then
    return true
  end
  return nil, l("printing error '%s'", err)
end

function printer.printData(printerName, lpData, docName, copies)
  if not copies then
    copies = 1
  end
  if util.isWin() then
    return printDataWin(printerName, lpData, docName, copies)
  else -- if util.isMac() then
    return printDataUnix(printerName, lpData, docName, copies)
--  else
--    local err = l("linux printing has not been done yet")
--    print(err)
--    return nil,err
  end
end

if not ... then
	util.printTable(printer.getPrinterList(), "printer.getPrinterList()")
  local data = util.readFile("report/pallet_label_ferrum1.pdf", "binary")
  local ret
  local time = util.seconds()
  local copies = 1
	local printerName = "default"
  if data and #data > 0 then
		ret = printer.printData(printerName, data, "pallet_label_ferrum1.pdf", copies)
    -- ret = printer.printData("HP LaserJet 4100 Series PS Class Driver", data, "pallet_label_ferrum1.pdf", copies)
    -- ret = printer.printData("HP Universal Printing PCL 6", data, "pallet_label_ferrum1.pdf")
    -- ret = printer.printData("HP Universal Printing PCL 6", data, "pallet_label_ferrum1.pdf")
    -- ret = printer.printData("Bullzip PDF Printer", data, "pallet_label_ferrum1.pdf")
    -- ret = printer.printData("Microsoft XPS Document Writer", data, "pallet_label_ferrum1.pdf")
  end
  time = util.seconds(time)
  util.print("\nprint to printer '%s', time: %d, data size: %d, return value: %s", printerName, time, #data, tostring(ret))
end

return printer