--- recdata.lua
--
local recdata = {}

local util = require "util"
local peg = require "peg"
local json = require "json"
local dschema -- require "convert"
local warningChar
local dot, startBracket = peg.toPattern("."), peg.toPattern("[")
local split, splitToArray, parseBefore, parseBetweenWithoutDelimiter = peg.split, peg.splitToArray, peg.parseBefore, peg.parseBetweenWithoutDelimiter

local dotTag, dotData, dotPos, dotPrevPos -- recdata.setDot is optimized to not to create any garbage
local dotPrevPosError
local function getDot(rec, tagName, showError)
	if type(rec) ~= "table" then
		return nil, util.printError("recdata get dot parameter 1 record '%s' is not a table", tostring(rec))
	end
	if rec[tagName] ~= nil then
		return rec[tagName]
	end
	dotData = rec
	dotPos = tagName:find(".", 2, true) -- tag can't start with a dot, optimize find to use 2 as start
	dotPrevPos = 1
	while dotPos ~= nil do
		dotTag = tagName:sub(dotPrevPos, dotPos - 1)
		if dotData[dotTag] ~= nil then -- should we check for type(dotData[dotTag]) == "table"?
			dotData = dotData[dotTag]
			-- elseif dotPrevPos == 1 then
			--   skip first start tag, for example rec = {name} and tagName = "per.name" should return per.name
		elseif dotPrevPos > 1 then
			if dotPrevPosError ~= tagName then
				dotPrevPosError = tagName
				if showError ~= false then
					util.printWarning("recdata get dot tag '%s' was not found from record: '%s'", tostring(tagName), rec)
				end
			end
			return nil -- this prevents accidental errors where last part tag name is in some other structure(?)
		end
		dotPrevPos = dotPos + 1
		dotPos = tagName:find(".", dotPos + 2, true) -- there can't be two dots in a row, optimize find to use dotPos + 2
	end
	return dotData[tagName:sub(dotPrevPos)]
end
recdata.getDot = getDot

function recdata.get(rec, tag, showError)
	if type(tag) == "string" and not tag:find("[", 1, true) then
		return getDot(rec, tag, showError) -- huge optimization for tags with only dots
	end
	if type(rec) ~= "table" then
		return nil, util.printError("recdata get parameter 1 record '%s' is not a table", tostring(rec))
	end
	if type(tag) == "table" and tag.tag then
		tag = tag.tag
	end
	if tag == nil then
		return nil, util.printError("recdata get parameter 1 record '%s', parameter 2 tag is nil", tostring(rec))
	end
	if rec[tag] ~= nil then
		return rec[tag] -- most of the cases when tag does not contain a dot
	end
	--[[ local fldName = dschema.fieldName(tag) -- without prefix
	if fldName ~= nil and rec[fldName] ~= nil then
		return rec[fldName]
	end --]]
	local err, rec2
	local nameArr = splitToArray(tag, dot)
	local data = rec
	local count = #nameArr
	for i = 1, count do
		if data[nameArr[i]] == nil then
			local tagName2 = parseBefore(nameArr[i], startBracket)
			if tagName2 ~= nameArr[i] then -- for example: meta_data[2] in "json_data.meta_data[2].value.street_address"
				if data[tagName2] == nil and tagName2 ~= "" then
					return nil, err
				end
				local tagContent = parseBetweenWithoutDelimiter(nameArr[i], "[", "]") or "" -- parseBetweenWithoutDelimiter() needs string, not peg pattern
				if tagContent:sub(1, 1) ~= "?" and util.isNumber(tagContent) then
					local arrIndex = tonumber(tagContent)
					if arrIndex and arrIndex > 0 then
						if tagName2 ~= "" then
							data = data[tagName2][arrIndex]
						else
							data = data[arrIndex]
						end
						if data == nil then
							return nil, err
						end
					else
						return nil, err
					end
				else
					-- for example: meta_data[?key=='_woo_carrier_agent_meta']
					local key, value = split(tagContent, "==") -- peg.split need string as param 2 because it calculates it's length, not peg pattern
					if key:sub(1, 1) ~= "?" then
						err = util.printError("tag content query key does not start with '?' character, tag: %s", nameArr[i])
					else
						key = key:sub(2)
					end
					if value:sub(1, 1) == "'" and value:sub(-1) == "'" then
						value = value:sub(2, -2)
					elseif util.isNumber(value) then
						value = tonumber(value)
					else
						err = util.printError("tag content query value is not string or number, tag: %s", nameArr[i])
					end
					if not util.isArray(data[tagName2]) then
						if data[tagName2][key] == value then
							data = data[tagName2]
							-- elseif util.isNumber(value) and data[tagName2][key] == tonumber(value) then
							-- data = data[tagName2]
						else
							return nil, err
						end
					else -- is array
						for arrIndex = 1, #data[tagName2] do
							rec2 = data[tagName2][arrIndex]
							if rec2[key] == value then
								data = rec2
								break
								-- elseif valueIsNumber and rec2[key] == tonumber(value) then
								-- data = rec2
								-- break
							elseif arrIndex == #data[tagName2] then
								return nil, err
							end
						end
					end
				end
			elseif i == 1 and count > 2 then
				dschema = dschema or require "dschema"
				local fldName = dschema.fieldName(nameArr[1] .. "." .. nameArr[2]) -- check field name like ord.json_data without parts after json_data
				if fldName == nil then
					return nil, err
				end
				--[[ elseif i == 1 and count == 2 then
				local fldName = dschema.fieldName(tag) -- without prefix
				data = data[fldName]
				return data ]]
			else
				return nil, err
			end
		else
			data = data[nameArr[i]]
		end
	end
	return data
end

local function setDot(rec, tagName, newValue)
	if type(rec) ~= "table" then
		return util.printError("recdata set dot parameter 1 '%s' is not a table", tostring(rec))
	end
	if type(tagName) ~= "string" then
		return util.printError("recdata set dot parameter 2 '%s' is not a string", tostring(tagName))
	end
	if warningChar == nil then
		warningChar = require("convert").warningChar()
	end
	dotData = rec
	dotPos = tagName:find(".", 2, true) -- tag can't start with a dot, optimize find to use 2 as start
	dotPrevPos = 1
	while dotPos ~= nil do
		dotTag = tagName:sub(dotPrevPos, dotPos - 1)
		if dotData[dotTag] == nil or dotData[dotTag] == warningChar then
			dotData[dotTag] = {}
			dotData = dotData[dotTag]
		elseif dotData[dotTag] ~= nil then
			dotData = dotData[dotTag]
		end
		dotPrevPos = dotPos + 1
		dotPos = tagName:find(".", dotPos + 2, true) -- there can't be two dots in a row, optimize find to use dotPos + 2
	end
	if dotData == nil then
		util.printError("record data set tag '%s' is not a valid, new value '%s', record '%s'", tostring(tagName), tostring(newValue), rec)
	elseif type(dotData) ~= "table" then
		util.printError("record data set tag '%s' destination type '%s' is not a table, new value '%s', record '%s'", tostring(tagName), type(dotData), tostring(newValue), rec)
	else
		dotData[tagName:sub(dotPrevPos)] = newValue
	end
end
recdata.setDot = setDot

function recdata.set(rec, tag, newValue) -- , useTag
	if type(tag) == "string" and not tag:find("[", 2, true) then
		return setDot(rec, tag, newValue) -- huge optimization for tags with only dots
	end
	if type(rec) ~= "table" then
		return util.printError("recdata set parameter 1 '%s' is not a table", tostring(rec))
	end
	if type(tag) == "table" and tag.tag then
		tag = tag.tag
	elseif tag == nil then
		return util.printError("recdata set parameter 2 '%s' is not a table", tostring(tag))
	end
	local err
	if warningChar == nil then
		warningChar = require("convert").warningChar()
	end
	-- local tagName = useTag and tag or dschema.fieldName(tag) or tag -- dotted with prefix or not
	local nameArr = splitToArray(tag, dot)
	local data = rec
	local count = #nameArr
	for i = 1, count do
		if i == count then -- this does not work with tags that end with arrays like arr[1], there must be tag after arr like arr[1].tag
			data[nameArr[i]] = newValue
		elseif data[nameArr[i]] == nil or (i < count and data[nameArr[i]] == warningChar) then
			if (i < count and data[nameArr[i]] == warningChar) then
				data[nameArr[i]] = {}
			end
			local tagName2 = parseBefore(nameArr[i], startBracket)
			if tagName2 == nameArr[i] then
				data[nameArr[i]] = {}
				data = data[nameArr[i]]
			else -- for example: meta_data[2] in "json_data.meta_data[2].value.street_address"
				if data[tagName2] == nil then
					data[tagName2] = {}
				end
				data = data[tagName2]
				local tagContent = parseBetweenWithoutDelimiter(nameArr[i], "[", "]")
				if util.isNumber(tagContent) then
					local arrIndex = tonumber(tagContent)
					if arrIndex and arrIndex > 0 then
						if data[arrIndex] == nil then
							data[arrIndex] = {}
						end
						data = data[arrIndex]
					else
						err = util.printError("recdata set tag '%s' is not correct, tag name part '%s' array index '%s' type '%s' is not a number", tostring(tag), nameArr[i], tostring(tagContent), type(tagContent))
						return err
					end
				end
			end
		elseif type(data[nameArr[i]]) ~= "table" then
			if type(data[nameArr[i]]) == "string" then
				local tbl
				tbl, err = json.fromJson(data[nameArr[i]])
				if not err then
					data[nameArr[i]] = tbl
					data = data[nameArr[i]]
				end
			end
			if err then
				err = util.printError("recdata set tag '%s' is not correct, tag name part '%s' type '%s' is not a table", tostring(tag), nameArr[i], type(data[nameArr[i]]))
				return err
			end
		else
			data = data[nameArr[i]]
		end
	end
end
-- recdata.setDot = recdata.set

return recdata

--[[
-- these create garbage and are very slow:

function recdata.setDot1(rec, fieldName, newValue) -- , useTag
	data = rec
	pos = fieldName:find(fieldName, dot)
	repeat
		dotTag = fieldName:sub(1, pos - 1)
		fieldName = fieldName:sub(pos + 1)
		if data[dotTag] == nil then
			data[dotTag] = {}
			data = data[dotTag]
		elseif data[dotTag] ~= nil then
			data = data[dotTag]
		end
		pos = find(fieldName, dot)
	until pos == 0
	data[fieldName] = newValue
end

function recdata.setDot2(rec, fieldName, newValue) -- , useTag
	local data = rec
	local prevTag
	local function dotTagValue(tag)
		if prevTag and data[prevTag] == nil then
			data[prevTag] = {}
			data = data[prevTag]
		elseif data[prevTag] ~= nil then
			data = data[prevTag]
		end
		prevTag = tag
	end
	iterateSeparators(fieldName, dot, dotTagValue)
	data[prevTag] = newValue
end
]]
