--- lib/rest.lua
--
-- rest.lua
-- if not from4d then
local util = require "util"
local json = require "json"
local date = require "dt"
local auth = require "auth"
local lz4 = require "lz4"
local l = require"lang".l
local peg = require "peg"
local server, gzip -- delay load
local pegReplace = peg.replace
local pegParseAfter, pegFind, pegFound = peg.parseAfter, peg.find, peg.found
local skip = util.skip
local dt = require "dt"
local http = require "http"
local fdClient = require "net/fd-client"
local brEncoder = require "brotli/encoder":new()
local brDecoder = require "brotli/decoder":new()
local from4d = util.from4d()
local fromJson, ioWrite, ioFlush = json.fromJson, io.write, io.flush
local websocketConvertFunc -- = calls["/rest/nc/ws"].func
local createTypeAnswer -- forward declaration

local pullBreakpoints = false
if os.getenv("LOCAL_LUA_DEBUGGER_VSCODE") == "1" then
	pullBreakpoints = require("lldebugger").pullBreakpoints or pullBreakpoints -- see: https://github.com/tomblind/local-lua-debugger-vscode/pull/67#issuecomment-1341493005
end
local restCache = {}

local option = util.prf("system/option.json")
local useCache = option.option and option.option.use_cache
if useCache == nil then
	useCache = true
end
--[[ if util.from4d() then
	useCache = false
end ]]
local stopCollectgarbage = option.stop_collectgarbage
if stopCollectgarbage == nil then
	stopCollectgarbage = true
end
if stopCollectgarbage then
	util.printInfo('rest.lua / collectgarbage("stop")')
	collectgarbage("stop")
	-- else
	-- print('rest.lua / use collectgarbage')
end
-- local useCompressDebug = option.compress_debug or false
-- end

local authList = {}
authList["bWFuYWdlYXBwOndlYm1hbmFnZXJpMy0="] = {
	-- manageapp:webm..3-
	"/rest/nc/ping",
	"/rest/nc/echo",
	"/rest/nc/echo2"
	-- "/rest/nc/timer*",
	-- "/rest/nc/gitupdate",
	-- "/rest/nc/distwin",
	-- "/rest/nc/distmacwin",
	-- "/rest/nc/restart",
	-- "/rest/nc/cache/clear",
	-- "/rest/nc/query/data"
}

local tcpCallCount = 0
local prevTcpCallCount = -1
local tcpPollCount = 0
local minCompressSize = 1000 -- 1400
local answerCount = 0
local answerSkip -- tonumber(arg[1]) or 0
local pollAfterSkip
local debugPrintChars = 1500
local debugLevel = 0
local calls = {}
-- local printIncompleteRequest = util.prf("system/option.json").option.print_incomplete_request or true

if from4d then
	answerSkip = 150000
	pollAfterSkip = 8 -- 25 -- 90000 -- how many poll()+yield() loops until server poll loop calls pollAfter()
else
	answerSkip = 150000
	pollAfterSkip = 750 -- 50000 --answerSkip
end

local function setDebugLevel(level)
	debugLevel = level
end

-- rest_answerOk and rest_answerError must be declared before plg.setCallbacks() -call
local brotliCompressOption = {quality = 6}
local function compress(compressData, compression, quality)
	local err
	local dataLen = 0
	if compression == "br" then
		-- print("gzip.gzip: "..#compressData)
		compressData, err = brEncoder:compress(compressData, quality and {quality = quality} or brotliCompressOption)
		if err then
			util.printError("brotli compression error: '%s'", tostring(err))
		end
		dataLen = #compressData
	elseif compression == "gzip" then
		-- print("gzip.gzip: "..#compressData)
		gzip = gzip or require "compress/zlib"
		compressData, dataLen = gzip.compress(compressData, quality or 9) -- CF-ZLIB-9 is the same speed as ZLIB-6, see: https://indico.fnal.gov/event/16264/contributions/36466/attachments/22610/28037/Zstd__LZ4.pdf
	elseif compression == "lz4" then -- => always compress
		-- if useCompressDebug then
		-- time = util.microSeconds()
		compressData, err = lz4.compress(compressData)
		-- time = util.microSeconds(time) / 1000
		if err then
			util.printError("lz4 compression error: '%s'", tostring(err))
		end
		dataLen = compressData and #compressData or 0
		--[[ util.printInfo(
					"lz4.compress: %d / %d = %d%%, time %.4f ms",
					dataLen,
					uncompressedSize,
					math.floor(dataLen / uncompressedSize * 100),
					time
				) ]]
		-- else
	else
		util.printError("unsupported compress '%s' type", tostring(compression))
	end
	return compressData, dataLen
end

local function uncompress(compressData, compression)
	local ret, err
	if compression == "br" then
		ret, err = brDecoder:decompress(compressData) -- , #compressData, uncompressedSize)
	elseif compression == "gzip" then
		gzip = gzip or require "compress/zlib"
		ret, err = gzip.uncompress(compressData) -- , #compressData, uncompressedSize)
		if ret ~= nil then
			err = nil -- err is compressData size
		end
	elseif compression == "lz4" then
		ret, err = lz4.decompress(compressData) -- , #compressData, uncompressedSize)
	else
		err = l("unsupported uncompress '%s' type", tostring(compression))
	end
	return ret, err
end

local function answerOk(retTbl)
	retTbl = retTbl or {}
	retTbl.status = "ok"
	return retTbl
end

---@return table
local function answerError(txt, ...)
	if txt == nil then
		txt = "missing error text in rest.answerError"
	end
	if ... then
		txt = l(txt, ...)
	end
	if type(txt) == "table" then
		if txt.error then
			txt = json.toJsonRaw(txt)
			return {error = txt} -- , txt
		end
		txt = json.toJsonRaw(txt)
	end
	return {error = txt} -- , txt -- '{"error": "'..txt..'"}'
end

local function requestType(sock)
	sock.is_options_call = nil
	local startPos = sock.header:find("GET ", 1, true)
	if startPos then
		return "GET"
	else
		startPos = sock.header:find("POST ", 1, true)
		if startPos then
			return "POST"
		else
			startPos = sock.header:find("OPTIONS ", 1, true)
			if startPos then
				-- end
				sock.is_options_call = true
				--[[if sock.header:find("Access-Control-Request-Method: GET", 1, true) then
					return "GET"
				elseif sock.header:find("Access-Control-Request-Method: POST", 1, true) then
					return "POST"
				else]]
				return "OPTIONS"
			else
				startPos = sock.header:find("PUT ", 1, true)
				if startPos then
					return "PUT"
				else
					startPos = sock.header:find("DELETE ", 1, true)
					if startPos then
						return "DELETE"
					end
				end
			end
		end
	end
	return "ERROR"
end

local function parseHeaderField(txt, tag, start)
	-- will return nil if not found
	local startPos, endPos = txt:find(tag, start, true)
	if not startPos then
		startPos, endPos = txt:lower():find(tag:lower(), start, true)
	end
	if startPos then
		local startPos2 = txt:find("\r\n", endPos + 1, true)
		if startPos2 then
			return txt:sub(endPos + 1, startPos2 - 1)
		end
	end
end

-- local httpStartErr = "HTTP/1.1 404 Not Found\nContent-Type: application/json\nContent-Length: "
-- local httpStartAuth = 'HTTP/1.1 401 Unauthorized\nContent-Type: application/json\nWWW-Authenticate: Basic realm="Manage Rest Server"\nContent-Length: '
-- httpStartErr = httpStartErr:gsub("\n", "\r\n")
-- httpStartAuth = httpStartAuth:gsub("\n", "\r\n")

-- X-XSS-Protection, see: https://dev.to/dotnetcoreblog/owasp---who-jck
local httpStartHello = "HTTP/1.1 200 OK\r\nConnection: keep-alive\r\nX-XSS-Protection: 1; mode=block\r\nContent-Length: "
local httpStart
if from4d then
	httpStart = "HTTP/1.1 200 OK\nAccess-Control-Allow-Origin: *\nContent-Type: application/json\nX-XSS-Protection: 1; mode=block\nContent-Length: "
else
	httpStart = "HTTP/1.1 200 OK\nContent-Type: application/json\nX-XSS-Protection: 1; mode=block\nContent-Length: "
end
-- \nAccess-Control-Allow-Origin: http://localhost:8080 works for dev machine, but we need others too
-- old: "HTTP/1.1 200 OK\nAccess-Control-Allow-Origin: *\nAccess-Control-Allow-Headers: Access-Control-Allow-Origin,Authorization,Content-Type,User-Agent\nContent-Type: application/json\nX-XSS-Protection: 1; mode=block\nContent-Length: "
-- \nConnection: keep-alive -- will be overridden by websocket upgrade

httpStart = pegReplace(httpStart, "\n", "\r\n")
local answerTbl = {}
answerTbl[2] = "" -- will contain sock.body length later
answerTbl[3] = "" -- will contain extra headers
-- answerTbl[4] = "Date: " -- ..dt.htmlDate()
answerTbl[4] = "\r\n\r\n"

local prevErrorText
local function createAnswerData(sock, dataParam)
	local data, err
	local answerTxt
	answerTbl[1] = httpStart -- httpStartAuth -- will contain sock.body length later
	if type(dataParam) == "table" and not sock.is_options_call then
		--[[ if dataParam.uncompressed_size then
			-- uncompressedSize = dataParam.uncompressed_size
			data = dataParam.data
		else ]]
		-- dataParam = nil
		-- answerTbl[1] = httpStart
		--[[elseif dataParam == "auth" then
		answerTbl[1] = httpStartAuth
		data = ""]]
		if dataParam.mime then
			answerTbl[1] = pegReplace(httpStart, "application/json", dataParam.mime)
			data = dataParam.data
		elseif dataParam.info and dataParam.info.error then
			if dataParam.error == nil then
				dataParam.error = {}
			end
			if dataParam.error ~= dataParam.info.error then
				if type(dataParam.error) == "string" then
					dataParam.error = {dataParam.error}
				end
				dataParam.error = util.addError(dataParam.error, dataParam.info.error)
			end
			dataParam.info.error = nil
			answerTxt = dataParam.error
		end

		if dataParam.cache_size then
			data = dataParam.data
			--[[ elseif false and dataParam.info and dataParam.data then
			-- force error and info part as first in return
			if dataParam.error then
				answerTxt = json.toJsonRaw(dataParam.error)
				data = '{"error":' .. answerTxt .. ',\n"info":' .. json.toJsonRaw(dataParam.info) .. ',\n"data":' .. json.toJsonRaw(dataParam.data) .. "}"
			else
				answerTxt = json.toJsonRaw(dataParam.info)
				data = '{"info":' .. answerTxt .. ',\n"data":' .. json.toJsonRaw(dataParam.data) .. "}"
			end ]]
		elseif not dataParam.http_header and dataParam.mime == nil then
			data, err = json.toJsonRaw(dataParam)
			if err then -- type(data) ~= "string" then
				if type(err) == "string" then
					answerTxt = l("Error when converting result data to json text. Check json for recursive tags, error: %s", err:sub(1, 500))
				else
					answerTxt = l("error when converting result data to json text")
				end
				data = '{"error":"' .. answerTxt:gsub('"', '\\"') .. '"}'
				util.printWarning(answerTxt)
				answerTxt = "  *** " .. answerTxt:gsub("error", "err") -- if answerTxt starts with "err" or contains "error" then 4d worker will stop!!!
			else
				answerTxt = ""
			end
		end
		-- end
	elseif sock.is_options_call then
		answerTbl[1] = pegReplace(answerTbl[1], "application/json", "text/plain")
		data = ""
	elseif dataParam then
		-- answerTbl[1] = httpStartErr
		data = dataParam
		err = tostring(data)
		data = '{"error":"' .. pegReplace(err, '"', "'") .. '"}'
		if pegFound(err, "authorization") then
			answerTbl[1] = pegReplace(answerTbl[1], "200 OK", "401 Unauthorized")
			-- else
			-- answerTbl[1] = pegReplace(answerTbl[1], "200 OK", "400 Bad Request")
		end
		if err ~= prevErrorText then -- optimize performance tests
			prevErrorText = err
			if pegFound(err, "unknown GET uri") then
				util.printWarning("  *** Rest call error: %s", err:sub(1, 1500))
			else
				util.printWarningWithCallPath("  *** Rest call error: %s", err:sub(1, 1500))
			end
		end
	elseif not sock.is_options_call then
		util.printError("worker call did not return any data parameter")
		-- answerTbl[1] = httpStartErr
		data = ""
	end
	if data == nil then
		data = ""
	elseif type(data) ~= "string" then
		data = tostring(data)
	end
	--[[if data == "" then -- all responses must contain json
		data = '{"error":"unknown error"}'
		answerTbl[1] = pegReplace(answerTbl[1], "200 OK", "400 Bad Request")
	end]]
	local bodyLen = #data
	if debugLevel > 1 and bodyLen < 2 * 1024 then -- 2 kb
		sock.uncompressed_answer_start = data:sub(1, 200)
	end
	--[[ if uncompressedSize then
		answerTbl[3] = "\r\nContent-Encoding: " .. sock.compress .. "\r\nContent-Uncompressed-Length: " .. uncompressedSize -- extra headers
	else ]]
	if dataParam and dataParam.cache_size then
		if dataParam[sock.compress] == nil then
			data, bodyLen = compress(data, sock.compress, 9)
			dataParam[sock.compress] = {data = data, size = bodyLen}
		else
			data = dataParam[sock.compress].data
			bodyLen = dataParam[sock.compress].size
		end
		util.printOk("* returned cache '%s' record '%s', size: %.1f kb, %s compressed size: %.1f kb", sock.uri, dataParam.cache_path, dataParam.cache_size / 1024, sock.compress, bodyLen / 1024)
		answerTbl[3] = "\r\nContent-Encoding: " .. sock.compress .. "\r\nContent-Uncompressed-Length: " .. dataParam.cache_size
	elseif bodyLen >= minCompressSize and sock.compress then
		-- print("do compress: "..tostring(doCompress), bodyLen)
		local uncompressedSize = #data
		data, bodyLen = compress(data, sock.compress)
		answerTbl[3] = "\r\nContent-Encoding: " .. sock.compress .. "\r\nContent-Uncompressed-Length: " .. uncompressedSize -- extra headers
	elseif dataParam and dataParam.http_header then
		-- extra headers like Connection: upgrade -answer
		-- websocket, tls, http2
		answerTbl[3] = "\r\n" .. dataParam.http_header
	else
		answerTbl[3] = "" -- no extra headers
	end
	if dataParam then
		if dataParam.http_status then
			-- answerTbl[1] = pegReplace(answerTbl[1], "Content-Type: application/json\r\n", "")
			answerTbl[1] = pegReplace(answerTbl[1], "200 OK", dataParam.http_status)
		end
		if dataParam.auth then
			dataParam.auth = nil
		end
	end
	if useCache and dataParam and dataParam.cache_key then
		local cachePath = sock.uri
		local cacheRec = restCache[cachePath]
		if cacheRec == nil then
			cacheRec = {cache_key = dataParam.cache_key}
			restCache[cachePath] = cacheRec
		end
		local cacheKey = dataParam[dataParam.cache_key]
		if cacheRec[cacheKey] == nil then
			local data2 = {data = util.clone(dataParam), cache_path = cacheKey}
			data2.data.auth = nil -- do not cache auth
			data2.data.cache_key = nil
			data2.data = json.toJsonRaw(data2.data)
			data2.cache_size = #data2.data
			local compressed, size = compress(data2.data, sock.compress, 9)
			data2[sock.compress] = {data = compressed, size = size}
			cacheRec[cacheKey] = data2
			util.printOk("* added cache '%s' record '%s', data size: %.1f kb", sock.uri, cacheKey, data2.cache_size / 1024)
		end
		-- end
	end
	answerTbl[2] = tostring(bodyLen) -- must be after possible gzip
	--[[if sock.type ~= "ws" then
		answerTbl[3] = answerTbl[3].."\r\nConnection: keep-alive"
	end]]
	-- answerTbl[4] = "Date: "..dt.htmlDate()
	sock.answer_text = answerTxt
	return table.concat(answerTbl) .. data
end

local function basicAuth(sock)
	sock.authenticated = nil
	--[[ do -- fix pm
		sock.authenticated = true
		return true
	end ]]
	if authList[sock.auth] then
		if type(authList[sock.auth]) == "string" then
			if authList[sock.auth] == sock.uri then
				sock.authenticated = true
				return true
			end
			if authList[sock.auth]:sub(-1) == "*" then
				if pegFind(sock.uri, authList[sock.auth]:sub(1, -2)) > 0 then
					sock.authenticated = true
					return true
				end
			end
		elseif type(authList[sock.auth]) == "table" then
			for _, uri in ipairs(authList[sock.auth]) do
				if uri == sock.uri then
					sock.authenticated = true
					return true
				elseif uri:sub(-1) == "*" and pegFind(sock.uri, uri:sub(1, -2)) > 0 then
					sock.authenticated = true
					return true
				end
			end
		end
	end
	return false
end

local function jsonValue(sock, callTbl, uri)
	local tag, func, authValid, authTable
	local param = sock.param
	tag = callTbl.param
	func = callTbl.func
	local err, ret
	if not func then
		return answerError("callback function is nil, call '%s'", tostring(uri))
	elseif not param then
		return answerError("json is missing, call '%s'", tostring(uri))
	end
	if tag and not param[tag] then
		ret, err = answerError("json tag is missing, call '%s', tag: '%s'", tostring(uri), tostring(tag))
	else
		if sock.auth then
			authValid = basicAuth(sock)
			authTable = sock.auth
			if not authValid then
				authValid, authTable = auth.authenticate(sock.uri, param) -- sock.base_uri
			end
		else
			authValid, authTable = auth.authenticate(sock.uri, param)
		end
		-- util.print(" ** REST call err: '%s', authValid: '%s', ret: '%s'\n\n  param: '%s'", tostring(err), tostring(authValid), ret, param)
		if authValid ~= true then
			return authTable
		else
			if authTable then
				param.auth = authTable
			end
			local cacheMainRec
			if useCache and (param == nil or param.refresh ~= true) then
				-- local cachePath = (param and param.base_path or "") .. sock.uri
				cacheMainRec = restCache[sock.uri]
			end
			if cacheMainRec then
				local cacheKey = param[cacheMainRec.cache_key]
				local cacheRec = cacheKey and cacheMainRec[cacheKey]
				if cacheRec then
					-- cacheRec.auth = authTable
					return cacheRec
				end
			end
			if tag then
				local cleanedParam = json.fixNull(param[tag], sock.uri)
				ret, err = func(cleanedParam) -- call function
			else
				-- local cleanedParam = json.fixNull(param, sock.uri)
				-- ret, err = func(cleanedParam)
				ret, err = func(param) -- call real worker function
			end
		end
	end

	if ret ~= nil and type(ret) ~= "table" then
		util.printError(" ** REST RETURN ERROR: return value is not a table, call '%s', return '%s'", tostring(uri), ret or "nil")
		if err == nil then
			err = tostring(ret)
		end
		ret = nil
	end
	if ret == nil and err then
		local errText = tostring(err)
		if type(err) == "table" then
			if type(err.error) == "string" then
				errText = err.error
			elseif type(err.error) == "table" then
				errText = json.toJson(err.error)
			else
				errText = json.toJson(err)
			end
		end
		util.printWarningWithCallPath(" ** REST ERROR: " .. errText)
		ret = {info = {error = errText}}
	elseif ret == nil then
		-- param.auth = param.auth -- ret.auth = param.auth
		local error = util.printError(" ** REST ERROR: return is nil without error, call '%s'", tostring(uri))
		ret = {info = {error = error}}
	else
		if ret.no_auth_return then
			if ret.data and not ret.mime then
				ret = ret.data
			else
				ret.no_auth_return = nil
			end
		elseif not ret.auth then
			if param and type(param.auth) == "table" then
				ret.auth = param.auth
			end
		elseif ret.auth and param.auth and param.auth.auth_token and ret.auth.auth_token ~= param.auth.auth_token then
			if uri ~= "/rest/nc/login" and uri:sub(1, 14) ~= "/rest/nc/auth/" then
				util.printError("return auth token is not equal to parameter auth token, call '%s'", tostring(uri))
				ret.auth = param.auth
			end
		end
		if ret.grid then
			for _, grid in pairs(ret.grid) do
				if type(grid) == "table" and grid.column then
					if grid.option == nil then
						if grid.options then
							grid.option = grid.options
							grid.options = nil -- todo: save fixed
						else
							grid.option = {}
						end
					end
					if grid.option and grid.option.enterable ~= false then
						if not grid.option.grid_option then
							grid.option.grid_option = {}
						end
						local gridOption = grid.option.grid_option
						if not (gridOption.cell_class_rules and gridOption.cell_class_rules["nc-grid-cell-changed"]) then
							if gridOption.cell_class_rules == nil then
								gridOption.cell_class_rules = {}
							end
							gridOption.cell_class_rules["nc-grid-cell-changed"] = "data.changeRecord && data.changeRecord[colDef.field] != null"
						end
						if gridOption.row_class_rules == nil then
							gridOption.row_class_rules = {}
						end
						gridOption.row_class_rules["nc-grid-row-new"] = gridOption.row_class_rules["nc-grid-row-new"] or "data.record_id == ''"
						gridOption.row_class_rules["nc-grid-row-deleted"] = gridOption.row_class_rules["nc-grid-row-deleted"] or "data.changeRecord && data.changeRecord.changeStatus === 'deleted'"
						gridOption.row_class_rules["nc-grid-row-changed"] = gridOption.row_class_rules["nc-grid-row-changed"] or "data.changeRecord != null"
						gridOption.row_class_rules["nc-grid-row-duplicate"] = gridOption.row_class_rules["nc-grid-row-duplicate"] or "data.changeRecord && data.changeRecord.duplicate != null"
					end
				end
			end
		end
	end
	if err then
		ret.error = err
	end
	return ret
end

local function createPostAnswer(sock)
	local data
	local answer = calls[sock.uri]
	if answer == nil then
		-- compatibility with older dbconn version
		sock.uri = pegReplace(sock.uri, "/rest/ma/", "/rest/nc/")
		answer = calls[sock.uri]
	end
	if answer and answer.callType == "POST" then
		if sock.type and sock.data then
			data = createTypeAnswer(sock)
		elseif sock.uri == "/rest/nc/soap" then
			data = answer.func(sock)
		else
			data = jsonValue(sock, answer, sock.uri)
		end
	else
		data = l("unknown POST uri: '%s', socket: %s", tostring(sock.uri):sub(1, 50), sock)
	end
	if type(data) == "table" and data.info and data.error == nil and data.info.error then
		data.error = data.info.error
	end
	return data
end

local function createGetAnswer(sock)
	-- sock.base_uri = pegParseBefore(sock.uri, "?")
	local data
	local answer = calls[sock.uri] -- calls[sock.base_uri]
	if answer and answer.callType == "GET" then
		if sock.uri == "/rest/nc/echo2" then
			data = answer.func(sock)
		elseif sock.uri == "/rest/nc/echo" then
			data = answer.func(sock.uri)
			-- elseif pegStartsWith(sock.base_uri, "/rest/nc/import/") then
			-- 	data = answer.func(sock)
		elseif sock.uri == "/rest/nc/oauth2" then
			data = answer.func(sock)
		elseif sock.type and sock.data then
			data = createTypeAnswer(sock)
		elseif sock.authenticated or sock.auth and basicAuth(sock) then
			data = answer.func(sock.uri) -- sock.uri
		elseif http.headerValueFound(sock.header, "Connection", "Upgrade") then
			-- TODO: call: basicAuth(sock) or other auth
			data = answer.func(sock) -- will set sock.type = "ws" or something else
		else
			data = l("invalid basic authorization for GET uri '%s'", sock.uri)
		end
	else
		if sock and sock.uri then
			data = l("unknown GET uri '%s'", sock.uri)
		else
			data = l("GET uri does not exist")
		end
	end
	if type(data) == "table" and data.info and data.error == nil and data.info.error then
		data.error = data.info.error
	end
	return data
end

createTypeAnswer = function(sock)
	local data
	if not websocketConvertFunc then
		websocketConvertFunc = calls["/rest/nc/ws"].func
	end
	-- local dataFunc = answer.func
	if sock.type == "ws" then
		data = websocketConvertFunc(sock) -- convert from websocket byte data, if 2. parameter is nil == receive
	end
	if sock.type == "ws" and data == "close" then
		-- sock.type = nil
		sock.do_close = true
		sock.raw_data = true
		data = websocketConvertFunc(sock, '{"connection":"close"}')
		return data -- will close
	else
		local err
		sock.data = nil
		sock.header = ""
		if type(data) == "table" then
			sock.body = data
		else
			sock.body, err = fromJson(data)
		end
		if err then
			data = l("socket type '%s' body json is invalid, error '%s'", tostring(sock.type), tostring(err))
		else
			local answer = calls[sock.body.uri]
			if answer then
				sock.uri = sock.body.uri
				sock.request_type = answer.callType
				if sock.request_type == "GET" then
					data = createGetAnswer(sock) -- recursive call
				elseif sock.request_type == "POST" then
					data = createPostAnswer(sock)
					data.time = dt.currentStringMs()
					data = json.toJsonRaw(data) -- json.toJsonRaw(data, "clean")
					if sock.type == "ws" then
						data = websocketConvertFunc(sock, data) -- convert from string to websocket byte data, 2. parameter == send
					end
					sock.raw_data = true
				else
					data = l("unknown socket type '%s' call type (not GET or POST)", tostring(sock.type))
				end
			else
				data = l("unknown socket type '%s' call '%s'", tostring(sock.type), sock.body.uri)
			end
		end
	end
	return data
end

local function createOptionsAnswer()
	-- sock.do_delete = true
	return {http_header = "Access-Control-Expose-Headers: Content-Length\nAccess-Control-Allow-Methods: POST, GET, OPTIONS"}
end

local function parseBody(sock)
	local param, err
	if sock.body and #sock.body >= 2 then
		local jsonData
		if sock.uncompress then
			jsonData, err = uncompress(sock.body, sock.uncompress)
			if err then
				sock.error = l("invalid socket body, call '%s', uncompress failed with error: '%s', socket: %s", tostring(sock.uri), tostring(err), sock)
				return
			end
		end
		param, err = fromJson(jsonData or sock.body)
		if err then
			sock.error = l("invalid json, call '%s', error: '%s', socket: %s", tostring(sock.uri), tostring(err), sock)
			return
		end
		param.base_path = param.pathname
		if param.base_path == nil then
			param.base_path = "/"
		elseif param.base_path ~= "/" then
			if param.base_path:sub(1, 4) == "/nc/" then
				param.base_path = "/nc/"
			else
				param.base_path = "/"
			end
		end
		if param.vue_path and param.vue_path:sub(1, 5) == "node_" then
			param.develop = true --  param.vue_path is node_modules/.vite/deps/vue.js
		end
		sock.param = param
		if param.payload then
			if param.payload.screen and param.payload.title and param.payload.referrer then
				sock.uri = "/api/send" -- we don't know base_path in this call, it's not coming from frontend code
				-- if param.payload.referrer then
				-- param.base_path = param.payload.referrer:sub(2) -- starts with /
				-- param.base_path = peg.parseBefore(param.base_path, "/")
			else
				l("call '%s' body contains payload but not payload.referrer, parameter type '%s', socket: %s", tostring(sock.uri), tostring(param.type), sock)
			end
		end
	else
		sock.error = l("json length is less than 2, call '%s', socket: %s", tostring(sock.uri), sock)
		return
	end
	if param.base_path and param.base_path ~= "/" and sock.uri:sub(1, #param.base_path) == param.base_path then
		sock.uri = sock.uri:sub(#param.base_path) -- base_path like /nc
	end
end

local function createAnswer(sock)
	local data
	util.clearError()
	if sock.httpErr then
		data = sock.httpErr
	elseif sock.request_type == "GET" then
		data = createGetAnswer(sock)
	elseif sock.request_type == "POST" then
		parseBody(sock)
		if sock.error then
			data = {error = sock.error}
		else
			data = createPostAnswer(sock)
		end
	elseif sock.is_options_call then -- or sock.request_type == "OPTIONS"
		data = createOptionsAnswer()
	elseif sock.request_type == "PUT" then
		data = {error = l("unsupported call type PUT")}
	elseif sock.request_type == "DELETE" then
		data = {error = l("unsupported call type DELETE")}
	else
		data = {error = l("unknown call type '%s' (not GET/POST/PUT/DELETE), call '%s'", tostring(sock.request_type):sub(1, 50), tostring(sock.uri):sub(1, 50))}
	end
	-- todo: check if commenting this causes problems?
	if sock.raw_data == nil and type(data) ~= "table" then
		sock.do_close = true -- sock.do_delete error or ws "close" -> close and delete after answer
	end
	if data.info and data.error == nil and data.info.error then
		data.error = data.info.error
	end
	if data.skip_answer then -- or data == "close" then
		return
	elseif sock.raw_data then -- or data == "close" then
		return data
	end
	return createAnswerData(sock, data)
end

local function setSockBody(sock, callData)
	if callData and #callData > 0 then
		sock.body = sock.body .. callData
	end
	if #sock.header > sock.header_length then
		sock.body = sock.header:sub(sock.header_length + 1, sock.header_length + sock.body_length)
		sock.header = sock.header:sub(1, sock.header_length)
	elseif #sock.header < sock.header_length then
		util.printError("request header length %d is bigger than header data length %d, socket: %s", sock.header_length, #sock.header, sock)
	end
	if #sock.body > sock.body_length then
		-- got more data than needed, save for next request in this sock
		sock.next_request = sock.body:sub(sock.body_length + 1)
		sock.body = sock.body:sub(1, sock.body_length)
		sock.incomplete_request = nil
	elseif #sock.body < sock.body_length then
		sock.incomplete_request = true
	elseif sock.incomplete_request then -- here we got exactly what we needed
		sock.incomplete_request = nil
	end
end

local function addToBody(sock)
	if sock.body_length > #sock.body then
		local needBytes = sock.header_length - #sock.header + sock.body_length - #sock.body
		local data, err, dataLen = sock:receive(needBytes)
		if dataLen > 0 then
			sock.body = sock.body .. data
		elseif err then
			util.printError("error when receiving request body data: '%s', socket: %s", tostring(err), sock)
			return false
		end
	end
	if sock.body_length < #sock.body then
		util.printError("request body length %d is bigger than body data length %d, socket: %s", sock.body_length, #sock.body, sock)
		return false
	end
	setSockBody(sock)
	return true
end

local function parseCall(callData, sock)
	sock.last_call = date.currentDateTime()
	if sock.type == "ws" then -- websocket
		sock.data = callData
		return
	end
	if sock.next_request then
		callData = sock.next_request .. callData
		sock.next_request = nil
	end
	if sock.incomplete_request == nil then -- and (sock.is_options_call or sock.header == nil or (sock.request_type and callData:sub(1, #sock.request_type) ~= sock.request_type)) then
		-- new call to socket
		sock.header = callData
		callData = ""
		sock.body = ""
		sock.header_length = nil
		sock.body_length = nil
		sock.next_request = nil
		sock.uri = nil
		sock.auth = nil
		sock.compress = nil
		sock.uncompress = nil
		if sock.tls_ctx then
			sock.protocol = "HTTPS"
		else
			sock.protocol = "HTTP"
		end
		sock.request_type = requestType(sock) -- sets sock.is_options_call
		if sock.request_type == nil then
			sock.incomplete_request = true
			return
		end
	end

	if sock.body_length and sock.request_type == "POST" then -- header has already been parsed, not enough body data
		if not addToBody(sock) then
			return -- addToBody() will set sock.incomplete_request = true
		end
	else
		-- parse headers
		if callData ~= "" then
			if sock.incomplete_request then
				sock.header = sock.header .. callData
			else
				sock.header = callData
			end
		end
		if not sock.request_type then
			sock.request_type = requestType(sock)
			if sock.request_type == nil then
				sock.incomplete_request = true
				return
			end
		end
		local pos = #sock.request_type + 1
		local startPos2, endPos2 = sock.header:find(" HTTP/", pos, true) -- HTTP/1.1 or other protocol, not HTTP vs HTTPS
		if not startPos2 then
			sock.incomplete_request = true
			return
		elseif sock.header_length == nil then
			-- find if full sock.header was found
			local startPos, endPos = sock.header:find("\r\n\r\n", endPos2 + 1, true)
			if not startPos then
				sock.incomplete_request = true
				return
			else -- full header exists
				sock.header_length = endPos
				sock.uri = sock.header:sub(pos + 1, startPos2 - 1)
			end
		else
			sock.header_length = sock.header_length + 0 -- debug
		end
		-- now we have full sock.header
		if sock.request_type ~= "POST" then
			-- GET, OPTIONS, CONNECT, HEAD, TRACE don't have body, DELETE may have a body
			if #sock.header > sock.header_length then -- no more than one request
				-- got more data than needed
				sock.next_request = sock.header:sub(sock.header_length + 1)
			elseif sock.next_request then
				sock.next_request = nil
			end
			if sock.incomplete_request then
				sock.incomplete_request = nil -- we have full sock.header and this request type does not have a body
			end
			sock.body_length = 0
		else
			-- POST, PUT, PATCH have body, DELETE may have a body
			sock.body_length = parseHeaderField(sock.header, "Content-Length: ", endPos2 + 1)
			if sock.body_length == nil then
				-- client error, could also assume that we got just what we needed
				sock.header = sock.header:sub(1, sock.header_length - 1)
				sock.httpErr = "411 Length Required"
				sock.incomplete_request = nil
				return -- answer with error
			else
				sock.body_length = tonumber(sock.body_length)
				if not addToBody(sock) then
					return -- addToBody() will set sock.incomplete_request = true
				end
			end
		end
	end

	-- we have full request with (optional) body, parse only needed headers
	sock.auth = parseHeaderField(sock.header, "Authorization: ", 1)
	if sock.auth then -- example: Authorization: Basic cmVzdDphdXRo
		sock.auth = pegParseAfter(sock.auth, "Basic ")
	end
	local acceptEncoding = parseHeaderField(sock.header, "Accept-Encoding: ", 1)
	if acceptEncoding then -- example: Accept-Encoding: gzip, deflate
		if acceptEncoding:find("br", 1, true) then
			sock.compress = "br"
		elseif acceptEncoding:find("gzip", 1, true) then
			sock.compress = "gzip"
		elseif acceptEncoding:find("lz4", 1, true) then
			sock.compress = "lz4"
		end
	end
	if sock.body_length > 0 then
		local contentEncoding = parseHeaderField(sock.header, "Content-Encoding: ", 1)
		if contentEncoding then
			if contentEncoding:find("br", 1, true) then
				sock.uncompress = "br"
			elseif contentEncoding:find("gzip", 1, true) then
				sock.uncompress = "gzip"
			elseif contentEncoding:find("lz4", 1, true) then
				sock.uncompress = "lz4"
			end
		end
		if #sock.body ~= sock.body_length then
			util.printError("#sock.body %d <> sock.body_length %d", #sock.body, sock.body_length)
			addToBody(sock)
		end
	end
	-- we have full parsed request now, continue to handle it
end

local basicAuthPatt = peg.toPattern("Authorization: Basic ") * peg.other(peg.define.endOfLine, 1) * peg.define.endOfLine
local function tcpAnswer(answerData, sock)
	local time = util.seconds()
	tcpCallCount = tcpCallCount + 1
	if pullBreakpoints then
		pullBreakpoints() -- see: https://github.com/tomblind/local-lua-debugger-vscode/pull/67#issuecomment-1341493005
	end
	parseCall(answerData, sock)
	-- if sock.incomplete_request then
	-- 	-- parseCall(answerData, sock)
	-- else
	if not sock.incomplete_request then
		if not sock.do_close then
			answerCount = answerCount + 1
			if sock.uri == "/" then
				sock.answer = httpStartHello .. "5\r\n\r\nHello"
				-- answer = httpStartHello.."5\r\nDate: "..dt.htmlDate().."\r\n\r\nHello"
				-- createAnswerData(sock, {hello = "world"})
				return
			end
			local printAnswer = answerCount == 1 or skip(answerCount, answerSkip) == 0
			-- local answerTxtStart = ""
			--[[ if util.isMac() then -- for debug
				printAnswer = true
			end]]
			if printAnswer then
				local str = answerCount .. ". uri: " .. sock.protocol .. ", " .. sock.request_type .. " '" .. sock.uri .. "' " .. date.toString(date.currentDateTime()) .. "\n'" .. (pegReplace(sock.header, basicAuthPatt, "Authorization: Basic ???\r\n") .. sock.body):sub(1, 900) .. "'"
				ioWrite("\n")
				util.printInfo(pegReplace(str, "\r\n", "\n"))
				--[[ else
				answerTxtStart = " " .. answerCount .. ". uri: " .. tostring(sock.request_type) .. " '" .. tostring(sock.uri) .. "' " ]]
			end
			sock.answer = createAnswer(sock)
			if printAnswer then
				time = util.seconds(time)
				util.printInfo(answerCount .. ". answer time: " .. util.seconds_to_clock(time, 5) .. "\n  " .. (sock.answer_text or "") .. "\n" .. (sock.param and sock.param.sql or "")) -- 4drest sets param.sql
				--[[ if sock.request_type == "POST" then
					collectgarbage()
				end ]]
				-- print(" answer: "..#answer.." bytes")
				-- print("\n")
			end
			sock.answer_text = nil
		end
	end
	if sock.do_delete then -- sock complete
		if sock.type and not sock.do_close then
			sock.do_delete = false
		else
			-- else
			-- need more sock.body
			if not sock.answer and not sock.do_close then -- ok to have nos answer with websocket close
				util.printWarning("sock.do_delete is true, but there is no answer")
			end
		end
	end
end

local function tcpClose(sock, reason) -- pollReturnValue
	local len
	if sock.header_length == nil then -- non-handled socket
		len = l("(no request)")
	else
		len = sock.header_length and sock.header_length + sock.body_length
		len = util.formatNum(len, 0)
	end
	sock.next_request = nil
	sock.body = nil -- don't not print these because of security
	sock.header = nil
	sock.data = nil
	fdClient.closeWorkerSocket(sock.socket)
	if not from4d and debugLevel > 0 then
		util.printInfo("*** closing rest socket '%s', type '%s', protocol '%s', reason: '%s', previous request length: %s bytes ***", tostring(sock.socket), tostring(sock.type), tostring(sock.protocol), reason or "", len)
	end
end

local function pollAfter()
	tcpPollCount = tcpPollCount + 1
	if debugLevel > 1 and not from4d then
		ioWrite(tcpPollCount .. " ")
		if skip(tcpPollCount, 20) == 0 then
			ioWrite("\n")
		end
		ioFlush()
	end
	if skip(tcpPollCount, pollAfterSkip) == 0 then
		if from4d then
			-- jit.flush()
			if server == nil then -- delay load ==
				server = require "system/server"
			end
			server.serverPause()
		else
			if prevTcpCallCount == tcpCallCount then
				-- ioWrite("| "..dt.currentString().." poll after: "..tcpPollCount)
				-- The ansi code for going to the beginning of the previous line is "\033[F"
				-- print("\033[F".."poll after: "..tcpPollCount..", "..dt.currentString())
				util.printToSameLine(" | poll after: " .. tcpPollCount .. ", " .. dt.currentString())
			else
				ioWrite("\n")
				util.printInfo("poll after: " .. tcpPollCount .. ", " .. dt.currentString())
			end
			prevTcpCallCount = tcpCallCount
		end
	end
	--[[ -- todo: fix this
	local current = date.currentDateTime()
	local sockets = poll.socketList()
	for socketNum, sock in pairs(sockets) do
		if date.secondDifference(sock.last_call, current) >= maxSecondsBeforeClose then
			if not sock.header then
				sock.header = ""
			end
			util.printInfo(
				l(
				"socket timeout close: %s, request:\n'%s', request length: %d bytes",
					tostring(socketNum),
				(sock.header..sock.body):sub(1, 400), #
				sock.header + #sock.body
				)
			)
			srv.closeSocket(sock)
		end
	end
	--]]
end

local function loadPlugins(plugins)
	local err
	for i, plugin in ipairs(plugins) do
		util.printInfo(i .. ". " .. "Loading plugin: " .. plugin) -- ..", database: "..dbName)
		local ok, plg = pcall(require, plugin)
		if not ok then
			err = l("failed to load plugin '%s', error: '%s'\n", plugin, plg)
			-- util.printRed(err) -- pegParseBefore(err, "\n"))
			break
			-- err = (err or "") .. err1
		else
			calls = util.tableCombine(calls, plg.calls) -- , "no-error"
			if plg.setCallbacks then
				plg.setCallbacks(answerError, answerOk)
			end
			if plg.init then
				plg.init()
			end
		end
	end
	if err then
		util.printRed("%s", err)
	else
		util.printInfo("All plugins have been loaded")
	end
	return err
end

local function callCount()
	return tcpCallCount + 1 -- tcpCallCount has not been set to tcpCallCount + 1 yet
end

local function pollCount()
	return tcpPollCount
end

return {
	setDebugLevel = setDebugLevel,
	loadPlugins = loadPlugins,
	parseCall = parseCall,
	pollAfter = pollAfter,
	tcpAnswer = tcpAnswer,
	tcpClose = tcpClose,
	callCount = callCount,
	pollCount = pollCount,
	answerError = answerError,
	answerOk = answerOk,
	createAnswerData = createAnswerData,
	-- variables
	debugPrintChars = debugPrintChars
}
