--- lib/spiderjs.lua
local spiderjs = {}

local ffi = require "mffi"
local util = require "util"

local lib
if ffi.os == "Windows" then
	if ffi.arch == "x64" then
		ffi.load("clang/libunwind.dll")
		ffi.load("clang/libc++.dll")
		util.loadDll("spiderjs/mozjs-124a1.dll")
	else
		ffi.load("vcruntime140.dll")
		ffi.load("vcruntime140_1.dll")
		util.loadDll("msvcp140.dll")
		util.loadDll("spiderjs/mozglue.dll") -- mozjs-108a1 115a1.dll
		util.loadDll("spiderjs/mozjs-115a1.dll") -- 108a1.dll 115a1.dll
	end
elseif ffi.os == "Linux" then
	if ffi.arch == "x64" then
		util.loadDll("spiderjs/libmozjs-106a1.so")
	else
		util.loadDll("spiderjs/libmozjs-122a1.so")
	end
end
lib = util.loadDll("spiderjs/spiderjs") or {}

ffi.cdef [[
typedef void JSContext;
const char* getVersion();
void setDebug(int8_t debugState);
int32_t getErrorTextLength();
int32_t getErrorText(const char** buffer);
int32_t setPath(const char* rootPath, const char* startPath);
void* newRuntime();
void freeRuntime(JSContext* ctx);
int32_t freeResultBuffer(char** resultBuffer);
int32_t setCode(JSContext* cx, const char* code, const char* functionName, int32_t parameterCount);
int32_t runCode(JSContext* cx, const char* parameter, int32_t parameterLength, char** resultBuffer);
int32_t setCallFunction(JSContext* cx, const char* functionName, int32_t parameterCount);
]]

local resultBuffer, errorBuffer
local resultBufferSize = 0
local errorBufferSize = 0

local function getErrorText()
	local size = errorBufferSize
	local len = lib.getErrorTextLength() + 1 -- +1 = room for trailing 0
	if len <= 0 then
		return "no error"
	end
	while len > errorBufferSize do
		errorBufferSize = errorBufferSize + 4096
	end
	if size ~= errorBufferSize then
		errorBuffer = ffi.newAnchor("const char*[?]", errorBufferSize)
	end
	local ret = lib.getErrorText(errorBuffer)
	if ret >= 0 then
		return ffi.string(errorBuffer[0], len) -- for some reason this loses the first char of error message: should be 'start-module', but is 'tart-module'
	end
	return "spiderjs error: getErrorText, error buffer is null pointer"
end

function spiderjs.getVersion()
	return ffi.string(lib.getVersion())
end

function spiderjs.setDebug(debugState)
	lib.setDebug(debugState)
end

function spiderjs.setPath(rootPath, startPath)
	return lib.setPath(ffi.cast("const char *", rootPath), ffi.cast("const char *", startPath))
end

function spiderjs.setCallFunction(ctx, functionName, parameterCount)
	local ret = lib.setCallFunction(ctx, ffi.cast("const char *", functionName), parameterCount)
	if ret < 0 then
		local err = getErrorText()
		util.printError("spiderjs setCallFunction() failed with error: %d. %s", ret, err)
		return ret, err
	end
	return ret
end

function spiderjs.setCode(ctx, code, functionName, parameterCount)
	local ret = lib.setCode(ctx, ffi.cast("const char *", code), ffi.cast("const char *", functionName), parameterCount)
	if ret < 0 then
		local err = getErrorText()
		util.printError("spiderjs setCode() failed with error: %d:\n%s", ret, err)
		return ret, err
	end
	return ret
end

function spiderjs.runCode(ctx, param)
	local ret = lib.runCode(ctx, ffi.cast("const char *", param), #param, resultBuffer)
	if ret > resultBufferSize then
		resultBufferSize = ret
	elseif ret < 0 then
		-- util.printError("spiderjs runCode() failed with error: %d. %s", ret, getErrorText())
		return getErrorText(), ret -- we normally return result, in case of error 1. param is error text
	end
	return ffi.string(resultBuffer[0], ret), ret
end

function spiderjs.newRuntime()
	if resultBuffer == nil then
		resultBuffer = ffi.newAnchor("char*[1]")
	end
	-- we must NOT allocate or windows version does not work, lib will manage *resultBuffer allocation
	-- no: resultBuffer[0] = ffi.C.malloc(resultBufferSize)
	return lib.newRuntime() -- creates a new JavaScript VM and returns context
end

function spiderjs.freeRuntime(ctx)
	lib.freeRuntime(ctx)
	lib.freeResultBuffer(resultBuffer)
	resultBufferSize = 0
end

return spiderjs
