-- svg.lua
-- https://github.com/cappelnord/EzSVG
local util = require "util"
local draw_util = require "draw/draw_util"
local peg = require "peg"
local base64 = require "base64"
-- local svg = require "EzSVG"

local replace = peg.replace
local find = peg.find
local pictCache = {}

local function xmlNum(num)
	return replace(tostring(num), ",", ".")
end

local function clipRect(ref, style, option, x, y, width, height)
	if option == "default" then
	elseif option == "restore" then
		-- todo: cr:restore()
		return
	elseif option == "set" then
		-- todo: cr:save() -- set x, y, width, height
	else
		util.printError(l("clipRect parameter '%s' is not 'set', 'restore' or 'default'", option))
		return
	end
	-- todo: cr:rectangle(x, y, width, height)
	-- todo: cr:clip()
end

local function rect(ref, style, x, y, width, height, rx, ry)
	if not style then
		style = ref.style["default-rect"]
	end
	local svg = '<rect '
	if style and type(style.style) == "string" then
		svg = svg .. 'class="' .. style.style .. '" '
	end
	svg = svg .. 'x="' .. xmlNum(x)
	svg = svg .. '" y="' .. xmlNum(y)
	svg = svg .. '" width="' .. xmlNum(width)
	svg = svg .. '" height="' .. xmlNum(height)
	if rx then
		svg = svg .. '" rx="' .. xmlNum(rx)
	end
	if ry then
		svg = svg .. '" ry="' .. xmlNum(ry)
	end
	svg = svg .. '"/>'
	ref.svg[#ref.svg + 1] = svg

	-- local obj = svg.Rect(x, y, width, height, 0, 0, s) --, rx, ry)
	-- ref.svg:add(obj)
end

local function pict(ref, style, x, y, width, height, fileName)
	local svg = '<image '
	if not style then
		style = ref.style["default-pict"]
	end
	if style and type(style.style) == "string" then
		svg = svg .. 'class="' .. style.style .. '" '
		style = ref.style[style.style]
	end
	svg = svg .. 'x="' .. xmlNum(x)
	svg = svg .. '" y="' .. xmlNum(y)
	svg = svg .. '" width="' .. xmlNum(width)
	svg = svg .. '" height="' .. xmlNum(height)
	if not style then
		svg = svg .. '" preserveAspectRatio="xMinYMin'
	elseif style.scaling == "fit" then
		svg = svg .. '" preserveAspectRatio="xMinYMin'
	end
	if not pictCache[fileName] then
		local pos = peg.findFromEnd(fileName, ".")
		if pos > 0 then
			pictCache[fileName] = {}
			pictCache[fileName].pict_type = fileName:sub(pos + 1)
			local data = util.readFile(fileName)
			-- must use temp var for base64.encode because it need to calculate data length
			if data == nil then
				data = ""
			end
			pictCache[fileName].pict_data = base64.encode(data)
		end
	end
	svg = svg .. '" xlink:href="data:image/' .. pictCache[fileName].pict_type .. ';base64,' .. pictCache[fileName].pict_data
	svg = svg .. '"/>'
	ref.svg[#ref.svg + 1] = svg
end

local function line(ref, style, x, y, x2, y2)
	if not style then
		style = ref.style["default_line"]
	end
	local svg = '<line '
	if style and type(style.style) == "string" then
		svg = svg .. 'class="' .. style.style .. '" '
	end
	svg = svg .. 'x1="' .. xmlNum(x)
	svg = svg .. '" y1="' .. xmlNum(y)
	svg = svg .. '" x2="' .. xmlNum(x2)
	svg = svg .. '" y2="' .. xmlNum(y2)
	svg = svg .. '"/>'
	ref.svg[#ref.svg + 1] = svg

	--[[
	local color = style.stroke_color
	local strokeWidth = style.stroke_width * scaleX
	if color then
		local r,g,b,a = draw_util.colorToRgba(color)
		cr:set_source_rgba(r, g, b, a)
	end
	cr:move_to(x, y)
	cr:line_to(x2, y2)
	if strokeWidth then
		cr:set_line_width(strokeWidth)
	end
	cr:stroke()
	--]]
end

local function text(ref, style, text, x, y)
	if not style then
		style = ref.style["default-text"]
	end
	local svg = '<text '
	if style and type(style.style) == "string" then
		svg = svg .. 'class="' .. style.style .. '" '
	end
	svg = svg .. 'x="' .. xmlNum(x)
	svg = svg .. '" y="' .. xmlNum(y)
	svg = svg .. '">' .. text .. '</text>'
	ref.svg[#ref.svg + 1] = svg
	local s = {}
	-- local obj = svg.Text(text, x, y)
	-- ref.svg:add(obj)
end

local function newDocument(ref)
	pictCache = {}
	local styleTxt = nil -- ref.style and json.toJson(ref.style)
	if styleTxt then
		styleTxt = replace(styleTxt, "_", "-")
	else
		styleTxt = [[
	image {
	}
	line {
	 stroke-width: 1;
	 stroke: black;
	}
	rect {
	 stroke-width: 1;
	 stroke: black;
	 fill: none;
	}
	text {
		font-name: Arial;
		font-size: 14px;
	}
	]]
	end
	--[[
		 // currentColor, https://css-tricks.com/cascading-svg-fill-color/
		rgba(0,0,0,1); == black
		]]
	local svg = [[<?xml version="1.0" encoding="UTF-8"?>
<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink"]]
	svg = svg .. ' width="' .. xmlNum(ref.paper.full_width) .. '"'
	svg = svg .. ' height="' .. xmlNum(ref.paper.full_width) .. '"'
	svg = svg .. [[
>
<defs>
<style type="text/css"><![CDATA[
]] .. styleTxt .. "]]>" .. [[

</style>
</defs>]]

	return {svg}
end

local function draw(ref)
	pictCache = nil
	ref.svg[#ref.svg + 1] = '</svg>'
	return table.concat(ref.svg, "\n") -- ref.svg:draw()
end

return {newDocument = newDocument, clipRect = clipRect, line = line, rect = rect, pict = pict, text = text, draw = draw}
