--- lib/system/crypto.lua
local crypto = {}

local ffi = require "mffi"
local util = require "util"
local peg = require "peg"

ffi.cdef [[
unsigned char *SHA1(const unsigned char *d, size_t n, unsigned char *md);
unsigned char *SHA256(const unsigned char *d, size_t n, unsigned char *md);
]]

-- tls
ffi.cdef [[
// crypto.h
typedef	unsigned char 	u_char;
int RAND_pseudo_bytes(u_char *buf, int num);

static const int TLS_API	= 20141031;
static const int TLS_PROTOCOL_TLSv1_0	= (1 << 1);
static const int TLS_PROTOCOL_TLSv1_1	= (1 << 2);
static const int TLS_PROTOCOL_TLSv1_2	= (1 << 3);
static const int TLS_PROTOCOL_TLSv1 = (TLS_PROTOCOL_TLSv1_0|TLS_PROTOCOL_TLSv1_1|TLS_PROTOCOL_TLSv1_2);

static const int TLS_PROTOCOLS_ALL = TLS_PROTOCOL_TLSv1;
static const int TLS_PROTOCOLS_DEFAULT = TLS_PROTOCOL_TLSv1_2;

static const int TLS_WANT_POLLIN		= -2;
static const int TLS_WANT_POLLOUT	= -3;


typedef void SSL_CTX;
typedef void SSL;
typedef void X509;

struct tls {
	struct tls_config *config;
	uint32_t flags;
	uint32_t state;

	char *errmsg;
	int errnum;

	char *servername;
	int socket;

	SSL *ssl_conn;
	SSL_CTX *ssl_ctx;
	X509 *ssl_peer_cert;
	struct tls_conninfo *conninfo;
};


struct tls_config {
	const char *ca_file;
	const char *ca_path;
	char *ca_mem;
	size_t ca_len;
	const char *cert_file;
	char *cert_mem;
	size_t cert_len;
	const char *ciphers;
	int ciphers_server;
	int dheparams;
	int ecdhecurve;
	const char *key_file;
	char *key_mem;
	size_t key_len;
	uint32_t protocols;
	int verify_cert;
	int verify_client;
	int verify_depth;
	int verify_name;
	int verify_time;
};

int tls_init(void);
struct tls *tls_new(void);
struct tls *tls_server_conn(struct tls *ctx);
const char *tls_error(struct tls *_ctx);
struct tls_config *tls_config_new(void);
void tls_config_free(struct tls_config *_config);

int tls_config_set_ca_file(struct tls_config *_config, const char *_ca_file);
int tls_config_set_ca_path(struct tls_config *_config, const char *_ca_path);
int tls_config_set_ca_mem(struct tls_config *_config, const uint8_t *_ca,
    size_t _len);
int tls_config_set_cert_file(struct tls_config *_config,
    const char *_cert_file);
int tls_config_set_cert_mem(struct tls_config *_config, const uint8_t *_cert,
    size_t _len);
int tls_config_set_ciphers(struct tls_config *_config, const char *_ciphers);
int tls_config_set_dheparams(struct tls_config *_config, const char *_params);
int tls_config_set_ecdhecurve(struct tls_config *_config, const char *_name);
int tls_config_set_key_file(struct tls_config *_config, const char *_key_file);
int tls_config_set_key_mem(struct tls_config *_config, const uint8_t *_key,
    size_t _len);
void tls_config_set_protocols(struct tls_config *_config, uint32_t _protocols);
void tls_config_set_verify_depth(struct tls_config *_config, int _verify_depth);

void tls_config_prefer_ciphers_client(struct tls_config *_config);
void tls_config_prefer_ciphers_server(struct tls_config *_config);

void tls_config_insecure_noverifycert(struct tls_config *_config);
void tls_config_insecure_noverifyname(struct tls_config *_config);
void tls_config_insecure_noverifytime(struct tls_config *_config);
void tls_config_verify(struct tls_config *_config);

void tls_config_verify_client(struct tls_config *_config);
void tls_config_verify_client_optional(struct tls_config *_config);

void tls_config_clear_keys(struct tls_config *_config);
int tls_config_parse_protocols(uint32_t *_protocols, const char *_protostr);

struct tls *tls_client(void);
struct tls *tls_server(void);
int tls_configure(struct tls *_ctx, struct tls_config *_config);
void tls_reset(struct tls *_ctx);
void tls_free(struct tls *_ctx);

int tls_accept_fds(struct tls *_ctx, struct tls **_cctx, int _fd_read,
    int _fd_write);
int tls_accept_socket(struct tls *_ctx, struct tls **_cctx, int _socket);
int tls_connect(struct tls *_ctx, const char *_host, const char *_port);
int tls_connect_fds(struct tls *_ctx, int _fd_read, int _fd_write,
    const char *_servername);
int tls_connect_servername(struct tls *_ctx, const char *_host,
    const char *_port, const char *_servername);
int tls_connect_socket(struct tls *_ctx, int _s, const char *_servername);
int tls_handshake(struct tls *_ctx);
ssize_t tls_read(struct tls *_ctx, void *_buf, size_t _buflen);
ssize_t tls_write(struct tls *_ctx, const void *_buf, size_t _buflen);
int tls_close(struct tls *_ctx);

int tls_peer_cert_provided(struct tls *ctx);
int tls_peer_cert_contains_name(struct tls *ctx, const char *name);

const char * tls_peer_cert_hash(struct tls *_ctx);
const char * tls_peer_cert_issuer(struct tls *ctx);
const char * tls_peer_cert_subject(struct tls *ctx);
time_t	tls_peer_cert_notbefore(struct tls *ctx);
time_t	tls_peer_cert_notafter(struct tls *ctx);

const char * tls_conn_version(struct tls *ctx);
const char * tls_conn_cipher(struct tls *ctx);

uint8_t *tls_load_file(const char *_file, size_t *_len, char *_password);

]]

crypto.crypto = util.loadDll("crypto", "no-error") -- new mac + win linux has no numbers
if crypto.crypto ~= nil then
	crypto.ssl = util.loadDll("ssl", "no-error")
	crypto.tls = util.loadDll("tls")
end
if util.isLinux() then -- linux
	local uname = util.runCommandLine("uname -a")
	util.print(uname)
end
if crypto.crypto == nil then
	-- util.printError()
	util.printWarning("libcrypto loading failed")
elseif crypto.ssl == nil then
	util.printWarning("libssl loading failed")
elseif crypto.tls == nil then
	util.printWarning("libtls loading failed")
end

crypto.tlsError = function(tls_ctx)
	local err = crypto.tls.tls_error(tls_ctx)
	if ffi.isNull(err) then
		return ""
	end
	return ffi.string(err)
end

return crypto
