--- lib/system/diff.lua
-- TODO: migrate to pure-lua version: https://github.com/google/diff-match-patch/blob/master/lua/diff_match_patch.lua
-- diff.h
-- example: https://gist.github.com/Drako/3f37d3a61200a5c2d1ec
local diff = {}

local ffi = require "mffi"
local C = ffi.C
local bit = require "bit"
local util = require "util"
local dt = require "dt"
local peg = require "peg"
local color = require "ansicolors"
local libDiff, fs

local C_DEFAULT = "%{reset}"
local C_DIFF = "%{red}"
local C_OLDFILE = "%{cyan}"
local C_NEWFILE = "%{red}"
local C_STATS = "%{green}"
local C_OLD = "%{bright red}"
local C_NEW = "%{bright green}"
local C_ONLY = "%{magenta}" -- pink
local C_RESET = "%{reset}"

--[[ https://github.com/git/git/blob/master/diff/diff.h
	/* xpparam_t flags */
define XDF_NEED_MINIMAL (1 << 0) // in diff.h: #define XDF_NEED_MINIMAL (1 << 1)


define XDF_IGNORE_WHITESPACE (1 << 1)
define XDF_IGNORE_WHITESPACE_CHANGE (1 << 2)
define XDF_IGNORE_WHITESPACE_AT_EOL (1 << 3)
define XDF_IGNORE_CR_AT_EOL (1 << 4)
define XDF_WHITESPACE_FLAGS (XDF_IGNORE_WHITESPACE | \
			      XDF_IGNORE_WHITESPACE_CHANGE | \
			      XDF_IGNORE_WHITESPACE_AT_EOL | \
			      XDF_IGNORE_CR_AT_EOL)

define XDF_IGNORE_BLANK_LINES (1 << 7)

define XDF_PATIENCE_DIFF (1 << 14)
define XDF_HISTOGRAM_DIFF (1 << 15)
define XDF_DIFF_ALGORITHM_MASK (XDF_PATIENCE_DIFF | XDF_HISTOGRAM_DIFF)
define XDF_DIFF_ALG(x) ((x) & XDF_DIFF_ALGORITHM_MASK)

define XDF_INDENT_HEURISTIC (1 << 23)

/* xdemitconf_t.flags */
define XDL_EMIT_FUNCNAMES (1 << 0)
define XDL_EMIT_FUNCCONTEXT (1 << 2)

define XDL_MMB_READONLY (1 << 0)

define XDL_MMF_ATOMIC (1 << 0)

define XDL_BDOP_INS 1
define XDL_BDOP_CPY 2
define XDL_BDOP_INSB 3

/* merge simplification levels */
define XDL_MERGE_MINIMAL 0
define XDL_MERGE_EAGER 1
define XDL_MERGE_ZEALOUS 2
define XDL_MERGE_ZEALOUS_ALNUM 3

/* merge favor modes */
define XDL_MERGE_FAVOR_OURS 1
define XDL_MERGE_FAVOR_THEIRS 2
define XDL_MERGE_FAVOR_UNION 3

/* merge output styles */
define XDL_MERGE_DIFF3 1
]]

ffi.cdef [[

/* patch flags */
static const int XDL_PATCH_NORMAL = '-'; // lib/diff.lua:89: ';' expected near '=' at line 2
static const int XDL_PATCH_REVERSE = '+';
static const int XDL_PATCH_IGNOREBSPACE = (1 << 8);

/* xpparam_t.flags */
static const int XDF_NEED_MINIMAL = (1 << 1);

static const int XDF_IGNORE_WHITESPACE = (1 << 1);
static const int XDF_IGNORE_WHITESPACE_CHANGE = (1 << 2);
static const int XDF_IGNORE_WHITESPACE_AT_EOL = (1 << 3);
static const int XDF_IGNORE_CR_AT_EOL = (1 << 4);
static const int XDF_IGNORE_BLANK_LINES = (1 << 7);

typedef struct s_mmbuffer {
	char *ptr;
	long size;
} mmbuffer_t;

int32_t patchString(const char *fileStr, size_t fileSize, const char *patchStr, size_t patchSize, int32_t mode, void* callbackFunc);
int32_t diffString(const char *fileStr1, size_t fileSize1, const char *fileStr2, size_t fileSize2, int32_t mode, void* callbackFunc);
int32_t areEqual(const char *fileStr1, size_t fileSize1, const char *fileStr2, size_t fileSize2);

]]

local function loadDll()
	if not libDiff then
		if util.isWin() then
			if ffi.arch == "x64" then
				util.loadDll("graphics/libwinpthread-1.dll")
				util.loadDll("graphics/libgcc_s_seh-1.dll")
			else
				util.loadDll("graphics/libgcc_s_sjlj-1.dll")
			end
			util.loadDll("graphics/libstdc++-6.dll")
		end
		libDiff = util.loadDll("diff_ma")
	end
end
loadDll()

local function makeHeader(filename1, filename2, path)
	--[[ example of header:
		--- diff_str1.txt	2018-01-20 02:10:45.000000000 +0200
		+++ diff_str2.txt	2018-01-20 03:32:16.000000000 +0200
	]]
	if path == nil or path == "" then
		return ""
	else
		if not fs then
			fs = require "fs"
		end
		local attr1 = fs.attributes(path .. filename1)
		local attr2 = fs.attributes(path .. filename2)
		local time1 = dt.formatNum(attr1.change, "%Y-%m-%d %H:%M:%S.000000000 ") .. dt.currentTimeZone()
		local time2 = dt.formatNum(attr2.change, "%Y-%m-%d %H:%M:%S.000000000 ") .. dt.currentTimeZone()
		return "--- " .. filename1 .. "\t" .. time1 .. "\n+++ " .. filename2 .. "\t" .. time2 .. "\n"
	end
end

function diff.areEqual(string1, string2)
	string1 = peg.removeFromEnd(string1, "\n") .. "\n"
	string2 = peg.removeFromEnd(string2, "\n") .. "\n"
	local areEqual = libDiff.areEqual(string1, #string1, string2, #string2) -- returns 0 if files are identical
	return areEqual == 0 -- == 0, err
end

local resultString = {}
local rejectString = {}
local function callbackFunction(priv, mmbuffer, count)
	--[[
	The first parameter of the callback is the same priv field specified inside the xdemitcb_t structure. The second parameter point to an array of mmbuffer_t (see above for a definition of the structure) whose element count is specified inside the last parameter of the callback itself.
	]]
	local stripAfterNewLine = ffi.cast("uint8_t*", priv)[0] -- set: ecb.priv = ffi.newAnchor("uint8_t[1]", 1) -- tells callback to strip after \n
	for i = 0, count - 1 do -- c struct index from 0
		local buf = mmbuffer[i]
		local str = ffi.string(buf.ptr, buf.size)
		if stripAfterNewLine == 1 then
			resultString[#resultString + 1] = str -- using buf.size, no need for this: peg.parseBeforeWithDivider(str, "\n")
			-- elseif stripAfterNewLine == 2 then
			-- rejectString[#rejectString + 1] = str
		elseif stripAfterNewLine == 3 then
			rejectString[#rejectString + 1] = str -- using buf.size, no need for this:peg.parseBeforeWithDivider(str, "\n")
		else
			util.print(" *** diff callback with unknown priv: " .. stripAfterNewLine)
			resultString[#resultString + 1] = str -- clared in init_xdiff()
		end
	end
	return 0 -- The function returns 0 if succeeded or -1 if an error occurred.
end

function diff.diffString(string1, string2, filename1, filename2, path)
	string1 = peg.removeFromEnd(string1, "\n") .. "\n"
	string2 = peg.removeFromEnd(string2, "\n") .. "\n"
	local mode = bit.bor(C.XDF_NEED_MINIMAL, C.XDF_IGNORE_WHITESPACE, C.XDF_IGNORE_WHITESPACE_CHANGE, C.XDF_IGNORE_WHITESPACE_AT_EOL, C.XDF_IGNORE_CR_AT_EOL, C.XDF_IGNORE_BLANK_LINES) -- option, see xpparam_t flags
	resultString = {}
	local cb = ffi.cast("int (*)(void *, mmbuffer_t *, int)", callbackFunction)
	local err = libDiff.diffString(string1, #string1, string2, #string2, mode, cb)
	local header = ""
	if #resultString > 0 then
		header = makeHeader(filename1, filename2, path)
	end
	local diffTxt = header .. table.concat(resultString)
	resultString = nil -- file local, release memory
	return diffTxt, err
end

--[[
function diff.diffFile(string1, string2)
end
]]

function diff.patchString(fileString, patchString, direction)
	if patchString:sub(1, 4) ~= "@@ -" then -- remove headers
		patchString = peg.parseAfterWithDivider(patchString, "@@ -")
	end
	fileString = peg.removeFromEnd(fileString, "\n") .. "\n"
	patchString = peg.removeFromEnd(patchString, "\n") .. "\n"
	local mode
	if direction == "normal" then
		mode = bit.bor(C.XDL_PATCH_NORMAL, C.XDL_PATCH_IGNOREBSPACE) -- this diws NOT work, don't know why
	elseif direction == "reverse" then
		mode = bit.bor(C.XDL_PATCH_REVERSE, C.XDL_PATCH_IGNOREBSPACE)
	else
		util.printError("parch direction is not 'normal' or 'reverse'")
		return "", "", -100
	end
	--[[
	XDL_PATCH_IGNOREBSPACE Ignore the whitespace at the beginning and the end of the line.
	XDL_PATCH_NORMAL Perform standard patching like if the patch memory file mmfp has been created using mmf as "old" fs.
XDL_PATCH_REVERSE Apply the reverse patch. That means that the mmf memory file has to be considered as if it was specified as "new" file during the differential operation ( xdl_diff() ). The result of the operation will then be the file content that was used as "old" file during the differential operation. ]]
	resultString = {}
	rejectString = {}
	local cb = ffi.cast("int (*)(void *, mmbuffer_t *, int)", callbackFunction)
	local err = libDiff.patchString(fileString, #fileString, patchString, #patchString, mode, cb)
	local resultTxt = table.concat(resultString)
	local rejectTxt = table.concat(rejectString)
	-- if #rejectString > 0 then
	-- local header = makeHeader(filename, filenameReject, path)
	-- rejectTxt = header..table.concat(rejectString)
	-- end
	resultString = nil -- free file-local vars
	rejectString = nil
	return resultTxt, rejectTxt, err
end

function diff.colorizeConsole(diffText, maxLines)
	-- colorize diff output for ANSI terminals
	-- based on "diff2html"
	-- (http://www.linuxjournal.com/content/convert-diff-output-colorized-html)
	-- style color definitions
	-- local C_COMMENT = WHITE
	local out = {}
	local colorTxt
	local lineNum = 0
	peg.iterateLines(diffText, function(line)
		lineNum = lineNum + 1
		if maxLines == nil or lineNum <= maxLines then
			-- determine line color
			if line:sub(1, 7) == 'Only in' then
				colorTxt = C_ONLY
			elseif line:sub(1, 4) == 'diff' then
				colorTxt = C_DIFF
			elseif line:sub(1, 3) == '---' then
				colorTxt = C_OLDFILE
			elseif line:sub(1, 3) == '+++' then
				colorTxt = C_NEWFILE
			elseif line:sub(1, 2) == '@@' then
				colorTxt = C_STATS
			elseif line:sub(1, 1) == '+' then
				colorTxt = C_NEW
			elseif line:sub(1, 1) == '-' then
				colorTxt = C_OLD
			else
				colorTxt = C_DEFAULT -- check
			end
			-- Output the line.
			if line ~= "\n" and line ~= "" then
				if color ~= "" then
					out[#out + 1] = color(colorTxt .. " " .. line .. C_RESET)
				else
					out[#out + 1] = line
				end
			end
		end
	end)
	return table.concat(out, "\n")
end

return diff

--[[  -- test binary diff formats, this code works inside diffString() -function
	if false then
		local buf1 = ffi.newAnchor("mmbuffer_t")
		buf1.ptr = string1
		buf1.size = #string1
		local buf2 = ffi.newAnchor("mmbuffer_t")
		buf2.ptr = string2
		buf2.size = #string2

		ecb.priv = ffi.newAnchor("uint8_t[1]", 0) -- do not strip after newline

		resultString = {}
		err = libDiff.xdl_rabdiff_mb(buf1, buf2, ecb)
		local diff1 = table.concat(resultString)
		fs.writeFile(path.."rabdiff_mb.diff", diff1)
		fs.openFile(path.."rabdiff_mb.diff")

		resultString = {}
		local bdp = ffi.newAnchor("bdiffparam_t")
		bdp.bsize = 32 -- Suggested values go from 16 to 64, with a preferred power of two characteristic.
		err = libDiff.xdl_bdiff_mb(buf1, buf2, bdp, ecb)
		local diff2 = table.concat(resultString)
		fs.writeFile(path.."bdiff_mb.diff", diff2)
		fs.openFile(path.."bdiff_mb.diff")
	end
]]
