--- net.lua
--
local net = {}

local ffi = require "mffi"
local C = ffi.C
local bit = require "bit"
local util = require "util"
local peg = require "peg"

local errorPrefix = "   NETWORK ERROR: "

local rpcRt4
if util.isWin() then
	C = util.loadDll("ws2_32") -- ffi.load("ws2_32")
	rpcRt4 = util.loadDll("rpcrt4")
end

if util.isWin() then

	function net.errorText(err)
		return util.winErrorText(err) -- util.winErrorText(err)
	end

	function net.printError(errNum, errText)
		if errText == nil then
			if errNum == nil then
				errNum = C.WSAGetLastError()
			end
			errText = net.errorText(errNum)
		end
		if errText and #errText > 0 then
			util.print(errorPrefix .. errText .. "\n" .. util.callPath())
		end
	end

	local wsaData = ffi.newAnchor("WSADATA[1]") -- usually called only once
	function net.initialize()
		local function MAKEWORD(low, high)
			return bit.bor(low, bit.lshift(high, 8))
		end
		-- if ffi.abi("64bit") then
		--	wsaData = ffi.newNoAnchor("WSADATA64[1]")
		-- else
		-- end
		local wVersionRequested = MAKEWORD(2, 2) -- ffi.cast("WORD", MAKEWORD(2, 2))
		local err = C.WSAStartup(wVersionRequested, wsaData)
		if err ~= 0 then
			util.print(errorPrefix .. "WSAStartup failed with error code: " .. err .. ", " .. net.errorText(err))
		end
		return err -- err,wsaData[0]
	end

	function net.poll(fdArray, fds, timeout)
		-- http://www.winsocketdotnetworkprogramming.com/winsock2programming/winsock2advancediomethod5e.html
		return C.WSAPoll(fdArray, fds, timeout)
	end

	local srcAddr = ffi.newAnchor("struct sockaddr_in") -- ffi.cast("struct sockaddr_in *", pAddr)
	local len_c = ffi.newAnchor("unsigned long[1]")
	function net.inetNtop(family, pAddr, strPtr)
		-- win XP: http://memset.wordpresC.com/2010/10/09/inet_ntop-for-win32/
		ffi.copy(srcAddr.sin_addr, pAddr, ffi.sizeof(srcAddr.sin_addr))
		len_c[0] = ffi.sizeof(strPtr)
		srcAddr.sin_family = family
		local ret = C.WSAAddressToStringA(ffi.cast("struct sockaddr *", srcAddr), ffi.sizeof("struct sockaddr"), nil, strPtr, len_c)
		if ret ~= 0 then
			util.print(errorPrefix .. "WSAAddressToString failed with error: " .. tonumber(ret))
			return nil
		end
		return strPtr
	end

	local uuid = ffi.newAnchor("UUID[1]") -- usually called only once
	function net.getMacAddress()
		local ret = rpcRt4.UuidCreateSequential(uuid) -- FIONBIO in win_socket.lua
		if ret == 0 or ret == 1824 then -- RPC_S_UUID_LOCAL_ONLY 1824L
			local macAddr = ffi.newNoAnchor("uint8_t[?]", 6)
			ffi.copy(macAddr, uuid[0].Data4 + 2, 6)
			return string.format("%02x:%02x:%02x:%02x:%02x:%02x", macAddr[0], macAddr[1], macAddr[2], macAddr[3], macAddr[4], macAddr[5])
		end
		util.printWarning("could not get system mac address, using random value for uuid last part")
		return string.format("%02x:%02x:%02x:%02x:%02x:%02x", math.random(0, 255), math.random(0, 255), math.random(0, 255), math.random(0, 255), math.random(0, 255), math.random(0, 255))
	end

	function net.getLocalIpAddress()
		local ip
		-- does NOT work, needs connected socket
		--[[
		local name = ffi.newNoAnchor("char[255]")
		local err = C.gethostname(name, ffi.sizeof(name))
		if err == 0 then
			local hostInfo = C.gethostbyname(name)
			if hostInfo ~= nil then
				-- ip = inet_ntoa (*(struct in_addr *)*hostInfo->h_addr_list);
				local h_addr_list = ffi.cast("struct in_addr *", hostInfo.h_addr_list)
				ip = C.inet_ntoa(h_addr_list[0])
				ip = ffi.string(ip)
			end
		end
		]]
		local ret = util.runCommandLineRaw("ipconfig /all") -- we need line endings
		peg.iterateLines(ret, function(line)
			if not ip and peg.find(line, "IPv4") > 0 then
				local pos2 = peg.find(line, "(") -- '(' is start of '(Preferred)'
				if pos2 > 0 then
					local pos1 = peg.find(line, ": ")
					if pos1 > 0 then
						ip = line:sub(pos1 + 2, pos2 - 1)
					end
				end
			end
		end)
		if not ip then -- non-english system, TEST this!
			peg.iterateLines(ret, function(line)
				if not ip and peg.find(line, "IPv4") > 0 then
					local pos1 = peg.find(line, ": ")
					if pos1 > 0 then
						ip = line:sub(pos1 + 2)
						pos1 = peg.find(ip, "(") -- todo: test this
						if pos1 > 0 then
							ip = ip:sub(pos1 - 1)
						end
						if ip == "" or ip == "127.0.0.1" then
							ip = nil
						end
					end
				end
			end)
		end
		if not ip then -- XP
			peg.iterateLines(ret, function(line)
				if not ip and peg.find(line, "IP Address") > 0 then
					local pos1 = peg.find(line, ": ")
					if pos1 > 0 then
						ip = line:sub(pos1 + 2)
					end
				end
			end)
		end
		return ip or ""
	end

else
	-- unix

	function net.errorText(err)
		if type(err) ~= "number" then
			if type(err) == "string" then
				return err
			end
			return util.printError("parameter type '%s' is not a number", type(err))
		end
		local txt = ffi.string(C.gai_strerror(err))
		if txt == "Unknown error" then -- needed at least for osx
			txt = ffi.string(C.strerror(err))
		end
		return tostring(err) .. ": " .. txt
	end

	function net.initialize()
		return 0 -- for win compatibility
	end

	function net.poll(fds, nfds, timeout)
		return C.poll(fds, nfds, timeout)
	end

	function net.printError(errNum, errText)
		errText = errText or ""
		if not errNum then
			errNum = ffi.errno()
		end
		local errText2 = net.errorText(errNum)
		if #errText > 0 or #errText2 > 0 then
			return util.printError(errorPrefix .. errText .. "(" .. tonumber(errNum) .. ") " .. errText2)
		end
		return errText2
	end

	function net.inetNtop(family, pAddr, strPtr)
		return C.inet_ntop(family, pAddr, strPtr, ffi.sizeof(strPtr))
	end

	function net.getMacAddress()
		local ret
		if util.isMac() then
			ret = util.runCommandLine("LC_ALL=C ifconfig en0 | awk '/ether/{print $2}'")
		elseif util.isLinux() then
			-- ret = util.runCommandLine("ip addr show eth0 | grep 'link/ether' | awk '{print $2}'")
			ret = util.runCommandLine("ip addr | grep 'link/ether' | awk '{print $2}'")
			if ret == "" then -- no ifconfig, ip addr
				ret = util.runCommandLine("LC_ALL=C ifconfig eth0 | awk '/HWaddr/ {print $NF}'")
				if ret == "" then -- no eth0, use first found uuid
					ret = util.runCommandLine("LC_ALL=C ifconfig | awk '/HWaddr/ {print $NF}'")
				end
			end
		end
		if #ret < 17 then
			util.printError("network card mac address was not found")
		end
		ret = ret:sub(1, 17)
		return ret:lower()
	end

	function net.getLocalIpAddress()
		local ip
		if util.isLinux() then
			-- ip = util.runCommandLineRaw("ip route get 1 | awk '{print $NF;exit}'")
			-- ip = util.runCommandLine("ip addr show eth0 | grep 'inet\b' | awk '{print $2}' | cut -d/ -f1")
			ip = util.runCommandLineRaw("ip addr | grep 'inet ' | awk '{print $2}' | cut -d/ -f1")
			if ip == "" then -- no ifconfig, ip addr
				ip = util.runCommandLineRaw("ifconfig eth0 | grep 'inet addr:' | cut -d: -f2 | awk '{ print $1}'")
			end
		else
			ip = util.runCommandLineRaw("ifconfig | grep -E 'inet.[0-9]' | grep -v '127.0.0.1' | awk '{ print $2}'") -- mac + linux
		end
		ip = peg.removeFromEnd(ip, "\n") -- util.runCommandLineRaw() returns \n at then end
		ip = peg.replace(ip, "\n", ", ") -- multiple results
		ip = peg.replace(ip, "127.0.0.1, ", "") -- no localhost
		return ip or ""
	end

end

function net.getnameinfo(sa, salen, host, hostlen, serv, servlen, flags)
	return C.getnameinfo(sa, salen, host, hostlen, serv, servlen, flags)
end

local hostNameBuf, servNameBuf
local hostNameBufLen = 0
local servNameBufLen = 0
function net.getaddrinfo(hostName, servName, hints, res)
	if not hostNameBuf or #hostName >= hostNameBufLen then -- url or ip
		if hostNameBuf then
			ffi.free(hostNameBuf)
		end
		hostNameBufLen = #hostName + 100
		hostNameBuf = ffi.createBuffer(hostNameBufLen)
	end
	hostNameBuf = ffi.copyStringToBuffer(hostNameBuf, hostName, #hostName)

	if not servNameBuf or #servName >= servNameBufLen then -- normally port number
		if servNameBuf then
			ffi.free(servNameBuf)
		end
		servNameBufLen = #servName + 10
		servNameBuf = ffi.createBuffer(servNameBufLen)
	end
	servNameBuf = ffi.copyStringToBuffer(servNameBuf, servName, #servName)
	return C.getaddrinfo(hostNameBuf, servNameBuf, hints, res)
end

function net.ntohs(netShort)
	return C.ntohs(netShort)
end

function net.htons(port)
	return C.htons(port)
end

function net.inet_addr(address)
	return C.inet_addr(address)
end

return net
