--  lib/server.lua
local ffi = require "mffi"
local peg = require "peg"
local util = require "util"
local socket = require "system/socket"
local poll = require "system/poll"
local pollFd = poll.pollFd
local dt = require "dt"
local net = require "system/net"
local l = require"lang".l
local tm = require "time"
local color = require "ansicolors"
local dconn = require "dconn"
local seconds = util.seconds
local ioWrite, ioFlush = io.write, io.flush
local print = util.print
local formatNum = util.formatNum

local coro = require "coro"
local useCoro = coro.useCoro()
local currentThreadSocket = coro.currentThreadSocket
local threadSocketArr = coro.threadSocketArr
local createSocketCoroutine = coro.createSocketCoroutine
local setSocket = coro.setSocket
local threadId = coro.threadId
local coroResume = coro.resume
local socketClosed = coro.socketClosed
local yield = coro.yield
local coroClose = coro.close
local coroStatus = coro.status
local releaseLock = coro.releaseLock
local currentThread = coro.currentThreadFunction()

local useUdp = true
local testDebugLevel = util.fromEditor() and 0 or nil -- 1 -- nil
local mainThread = coro.setMainThread(nil, testDebugLevel)
local from4d = util.from4d()
local yield4d
if from4d then
	yield4d = require"plg4d".yieldAbsolute -- plg4d.yieldAbsolute -- or plg4d.yield
	--[[
	local delayInTicks = 10
	local currentProcess = plg4d.currentProcessNumber()
	yield4d = function()
    plg4d.sleep(currentProcess, delayInTicks)
	end --]]
end

-- local prevPollTime = 0
local localIpAddress, tcpPort, acceptSocket, prevPollPrintTime
local debugLevel, useProfiler, profiler, profilerEndCount, debugPrintChars
local pollAfterFunction, answerFunction, beforeCloseFunction, callCountFunction
--[[	If timeout is greater than zero, it specifies a maximum interval (in milliseconds) to wait for any file
			descriptor to become ready.  If timeout is zero, then poll() will return without blocking. If the value
			of timeout is -1, the poll blocks indefinitely.]]
--[[POLLPRI	Priority data may be read without blocking. This flag is not supported by the Microsoft Winsock provider.]]
-- local connectEvents = bit.bor(C.POLLIN, C.POLLOUT)

local loc = {}
loc.prevPollPrintCount = 0
loc.prevPollPrintTime = 0
loc.prevCollectGarbageTime = 0
-- loc.collectGarbageAnswerCount = 0 -- make collectgarbage at start
loc.collectGarbageTimeout = nil
loc.collectGarbageRestartMemoryUse = nil

local answerCount = 0
local pollInCount = 0
local pollOutCount = 0
local pollCloseCount = 0
local pollErrCount = 0

local totalBytesReceived = 0
local totalBytesSent = 0
local listenSocketTcp = nil
local listenSocketUdp = nil
local stop = false
local pause = false

local closeSocket
do
	local sockNum
	closeSocket = function(sock, reason)
		if not sock then
			util.printError("-*-ERR: close socket is nil")
			return
		end
		if sock.closed then
			util.printRed("-*-ERR: socket has been closed")
			return
		end
		sockNum = tonumber(sock.socket)
		if sockNum == nil or sockNum < 1 then
			util.printRed("-*-ERR: close socket: '%s'", tostring(sockNum))
		end
		if beforeCloseFunction then
			beforeCloseFunction(sock, reason)
		end
		if not sock.closed then -- connection sockets have been closed in dconn.disconnectAll(thread)
			sock:close() -- or sock:shutdown(), removes sock.thread and other keys
		end
		--[[ if socketIdx[sockStr] then
			-- sock:close() has a callback to socketClosed() from poll.removeFd(), socketIdx[sockStr] should be nil
			util.printRed("-*-ERR: socket '%s' was found from the socket index", sockStr)
		end ]]
	end
end

-- local skip = util.skip
--[[ local noFastAnswerWarningUri = util.invertTable({"/rest/nc/auth/init"})
local prevAnswerTime = util.seconds()
local currentAnswerTime, currentAnswerCount ]]
local function answer(sock)
	if sock.closed then
		return
	end
	answerCount = answerCount + 1
	-- a % b == a - math.floor(a/b)*b
	-- if answerCount % 500 == 0 then
	-- if skip(answerCount, 5) == 0 then
	--[[ currentAnswerTime = util.seconds()
	if currentAnswerTime - prevAnswerTime < 0.05 and not sock.do_close then
		currentAnswerCount = currentAnswerCount + 1
		if currentAnswerCount > 1 then
			if not from4d and noFastAnswerWarningUri[sock.uri] == nil then
				-- util.printToSameLine("answer: "..answerCount.."\n") -- formatNum(answerCount, 0)
				util.printWarning("fast answer %d, time: %.5f, socket: %s", formatNum(answerCount, 0), currentAnswerTime - prevAnswerTime, sock)
				-- sock.do_close = true
			end
		end
	else
		currentAnswerCount = 0
	end
	prevAnswerTime = currentAnswerTime ]]
	local request, requestErr, requestLen, addr, answerData, bytesSent
	if useUdp and sock == listenSocketUdp then
		request, requestErr, requestLen, addr = nil, nil, nil, nil -- sock:receivefrom() -- TODO: fix udp
	else
		request, requestErr, requestLen = sock:receive("\r\n\r\n", true)
	end
	if requestErr then
		if requestErr == 0 then
			print("  socket '%s' receive failed with error '%s'", tostring(sock.socket), tostring(requestErr))
			-- closeSocket(sock, "receive failed")
		elseif sock.do_close then
			return
		else
			if requestErr ~= 54 and requestErr ~= 2 then -- error 54: Connection reset by peer,  error 2: Temporary failure in name resolution
				util.printWarning("  socket '%s' receive failed with error %s", tostring(sock.socket), net.errorText(requestErr))
				closeSocket(sock, "no resume")
			else
				closeSocket(sock, "receive failed")
			end
			return
		end
	end
	-- todo: handle requestErr
	if requestLen == nil then
		requestLen = 0
	end
	if requestLen > 0 then
		totalBytesReceived = totalBytesReceived + requestLen
		if debugLevel >= 2 then
			print(" -- Bytes received: ", requestLen .. ", " .. totalBytesReceived .. " total")
			print(" -- Data  received: \n'" .. string.sub(request, 1, debugPrintChars) .. "'\n")
		end
		if answerFunction then
			if useUdp and sock == listenSocketUdp then
				answerData = " - nc UDP server ip: <" .. localIpAddress .. ">, you said: '" .. request .. "'"
				if addr then
					bytesSent = sock:sendTo(answerData, addr) -- localIpAddress installed in serverRun()
				else
					bytesSent = 0
				end
				print(answerData)
			else
				answerFunction(request, sock)
				if sock.answer and sock.answer ~= "" then
					bytesSent = sock:send(sock.answer)
				else
					bytesSent = 0
				end
			end
			if bytesSent > 0 then
				totalBytesSent = totalBytesSent + bytesSent
			end
			if sock.answer == nil then
				util.printRed(" -- bytes sent: %s, no answer sent", tostring(bytesSent))
			else
				if debugLevel >= 2 then
					util.print(" -- bytes sent: %s / %s", tostring(bytesSent), tostring(#sock.answer))
				end
				if debugLevel >= 3 then
					if peg.find(sock.answer, "Content-Encoding: ") == 0 then
						print(" -- send data:\n" .. sock.answer:sub(1, 600) .. "\n")
					end
				end
			end
			if sock.answer and bytesSent == #sock.answer then
				sock.answer = nil
			end
		end
		if sock.do_close then
			closeSocket(sock)
		end
		--[[ -- Linux has POLLRDHUP event, not POLLHUP
	elseif requestLen == 0 then
		if isLinux() then
			-- we must close socket in Linux or code will come here in a loop forever
			closeSocket(sock, "nothing received")
			return
		end ]]
		--[[ else
		util.printWarning(" -- socket %s, socket receive failed with error: %d", tostring(sock.socket), requestLen)
		if requestLen ~= -1 then
			closeSocket(sock)
			return
		end ]]
	end
end

local function acceptLoop(sock)
	return function()
		local thread = currentThread()
		-- coro.clearLoadedPackages(thread)
		while not (sock.do_close or sock.closed) do
			yield() -- direct coro.yield, not coro.yieldFunction
			if not (sock.do_close or sock.closed) then
				answer(sock) -- answer() or resume may close the socket
				if not sock.closed then
					releaseLock(sock, true) -- true means "allow no locks"
				end
			end
		end
		if debugLevel > 0 then
			print("-- closing %s, socket: %s%s%s, thread count: %d, connection count: %d", threadId(thread), tostring(sock.socket or sock.closed), sock.closed and " (closed)" or "", (sock.host and ">" or "<") or "-", coro.threadCount(), dconn.connectionCount())
		end
		dconn.disconnectAll(thread)
		if sock.do_close then
			closeSocket(sock)
		end
		local sockArr = threadSocketArr(thread)
		if #sockArr > 0 then
			for _, sock2 in pairs(sockArr) do
				if not sock2.closed then
					print("  -- closing %s additional unclosed socket, socket: %s%s%s", threadId(thread), tostring(sock2.socket or sock2.closed), sock2.closed and " (closed)" or "", (sock2.host and ">" or "<") or "-")
					closeSocket(sock2)
				end
			end
		end
		coroClose(thread)
	end
end

local function addSocket(sock, socketType)
	if not useCoro then
		return
	end
	local coroFunction
	if socketType == "listen" then
		return -- 'listen' is always in the main thread
	elseif socketType == "accept" then
		coroFunction = acceptLoop
		if useCoro then
			createSocketCoroutine(sock, coroFunction(sock))
		end
	elseif socketType == "connect" then
		local threadSocket, thread, nativeThread = currentThreadSocket()
		if sock.thread == nil then
			if thread == mainThread then
				sock.main_thread = thread
				return
			end
			sock.thread = nativeThread
		end
		if threadSocket == nil then
			util.printWarning("server connect socket '%s' coroutine '%s' was not found", tostring(sock.socket), threadId(thread))
		elseif thread == mainThread then
			util.printError("server connect socket '%s' coroutine '%s' is main thread", tostring(sock.socket), threadId(thread))
		elseif threadSocket.thread ~= nativeThread then
			util.printError("server connect socket '%s' coroutine '%s' is not same as current socket thread", tostring(sock.socket), threadId(thread))
		end
	else
		util.printError("server add socket type '%s' is not 'accept' or 'connect'", tostring(socketType))
	end
	setSocket(sock)
end

local function resume(sock, operation)
	if operation == "in" and sock.do_close then
		operation = "in + close" -- for debug
	end
	local resumeOk, resumeError = coroResume(sock.thread) -- write in acceptLoop()
	if resumeOk ~= true then
		if sock.thread and coroStatus(sock.thread) == "dead" then
			util.printError("coroutine %s-callback, coroutine is dead, resume error: '%s', socket: '%s'", operation, tostring(resumeError), tostring(sock.socket))
			closeSocket(sock, "coroutine is dead")
		else
			util.printError("coroutine %s-callback resume error '%s', socket: '%s'", operation, tostring(resumeError), tostring(sock.socket))
		end
	end
end

local function setThreadSocksClosed(sock)
	if sock.socket_type ~= "accept" then
		sock.do_close = true
	else
		local thread = sock.thread
		local sockArr = threadSocketArr(thread)
		for _, sock2 in pairs(sockArr) do
			if not sock2.closed and not sock2.do_close then
				if debugLevel >= 0 then
					print("  -- set %s socket to be closed: %s%s%s", threadId(thread), tostring(sock2.socket), sock2.closed and " (closed)" or "", (sock.host and ">" or "<") or "-")
				end
				sock2.do_close = true
			end
		end
	end
end

local function outCallback(sock)
	-- runs this function when you can write out
	if debugLevel > 0 then
		print("outCallback, socket: '%s'", tostring(sock.socket))
	end
	pollOutCount = pollOutCount + 1
	if useCoro then
		if sock.thread == nil then -- coroIndex(sock.socket)
			util.printRed("coroutine out-callback resume error, socket has no coroutine: '%s'", tostring(sock.socket))
		else
			resume(sock, "out")
		end
	end
end

local function closeCallback(sock, reason)
	-- if debugLevel >= 0 then
	-- print("close_callback, socket: '%s'", tostring(sock.socket))
	-- end
	if sock.closed then
		return
	end
	pollCloseCount = pollCloseCount + 1
	if useCoro then
		if sock.thread == nil then
			if not sock.main_thread then
				util.printRed("close-callback: socket thread is missing, socket: '%s'", tostring(sock.socket))
			end
			-- closeSocket(sock)
			sock.do_close = true
			return
		else
			setThreadSocksClosed(sock)
			resume(sock, "close")
		end
	else
		closeSocket(sock, reason)
	end
end

local function inCallback(sock)
	pollInCount = pollInCount + 1
	if sock == listenSocketTcp then
		acceptSocket = sock:accept()
		--[[
		local result = acceptSocket:setsockopt(C.SOL_SOCKET, C.SO_NOSIGPIPE, 1)
		if result ~= 0 then
			acceptSocket:cleanup(result, "socket setsockopt SO_NOSIGPIPE failed with error: "..result)
		end
		]]
		if acceptSocket ~= nil then
			-- addSocket(acceptSocket, "accept")
			if debugLevel > 0 then
				print("  -- new client, ip:port = %s, socket=%s, %s", acceptSocket:socket_address(), tostring(acceptSocket.socket), threadId(acceptSocket.thread))
			end
		end
		-- until acceptSocket < 1 -- socket accept will give always the same socket na n windows
	elseif useCoro and not (useUdp and sock == listenSocketUdp) then
		-- TODO: fix udp
		-- elseif useCoro then
		if sock.thread == nil then
			if sock.main_thread then
				closeSocket(sock)
				return
			end
			util.printRed("in-callback: socket thread is missing, socket: '%s'", tostring(sock.socket))
			closeSocket(sock)
		else
			resume(sock, "in")
		end
	else
		answer(sock)
	end
	--[[ if from4d then
		yield4d()
	end ]]
end

local function errorCallback(sock, eventType)
	util.printWarning("*-*ERR: errorCallback: " .. eventType .. ", fd=" .. tostring(sock))
	-- runs this function when you can write out
	if eventType == "POLLNVAL" then
		closeSocket(sock)
	else -- eventType == "POLLERR"
		closeSocket(sock)
	end
	pollErrCount = pollErrCount + 1
end

local collectGarbage
do
	local usedMem, unit, msg
	collectGarbage = function(pollReturn) -- , tcpPollCount) -- , forceTest)
		-- in calling side: if not from4d then
		--[[ if forceTest or loc.collectGarbageTimeout > 0 and loc.collectGarbageAnswerCount < answerCount then
		local now, timeDiff, msg, usedMem
		now = dt.currentDateTime()
		timeDiff = dt.secondDifference(loc.prevCollectGarbageTime, now)
		if timeDiff > loc.collectGarbageTimeout then ]]
		-- if not from4d then
		usedMem = collectgarbage("count")
		unit = "kb"
		-- if usedMem > 10 ^ 6 then
		-- 	usedMem = usedMem / 1000
		-- 	unit = "Mb"
		-- end
		msg = "\n\n" .. dt.currentString() .. ", answer count: " .. formatNum(answerCount, 0) .. ", poll return: " .. pollReturn .. ", fd count: " .. formatNum(poll.fdCount(), 0) -- .. ", poll count: " .. formatNum(tcpPollCount, 0)
		if loc.collectGarbageRestartMemoryUse <= 0 then
			msg = msg .. l(" | used memory: %s %s", formatNum(usedMem, 1), unit)
		else
			msg = msg .. l(" | check garbage, used memory: %s %s", formatNum(usedMem, 1), unit)
		end
		--[[
				if true or loc.prevCollectGarbageTime == 0 then
					msg = msg.."\n\n"
				else
					msg = msg..", delay from previous: "..util.seconds_to_clock(timeDiff, 0).."\n"
				end
				]]
		-- ioWrite(msg)
		-- ioFlush() -- write to screen before possible crash
		print(msg)
		if loc.collectGarbageRestartMemoryUse <= 0 then
			collectgarbage() -- may crash here
			usedMem = collectgarbage("count")
			if unit == "Mb" then
				usedMem = usedMem / 1024
			end
			ioWrite(color(l(" | after garbage collection: %%{bright cyan}%s %s", formatNum(usedMem, 1), unit)) .. "\n")
		elseif usedMem >= loc.collectGarbageRestartMemoryUse then
			ioWrite(color("%{bright cyan}" .. l(" | check garbage, planned restart because used memory is bigger than %d kb", loc.collectGarbageRestartMemoryUse)))
			ioWrite("\n\n")
			ioFlush()
			os.exit(2) -- restart app
		end
		-- ioWrite("\n\n")
		-- end
		-- loc.collectGarbageAnswerCount = answerCount
		--[[ 	loc.prevCollectGarbageTime = now
		end
	end ]]
	end
end

local function printPoll(pollRet)
	if not from4d then
		if callCountFunction then
			prevPollPrintTime = seconds()
			if answerCount > loc.prevPollPrintCount + 50000 or prevPollPrintTime - loc.prevPollPrintTime > 10 then -- > 0.2
				loc.prevPollPrintCount = answerCount
				local msg
				if pollRet == 0 and not from4d then
					msg = dt.currentString() .. ", call: " .. formatNum(callCountFunction(), 0) .. ", poll: " .. formatNum(poll.pollCount(), 0) .. ", fd count:" .. poll.fdCount() .. ", collectgarbage() "
				else
					msg = dt.currentString() .. ", call: " .. formatNum(callCountFunction(), 0) .. ", poll: " .. formatNum(poll.pollCount(), 0) .. ", fd count: " .. poll.fdCount() .. " " -- ..(" ")::rep(21) -- empy text to overwrite ", collectgarbage()"
				end
				util.print(msg) -- util.printToSameLine(msg)
			end
			loc.prevPollPrintTime = prevPollPrintTime
		end
		if pollRet == 0 and not from4d then
			collectGarbage(" > collectgarbage\n")
		end
	end
end

local pollLoop
do
	local pollRet, noPollCount
	pollLoop = function() -- main loop
		if not listenSocketTcp then
			local err = "error: server has not been started - quit-lx-server"
			util.printRed("server error: '%s'", tostring(err))
			return err
		end
		pause = false
		repeat
			pollRet, noPollCount = pollFd()
			if pollRet < 0 then -- if pollRet ~= 0 then -- if pollRet < 0 then
				util.printError("  poll() return: " .. pollRet)
			end
			if pollRet > 0 or not pollAfterFunction then
				printPoll(pollRet)
			end
			if pollAfterFunction then
				pollAfterFunction()
			end
			if from4d then
				yield4d()
			elseif pollRet == 0 and noPollCount > 1 then
				collectGarbage(pollRet)
				--[[ else -- something came in or went out
					loc.prevCollectGarbageTime = dt.currentDateTime()
					end ]]
			end
		until stop or pause or (useProfiler and pollInCount > profilerEndCount)
	end
end

local function serverPause()
	pause = true
end

local function serverStop()
	-- util.printError("serverStop() called")
	stop = true
	closeSocket(listenSocketTcp) -- must do this to unbind listen
	if useUdp and listenSocketUdp then
		closeSocket(listenSocketUdp)
	end

	if useProfiler then
		profiler.stop()
	end

	local txt = "\n -- Server statistics --"
	txt = txt .. "\nanswerCount:          " .. answerCount
	txt = txt .. "\npoll.fdCount:        " .. poll.fdCount()
	txt = txt .. "\npollCount:            " .. poll.pollCount()
	txt = txt .. "\npollInCount:          " .. pollInCount
	txt = txt .. "\npollOutCount:         " .. pollOutCount
	txt = txt .. "\npollCloseCount:       " .. pollCloseCount
	txt = txt .. "\npollErrCount:         " .. pollErrCount
	txt = txt .. "\nfd add/remove count:  " .. poll.fdAddCount() .. "/" .. poll.fdRemoveCount()
	txt = txt .. "\ntotalBytesReceived:   " .. totalBytesReceived
	txt = txt .. "\ntotalBytesSent:       " .. totalBytesSent
	util.printInfo(txt)

	dconn.disconnectAll()
	if poll.fdCount() > 0 then
		poll.removeAll(socket.close) -- must close after statistics
	end
end

local function setTimeout(timeout)
	-- print("setTimeout: "..tostring(timeout))
	if timeout > 0 then
		poll.setTimeout(timeout)
	elseif timeout < 0 then
		util.printError(l("timeout '%s' < 0", tostring(timeout)))
		poll.setTimeout(0)
	elseif from4d then
		poll.setTimeout(0)
	else
		poll.setTimeout(1)
	end
end

local function printSystemInfo()
	local hardware = require "hardware"
	local wineVrs = ""
	if not from4d then
		local _, vrs = util.isWine("get-version")
		if vrs then
			wineVrs = ", Wine version: '" .. vrs .. "'"
		end
	end
	local json = require "json"
	local jsonLib = json.jsonLibraryName()
	local winVer = util.isWin() and " " .. hardware.windowsVersionString() or ""
	local lfs = require "lfs_load"
	--[[ local scrypt = require "scrypt"
	if type(scrypt) ~= "table" or scrypt.cpuperf == nil then ]]
	util.printInfo(ffi.os .. winVer .. " " .. ffi.arch .. wineVrs .. ", using: " .. jsonLib .. ", file library: " .. lfs._VERSION .. (useCoro and ", coroutine" or ""))
	--[[ else
		util.printInfo(ffi.os .. winVer .. " " .. ffi.arch .. wineVrs .. ", using: " .. jsonLib .. ", " .. lfs._VERSION .. "\nCPU performance: " .. formatNum(scrypt.cpuperf(), 0))
	end ]]
	hardware.printInfo()
end

local function serverRun(port_, debug_level_, timeout_, debugPrintChars_, pollAfterFunction_, answerFunction_, beforeCloseFunction_, callCountFunction_, startTime, pluginTime)
	loc.collectGarbageTimeout = util.prf("system/option.json").option.collect_garbage_timeout or 5 -- in seconds
	loc.collectGarbageRestartMemoryUse = util.prf("system/option.json").option.restart_memory_use_kb or 500000 -- in kilobytes
	if util.isWin() then
		local hardware = require "hardware"
		if not hardware.windowsMinVersion("vista") then
			local txt = "error: " .. l("socket server does not work in older operating system than Windows Vista, current operating system: '%s'", hardware.windowsVersionString())
			util.printRed(txt)
			return txt
		end
	end
	-- copy params to local vars, used in other functions
	tcpPort = port_ -- udp listens the same port as socket
	debugLevel = debug_level_ or testDebugLevel
	if useCoro then
		mainThread = coro.setMainThread(nil, debugLevel)
	end
	debugPrintChars = debugPrintChars_
	pollAfterFunction = pollAfterFunction_
	answerFunction = answerFunction_
	beforeCloseFunction = beforeCloseFunction_
	callCountFunction = callCountFunction_

	localIpAddress = net.getLocalIpAddress()

	useProfiler = debugLevel < 0
	if useProfiler then
		useProfiler = debugLevel
	end
	-- set poll timeout, callbacks and sockets
	setTimeout(timeout_)
	poll.setDebugLevel(debugLevel)
	poll.setInCallback(inCallback)
	poll.setOutCallback(outCallback)
	poll.setCloseCallback(closeCallback)
	poll.setErrorCallback(errorCallback)

	local mac = require"system/net".getMacAddress()
	util.printOk("Mac address: %s", peg.replace(tostring(mac), ":", ""))
	-- http://beej.us/guide/bgnet/output/html/multipage/syscalls.html#bind
	-- print("listen socket port: "..tcpPort)
	listenSocketTcp = socket.listen("0.0.0.0", tcpPort, "tcp")
	if not listenSocketTcp then
		return
	end
	-- poll.addFd(listenSocketTcp, listenEvents) -- add listen socket to poll arrays

	if useUdp then
		listenSocketUdp = socket.listen("0.0.0.0", tcpPort, "udp")
		if not listenSocketUdp then
			util.printError("udp listen failed")
		end
	end

	if useProfiler then
		profilerEndCount = 250000
		profiler = require("jit.p") -- run/jit.p is path
		local options = "fi1"
		local output = "server.txt"
		profiler.start(options, output)
	end
	local localIpAddressWithPort = peg.replace(localIpAddress, ",", ":" .. tcpPort .. ",") .. ":" .. tcpPort
	local txt = "NC socket server waiting on: http://127.0.0.1:" .. tcpPort .. "/,  http://" .. peg.replace(localIpAddressWithPort, ", ", "/,  http://") .. "/"
	if socket.setupTls() then
		txt = txt .. "\n                             https://127.0.0.1:" .. tcpPort .. "/, https://" .. peg.replace(localIpAddressWithPort, ", ", "/, https://") .. "/"
	end
	util.printOk(txt)
	if listenSocketUdp then
		util.printOk("NC udp server waiting on:    127.0.0.1:" .. tcpPort .. "/, " .. localIpAddressWithPort .. "/")
	end
	startTime = tm.seconds(startTime)
	util.printInfo("--- server started in %%{reset}%.4f seconds%%{bright cyan}, plugins loaded in %%{reset}%.4f seconds%%{bright cyan}, %s, used memory %%{reset}%.1f kb%%{bright cyan} ---", startTime, pluginTime, dt.currentString(), collectgarbage("count"))
	printSystemInfo()
	local copyrightSign = util.isWin() and "(C)" or "Ⓒ"
	util.printOk("\nCapacic server. Copyright %s 2021-2024 Capacic Ltd. All Rights Reserved.\n", copyrightSign) -- U+24B8 Ⓒ is bigger than U+00A9 © https://en.wikipedia.org/wiki/Copyright_symbol

	local ret = pollLoop()
	if ret == true then -- main loop
		return "ok"
	end
	return ret -- error
end

return {currentThreadSocket = currentThreadSocket, closeSocket = closeSocket, socketClosed = socketClosed, collectGarbage = collectGarbage, pollLoop = pollLoop, addSocket = addSocket, serverPause = serverPause, serverStop = serverStop, setTimeout = setTimeout, serverRun = serverRun}
