--- lib/system/socket.lua
--
-- http://tangentsoft.net/wskfaq/
-- http://tangentsoft.net/wskfaq/advanced.html#backlog
-- https://msdn.microsoft.com/en-us/library/windows/desktop/aa365603(v=vs.85).aspx
--
local socket = {__name = "system/socket"} -- set metatable to new socket in create() function
local ffi = require "mffi"
local C = ffi.C
local stringBuffer = require "string.buffer" -- luacheck: ignore
local util = require "util"
local fs = require "fs"
local peg = require "peg"
local bit = require "bit"
local net = require "system/net"
local fdClient = require "net/fd-client"
local fdSend = require "net/fd-send"
local crypto = require "system/crypto"
local tls = crypto.tls -- we need only tls part here
local l = require"lang".l
local dprf = require "dprf"
local coro = require "coro"
local poll -- forward declaration
local coroYield = coro.yieldFunction(1, false) -- yield every 1 calls, do NOT resume without real poll event
local currentThread = coro.currentThreadFunction()
local threadId = coro.threadId
local netErrorText = net.errorText
local seconds = util.seconds
local sleep = util.sleep
local clearRecord = util.clearRecord
local isWin = util.isWin()
local print = util.print
local ioWrite = io.write
local debugCloseSocket = false
local defaultSocketReadTimeoutSeconds, loopPrintInterval
-- local floor = math.floor

local usePollout = not util.from4d()

local debugLevel = 0
function socket.setDebugLevel(level)
	debugLevel = level
end

local isTest = false
function socket.setTest(level)
	isTest = level
end

local socketIdx = {} -- we do not need thread as index because system sockets are unique
socket.socketIdxAnchor = socketIdx -- prevent garbage collect using socketIdxAnchor
local useWinNativeMethod = nil
local argIntC = ffi.newAnchor("int[1]")

local function useNativeMethod(pref)
	useWinNativeMethod = false
	pref = pref or {} -- util.prf("system/option.json").option -- load loop error here
	if isWin then
		if pref.use_native_method_win ~= nil then
			useWinNativeMethod = pref.use_native_method_win == true
		else
			useWinNativeMethod = false
		end
		if useWinNativeMethod then -- and not util.from4d() then
			-- util.printInfo(l "Using native Windows sockets.")
			print("Using native Windows sockets.")
		end
	elseif pref.use_native_method_osx == true and util.isMac() then
		-- util.printInfo(l "Using native OSX kqueue.")
		print("Using native OSX kqueue.")
	elseif pref.use_native_method_osx == true and util.isLinux() then
		-- util.printInfo(l "Using native Linux epoll.")
		print("Using native Linux epoll.")
	end
end

useNativeMethod()

local function clearSockRecord(sock)
	--[[ if sock.lock_topic then
		util.printWarningWithCallPath("closed socket '%s', %s has active lock '%s'", tostring(sock.socket), threadId(sock.thread), sock.lock_topic)
		coro.releaseLock(sock)
	end ]]
	local sockNum = sock.socket or sock.closed
	if socket.send_buf then
		socket.send_buf:free()
	end
	if socket.receive_buf then
		socket.receive_buf:free()
	end
	if socket.receive_data then
		socket.receive_data:free()
	end
	-- local option = sock.option
	clearRecord(sock)
	-- sock.option = option
	sock.closed = sockNum
end

local function loadPoll()
	if poll == nil then
		poll = require "system/poll"
	end
end

local function allocate(n)
	return C.malloc(n)
end

local function realloc(ptr, n)
	return C.realloc(ptr, n)
end

local errorPrefix = "   SOCKET ERROR: "
local INVALID_SOCKET
local closeFunction
if isWin then
	C = util.loadDll("ws2_32")
	if C == nil then
		util.printError("C is nil")
	end
	local C2 = ffi.loadMsvcr()
	if C2 == nil then
		util.printError("C is nil")
	end
	allocate = function(n)
		return C2.malloc(n)
	end
	realloc = function(ptr, n)
		return C2.realloc(ptr, n)
	end
	-- C = ffi.loadMsvcr()
	INVALID_SOCKET = ffi.newNoAnchorBasic("SOCKET", bit.bnot(0))
	closeFunction = C.closesocket
else
	INVALID_SOCKET = -1
	closeFunction = C.close
end
if tls == nil then
	util.printError("tls is nil")
end
socket.INVALID_SOCKET = INVALID_SOCKET

local kb = 1024
local socketSendBufferSize = 256 * kb
local socketReceiveBufferSize = 256 * kb
local receiveDataInitialSize = socketReceiveBufferSize * 12 -- when sendBufferSize 655360 b = 640kb, receiveBufferSize * 11 is enough for big pg test

local Mb = 1024 * 1024
local minDebugSendSize = 2 * kb
local receiveBufferSize = 32 * Mb

local receiveBuffer = allocate(receiveBufferSize)
if not receiveBuffer then
	util.printError("socket receive buffer create failed, buffer size: " .. receiveBufferSize)
end
local receiveBufferPtr = ffi.cast("void *", receiveBuffer)

local sendBufferSize = 32 * Mb
local sendBuffer = allocate(sendBufferSize)
local sendBufferPtr = ffi.cast("void *", sendBuffer)
local sendBufferSendPtr = ffi.cast("const void *", sendBuffer)
-- prevent garbage collection using anchor:
socket.anchor = {receiveBuffer = receiveBuffer, receiveBufferPtr = receiveBufferPtr, sendBuffer = sendBuffer, sendBufferPtr = sendBufferPtr, sendBufferSendPtr = sendBufferSendPtr}

--[[
-- local SOCKET_ERROR	= -1	-- 0xffffffff
local sendBufLen = 32 * 1024 * 1024 -- x Mb
local receiveBufferSize = sendBufLen -- 128*1024 -- 128kb
-- create answer buffer and default answer
local recvBuffer = allocate(receiveBufferSize)
if not recvBuffer then
	util.printError("Receive buffer createBuffer() failed, buffer length: " .. receiveBufferSize)
end
local sock.receive_buf = ffi.cast("void *", recvBuffer)

local sendBuffer = allocate(sendBufLen)
if not sendBuffer then
	util.printError("Send buffer createBuffer() failed, buffer length: " .. sendBufLen)
end
local sendBufferPtr = ffi.cast("void *", sendBuffer)
local sendBufSendPtr = ffi.cast("const void *", sendBuffer)
local anchor = {recvBuffer = recvBuffer, sock.receive_buf = sock.receive_buf, sendBuffer = sendBuffer, sendBufferPtr = sendBufferPtr, sendBufSendPtr = sendBufSendPtr}
socket.anchor = anchor ]]
--[[ local function printBuf()
	util.printTable(anchor, "anchor")
end ]]

local tcpNoDelay = 1
local tcpQuickAck = 1
local sendFlags, receiveFlags = 0, 0
if util.isLinux() then
	sendFlags = C.MSG_NOSIGNAL
	receiveFlags = C.MSG_NOSIGNAL
end

do
	local err = net.initialize()
	if err ~= 0 then
		net.printError(err, "error in net.initialize(): ")
	end
end

local function closeSocketByNumber(socketNum)
	if socketIdx[tostring(socketNum)] == nil then
		util.printWarning("socket '%s' does not exist in socket array", tostring(socketNum))
	else
		socketIdx[tostring(socketNum)] = nil
	end
	-- local socket_c = ffi.cast("int", socketNum)
	fdClient.closeWorkerSocket(socketNum)
	if debugCloseSocket and not util.from4d() then
		util.printOk("closed socket '%s'", tostring(socketNum))
	end
	return closeFunction(socketNum) -- C.close(socketNum)
end

function socket:closeSocket(showWarning)
	if self.closed then
		util.printWarningWithCallPath("socket:closeSocket(), self.closed is true")
		return 0
	end
	loadPoll()
	self.do_close = true
	poll.removeFd(self)
	local ret
	if self.socket then
		ret = closeSocketByNumber(self.socket)
	else
		ret = 0
		if showWarning ~= false then
			util.printWarning("socket:closeSocket(), self.socket is nil")
		end
	end
	clearSockRecord(self)
	return ret
end

function socket.shutdown(sock, how)
	local ret
	if sock.socket then
		how = how or C.SHUT_RDWR
		local time = seconds()
		ret = C.shutdown(sock.socket, how) -- ret = closeSocketByNumber(sock.socket)
		util.printInfo("  shutting down socket '%s' in %.6f seconds", tostring(sock.socket), seconds(time))
		socketIdx[tostring(sock.socket)] = nil
	else
		ret = 0
		-- if showWarning ~= false then
		util.printWarning("socket:shutdown(), sock.socket is nil")
		-- end
	end
	loadPoll()
	sock.shutdown = true
	poll.removeFd(sock)
	clearSockRecord(sock)
	return ret
end

function socket.closeAll()
	-- called from poll.lua program quitting __gc, so no need to check for threads
	util.printInfo("quitting program: closing all sockets")
	for _, sock in pairs(socketIdx) do
		sock.socket.do_close = true
		sock.socket:shutdown()
	end
end

function socket.eagain(err)
	if useWinNativeMethod then
		return err == C.WSAEWOULDBLOCK
	end
	return err == C.EAGAIN
end

function socket.econnreset(err)
	if useWinNativeMethod then
		return err == C.WSAECONNRESET -- C.WSAEPIPE does not exist
	end
	return err == C.ECONNRESET or err == C.EPIPE or util.isLinux() and err == 0
end

local lastError
if useWinNativeMethod then
	lastError = function()
		return C.WSAGetLastError()
	end
else
	lastError = function()
		return ffi.errno()
	end
end
socket.lastError = lastError

function socket:invalidSocket()
	return self.socket == INVALID_SOCKET
end

--[[
local function LOWBYTE(word)
	return band(word, 0xff)
end

local function HIGHBYTE(word)
	return band(rshift(word,8), 0xff)
end
]]

do
	local findIdx
	local function findSocket(socketNum)
		findIdx = socketIdx[tostring(socketNum)]
		if findIdx == nil then
			util.printError("socket was not found: " .. tostring(socketNum))
			util.printTable(socketIdx, "socketIdx")
		end
		return findIdx
	end
	socket.find = findSocket
end

if isWin then
	local IOCPARM_MASK = 0x7f -- parameters must be < 128 bytes
	-- local IOC_VOID        = 0x20000000      -- no parameters
	-- local IOC_OUT         = 0x40000000      -- copy out parameters
	local IOC_IN = 0x80000000 -- copy in parameters
	-- local IOC_INOUT       = bor(IOC_IN,IOC_OUT)
	local function _IOW(x, y, t)
		return bit.bor(IOC_IN, bit.lshift(bit.band(ffi.sizeof(t), IOCPARM_MASK), 16), bit.lshift(x, 8), y)
	end
	local FIONBIO = _IOW(string.byte "f", 126, "uint32_t") -- set/clear non-blocking i/o

	-- require "win_socket"
	local wsa_err_num, wsa_err_text
	function socket.cleanup(sock, errnum, errtext)
		-- get WSAGetLastError() before close and WSACleanup
		wsa_err_num = C.WSAGetLastError()
		wsa_err_text = netErrorText(wsa_err_num)
		if errnum and errnum ~= -1 and errnum ~= wsa_err_num then
			wsa_err_text = netErrorText(wsa_err_num) .. ", WSAGetLastError: " .. tonumber(wsa_err_num) .. ". " .. wsa_err_text
		end
		sock:closeSocket() -- socket.destroy(sock)
		C.WSACleanup()
		if errtext and #errtext > 0 then
			print(errorPrefix .. errtext .. "(" .. tonumber(errnum) .. ") " .. wsa_err_text)
		end
	end

	function socket:setBlocking(arg)
		--[[
		WSAIoctl(self:getNativeSocket(), SIO_KEEPALIVE_VALS,
		keeper, ffi.sizeof(tcp_keepalive),
		outbuff, outbuffsize,
		pbytesreturned);
		]]
		if arg == false then
			arg = 1 -- 1 == non-blocking
		elseif arg == true then
			arg = 0 -- 1 == blocking
		else
			util.printError("setBlocking mode must be false or true")
			return
		end
		argIntC[0] = arg
		return C.ioctlsocket(self.socket, FIONBIO, argIntC) -- FIONBIO in win_socket.lua
	end
else
	-- unix

	function socket.cleanup(sock, errnum, errtext, option)
		-- print("socket.cleanup: ")
		sock:closeSocket() -- socket.destroy(sock)
		if errtext and #errtext > 0 then
			if option == "no-trace" then
				util.printRed(errorPrefix .. errtext .. "(" .. tonumber(errnum) .. ") " .. netErrorText(errnum))
			else
				util.printError(errorPrefix .. errtext .. "(" .. tonumber(errnum) .. ") " .. netErrorText(errnum))
			end
		end
	end

	function socket:setBlocking(arg)
		local flags = C.fcntl(self.socket, C.F_GETFL, 0)
		if flags < 0 then
			return flags
		end
		if arg == false then
			flags = bit.bor(flags, C.O_NONBLOCK)
		elseif arg == true then
			flags = bit.band(flags, bit.bnot(C.O_NONBLOCK))
		else
			util.printError("setBlocking mode must be false or true")
			return
		end
		-- local flags_c = ffi.newNoAnchor("int", flags) -- or ffi.cast() ?
		return C.fcntl(self.socket, C.F_SETFL, flags) -- ffi.cast("int", flags)) -- , flags_c)
		--[[ https://learn.microsoft.com/en-us/windows/win32/api/winsock/nf-winsock-ioctlsocket?redirectedfrom=MSDN
        Set the socket I/O mode: In this case FIONBIO
        enables or disables the blocking mode for the
        socket based on the numerical value of iMode.
        If iMode = 0, blocking is enabled;
        If iMode != 0, non-blocking mode is enabled. ]]
		-- local iMode = ffi.newNoAnchor("u_long[1]")
		-- local ret = C.ioctlsocket(self.socket, C.FIONBIO, iMode);
		--[[ if (ret != NO_ERROR)
      printf("ioctlsocket failed with error: %ld\n", iResult);
    } ]]
	end
end
local cleanup = socket.cleanup
--[[
		-- http://www.winsocketdotnetworkprogramming.com/winsock2programming/winsock2advancediomethod5e.html
		-- http://www.winsocketdotnetworkprogramming.com/winsock2programming/winsock2advancediomethod5f.html
		-- http://www.winsocketdotnetworkprogramming.com/winsock2programming/winsock2advancediomethod5g.html
		-- functions we may need
		WSADATA wsaData;
		WSACreateEvent() // WSA_INVALID_EVENT
		WSAResetEvent(EventArray[0])
		WSASetEvent(EventArray[0])
    WSACloseEvent(EventArray[Index - WSA_WAIT_EVENT_0])
		CreateThread(NULL, 0, ProcessIO, NULL, 0, &ThreadId)
		GlobalAlloc(GPTR, sizeof(SOCKET_INFORMATION)))
		GlobalFree(SI)
		InitializeCriticalSection(&CriticalSection)
		EnterCriticalSection(&CriticalSection)
		LeaveCriticalSection(&CriticalSection)
		WSASocket(AF_INET, SOCK_STREAM, 0, NULL, 0, WSA_FLAG_OVERLAPPED)
		WSASend(SI->Socket, &(SI->DataBuf), 1, &SendBytes, 0, &(SI->Overlapped), NULL) == SOCKET_ERROR
		WSARecv(SocketArray[EventTotal]->Socket, &(SocketArray[EventTotal]->DataBuf), 1, &RecvBytes, &Flags, &(SocketArray[EventTotal]->Overlapped), NULL) == SOCKET_ERROR
		WSAWaitForMultipleEvents(EventTotal, EventArray, FALSE,  WSA_INFINITE, FALSE) == WSA_WAIT_FAILED
		WSAGetOverlappedResult(SI->Socket, &(SI->Overlapped), &BytesTransferred, FALSE, &Flags) == FALSE || BytesTransferred == 0)

	]]
local function createSocketBuffres(sock)
	if sock.receive_data == nil then
		sock.receive_data = stringBuffer.new(receiveDataInitialSize)
	end
	--[[ if sock.send_buf == nil then
		sock.send_buf = stringBuffer.new(sendSocketBufferSize) -- , bufferOptions
		sock.send_buf_size = sendSocketBufferSize
	end ]]
end

local create
if useWinNativeMethod then
	create = function(self) -- ai_family, ai_socktype, ai_protocol, flags)
		-- https://msdn.microsoft.com/en-us/library/windows/desktop/ms742212(v=vC.85).aspx
		-- ai_family AF_INET = 2, AF_INET6 = 23
		-- ai_socktype SOCK_STREAM = 1, SOCK_DGRAM = 2
		-- ai_protocol IPPROTO_TCP = 6, IPPROTO_UDP = 17
		if not self.socket then
			self.socket = C.WSASocketA(self.ai_family, self.ai_socktype, self.ai_protocol, nil, 0, self.flags or 0) -- WSA_FLAG_OVERLAPPED
		end
		createSocketBuffres(self)
		local sock = tostring(self.socket)
		if self.socket == INVALID_SOCKET then
			util.printError("socket '%s' is invalid socket", tostring(sock.socket))
		end
		if socketIdx[sock] then
			util.printError("socket '%s' was already in socket array", tostring(sock.socket))
		end
		socketIdx[sock] = {thread = currentThread(), socket = setmetatable(self, {__index = socket})} -- setmetatable returns self
		return self
	end
else
	create = function(self) -- ai_family, ai_socktype, ai_protocol, flags)
		if not self.socket then
			self.socket = C.socket(self.ai_family, self.ai_socktype, self.ai_protocol)
		end
		createSocketBuffres(self)
		local sock = tostring(self.socket)
		if socketIdx[sock] then
			util.printError("socket '%s' was already in socket array", tostring(sock.socket))
		end
		socketIdx[sock] = {thread = currentThread(), socket = setmetatable(self, {__index = socket})}
		return self
	end
end
socket.create = create
--[[
function socket.destroy(sock)
	sock:closeSocket()
	for key in pairs(sock) do
		sock[key] = nil
	end
	-- sock = nil -- does this help anything?
end
]]
function socket:bind(sockaddr, addrLen)
	-- https://msdn.microsoft.com/en-us/library/windows/desktop/ms740621(v=vC.85).aspx
	return C.bind(self.socket, sockaddr, addrLen) -- same for win and mac
end

--[[ local sockAddr = ffi.newAnchor("struct sockaddr_in[1]") -- ffi.newNoAnchor("struct sockaddr_storage")
local addr_len = ffi.newAnchor("socklen_t[1]") -- unsigned int[1]")
addr_len[0] = ffi.sizeof(sockAddr) ]]
--[=[
if useWinNativeMethod then
	function socket:accept(sockaddr, addrLen)
		-- https://msdn.microsoft.com/en-us/library/windows/desktop/ms737524(v=vs.85).aspx
		--[[
		BOOL AcceptEx(
			_In_  SOCKET       sListenSocket,
			_In_  SOCKET       sAcceptSocket,
			_In_  PVOID        lpOutputBuffer,
			_In_  DWORD        dwReceiveDataLength,
			_In_  DWORD        dwLocalAddressLength,
			_In_  DWORD        dwRemoteAddressLength,
			_Out_ LPDWORD      lpdwBytesReceived,
			_In_  LPOVERLAPPED lpOverlapped
		);

		local dwReceiveDataLength = 0
		local dwLocalAddressLength = ffi.sizeof("struct sockaddr_in")+16
		local dwRemoteAddressLength = ffi.sizeof("struct sockaddr_in")+16
		local lpOutputBufferLen = dwReceiveDataLength + dwLocalAddressLength + dwRemoteAddressLength
		local lpOutputBuffer = ffi.newNoAnchor("char[?]", lpOutputBufferLen)
		local lpdwBytesReceived = ffi.newNoAnchor("DWORD[1]")
		local lpOverlapped = self:createOverlapped(lpOutputBuffer, lpOutputBufferLen, SocketOps.ACCEPT)
		-- local mswsock = require("mswsock");
		local status = mswsock.AcceptEx(self:getNativeSocket(), newsock,
			lpOutputBuffer,
			dwReceiveDataLength,
			dwLocalAddressLength,
			dwRemoteAddressLength,
			lpdwBytesReceived,
			ffi.cast("OVERLAPPED *", lpOverlapped))
		local err = ws2_32.WSAGetLastError()
		-- print("NativeSocket.accept, AcceptEx(), STATUS, ERR: ", newsock, status, err);
		if err ~= WSA_IO_PENDING then
			return false, err;
		end
		-- If we've gotten this far, it means an accept was queued
		-- so we should yield, and we'll continue when completion is indicated
		local key, bytes, ovl = waitForIO(self, lpOverlapped);
		-- print("++ NativeSocket.accept(), after waitForIO: ", key, bytes, ovl)
		-- if no error, then return the socket
    return newsock;
		--]]
		local newSockNum = C.accept(self.socket, sockaddr, addrLen)
		if newSockNum ~= INVALID_SOCKET then
			local newSock = util.clone(self)
			newSock.socket = newSockNum
			-- newSock.sockaddr = sockaddr
			-- newSock.addrLen = addrLen
			return create(newSock)
		end
	end
else
	function socket:accept(sockaddr, addrLen)
		local newSockNum = C.accept(self.socket, sockaddr, addrLen)
		if newSockNum ~= INVALID_SOCKET then
			local newSock = util.clone(self)
			newSock.socket = newSockNum
			-- newSock.sockaddr = sockaddr
			-- newSock.addrLen = addrLen
			return create(newSock)
		end
	end
end
]=]

local function resizeReceiveBuffer(len)
	util.printError("length %d is bigger than receive buffer size %d", len, receiveBufferSize)
	return receiveBufferSize
	--[[ C.free(receiveBuffer)
	receiveBuffer = allocate(len)
	receiveBufferPtr = ffi.cast("char *", receiveBuffer) ]]
end

local function receiveLength(flags, len)
	-- recv has len because flags can be MSG_PEEK
	if flags and len then
		if len > receiveBufferSize then
			len = resizeReceiveBuffer(len)
		end
	elseif len then
		util.printError("socket recv can't have length without flags like MSG_PEEK")
		len = receiveBufferSize
	else
		len = receiveBufferSize
	end
	return len
end

local recv
if useWinNativeMethod then
	local dwBufferCount = 1
	local numberOfBytesReceived = ffi.newAnchor("DWORD[1]")
	local lpFlags = ffi.newAnchor("DWORD[1]")
	local lpBuffers = ffi.newAnchor("WSABUF")
	local ret
	recv = function(sock, flags, len)
		-- https://msdn.microsoft.com/en-us/library/windows/desktop/ms741688(v=vs.85).aspx
		--[[
			int WSARecv(
				_In_    SOCKET                             s,
				_Inout_ LPWSABUF                           lpBuffers,
				_In_    DWORD                              dwBufferCount,
				_Out_   LPDWORD                            lpNumberOfBytesRecvd,
				_Inout_ LPDWORD                            lpFlags,
				_In_    LPWSAOVERLAPPED                    lpOverlapped,
				_In_    LPWSAOVERLAPPED_COMPLETION_ROUTINE lpCompletionRoutine
			);
		--]]
		if sock.socket == nil then
			util.printError("sock recveive data failed, socket is nil")
			return -1
		end
		lpBuffers.len = receiveLength(flags, len)
		lpBuffers.buf = receiveBufferPtr
		lpFlags[0] = flags or 0
		numberOfBytesReceived[0] = 0
		-- local lpBuffers = ffi.newNoAnchor("WSABUF", bufflen, ffi.cast("char *", buffer))
		ret = C.WSARecv(sock.socket, lpBuffers, dwBufferCount, numberOfBytesReceived, lpFlags, nil, -- lpOverlapped, -- ffi.cast("OVERLAPPED *",lpOverlapped),
		nil -- lpCompletionRoutine)
		)
		if numberOfBytesReceived[0] == 0 then
			return numberOfBytesReceived[0]
		end
		if C.WSAGetLastError() == C.WSAEWOULDBLOCK then
			return numberOfBytesReceived[0]
		end
		net.printError()
		return ret
	end
else
	recv = function(sock, flags, len)
		if sock.socket == nil then
			util.printRed("socket recv() socket is nil")
			return -100
		end
		return tonumber(C.recv(sock.socket, receiveBufferPtr, receiveLength(flags, len), flags))
	end
end

local recvFrom
if useWinNativeMethod then
	local dwBufferCount = 1
	local numberOfBytesReceived = ffi.newAnchor("DWORD[1]")
	local lpFlags = ffi.newAnchor("DWORD[1]")
	local lpBuffers = ffi.newAnchor("WSABUF")
	local ret
	recvFrom = function(sock, flags, addr, addrLen, len)
		-- https://msdn.microsoft.com/en-us/library/windows/desktop/ms741686(v=vs.85).aspx
		--[[
			int WSARecvFrom(
				_In_    SOCKET                             s,
				_Inout_ LPWSABUF                           lpBuffers,
				_In_    DWORD                              dwBufferCount,
				_Out_   LPDWORD                            lpNumberOfBytesRecvd,
				_Inout_ LPDWORD                            lpFlags,
				_Out_   struct sockaddr                    *lpFrom,
				_Inout_ LPINT                              lpFromlen,
				_In_    LPWSAOVERLAPPED                    lpOverlapped,
				_In_    LPWSAOVERLAPPED_COMPLETION_ROUTINE lpCompletionRoutine
			);
		--]]
		lpBuffers.len = receiveLength(flags, len)
		lpBuffers.buf = receiveBufferPtr
		if flags then
			lpFlags[0] = flags
		end
		numberOfBytesReceived[0] = 0
		-- local lpNumberOfBytesReceived = ffi.cast("LPDWORD", numberOfBytesReceived) -- nil, can be nil only if overlapped is in use
		ret = C.WSARecvFrom(sock.socket, lpBuffers, dwBufferCount, numberOfBytesReceived, -- lpNumberOfBytesReceived,
		lpFlags, ffi.cast("struct sockaddr *", addr), -- call will change this, can not be "const struct sockaddr *"
		addrLen, nil, -- lpOverlapped, -- ffi.cast("OVERLAPPED *",lpOverlapped),
		nil) -- lpCompletionRoutine)
		if numberOfBytesReceived[0] == 0 then
			return numberOfBytesReceived[0], addr
		end
		if C.WSAGetLastError() == C.WSAEWOULDBLOCK then
			return numberOfBytesReceived[0], addr
		end
		net.printError()
		return ret, addr
	end
else
	recvFrom = function(sock, flags, addr, addrLen, len)
		--[[extern ssize_t recvfrom (int __fd, void *__restrict __buf, size_t __n,
			int __flags, struct sockaddr *__restrict __addr,
			socklen_t *__restrict __addr_len);]]
		-- socket.getpeername(sock, ffi.cast("struct sockaddr *", addr), len_c)
		-- call will change addr, can not be "const struct sockaddr *"
		return tonumber(C.recvfrom(sock.socket, receiveBufferPtr, receiveLength(flags, len), flags, ffi.cast("struct sockaddr *", addr), addrLen)), addr -- , addr_len
	end
end

--[[ local function socketError(sock, err)
	util.printError(err)
	util.printTable(sock, "socket.self")
	util.printTable(socketIdx, "socketIdx")
end ]]

--[[ local peekNoWait = bit.bor(C.MSG_PEEK, C.MSG_DONTWAIT)
local function checkSocketClosed(sock)
	local ret = C.recv(sock.socket, nil, 1, peekNoWait) -- sock:getsockopt(level, option_name, option_value, option_len)
	-- Upon successful completion, recv() shall return the length of the message in bytes. If no messages are available to be received and the peer has performed an orderly shutdown, recv() shall return 0. Otherwise, -1 shall be returned and errno set to indicate the error.
	if ret == 0 then
		return true
	end
	ret = ffi.errno()
	if ret == C.EAGAIN or ret == C.EWOULDBLOCK then
		return false
	end
	return true
end ]]

local sendBufferInfoSize = 2 * Mb
local send
local formatNum = util.formatNumFunction(0)
if useWinNativeMethod then
	-- lpOverlapped = nil -- self:createOverlapped(ffi.cast("uint8_t *",buff), bufflen, SocketOps.WRITE);
	-- lpCompletionRoutine = nil -- can be nil only if overlapped is in use
	local dwBufferCount = 1
	local lpBuffers = ffi.newAnchor("WSABUF")
	local numberOfBytesSent = ffi.newAnchor("DWORD[1]")
	local ret, dwFlags
	send = function(sock, len, flags)
		-- https://msdn.microsoft.com/en-us/library/windows/desktop/ms742203(v=vs.85).aspx
		--[[
			int WSASend(
				_In_  SOCKET                             s,
				_In_  LPWSABUF                           lpBuffers,
				_In_  DWORD                              dwBufferCount,
				_Out_ LPDWORD                            lpNumberOfBytesSent,
				_In_  DWORD                              dwFlags,
				_In_  LPWSAOVERLAPPED                    lpOverlapped,
				_In_  LPWSAOVERLAPPED_COMPLETION_ROUTINE lpCompletionRoutine
			);
		--]]
		--
		lpBuffers.len = len
		lpBuffers.buf = sendBufferSendPtr
		if flags then
			dwFlags = ffi.cast("DWORD", flags)
		else
			dwFlags = nil
		end
		numberOfBytesSent[0] = 0
		ret = C.WSASend(sock.socket, lpBuffers, -- ffi.newNoAnchor("WSABUF", bufflen, ffi.cast("char *", buffer)),
		dwBufferCount, numberOfBytesSent, -- lpNumberOfBytesSent,
		dwFlags, nil, -- lpOverlapped, -- ffi.cast("OVERLAPPED *",lpOverlapped),
		nil -- lpCompletionRoutine)
		)
		if ret == 0 then
			return numberOfBytesSent[0]
		end
		if C.WSAGetLastError() == C.WSAEWOULDBLOCK then
			return numberOfBytesSent[0]
		end
		net.printError()
		return ret
	end
else
	local time
	--[[ local sendFlagsNoBlock
	if not isWin then
		sendFlagsNoBlock = bit.bor(C.MSG_DONTWAIT, C.O_NONBLOCK)
	end ]]
	send = function(sock, len, flags)
		if len >= sendBufferSize then
			util.printError("socket '%s' send length %s is bigger than send buffer size %s", tostring(sock.socket), formatNum(len), formatNum(sendBufferSize))
			len = sendBufferSize
		end
		if len >= sendBufferInfoSize then
			util.ioWrite("    starting socket '%s' send, length %s bytes...", tostring(sock.socket), formatNum(len))
			time = util.seconds()
		end
		--[=[ todo: test this code, previous was only if sock.do_close then
		coroYield(sock)
		if sock.closed then
			return -2, util.printRed("socket '%s' is closed, send length %d", tostring(sock.socket), len)
			--[[ elseif sock.do_close then --  send remaining data before real close
			return -2, util.printRed("socket '%s' will be closed, send length %d", tostring(sock.socket), len) ]]
		end -- ]=]
		--[[ if checkSocketClosed(sock) then
			return -2, util.printRed("socket '%s' is not alive any more, send length %d", tostring(sock.socket), len)
		end ]]
		--[[ if flags == 0 and not isWin then
			flags = sendFlagsNoBlock
		end ]]
		-- tls socket answer
		local ret
		if sock.tls_ctx then
			ret = tonumber(tls.tls_write(sock.tls_ctx, sendBufferSendPtr, len))
		else
			ret = tonumber(C.send(sock.socket, sendBufferSendPtr, len, flags))
		end
		if len >= sendBufferInfoSize then
			if ret ~= len then
				util.printRed("socket '%s' send done, result %d, length %s, in %.3f seconds", tostring(sock.socket), ret, formatNum(len), util.seconds(time))
			else
				util.printOk("socket '%s' send done, length %s, in %.3f seconds", tostring(sock.socket), formatNum(len), util.seconds(time))
			end
			time = nil
		end
		return ret
	end
	--[[
	local ret, ptr, bufLen
	send = function(sock, len, flags)
		ptr, bufLen = sock.send_buf:ref()
		if len > bufLen then
			util.printError("socket '%s' send buffer data size %d is smaller than send length %d", tostring(sock.socket), bufLen, len)
			len = bufLen
		elseif bufLen <= 0 then
			util.printError("socket '%s' send buffer does not contain data, send length %d bytes", tostring(sock.socket), len)
		end
		ret = tonumber(C.send(sock.socket, ptr, len, flags))
		if ret == len then
			sock.send_buf:reset()
		elseif ret > 0 then
			sock.send_buf = sock.send_buf:skip(ret)
		end
		return ret
	end ]]
end

local sendTo
if useWinNativeMethod then
	local dwBufferCount = 1
	local numberOfBytesSent = ffi.newAnchor("DWORD[1]")
	local lpBuffers = ffi.newAnchor("WSABUF")
	local ret, dwFlags
	sendTo = function(sock, len, flags, addr, addrLen)
		-- https://msdn.microsoft.com/en-us/library/windows/desktop/ms741693(v=vs.85).aspx
		--[[
			int WSASendTo(
				_In_  SOCKET                             s,
				_In_  LPWSABUF                           lpBuffers,
				_In_  DWORD                              dwBufferCount,
				_Out_ LPDWORD                            lpNumberOfBytesSent,
				_In_  DWORD                              dwFlags,
				_In_  const struct sockaddr              *lpTo,
				_In_  int                                iToLen,
				_In_  LPWSAOVERLAPPED                    lpOverlapped,
				_In_  LPWSAOVERLAPPED_COMPLETION_ROUTINE lpCompletionRoutine
			);
		--]]
		-- local lpBuffers = ffi.newNoAnchor("WSABUF", bufflen, ffi.cast("char *", buffer))
		lpBuffers.len = len
		lpBuffers.buf = sendBufferSendPtr -- ffi.cast("char *", buffer)
		if flags then
			dwFlags = ffi.cast("DWORD", flags)
		else
			dwFlags = nil
		end
		numberOfBytesSent[0] = 0
		-- local lpNumberOfBytesSent = ffi.cast("LPDWORD", numberOfBytesSent) -- nil, can be nil only if overlapped is in use
		ret = C.WSASendTo(sock.socket, lpBuffers, dwBufferCount, numberOfBytesSent, -- lpNumberOfBytesSent,
		dwFlags, ffi.cast("const struct sockaddr *", addr), addrLen, nil, -- lpOverlapped, -- ffi.cast("OVERLAPPED *",lpOverlapped),
		nil -- lpCompletionRoutine)
		)
		if ret == 0 then
			return numberOfBytesSent[0]
		end
		if C.WSAGetLastError() == C.WSAEWOULDBLOCK then
			return numberOfBytesSent[0]
		end
		net.printError()
		return ret
	end
else
	sendTo = function(sock, len, flags, addr, addrLen) -- , addr_len)
		--[[sendto (int __fd, __const void *__buf, size_t __n,
				 int __flags, __const struct sockaddr * __addr,
				 socklen_t __addr_len);]]
		return tonumber(C.sendto(sock.socket, sendBufferSendPtr, len, flags, ffi.cast("const struct sockaddr *", addr), addrLen))
	end
	--[[ sendTo = function (sock, len, flags, addr, addrLen)
		ptr, bufLen = sock.send_buf:ref()
		if len > bufLen then
			util.printError("socket '%s' send to -buffer data size %d is smaller than send length %d", tostring(sock.socket), bufLen, len)
			len = bufLen
		elseif bufLen <= 0 then
			util.printError("socket '%s' send to -buffer buffer does not contain data, send length %d bytes", tostring(sock.socket), len)
		end
		ret = tonumber(C.sendto(sock.socket, ptr, len, flags, ffi.cast("const struct sockaddr *", addr), addrLen))
		if ret == len then
			sock.send_buf:reset()
		elseif ret > 0 then
			sock.send_buf = sock.send_buf:skip(ret)
		end
		return ret
	end ]]
end

--[[ local function toTxt(txt)
	return txt ~= nil and ffi.string(txt) or tostring(txt)
end ]]

local function connectTls(host, port)
	local err = tls.tls_init()
	if err ~= 0 then
		return nil, nil, l("Could not initialize TLS library, error %n - '%s'", err, crypto.tlsError())
	end
	local ctx = tls.tls_client()
	if ffi.isNull(ctx) then
		return nil, nil, l("Could not create TLS client, error %n - '%s'", err, crypto.tlsError(ctx))
	end
	local tlsConfig = tls.tls_config_new()
	if ffi.isNull(tlsConfig) then
		return nil, nil, l("Could not create TLS config, error %n - '%s'", err, crypto.tlsError(ctx))
	end
	tls.tls_config_set_protocols(tlsConfig, C.TLS_PROTOCOLS_ALL)
	tls.tls_config_set_ciphers(tlsConfig, "secure") -- secure, compat, legacy, insecure, default
	tls.tls_config_insecure_noverifycert(tlsConfig)
	tls.tls_config_insecure_noverifyname(tlsConfig)
	err = tls.tls_configure(ctx, tlsConfig)
	if err ~= 0 then
		return nil, nil, l("Could not configure TLS library, error %n - '%s'", err, crypto.tlsError(ctx))
	end
	port = tostring(port)
	print(l("Start TLS connection with server: '%s:%s'", host, port))
	err = tls.tls_connect(ctx, host, port)
	if err ~= 0 then
		return nil, nil, l("Could not connect TLS, error %s, '%s'", tostring(err), crypto.tlsError(ctx))
	end
	--[[
	local buf = ffi.newNoAnchor("char[8192]")
	local msg = "HEAD / HTTP/1.1\r\n"..
							"Host: nc\r\n\r\n"..
							"Connection: close\r\n\r\n"
	local hello = msg
	print(l("Write to TLS socket: '%s'", msg))
	err = tls.tls_write(ctx, hello, #msg) -- ffi.sizeof(hello)
	if err ~= #msg then -- ffi.sizeof(hello)
		print("Could not write to TLS socket")
	end
	print(l("Read from TLS socket"))
	err = tls.tls_read(ctx, buf, 8192)
	if tonumber(err) < 0 then
		print("Could not read from TLS socket")
	end
	print(l("Returned read buffer from TLS socket: '\n%s\n'", ffi.string(buf):sub(1, 1000)))
	local txt = tls.tls_conn_version(ctx)
	print(l("tls_conn_version: '%s'", toTxt(txt)))
	txt = tls.tls_conn_cipher(ctx)
	print(l("tls_conn_cipher: '%s'", toTxt(txt)))
	txt = tls.tls_peer_cert_issuer(ctx)
	print(l("tls_peer_cert_issuer: '%s'", toTxt(txt)))
	txt = tls.tls_peer_cert_subject(ctx)
	print(l("tls_peer_cert_subject: '%s'", toTxt(txt)))
	--]]
	return ctx, tlsConfig
end
socket.connectTls = connectTls

--[=[
local connect
if useWinNativeMethod then
	connect = function(sock, sockaddr, address_len)
		--[[
		BOOL PASCAL ConnectEx(
			_In_     SOCKET                s,
			_In_     const struct sockaddr *name,
			_In_     int                   namelen,
			_In_opt_ PVOID                 lpSendBuffer,
			_In_     DWORD                 dwSendDataLength,
			_Out_    LPDWORD               lpdwBytesSent,
			_In_     LPOVERLAPPED          lpOverlapped
		);
		--]]
		-- https://msdn.microsoft.com/en-us/library/windows/desktop/ms737606(v=vC.85).aspx
		return C.connect(sock, sockaddr, address_len)
	end
else
	connect = function(sock, sockaddr, address_len)
		return C.connect(sock, sockaddr, address_len)
	end
end ]=]

function socket:getsockopt(level, option_name, option_value, option_len)
	return C.getsockopt(self.socket, level, option_name, option_value, option_len)
end

-- if isWin then -- and (option_name == C.SO_REUSEADDR or option_name == C.SO_SNDBUF) then
-- argIntC = ffi.cast("const char *", argIntC)
-- option_len = ffi.sizeof(argIntC)
-- end
local arg_cLen = ffi.sizeof(argIntC)
function socket:setsockopt(level, option_name, option_value)
	-- local argIntC, option_len
	-- argIntC = ffi.newNoAnchor("int[1]", option_value)
	argIntC[0] = option_value
	return C.setsockopt(self.socket, level, option_name, ffi.cast("const char*", argIntC), arg_cLen)
end

function socket:getpeername(name, namelen)
	if C.getpeername(self.socket, name, namelen) ~= 0 then
		net.printError()
	end
end

-- old system/socket.lua code ---

local sendTimeout = 45 -- will be loaded later
local sendNumber = 0

local function loadTls()
	if crypto == nil then
		sendTimeout = util.prf("system/option.json").option.tcp_send_timeout or 45 -- in seconds
		crypto = require "system/crypto"
		tls = crypto.tls -- may be nil == crypto load failed
	end
end

--[[
local function free(buf)
	--print("free: ", buf, tonumber(buf))
	C.free(buf)
end ]]

local newSocket
if useWinNativeMethod then
	newSocket = function(tbl) -- ai_family, ai_socktype, ai_protocol, flags)
		-- https://msdn.microsoft.com/en-us/library/windows/desktop/ms742212(v=vC.85).aspx
		-- ai_family AF_INET = 2, AF_INET6 = 23
		-- ai_socktype SOCK_STREAM = 1, SOCK_DGRAM = 2
		-- ai_protocol IPPROTO_TCP = 6, IPPROTO_UDP = 17
		if not tbl.socket then
			tbl.socket = C.WSASocketA(tbl.ai_family, tbl.ai_socktype, tbl.ai_protocol, nil, 0, tbl.flags or 0) -- WSA_FLAG_OVERLAPPED
		end
	end
else
	newSocket = function(tbl) -- ai_family, ai_socktype, ai_protocol, flags)
		if not tbl.socket then
			tbl.socket = C.socket(tbl.ai_family, tbl.ai_socktype, tbl.ai_protocol)
		end
	end
end

function socket.new(tbl)
	if tbl then
		newSocket(tbl)
		local sock = tostring(tbl.socket)
		if tbl.socket == INVALID_SOCKET then
			util.printError("socket '%s' is invalid socket", tostring(sock.socket))
		end
		if socketIdx[sock] then
			util.printError("socket '%s' was already in socket array", tostring(sock.socket))
		end
		socketIdx[sock] = {thread = currentThread(), socket = setmetatable(tbl, {__index = socket})} -- this sets functions to tbl and returns tbl
		return tbl
	end
	return setmetatable({}, {__index = socket})
end

local function close(sock)
	if type(sock) == "number" then
		util.printWarning("socket type '%s' is not a table, socket number: %d", type(sock), tostring(socket))
		closeSocketByNumber(sock)
	elseif type(sock) ~= "table" then
		util.printError("socket type '%s' is not a table", type(sock))
	elseif type(sock.close) ~= "function" then
		util.printError("sock:close type '%s' is not a function", type(sock))
	else
		-- util.printOk("close, sock:closeSocket(): "..tostring(sock).." - "..tostring(sock.socket))
		if sock.tls_ctx then
			tls.tls_close(sock.tls_ctx)
			tls.tls_free(sock.tls_ctx)
			tls.tls_config_free(sock.tls_config)
		end
		return sock:closeSocket(false) -- destroy(sock) -- sock:closeSocket()
	end
	clearSockRecord(sock)
	return 1 -- compatible with luaSocket
end
socket.close = close

--[[
-- FIX: with this code: http://beej.us/guide/bgnet/output/html/multipage/syscalls.html#bind
local sockAddr = ffi.newNoAnchor("struct sockaddr_in")
sockAddr.sin_family = C.AF_INET
sockAddr.sin_addr.s_addr = C.INADDR_ANY -- does not work in win without changing in_addr.S_addr to in_addr.s_addr
sockAddr.sin_port = net.htons(port)
]]

local tlsServer, tlsConfig
local function setupTls()
	local prf = dprf.prf("cert/tls.json")
	if util.from4d() and prf.use_tls_in_4d == false then
		return nil, l("use_tls_in_4d is set off in preference cert/tls.json")
	elseif prf.use_tls == false then
		return nil, l("use_tls is set off in preference cert/tls.json")
	end
	loadTls()
	if tls == nil then
		return nil, l("tls is not supported")
	end
	if tlsServer == nil then
		local caCertFile, keyFile
		if util.from4d() then
			caCertFile = util.path4d() .. prf["4d"].cert -- "crt": "Resources/Lua/preference/cert/nc-server.crt",
			keyFile = util.path4d() .. prf["4d"].key -- "key": "Resources/Lua/preference/cert/nc-server.key"
			if not fs.fileExists(caCertFile) or not fs.fileExists(keyFile) then
				caCertFile = nil
				keyFile = nil
			end
		end
		if caCertFile == nil then
			if prf.server.cert:sub(1, 1) == "/" then
				caCertFile = prf.server.cert -- "crt": "/xxx/server.crt",
			else
				caCertFile = util.mainPath() .. prf.server.cert -- "crt": "preference/cert/nc-server.crt",
			end
			if prf.server.key:sub(1, 1) == "/" then
				keyFile = prf.server.key -- "key": "/xxx/server.key"
			else
				keyFile = util.mainPath() .. prf.server.key -- "key": "preference/cert/nc-server.key"
			end
		end
		if tls.tls_init() ~= 0 then
			return nil, l("tls.tls_init() failed")
		end

		tlsServer = tls.tls_server()
		if ffi.isNull(tlsServer) then
			return nil, l("tls.tls_server() failed")
		end

		tlsConfig = tls.tls_config_new()
		if ffi.isNull(tlsConfig) then
			return nil, l("tlsConfig_new() failed")
		end

		local confRet = tls.tls_config_set_cert_file(tlsConfig, caCertFile)
		if confRet ~= 0 then
			local content = fs.readFile(caCertFile) or ""
			return nil, l("unable to set TLS certificate file '%s', content:\n'%s'", caCertFile, content)
		end
		confRet = tls.tls_config_set_ca_file(tlsConfig, caCertFile)
		if confRet ~= 0 then
			local content = fs.readFile(caCertFile) or ""
			return nil, l("unable to set root CA file '%s', content:\n'%s'", caCertFile, content)
		end
		confRet = tls.tls_config_set_key_file(tlsConfig, keyFile)
		if confRet ~= 0 then
			local content = fs.readFile(keyFile) or ""
			return nil, l("unable to set TLS key file '%s', content:\n'%s'", keyFile, content)
		end
		util.printInfo("TLS certificate files: %s, %s", caCertFile, keyFile)
		--[[ https://sortix.org/man/man3/tlsConfig_set_ciphers.3.html
			TLS_PROTOCOL_TLSv1_0
			TLS_PROTOCOL_TLSv1_1
			TLS_PROTOCOL_TLSv1_2
			Additionally, the values TLS_PROTOCOL_TLSv1 (TLSv1.0, TLSv1.1 and TLSv1.2), TLS_PROTOCOLS_ALL (all supported protocols) and TLS_PROTOCOLS_DEFAULT (TLSv1.2 only) may be used. (Client and server)
		]]
		tls.tls_config_set_protocols(tlsConfig, C.TLS_PROTOCOLS_DEFAULT)
		if tls.tls_configure(tlsServer, tlsConfig) ~= 0 then
			return nil, l("tls_configure() failed: '%s'", crypto.tlsError(tlsServer))
		end
		tls.tls_config_set_ciphers(tlsConfig, "default") -- secure, compat, legacy, insecure, default
		tls.tls_config_insecure_noverifyname(tlsConfig)
		tls.tls_config_insecure_noverifycert(tlsConfig)
		tls.tls_config_verify_client_optional(tlsConfig)
		-- tls.tls_config_insecure_noverifytime(tlsConfig)
	end
	return tlsServer, tlsConfig
end
socket.setupTls = setupTls

local function tlsHandshake(sock)
	-- https://github.com/libressl-portable/openbsd/blob/master/src/lib/libtls/tls.c
	local data, dataLen
	local i = 0
	repeat
		i = i + 1
		dataLen = recv(sock, C.MSG_PEEK, 4)
		if dataLen > 0 then
			sock.receive_data:putcdata(receiveBuffer, dataLen)
			data = sock.receive_data:get(dataLen)
			-- data = ffi.string(receiveBufferPtr, dataLen)
		else
			sleep(1) -- 1 ms
		end
	until dataLen >= 4 or i >= 500 -- 500 times of 1 ms sleep = 0.5 sec
	-- util.printInfo("%d. loops for accept read data length: %s bytes", i, tostring(dataLen))

	if not data then -- no data in accept, it is ok, try again in receive
		return
	end

	--[[
	http://blog.fourthbit.com/2014/12/23/traffic-analysis-of-an-ssl-slash-tls-session/

		 Record Type Values       dec      hex
	 -------------------------------------
	 CHANGE_CIPHER_SPEC        20     0x14
	 ALERT                     21     0x15
	 HANDSHAKE                 22     0x16
	 APPLICATION_DATA          23     0x17
	]]
	if data:sub(1, 1) ~= "\22" then -- // A TLS handshake record starts with byte 22
		return
	end
	-- tls.tls_accept_socket and tls.tls_handshake must be in blocking mode
	-- we have enough data for handshake, this should be fast and not to block forever
	local ret = sock:setBlocking(true)
	if ret ~= 0 then
		util.printError("socket %s, sock:setBlocking (set to blocking mode) failed with error: %d", tostring(sock.socket), ret)
	end
	-- https://github.com/libressl-portable/openbsd/blob/master/src/lib/libtls/tls.c
	-- util.printInfo("* tls_accept_socket(), socket: %s", tostring(sock.socket))
	local tlsConnCtxPtr = ffi.newNoAnchor("struct tls*[1]")
	ret = tls.tls_accept_socket(tlsServer, tlsConnCtxPtr, sock.socket)
	local tlsConnCtx = tlsConnCtxPtr[0]
	-- tls_accept_socket will create a new "struct tls" and sets "struct tls*" to tlsConnCtxPtr[0]
	if ret ~= 0 then
		util.printError("socket %s, tls.tls_accept_socket() failed with error: %d, '%s'", tostring(sock.socket), ret, crypto.tlsError(tlsConnCtx))
	else
		-- util.printInfo("* tls_handshake(), socket: %s", tostring(sock.socket))
		ret = tls.tls_handshake(tlsConnCtx) -- handshake may fail few times with insecure or other wrong certs
		if ret ~= 0 then -- and ret ~= -1 then
			util.printWarning("socket %s, tls.tls_handshake() failed with error: %d, '%s'", tostring(sock.socket), ret, crypto.tlsError(tlsConnCtx))
		elseif ret == 0 then
			sock.tls_ctx_ptr = tlsConnCtxPtr -- anchor pointer list
			sock.tls_ctx = tlsConnCtx
		end
	end
	-- [[ after tls handshake set socket back to non-blocking mode
	local ret2 = sock:setBlocking(false)
	if ret2 ~= 0 then
		util.printError("socket %s, sock:setBlocking (set to non-blocking mode) failed with error: %d", tostring(sock.socket), ret) -- ]]
	end
	if sock.tls_ctx then
		util.printOk("\n* tls handshake ok, socket: %s", tostring(sock.socket))
	end
end

local readOne
do
	local read
	local function recvTls(sock)
		read = tonumber(tls.tls_read(sock.tls_ctx, receiveBufferPtr, receiveBufferSize))
		if read < -2 then
			util.printWarning("tls read failed for socket '%s', error: %d, '%s'", tostring(sock.socket), read, crypto.tlsError(sock.tls_ctx))
		end
		return read
	end

	local readOneRet, readOneErr
	readOne = function(sock, flags)
		if sock.tls_ctx then
			readOneRet = recvTls(sock)
		elseif flags then
			readOneRet = recv(sock, flags)
		else
			readOneRet = recv(sock, receiveFlags)
		end
		if readOneRet > 0 then
			sock.receive_data:putcdata(receiveBuffer, readOneRet)
		end
		if isWin and readOneRet == -1 then
			readOneErr = lastError()
			if readOneErr == 0 then
				return 0
			elseif readOneErr == 2 and sock.thread == nil then -- not useCoro
				return 0
			end
		end
		return readOneRet
	end
end

local function findString(ptr, len, find, findLen)
	local i = 0
	findLen = findLen - 1 -- 0 based index
	while i < len do -- we use while because i can be advanced more that 1 inside second loop
		for j = 0, findLen do
			if ptr[i] == find[j] then
				if j == findLen then
					return i + 1
				end
				i = i + 1
			else
				i = i + 1
				break
			end
		end
	end
	return 0
end

local receiveFlagsNoWait = 0
local maxReadLoop = util.from4d() and 1000 or util.fromEditor() and 8000 or 50000
if not util.isWin() then
	receiveFlagsNoWait = bit.bor(C.MSG_DONTWAIT, receiveFlags)
end
function socket.receive(sock, remainingLen, readToBuffer) -- sock, flags, remainingLen, raw
	if type(remainingLen) == "number" and remainingLen > 0 and sock.receive_data and #sock.receive_data >= remainingLen then
		if #sock.receive_data ~= remainingLen then
			util.printRed("socket '%s' reinig data length %d is bigger than needed length %d ", tostring(sock.socket), #sock.receive_data, remainingLen)
		end
		return sock.receive_data:get(remainingLen), nil, remainingLen
	end
	if sock.closed then
		return nil, "socket is closed", 0
	elseif sock.do_close then
		return nil, "socket will be closed", 0
	end
	local findText, findLen
	if type(remainingLen) == "string" then
		findLen = #remainingLen
		findText = ffi.cast("char *", remainingLen)
		remainingLen = nil
	elseif remainingLen == nil then
		local err = util.printError("socket '%s' read length is nil", tostring(sock.socket))
		return nil, err, 0
	else
		if remainingLen > receiveDataInitialSize then
			util.printWarning("      socket receive data buffer size %d is smaller than read length %d ", receiveDataInitialSize, remainingLen)
		end
		if #sock.receive_data >= remainingLen and #sock.receive_data > 0 and not readToBuffer then
			if remainingLen == 0 then
				remainingLen = #sock.receive_data
			end
			return sock.receive_data:get(remainingLen), nil, remainingLen
		end
	end
	if sock.listen_tls and not sock.tls_ctx then -- if accept handshake failed (no data then), try again here
		tlsHandshake(sock)
	end
	local useCoro = sock.thread ~= nil
	local length
	if useCoro then
		length = readOne(sock, receiveFlagsNoWait) -- we must read before yield(), othwrwise we will not get pollin event
		if length <= 0 then
			if debugLevel > 0 then
				util.print("\n    waiting for data, socket '%s', need: %d bytes...", tostring(sock.socket), findLen or remainingLen)
			end
			if length == -1 then -- readOne() with receiveFlagsNoWait -1 is returned when there is no data
				coroYield(sock) -- return when there is a pollin event
				length = readOne(sock, receiveFlagsNoWait)
			end
			if length < 0 then
				local tcpErrorNum = lastError()
				if tcpErrorNum == C.EAGAIN then -- with receiveFlagsNoWait we get 35: Resource temporarily unavailable
					util.printWarning("tcp read error: '%s EAGAIN', %s, socket '%s'", tostring(tcpErrorNum), threadId(sock.thread), tostring(sock.socket))
					length = 0 -- allow one error, continue to read loop that return on next error
				else -- ]]
					local err = util.printRed("tcp read error: '%s', %s, socket '%s'", tostring(tcpErrorNum), threadId(sock.thread), tostring(sock.socket))
					sock.do_close = true
					return nil, err, 0
				end
			end
		end
	else
		-- not coro
		length = readOne(sock, receiveFlagsNoWait)
		if remainingLen == 0 then
			if length == -1 then -- nothing to read
				return ""
			end
		else
			if length == -1 then -- resource temporarily unavailable
				length = 0
			end
			-- 	length = readOne(sock) -- don't use receiveFlagsNoWait
		end
		if length == 0 then
			remainingLen = 1 -- go to read loop
		end
	end
	if defaultSocketReadTimeoutSeconds == nil then
		local option = isTest and {} or dprf.prf("system/option.json")
		defaultSocketReadTimeoutSeconds = option.option and option.option.default_socket_read_timeout_seconds or 30
		loopPrintInterval = option.option and option.option.default_socket_read_print_interval_seconds or 0.75
	end
	local timeout = tonumber(sock.timeout) or defaultSocketReadTimeoutSeconds
	local readStart = util.seconds()
	local prevPrintTime = readStart
	if length >= 0 then
		if findText then
			local ptr, findEnd
			local readLoop = 0
			local findStart = 0
			repeat
				readLoop = readLoop + 1
				ptr, length = sock.receive_data:ref()
				findEnd = length - findStart
				if findEnd > 0 then
					findEnd = findString(ptr + findStart, findEnd, findText, findLen)
					if findEnd > 0 then
						if findStart + findEnd > #sock.receive_data then
							util.printRed("socket '%s' receive data find end %d is bigger that data length %d", tostring(sock.socket), findStart + findEnd, #sock.receive_data)
							findEnd = #sock.receive_data
						end
						--[[ if #sock.receive_data > findStart + findEnd then
							util.print("socket '%s' received more data than needed %d > %d", tostring(sock.socket), #sock.receive_data, findStart + findEnd)
						end ]]
						return sock.receive_data:get(findStart + findEnd), nil, findStart + findEnd
					end
				end
				findStart = #sock.receive_data - findLen + 1
				if findStart < 0 then
					findStart = 0
				end
				length = readOne(sock, receiveFlagsNoWait)
				if length < 1 then
					if useCoro then
						coroYield(sock)
						--[[ if sock.do_close then
						return nil, "socket will be closed", 0
					end ]]
					elseif readLoop <= 10 then
						util.sleep(readLoop - 1)
					else
						util.sleep(10) -- 50000 loops * 10 ms = about 500 seconds wait time for header response to come in
					end
					length = readOne(sock, receiveFlagsNoWait)
				end
				--[[ if length > 0 then
					sock.receive_data:putcdata(receiveBuffer, length)
				else ]]
				if length == -1 then -- and lastError() == 35 -- we are using receiveFlagsNoWait, then -1 is valid answer (Resource temporarily unavailable)
					if util.isWine() then
						local tcpErrorNum = lastError()
						util.printRed("tcp read error: '%s', %s, socket '%s'", netErrorText(tcpErrorNum), threadId(sock.thread), tostring(sock.socket))
						os.exit()
					end
					length = 0
				elseif length < 0 then
					local tcpErrorNum = lastError()
					local err = util.printRed("tcp read error: '%s', %s, socket '%s'", netErrorText(tcpErrorNum), threadId(sock.thread), tostring(sock.socket))
					--[[ if tcpErrorNum == C.EAGAIN then -- with receiveFlagsNoWait we get 35: Resource temporarily unavailable
						-- length = 0
					elseif tcpErrorNum == 54 then -- error 54: Connection reset by peer, error 2: Temporary failure in name resolution
					else ]]
					sock.do_close = true
					return nil, err, 0
					-- end
				end
				if not sock.do_close and (length > 0 and readLoop <= 2 or readLoop % 100 == 0 or util.seconds(prevPrintTime) > loopPrintInterval) then
					if readLoop == 1 then
						ioWrite("\n")
					end
					util.print("      socket '%s' text read loop %d, last read %d", tostring(sock.socket), readLoop, length)
					prevPrintTime = util.seconds()
				end
			until length < 0 or readLoop >= maxReadLoop or sock.do_close or sock.closed or util.seconds(readStart) >= timeout -- TODO: test for sock.timeout in poll.lua
			local recvErr
			if length < 0 then
				recvErr = util.printError("tcp read error: '%s', %s, socket '%s'", netErrorText(lastError()), threadId(sock.thread), tostring(sock.socket))
			elseif readLoop >= maxReadLoop then
				recvErr = util.printError("tcp read error: max read loops reached, %s, socket '%s'", threadId(sock.thread), tostring(sock.socket))
			elseif sock.closed then
				recvErr = util.printError("tcp read error: socket is closed, %s, socket '%s'", threadId(sock.thread), tostring(sock.socket))
			elseif sock.do_close then
				recvErr = util.printError("tcp read error: socket will be closed, %s, socket '%s'", threadId(sock.thread), tostring(sock.socket))
			elseif util.seconds(readStart) >= timeout then
				recvErr = util.printError("tcp read error: max read timeout %.1f seconds reached, %s, socket '%s'", timeout, threadId(sock.thread), tostring(sock.socket))
			end
			length = #sock.receive_data
			return sock.receive_data:get(length), recvErr, length
		elseif length < remainingLen then
			local startLoopTime = util.seconds()
			local readLenOrig = remainingLen
			if length > 0 then
				-- sock.receive_data:putcdata(receiveBuffer, length)
				length = #sock.receive_data -- old buffer + new read length
				if length >= remainingLen then
					remainingLen = 0
				else
					remainingLen = remainingLen - length
				end
			end
			local readLoop = 0
			ioWrite("\n")
			if remainingLen > 0 then
				readLoop = readLoop + 1
				if readLoop == 1 then
					util.print("      socket '%s' read loop %d, need %d, read %d, done %.0f%%, remaining %d bytes", tostring(sock.socket), readLoop, readLenOrig, length, (readLenOrig - remainingLen) / readLenOrig * 100, remainingLen)
				else
					util.print("      socket '%s' read loop %d, last read %d, done %.0f%%, remaining %d bytes", tostring(sock.socket), readLoop, length, (readLenOrig - remainingLen) / readLenOrig * 100, remainingLen)
				end
			end
			local lastErrorNum
			local errorCount = 0
			local maxErrorCount = 10
			repeat
				readLoop = readLoop + 1
				if remainingLen > 0 then -- still something to read
					length = readOne(sock, receiveFlagsNoWait)
					if length < 1 then
						if useCoro then
							coroYield(sock)
							--[[ if sock.do_close then
							return nil, "socket will be closed", 0
						end ]]
						elseif readLoop <= 10 then
							util.sleep(readLoop - 1)
						else
							util.sleep(10) -- 50000 loops * 10 ms = about 500 seconds, these read loops come after reading header so data is on the way and we can loop less
						end
					end
					if length > 0 then
						-- sock.receive_data:putcdata(receiveBuffer, length)
						if length >= remainingLen then
							remainingLen = 0
						else
							remainingLen = remainingLen - length
						end
					elseif length < 0 then
						lastErrorNum = lastError()
						if lastErrorNum == C.EAGAIN then -- with receiveFlagsNoWait we get 35: Resource temporarily unavailable
							length = 0
						elseif lastErrorNum == C.ETIMEDOUT then -- 60: Operation timed out
							-- this happens in OSX when reading from 4D, but usually next read succeeds
							if errorCount == 0 then
								errorCount = errorCount + 1
							end
							-- util.printColor("bright magenta",
							util.print("      socket '%s' read loop %d, error count %d / %d, tcp read error ETIMEDOUT: '%s'", tostring(sock.socket), readLoop, errorCount, maxErrorCount, netErrorText(lastErrorNum))
							length = 0
						else
							errorCount = errorCount + 1
							util.printRed("      socket '%s' read loop %d, error count %d / %d, tcp read error: '%s'", tostring(sock.socket), readLoop, errorCount, maxErrorCount, netErrorText(lastErrorNum))
						end
					end
				end
				if not sock.do_close and (remainingLen <= 0 or readLoop <= 3 or readLoop % 100 == 0 or util.seconds(prevPrintTime) > loopPrintInterval) then
					if remainingLen > 0 then
						util.print("      socket '%s' read loop %d, last read %d, done %.0f%%, remaining %d bytes", tostring(sock.socket), readLoop, length, (readLenOrig - remainingLen) / readLenOrig * 100, remainingLen)
					elseif readLoop > 1 then
						util.print("      socket '%s' read loop %d, needed %d, last read %d, done 100%% in %.3f seconds", tostring(sock.socket), readLoop, readLenOrig, length, util.seconds(startLoopTime))
					end
					prevPrintTime = util.seconds()
				end
			until remainingLen <= 0 or (length < 0 and errorCount >= maxErrorCount) or readLoop >= maxReadLoop or sock.do_close or sock.closed or util.seconds(readStart) >= timeout -- TODO: test for sock.timeout in poll.lua
			local recvErr
			if length < 0 then
				recvErr = util.printError("socket '%s' receive data needed length: %s bytes, read length: %s bytes, error %s", tostring(sock.socket), tostring(readLenOrig), tostring(remainingLen), netErrorText(lastError()))
			elseif remainingLen > 0 then
				recvErr = util.printError("socket '%s' receive data needed length: %s bytes, read length: %s bytes, read loop: %d, max read loop: %d", tostring(sock.socket), tostring(readLenOrig), tostring(remainingLen), readLoop, maxReadLoop)
			end
			if sock.closed then
				if recvErr then
					recvErr = recvErr .. ", socket is closed"
					recvErr = util.printError(recvErr)
				else
					recvErr = util.printError("socket '%s' receive data needed length: %s bytes, read length: %s bytes, socket is closed", tostring(sock.socket), tostring(readLenOrig), tostring(remainingLen))
				end
			elseif sock.do_close then
				if recvErr then
					recvErr = recvErr .. ", socket will be closed"
					recvErr = util.printError(recvErr)
				else
					recvErr = util.printError("socket '%s' receive data needed length: %s bytes, read length: %s bytes, socket will be closed", tostring(sock.socket), tostring(readLenOrig), tostring(remainingLen))
				end
			elseif util.seconds(readStart) >= timeout then
				if recvErr then
					recvErr = recvErr .. l(", max read timeout %.1f seconds was reached", timeout)
					recvErr = util.printError(recvErr)
				else
					recvErr = util.printError("tcp read error: max read timeout %.1f seconds was reached, %s, socket '%s'", timeout, threadId(sock.thread), tostring(sock.socket))
				end
			end

			length = #sock.receive_data -- we must get this before buffer:get() because it makes length #receive_data to return zero
			if readToBuffer then
				if readLenOrig < length then
					return nil, recvErr, readLenOrig
				end
				return nil, recvErr, length
			end
			if readLenOrig <= length then
				return sock.receive_data:get(readLenOrig), recvErr, readLenOrig
			end
			return sock.receive_data:get(length), recvErr, length
		end
	end

	-- after one readOne() call
	if length > 0 then
		if readToBuffer then
			-- sock.receive_data:putcdata(receiveBuffer, length)
			return nil, 0, #sock.receive_data
		end
		if length == remainingLen and #sock.receive_data == 0 then
			return ffi.string(receiveBuffer, length), nil, length -- fast path, set no data to receive_data buffer, return directly from receiveBuffer
		end
		-- sock.receive_data:putcdata(receiveBuffer, length) -- more data than needed, put received data to the end of the buffer and return needed from buffer start
		length = #sock.receive_data
		if remainingLen > 0 and remainingLen <= length then
			return sock.receive_data:get(remainingLen), nil, remainingLen
		end
		return sock.receive_data:get(length), nil, length
	end
	-- util.printError("socket '%s' receive data needed length: %s bytes, read length: %s bytes", tostring(sock.socket), tostring(remainingLen), tostring(length))
	return nil, netErrorText(lastError()), 0 -- , netErrorText(ffi.errno())
end

function socket.receiveFrom(sock)
	local len, receivefromAddr = recvFrom(sock, receiveFlags)
	if len > 0 then
		return ffi.string(sock.receive_buf, len), len, receivefromAddr
	end
	return nil, len, receivefromAddr, lastError()
end

function socket.settimeout(sock, timeout)
	sock.timeout = timeout
	sock = sock.sock or sock
	if not util.isWindows() then
		local result = sock:setsockopt(C.SOL_SOCKET, C.SO_SNDTIMEO, timeout)
		if result ~= 0 then
			net.printError(result, "connectSocket:setsockopt SO_SNDTIMEO failed with error: " .. ffi.errno())
		end
		result = sock:setsockopt(C.SOL_SOCKET, C.SO_RCVTIMEO, timeout)
		if result ~= 0 then
			net.printError(result, "connectSocket:setsockopt SO_RCVTIMEO failed with error: " .. ffi.errno())
		end
	end
end

local function copyToSendBuffer(data, dataLen)
	if dataLen > sendBufferSize then
		util.printWarning("answer is bigger than send buffer size: " .. dataLen .. " > " .. sendBufferSize)
		local count = 0
		repeat
			count = count + 1
			sendBufferSize = sendBufferSize + (4 * Mb)
		until sendBufferSize > dataLen
		sendBuffer = realloc(sendBuffer, sendBufferSize)
		if ffi.isNull(sendBuffer) then
			util.printError("send buffer reallocation by %d Mb to size: %d failed", count * 4, sendBufferSize)
			util.closeProgram()
		end
		sendBufferPtr = ffi.cast("void *", sendBuffer)
		sendBufferSendPtr = ffi.cast("const void *", sendBuffer)
		util.printOk("resized send buffer by %d Mb to size: %d", count * 4, sendBufferSize)
	end
	ffi.copy(sendBufferPtr, data, dataLen)
end
--[[
local function copyToSendBuffer(sock, data, dataLen)
	if sock.send_buf_size < dataLen then
		if debugLevel > 0 then
			util.print("  resizing socket '%s' send buffer from %.1f kb to %.1f kb", tostring(sock.socket), sock.send_buf_size / 1024, dataLen / 1024)
		end
		sock.send_buf_size = dataLen
		-- sock.send_buf = sock.send_buf:reserve(dataLen)
	end
	if type(data) == "cdata" then
		sock.send_buf = sock.send_buf:set(data, dataLen)
	else
		sock.send_buf = sock.send_buf:set(data)
	end
end
]]

function socket.sendTo(sock, data, addr, dataLen)
	if dataLen == nil then
		dataLen = #data
	end
	-- local bytesSent = 0
	local sendResult = 0
	if dataLen > 0 then
		copyToSendBuffer(data, dataLen)
		sendResult = sendTo(sock, dataLen, sendFlags, addr) -- send answer buffer
		if sendResult == -1 then
			-- close(sock)
			-- no bad error, connection was closed??? hah?
			local errno = ffi.errno()
			util.printColor("bright red", " *** sock:sendTo failed with error -1, errno: %d, error '%s'", errno, netErrorText(errno))
		elseif sendResult < 0 then
			util.printColor("bright red", " *** sock:sendTo failed with error: %d", sendResult)
			close(sock)
		elseif sendResult ~= dataLen then
			util.printColor("bright red", " *** sock:sendTo did not send all data: %d / %d", sendResult, dataLen)
			-- bytesSent = sendResult
		end
	end
	return sendResult
end

local function setPollout(sock, value)
	if usePollout then
		poll.setOutFlag(sock, value)
	end
end

local function sendData(sock, dataLen, flags)
	if sock == nil then
		return -1, util.printError("sock send data failed, sock is nil")
	end
	if sock.socket == nil then
		return -1, util.printError("sock send data failed, socket is nil")
	end
	if usePollout then
		setPollout(sock, true)
		if sock.send_count then
			coroYield(sock) -- no yield on first send
		end
	end
	if debugLevel > 1 and dataLen > minDebugSendSize then
		if sock.uncompressed_answer_start then
			util.ioWrite("\n    sending data to socket '%s', %s, send to '%s', size %d bytes, data:\n  '%s'... ", tostring(sock.socket), tostring(sock.thread or "main thread"), tostring(sock.uri or sock.info), dataLen, sock.uncompressed_answer_start)
			sock.uncompressed_answer_start = nil
		elseif dataLen >= 50 * kb then
			util.ioWrite("\n    sending data to socket '%s', %s, send to '%s', size %.1f kb... ", tostring(sock.socket), tostring(sock.thread or "main thread"), tostring(sock.uri or sock.info), dataLen / kb)
		else
			util.ioWrite("\n    sending data to socket '%s', %s, send to '%s', size %d bytes... ", tostring(sock.socket), tostring(sock.thread or "main thread"), tostring(sock.uri or sock.info), dataLen)
		end
	end
	if sock.closed then
		setPollout(sock, false)
		return -2, "socket is closed"
		-- sock.do_close are still alive and need to send remaining data
	end
	local ret
	-- if sock.tls_ctx then
	-- 	ret = sendTls(sock, dataLen)
	-- else
	ret = send(sock, dataLen, flags)
	-- end
	-- sock.answer_size = ret -- for debugging
	if usePollout then
		setPollout(sock, false)
	end
	-- [[ todo: test this code, previous was only if sock.do_close then
	if sock.closed then
		return -2, "socket is closed after send"
	elseif sock.do_close and ret > 0 then -- ret >= 0
		return ret, "socket will be closed after succesful send" -- ]]
	elseif sock.do_close then --	if sock.do_close then
		return -2, "socket will be closed after send"
	end
	if ret < 0 then
		return -1, netErrorText(lastError())
	end
	return ret
end

function socket.clearSocketData(sock)
	if sock.receive_data then
		sock.receive_data:reset()
	end
end

function socket.send(sock, data, dataLen)
	--[[ if sock.do_close then -- not dead yet
		return -2, "socket will be closed"
	else ]]
	if sock.closed then
		return -2, "socket is closed"
	end
	if type(data) ~= "string" then
		return -2, util.printError("socket data type '%s' is not string", type(data))
	end
	sendNumber = sendNumber + 1
	if data:sub(1, 5) ~= "POST " and data:sub(1, 5) ~= "HTTP/" and data:sub(1, 5) ~= "GET /" then
		dataLen = #data
	end
	if dataLen == nil then
		dataLen = #data
	end
	-- print("dataLen: "..dataLen)
	local sendResult = 0
	if dataLen <= 0 then
		util.printError(" *** sock:send data length %d is smaller than 1", dataLen)
	else
		copyToSendBuffer(data, dataLen)
		local time
		if debugLevel > 1 and dataLen > minDebugSendSize then
			time = util.seconds()
		end
		local errText
		sendResult, errText = sendData(sock, dataLen, sendFlags)
		if time then
			ioWrite(l(" in %.5f seconds", util.seconds(time)))
		end
		if sendResult == dataLen or errText then
			return sendResult, errText
		end
		if sendResult == -2 then
			return sendResult
		elseif sendResult == -1 then
			-- close(sock)
			-- no bad error, connection was closed??? hah?
			local errno = ffi.errno()
			util.printColor("bright red", " *** socket send failed with error -1, errno: %d, error '%s'", errno, netErrorText(errno))
		elseif sendResult < 0 then
			util.printColor("bright red", " *** socket send failed with error: %d", sendResult)
			-- close(sock)
		else -- if sendResult >= 0 then
			util.printError(" *** sock:send result %d is not equal to data length %d -- TODO: fix loop send, keep remaining data in socket and use pollout", sendResult, dataLen)
			local startTime = seconds()
			local bytesSent = sendResult
			local loopCount = 0 -- TODO: rewrite without locals / coroutine
			local loopCountPrev = 0
			local sendCount = 1 -- already sent first answer
			local timeUs = util.microSeconds()
			-- sendResult -1 == ????
			while bytesSent < dataLen and sendResult >= -1 and seconds(startTime) < sendTimeout do
				repeat
					loopCount = loopCount + 1
					sleep(0)
					sock.send_count = loopCount
					if bytesSent > 0 then -- TODO: test this code
						data = data:sub(bytesSent + 1)
						dataLen = #data
					end
					copyToSendBuffer(data, dataLen)
					sendResult = sendData(sock, dataLen, sendFlags) -- send answer buffer
				until sendResult ~= 0
				if sendResult == -1 and loopCount - loopCountPrev > 200 then
					if loopCountPrev == 0 then
						loopCountPrev = loopCount
						sleep(250)
					else
						sendResult = -100 -- exit while loop
					end
				elseif sendResult < 0 then
					if sendCount > 4 then
						util.printInfo(" -- sock answer %d, send result: %d, send count: %d, loop count: %d, bytes sent %d / %d, %d remaining, total send time %d µs", sendNumber, sendResult, sendCount, loopCount, bytesSent, dataLen, dataLen - bytesSent, util.microSeconds(timeUs))
						if sendResult == -1 then
							sleep(200) -- extra 200ms sleep
						end
					end
					sleep(250)
				elseif sendResult > 0 then
					sendCount = sendCount + 1
					bytesSent = bytesSent + sendResult
					if bytesSent > dataLen then
						bytesSent = dataLen
					end
				end
			end
			if bytesSent < dataLen then
				-- if sendResult ~= -1 then
				util.printError(" -- sock answer %d, remaining bytes send failed with result code: %d, bytes sent: %d / %d, total send time %d µs", sendNumber, sendResult, bytesSent, dataLen, util.microSeconds(timeUs))
				-- end
			elseif sendCount > 4 then
				util.printInfo(" -- sock answer %d, send count: %d, loop count: %d, bytes sent %d / %d, %d remaining, total send time %d µs", sendNumber, sendCount, loopCount, bytesSent, dataLen, dataLen - bytesSent, util.microSeconds(timeUs))
			end
		end
	end
	sock.send_count = nil
	if sendResult >= 0 then
		return sendResult
	elseif sendResult < 0 then
		sock.do_close = true
	end
	return sendResult, netErrorText(lastError())
end

--[[ function socket.setBlocking(sock, mode)
	if not (mode == true or mode == false) then
		util.printError("setBlocking mode must be true or false")
		return
	end
	local result = sock:setBlocking(mode)
	if result ~= 0 then
		if mode == true then
			util.printError("sock:setBlocking (set to blocking mode) failed with error: %s", tostring(result))
		else
			util.printError("sock:setBlocking (set to non-blocking mode) failed with error: ", tostring(result))
		end
	end
end ]]

local addressInfo
do
	local hints_c = ffi.newAnchor("struct addrinfo")
	hints_c.ai_flags = 0 -- bit.bor(C.AI_NUMERICHOST) -- fill in my IP for me
	local addrInfo_c = ffi.newAnchor("struct addrinfo*[1]")
	addressInfo = function(host, port, socketType)
		-- http://beej.us/guide/bgnet/output/html/multipage/syscalls.html#bind

		-- DOES NOT work in windows: AF_UNSPEC,  AF_UNSPEC == use IPv4 or IPv6, whichever
		-- socketType == 0 == PF_UNSPEC, http3 uses it with IPPROTO_UDP
		if socketType == "unix" then
			hints_c.ai_protocol = 0
			hints_c.ai_socktype = C.SOCK_STREAM
			hints_c.ai_family = C.AF_UNIX
			addrInfo_c[0] = hints_c -- net.getaddrinfo() does not work with AF_UNIX
			return addrInfo_c
		elseif socketType == "udp" or socketType == "broadcast" or socketType == "http3" then
			hints_c.ai_socktype = C.SOCK_DGRAM
			hints_c.ai_protocol = C.IPPROTO_UDP
			if socketType == "http3" then
				hints_c.ai_family = C.PF_UNSPEC
			else
				hints_c.ai_family = C.AF_INET
			end
		else
			hints_c.ai_family = C.AF_INET
			hints_c.ai_socktype = C.SOCK_STREAM
			hints_c.ai_protocol = C.IPPROTO_TCP
		end
		local serv = tostring(port)

		--[[
	local addr = ffi.newNoAnchor("struct addrinfo[1]")
	local addr_c = ffi.cast("struct addrinfo *", addr[0])
	local addr_ptr = ffi.cast("struct addrinfo **", addr_c)
	]]
		-- local addrInfo_c = ffi.newNoAnchor("struct addrinfo*[1]")
		local err2 = net.getaddrinfo(host, serv, hints_c, addrInfo_c)
		-- sets addr struct content using host, serv and hints
		if err2 ~= 0 then
			err2 = net.printError(err2, l("sock.getaddrinfo() to host '%s:%s' failed with error: ", tostring(host), serv))
			return nil, err2
		end
		return addrInfo_c
	end
	socket.addressInfo = addressInfo
end

function socket.address(host, port)
	-- set up destination address
	local addr_s = ffi.newNoAnchor("struct sockaddr_in[1]")
	ffi.fill(addr_s[0], 0, ffi.sizeof(addr_s)) -- C.memset(addr_s[0], 0 ,ffi.sizeof(addr_s))
	local addr = addr_s[0]
	addr.sin_family = C.AF_INET -- do not set this or udp send will not work
	if isWin then
		addr.sin_addr.S_un.S_addr = net.inet_addr(host)
	else
		addr.sin_addr.s_addr = net.inet_addr(host)
	end
	addr.sin_port = net.htons(port)
	return addr_s
end

local function setSocketOption(sock)
	local ret
	-- local err = "accept socket: setsockopt TCP_NODELAY failed with error: "
	if not util.isLinux() then
		ret = sock:setsockopt(C.SOL_SOCKET, C.TCP_NODELAY, tcpNoDelay)
		if ret ~= 0 then
			return ret, "TCP_NODELAY"
		end
	end
	return ret
end

local function connect(host, port, socketType, connectTimeout, blocking, option)
	local udp, broadcast
	if socketType == "broadcast" then
		udp = true
		broadcast = true
	elseif socketType == "udp" or socketType == "http3" then
		udp = true
	end

	if blocking == nil or blocking == false then
		blocking = 0 -- default to non-blocking after connect
	end
	if not connectTimeout or connectTimeout < 0 then
		connectTimeout = 4 -- seconds
	end
	if socketType ~= "unix" then
		local host2 = peg.parseAfter(host, "://")
		if socketType ~= "unix" and peg.found(host2, "/") then
			util.printError("sock connect host '%s' must be plain host address, it must not contain subpaths '/'", host)
			return nil
		end
		if host and host:find("https://", 1, true) == 1 then
			host = host:sub(("https://"):len() + 1)
			if not port or port < 1 then
				port = 443
			end
		elseif host and host:find("http://", 1, true) == 1 then
			host = host:sub(("http://"):len() + 1)
			port = tonumber(port)
			if port == nil or port < 1 then
				port = 80
			end
		end
	end

	local addrInfo = addressInfo(host, port, socketType)
	if not addrInfo then
		return nil
	end
	local connectSocket = create({ai_family = addrInfo[0].ai_family, ai_socktype = addrInfo[0].ai_socktype, ai_protocol = addrInfo[0].ai_protocol, flags = nil, host = host, port = port})
	if connectSocket == INVALID_SOCKET then
		cleanup(connectSocket, nil, "new failed with error: ")
		return nil
	end

	if socketType == "tls" or port == 443 and socketType ~= "http3" then
		local ctx, tlsConfig2, errTxt = connectTls(host, port)
		if ctx then
			connectSocket.tls_ctx = ctx
			connectSocket.tls_config = tlsConfig2
		elseif option ~= "no-error" then
			util.printError(errTxt)
		end
	else
		-- Setup the TCP or UDP connecting socket
		local result

		--[=[
		if socketType == "http3" then
			-- see: lib/net/http3-lsquic-test_common.lua
			ffi.cdef [[ static const int IP_DONTFRAGMENT = 14; ]]
			result = connectSocket:setsockopt(C.SOL_SOCKET, C.IP_DONTFRAGMENT, 1)
			if result ~= 0 then
				cleanup(connectSocket, result, "connectSocket:setsockopt IP_DONTFRAGMENT failed with error: ")
				return nil
			end
		end --]=]

		-- if socketType ~= "http3" then
		-- SO_REUSEADDR, set reuse address for listen socket before bind
		result = connectSocket:setsockopt(C.SOL_SOCKET, C.SO_REUSEADDR, 1)
		if result ~= 0 then
			cleanup(connectSocket, result, "connectSocket:setsockopt SO_REUSEADDR failed with error: ")
			return nil
		end

		if util.isMac() then
			result = connectSocket:setsockopt(C.SOL_SOCKET, C.SO_NOSIGPIPE, 1)
			if result ~= 0 then
				cleanup(connectSocket, result, "connectSocket:setsockopt SO_NOSIGPIPE failed with error: " .. result)
				return nil
			end
			--[[elseif util.isLinux() then
			-- use in send() and recv() in server.lua: sendFlags = C.MSG_NOSIGNAL
			result = connectSocket:setsockopt(C.SOL_SOCKET, C.MSG_NOSIGNAL, 1)
			if result ~= 0 then
				cleanup(connectSocket, result, "connectSocket:setsockopt MSG_NOSIGNAL failed with error: "..result)
				return nil
			end]]
		end

		-- SO_USELOOPBACK, use always loopback when possible
		--[[
		if util.isMac() or isWin then -- SO_USELOOPBACK is not supported by Linux
			result = connectSocket:setsockopt(C.SOL_SOCKET, C.SO_USELOOPBACK, 1)
			if result ~= 0 then
				net.printError(result, "connectSocket:setsockopt SO_USELOOPBACK failed with error: ")
			end
		end --]]
		-- SO_SNDBUF, send buffer size
		result = connectSocket:setsockopt(C.SOL_SOCKET, C.SO_SNDBUF, socketSendBufferSize)
		if result ~= 0 then
			net.printError(result, "connectSocket:setsockopt SO_SNDBUF failed with error: ")
		end

		-- SO_RCVBUF, receive buffer size
		result = connectSocket:setsockopt(C.SOL_SOCKET, C.SO_RCVBUF, socketReceiveBufferSize)
		if result ~= 0 then
			net.printError(result, "connectSocket:setsockopt SO_RCVBUF failed with error: ")
		end

		-- TCP_NODELAY, sock-nodelay to 1
		-- print("tcpNoDelay: "..tcpNoDelay)
		local err
		result, err = setSocketOption(connectSocket)
		if err then
			connectSocket:cleanup(result, "connect socket: setsockopt " .. err .. " failed with error: ")
			return nil
		end

		-- TCP_QUICKACK, tcp-quickack to 1
		if util.isLinux() then
			result = connectSocket:setsockopt(C.SOL_SOCKET, C.TCP_QUICKACK, tcpQuickAck)
			if result ~= 0 then
				connectSocket:cleanup(result, "connectSocket:setsockopt TCP_QUICKACK failed with error: ")
				return nil
			end
		end

		if util.isMac() and not udp and socketType ~= "http3" and socketType ~= "unix" then
			result = connectSocket:setsockopt(C.IPPROTO_TCP, C.TCP_CONNECTIONTIMEOUT, connectTimeout) -- connectTimeout in seconds
			if result ~= 0 then
				net.printError(result, "connectSocket:setsockopt TCP_CONNECTIONTIMEOUT failed with error: ")
			end
		end
		-- end

		if broadcast then
			-- bind
			result = connectSocket:bind(addrInfo[0].ai_addr, addrInfo[0].ai_addrlen)
			if result ~= 0 then
				cleanup(connectSocket, result, "broadcast connectSocket:bind failed with error: ")
				return nil
			end

			-- SO_BROADCAST, set udp broadcast on
			result = connectSocket:setsockopt(C.SOL_SOCKET, C.SO_BROADCAST, 1)
			if result ~= 0 then
				cleanup(connectSocket, result, "connectSocket:setsockopt SO_BROADCAST failed with error: ")
				return nil
			end
		elseif socketType == "unix" then
			-- connect
			result = fdSend.connectToUnixServer(host, connectSocket.socket)
			if result ~= connectSocket.socket then
				if option ~= "no-error" then
					net.printError(result, "connectSocket connect to UNIX socket failed with error: ")
				end
				return nil
			end
		else
			-- connect
			-- http://stackoverflow.com/questions/2597608/c-socket-connection-timeout
			result = C.connect(connectSocket.socket, addrInfo[0].ai_addr, addrInfo[0].ai_addrlen)
			if result ~= 0 then -- -1 = connection could not be done
				if option ~= "no-error" then
					net.printError(result, "connectSocket connect failed with error: ")
				end
				return nil
			end
		end
		-- set to non-blocking mode
		if blocking == 0 then
			connectSocket:setBlocking(false)
		end
	end
	connectSocket.type = socketType
	connectSocket.info = tostring(connectSocket.socket) .. ". " .. socketType .. " connect " .. host .. ":" .. port
	loadPoll()
	poll.addFd(connectSocket, "connect")
	return connectSocket
end

function socket.connect(sock, host, port, socketType, connectTimeout, blocking, option)
	if type(sock) == "string" then
		host, port, socketType, connectTimeout, blocking, option = sock, host, port, socketType, connectTimeout, blocking
	elseif type(sock) ~= "table" then -- pgmoon uses this in sock:conncet() form
		util.printError("socket connect sock is not a table")
		return
	end
	if host:sub(1, 1) == "/" then
		socketType = "unix"
	end
	if socketType == nil then
		util.printWarningWithCallPath("socket connect type is nil, using 'tcp'")
		socketType = "tcp"
	end
	local connectSock = connect(host, port, socketType, connectTimeout, blocking, option)
	-- connectSock.url = tostring(host) .. ":" .. tostring(port or "")
	--[[ if connectSock and blocking == nil then
		connectSock:setBlocking(true)
	end ]]

	if connectSock and type(sock) == "table" then
		-- copy connectSock to sock that was parameter self
		for key, value in pairs(connectSock) do
			sock[key] = value
		end
		setmetatable(sock, {__index = socket})
		connectSock = sock
	end
	return connectSock
end

function socket.listen(host, port, socketType, socketFlags, block)
	local udp, broadcast, isTls
	if socketType == "broadcast" then
		udp = true
		broadcast = true
	elseif socketType == "udp" or socketType == "http3" then
		udp = true
	elseif socketType and peg.found(socketType, "isTls") then
		isTls = true
		if not tlsServer then
			local ok, err = setupTls()
			if not ok then
				isTls = false
				util.printWarning("TLS listen did not start: '%s'", err)
			end
		end
	end

	local addrInfo = addressInfo(host, port, socketType)
	if not addrInfo then
		return nil
	end
	-- Create a SOCKET for connecting to server
	local listenSocket = create({ai_family = addrInfo[0].ai_family, ai_socktype = addrInfo[0].ai_socktype, ai_protocol = addrInfo[0].ai_protocol, flags = socketFlags})
	if listenSocket == INVALID_SOCKET then
		cleanup(listenSocket, nil, "new failed with error: ")
		return nil
	end

	listenSocket.type = socketType
	listenSocket.info = tostring(listenSocket.socket) .. ". " .. socketType .. " listen " .. host .. ":" .. port
	if isTls then
		listenSocket.listen_tls = true
	end
	-- Setup the socket.listening socket
	local result
	-- SO_REUSEADDR, set reuse address for listen socket before bind
	if isWin then
		-- https://msdn.microsoft.com/en-us/library/windows/desktop/ms737550(v=vs.85).aspx
		-- https://msdn.microsoft.com/en-us/library/windows/desktop/ms740621(v=vs.85).aspx
		-- https://msdn.microsoft.com/en-us/library/windows/desktop/ms740532(v=vs.85).aspx
		-- local SO_EXCLUSIVEADDRUSE = bit.bnot(C.SO_REUSEADDR) -- disallow local address reuse
		result = listenSocket:setsockopt(C.SOL_SOCKET, bit.bnot(C.SO_REUSEADDR), 1)
	else
		result = listenSocket:setsockopt(C.SOL_SOCKET, C.SO_REUSEADDR, 1) -- SO_REUSEADDR == only one can listen to same address+port, SO_REUSEPORT = multiple can listen to same port, see: https://stackoverflow.com/questions/14388706/how-do-so-reuseaddr-and-so-reuseport-differ
		-- result = listenSocket:setsockopt(C.SOL_SOCKET, C.SO_REUSEPORT, 1)
	end
	if result ~= 0 then
		cleanup(listenSocket, result, "listenSocket:setsockopt SO_REUSEADDR failed with error: ")
		return nil
	end

	-- set to non-blocking mode
	result = listenSocket:setBlocking(block or false)
	if result ~= 0 then
		cleanup(listenSocket, result, "listenSocket:setBlocking (set to non-blocking mode) failed with error: ")
		return nil
	end

	if util.isMac() then
		result = listenSocket:setsockopt(C.SOL_SOCKET, C.SO_NOSIGPIPE, 1)
		if result ~= 0 then
			cleanup(listenSocket, result, "listenSocket:setsockopt SO_NOSIGPIPE failed with error: " .. result)
			return nil
		end
		--[[elseif util.isLinux() then
    -- use in send() and recv() in server.lua: sendFlags = C.MSG_NOSIGNAL
		result = listenSocket:setsockopt(C.SOL_SOCKET, C.MSG_NOSIGNAL, 1)
		if result ~= 0 then
			cleanup(listenSocket, result, "listenSocket:setsockopt MSG_NOSIGNAL failed with error: "..result)
			return nil
		end]]
	end

	-- SO_USELOOPBACK, use always loopback when possible
	--[[
	if util.isMac() or isWin then -- SO_USELOOPBACK is not supported by Linux
		result = listenSocket:setsockopt(C.SOL_SOCKET, C.SO_USELOOPBACK, 1)
		if result ~= 0 then
			net.printError(result, "setsockopt SO_USELOOPBACK failed with error: ")
		end
	end --]]
	-- SO_SNDBUF, send buffer size
	result = listenSocket:setsockopt(C.SOL_SOCKET, C.SO_SNDBUF, socketSendBufferSize)
	if result ~= 0 then
		net.printError(result, "listenSocket:setsockopt SO_SNDBUF failed with error: ")
	end

	-- SO_RCVBUF, receive buffer size
	result = listenSocket:setsockopt(C.SOL_SOCKET, C.SO_RCVBUF, socketReceiveBufferSize)
	if result ~= 0 then
		net.printError(result, "listenSocket:setsockopt SO_RCVBUF failed with error: ")
	end

	-- TCP_NODELAY, tcp-nodelay to 1
	-- print("tcpNoDelay: "..tcpNoDelay)
	if not udp and not util.isLinux() then
		result = listenSocket:setsockopt(C.SOL_SOCKET, C.TCP_NODELAY, tcpNoDelay)
		if result ~= 0 then
			cleanup(listenSocket, result, "listenSocket:setsockopt TCP_NODELAY failed with error: ")
			return nil
		end
	end

	-- bind
	result = listenSocket:bind(addrInfo[0].ai_addr, addrInfo[0].ai_addrlen)
	if result ~= 0 then
		cleanup(listenSocket, result, "listenSocket:bind for port " .. port .. " failed with error: ", "no-trace")
		util.printRed("\n*** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** ***\n***  \n***  " .. l("there is another instance of this program running, or some other program using the port %d", port) .. "   ***\n***\n*** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** *** ***")
		if util.fromEditor() then
			os.exit()
		elseif util.from4d() == false then
			util.printInfo("waiting 10 seconds for restart")
			sleep(10 * 1000)
		end
		return nil
	end

	if udp then
		if broadcast then
			-- SO_BROADCAST, set udp broadcast on
			result = listenSocket:setsockopt(C.SOL_SOCKET, C.SO_BROADCAST, 1)
			if result ~= 0 then
				cleanup(listenSocket, result, "listenSocket:setsockopt SO_BROADCAST failed with error: ")
				return nil
			end
		end
	else
		-- listen
		result = C.listen(listenSocket.socket, C.SOMAXCONN) -- C.SOMAXCONN is backlog
		if result ~= 0 then
			cleanup(listenSocket, result, "listenSocket:listen failed with error: ")
			return nil
		end
	end
	loadPoll()
	poll.addFd(listenSocket, "listen")
	return listenSocket
end

--[[ function socket.listen(sock, host, port, socketType, socketFlags, block)
	if type(sock) ~= "table" then
		util.printError("socket.listen sock is not a table")
		return
	end
	sock.sock = listen(host, port, socketType, socketFlags, block)
	return sock.sock
end
]]

-- local clientSocket, newSockNum, newSock
function socket.accept(sock)
	local client_addr_size = ffi.newAnchor("int[1]")
	client_addr_size[0] = ffi.sizeof("struct sockaddr")
	local client_addr = ffi.newAnchor("struct sockaddr_in[1]")
	local client_addr_ptr = ffi.cast("struct sockaddr *", client_addr)

	-- Accept a client socket
	local clientSocket, newSockNum, newSock
	newSockNum = C.accept(sock.socket, client_addr_ptr, client_addr_size)
	if newSockNum ~= INVALID_SOCKET then
		newSock = util.clone(sock)
		newSock.socket = newSockNum
		-- newSock.sockaddr = sockaddr
		-- newSock.addrLen = addrLen
		clientSocket = create(newSock)
	end
	if clientSocket == nil then
		return nil
	end
	if tonumber(clientSocket.socket) < 0 then
		util.printColor("bright red", "listenSocket:accept failed with error: " .. tonumber(clientSocket.socket))
		return nil
	end
	local result, err = setSocketOption(clientSocket)
	if err then
		clientSocket:cleanup(result, "client socket: setsockopt " .. err .. " failed with error: ")
		return nil
	end
	-- set to non-blocking mode
	-- if blocking == 0 then
	-- clientSocket:setBlocking(false)
	-- end
	--[[
	if clientSocket < 0 then
		return clientSocket -- poll error, ok
		-- cleanup(listenSocket, clientSocket, "accept failed with error: ")
		return nil
	end

	-- SO_SNDBUF, send buffer size
	result = setsockopt(clientSocket, C.SOL_SOCKET, C.SO_SNDBUF, socketSendBufferSize)
	if result ~= 0 then
		cleanup(clientSocket, result, "clientSocket:setsockopt SO_SNDBUF failed with error: ")
		return nil
	end

	-- SO_RCVBUF, receive buffer size
	result = setsockopt(clientSocket, C.SOL_SOCKET, C.SO_RCVBUF, socketReceiveBufferSize)
	if result ~= 0 then
		cleanup(clientSocket, result, "clientSocket:setsockopt SO_RCVBUF failed with error: ")
		return nil
	end

	-- TCP_NODELAY, socket-nodelay to 1
	result = clientSocket:setsockopt(C.SOL_SOCKET, C.TCP_NODELAY, tcpNoDelay)
	if result ~= 0 then
		cleanup(clientSocket, result, "clientSocket:setsockopt TCP_NODELAY failed with error: ")
		return nil
	end
	]]
	-- [[
	if clientSocket.listen_tls and not clientSocket.tls_ctx then
		tlsHandshake(clientSocket) -- try handshake if there is data, try again in receive if there was not data
	end -- ]]
	-- local sock = socket.create()
	-- sock.sock = clientSocket
	clientSocket.type = sock.type
	clientSocket.info = tostring(clientSocket.socket) .. ". " .. clientSocket.type .. " accept" .. peg.parseAfter(sock.info, "listen")
	loadPoll()
	poll.addFd(clientSocket, "accept")
	return clientSocket -- ,client_addr_ptr
end

do
	local ipstr = ffi.newAnchor("char[?]", C.INET_ADDRSTRLEN)
	function socket.address_to_string(addr)
		local s = ffi.cast("struct sockaddr_in *", addr)
		---@diagnostic disable-next-line: undefined-field
		local port = net.ntohs(s.sin_port)
		---@diagnostic disable-next-line: undefined-field
		net.inetNtop(C.AF_INET, s.sin_addr, ipstr)
		return ffi.string(ipstr) .. ":" .. port
	end
end

do
	local ipstr = ffi.newAnchor("char[?]", C.INET_ADDRSTRLEN)
	local ipstr6 = ffi.newAnchor("char[?]", C.INET6_ADDRSTRLEN)
	local len_c = ffi.newAnchor("unsigned int[1]")
	local addr_c = ffi.newAnchor("struct sockaddr_storage")
	len_c[0] = ffi.sizeof(addr_c) -- ffi.sizeof("struct sockaddr")
	function socket.socket_address(sock)
		-- local addr_c = ffi.newNoAnchor("struct sockaddr_storage")
		sock:getpeername(ffi.cast("struct sockaddr *", addr_c), len_c)
		-- deal with both IPv4 and IPv6:
		local port
		---@diagnostic disable-next-line: undefined-field
		if addr_c.ss_family == C.AF_INET then
			-- use address_to_string() ?
			local s = ffi.cast("struct sockaddr_in *", addr_c)
			---@diagnostic disable-next-line: undefined-field
			port = net.ntohs(s.sin_port)
			---@diagnostic disable-next-line: undefined-field
			net.inetNtop(C.AF_INET, s.sin_addr, ipstr)
			return ffi.string(ipstr) .. ":" .. port
			---@diagnostic disable-next-line: undefined-field
		elseif addr_c.ss_family == C.AF_INET6 then
			local s = ffi.cast("struct sockaddr_in6 *", addr_c)
			---@diagnostic disable-next-line: undefined-field
			port = net.ntohs(s.sin6_port)
			---@diagnostic disable-next-line: undefined-field
			net.inetNtop(C.AF_INET6, s.sin6_addr, ipstr6)
			return ffi.string(ipstr6) .. ":" .. port
		end
		return l("could not resolve socket address")

		--[[
	-- http://stackoverflow.com/questions/4282292/convert-struct-in-addr-to-text
	-- http://www.freelists.org/post/luajit/FFI-pointers-to-pointers,1
	-- https://gist.github.com/neomantra/2644943

	local hostname = createBuffer(C.NI_MAXHOST)
	local servInfo = createBuffer(C.NI_MAXSERV)
	local result = net.getnameinfo(client_addr_ptr, ffi.sizeof("struct sockaddr_in"), hostname, C.NI_MAXHOST, servInfo, C.NI_MAXSERV, 0)
	if result ~= 0 then
		cleanup(result, "net.getnameinfo failed with error: ")
		return nil
	end
	--print("client ip:port : "..ffi.string(hostname)..":"..ffi.string(servInfo)) -- servInfo is post number
	return ffi.string(hostname)..":"..ffi.string(servInfo)
	]]
	end
end

return socket

--[[
-- OLD:
int l_socket_set_nonblock(LSocketFD sock, bool val) {
#ifdef __WINDOWS__
	unsigned long flag = val;
	return ioctlsocket(sock, FIONBIO, &flag);
#else
	int flags;
	flags = fcntl(sock, F_GETFL);
	if(flags < 0) return flags;
	if(val) {
		flags |= O_NONBLOCK;
	} else {
		flags &= ~(O_NONBLOCK);
	}
	return fcntl(sock, F_SETFL, flags);
#endif
}

int l_socket_set_close_on_exec(LSocketFD sock, bool val) {
#ifdef __WINDOWS__
	return 0;
#else
	int flags;
	flags = fcntl(sock, F_GETFD);
	if(flags < 0) return flags;
	if(val) {
		flags |= FD_CLOEXEC;
	} else {
		flags &= ~(FD_CLOEXEC);
	}
	return fcntl(sock, F_SETFD, flags);
#endif
}

int l_socket_get_option(LSocketFD sock, int level, int opt, void *val, socklen_t *len) {
	return getsockopt(sock, level, opt, val, len);
}

int l_socket_set_option(LSocketFD sock, int level, int opt, const void *val, socklen_t len) {
	return setsockopt(sock, level, opt, val, len);
}

int l_socket_get_int_option(LSocketFD sock, int level, int opt, int *val) {
	socklen_t len = sizeof(*val);
	return l_socket_get_option(sock, level, opt, val, &len);
}

int l_socket_set_int_option(LSocketFD sock, int level, int opt, int val) {
	return l_socket_set_option(sock, level, opt, &val, sizeof(val));
}

int l_socket_pair(int type, int flags, LSocketFD sv[2]) {
	type |= flags;
#ifdef __WINDOWS__
	/* todo: use TCP socketC. */
	errno = EAFNOSUPPORT;
	return -1;
#else
	return socketpair(AF_UNIX, type, 0, sv);
#endif
}

LSocketFD l_socket_open(int domain, int type, int protocol, int flags) {
	type |= flags;
	return socket(domain, type, protocol);
}

LSocketFD l_socket_close_internal(LSocketFD sock) {
#ifdef __WINDOWS__
	return closesocket(sock);
#else
	return close(sock);
#endif
}

int l_socket_shutdown(LSocketFD sock, int how) {
	return shutdown(sock, how);
}

int l_socket_connect(LSocketFD sock, LSockAddr *addr) {
	return connect(sock, L_SOCKADDR_TO_CONST_ADDR_AND_LEN(addr));
}

int l_socket_bind(LSocketFD sock, LSockAddr *addr) {
	return bind(sock, L_SOCKADDR_TO_CONST_ADDR_AND_LEN(addr));
}

int l_socket_listen(LSocketFD sock, int backlog) {
	return listen(sock, backlog);
}

int l_socket_get_sockname(LSocketFD sock, LSockAddr *addr) {
	MAKE_TEMP_ADDR(tmp1);
	int rc;

	rc = getsockname(sock, GET_TEMP_ADDR_AND_PTR_LEN(tmp1));
	if(rc == 0) {
		L_SOCKADDR_FILL_FROM_TEMP(addr, tmp1);
	}
	return rc;
}

int l_socket_get_peername(LSocketFD sock, LSockAddr *addr) {
	MAKE_TEMP_ADDR(tmp1);
	int rc;

	rc = getpeername(sock, GET_TEMP_ADDR_AND_PTR_LEN(tmp1));
	if(rc == 0) {
		L_SOCKADDR_FILL_FROM_TEMP(addr, tmp1);
	}
	return rc;
}

LSocketFD l_socket_accept(LSocketFD sock, LSockAddr *peer, int flags) {
	MAKE_TEMP_ADDR(tmp1);
#ifdef __WINDOWS__
	LSocketFD rc;
	if(peer != NULL) {
		rc = accept(sock, GET_TEMP_ADDR_AND_PTR_LEN(tmp1));
		if(rc == 0) {
			L_SOCKADDR_FILL_FROM_TEMP(peer, tmp1);
		}
	} else {
		rc = accept(sock, NULL, NULL);
	}
	if(rc == INVALID_SOCKET) {
		return rc;
	}
	if((flags & SOCK_NONBLOCK) == SOCK_NONBLOCK) {
		l_socket_set_nonblock(rc, 1);
	}
	return rc;
#else
	if(peer != NULL) {
		LSocketFD rc;
		rc = accept4(sock, GET_TEMP_ADDR_AND_PTR_LEN(tmp1), flags);
		if(rc == 0) {
			L_SOCKADDR_FILL_FROM_TEMP(peer, tmp1);
		}
		return rc;
	}
	return accept4(sock, NULL, NULL, flags);
#endif
}

int l_socket_send(LSocketFD sock, const void *buf, size_t len, int flags) {
	return send(sock, buf, len, flags);
}

int l_socket_recv(LSocketFD sock, void *buf, size_t len, int flags) {
	return recv(sock, buf, len, flags);
}
--]]
