--- lib/table/qsort_op.lua
-- https://github.com/DarkRoku12/lua_sort
-- Luajit optimized quicksort
-- Copyright (C) 2017 - DarkRoku12
-- Optimized version.
-- Stack slot #1 = tbl.
local floor = math.floor

local coro = require "coro"
local useCoro = coro.useCoro()
local currentThreadSocket = coro.currentThreadSocket
local coroYield = coro.yieldFunction(5000, true) -- return without poll event
local minYieldArrayLength = 5000
local rget = rawget
local rset = rawset

local function defaultCompare(a, b)
	return a < b
end

local function set2(tbl, i, j, ival, jval)
	rset(tbl, i, ival) -- tbl[i] = ival
	rset(tbl, j, jval) -- tbl[j] = jval
end

local function auxSort(tbl, l, u, compareFunc, sock)
	local a, b, i, j, P, P2
	while l < u do
		if sock then
			coroYield(sock)
			if sock.do_close then
				return false
			end
		end
		-- sort elements a[l], a[(l+u)/2] and a[u]
		-- do
		a = rget(tbl, l) -- tbl[l]
		b = rget(tbl, u) -- tbl[u]
		if compareFunc(b, a) then
			set2(tbl, l, u, b, a) -- /* swap a[l] - a[u] */
		end
		-- end
		if u - l == 1 then
			break
		end -- only 2 elements
		i = floor((l + u) / 2) -- -- for tail recursion (i).
		-- do
		a = rget(tbl, i) -- tbl[i]
		b = rget(tbl, l) -- tbl[l]
		if compareFunc(a, b) then -- a[i] < a[l] ?
			set2(tbl, i, l, b, a)
		else
			-- b = nil -- remove a[l]
			b = rget(tbl, u) -- tbl[u]
			if compareFunc(b, a) then -- a[u]<a[i] ?
				set2(tbl, i, u, b, a)
			end
		end
		-- end
		if u - l == 2 then
			break
		end -- only 3 elements
		P = rget(tbl, i) -- tbl[i] -- Pivot.
		P2 = P -- lua_pushvalue(L, -1)
		b = rget(tbl, u - 1) -- tbl[u - 1]
		set2(tbl, i, u - 1, b, P2)
		-- a[l] <= P == a[u-1] <= a[u], only need to sort from l+1 to u-2 */
		i = l
		j = u - 1 -- for tail recursion (j).
		while true do -- for( ; ; )
			-- invariant: a[l..i] <= P <= a[j..u]
			-- repeat ++i until a[i] >= P
			i = i + 1 -- ++i
			a = rget(tbl, i) -- tbl[i]
			while compareFunc(a, P) do
				i = i + 1 -- ++i
				a = rget(tbl, i) -- tbl[i]
			end
			-- repeat --j until a[j] <= P
			j = j - 1 -- --j
			b = rget(tbl, j) -- tbl[j]
			while compareFunc(P, b) do
				j = j - 1 -- --j
				b = rget(tbl, j) -- tbl[j]
			end
			if j < i then
				break
			end
			set2(tbl, i, j, b, a)
		end -- end for
		-- can't use rawset() because of swap operation, it would need temporary variables
		tbl[u - 1], tbl[i] = rget(tbl, i), rget(tbl, u - 1) -- tbl[u - 1], tbl[i] = tbl[i], tbl[u - 1]

		-- a[l..i-1] <= a[i] == P <= a[i+1..u] */
		-- adjust so that smaller half is in [j..i] and larger one in [l..u] */
		if i - l < u - i then
			j = l
			i = i - 1
			l = i + 2
		else
			j = i + 1
			i = u
			u = j - 2
		end
		auxSort(tbl, j, i, compareFunc, sock) -- call recursively the smaller one */
	end
	-- end of while
	-- repeat the routine for the larger one
end

-- sort function.
local function sorter(tbl, compareFunc, sock)
	if type(tbl) ~= "table" then
		local util = require "util"
		local l = require"lang".l
		util.printError(l("quicksort first parameter type '%s' is not a table", type(tbl)))
		return
	end
	if compareFunc and type(compareFunc) ~= "function" then
		local util = require "util"
		local l = require"lang".l
		util.printError(l("quicksort second parameter type '%s' is not a (comparison) function", type(compareFunc)))
		return
	end
	if not useCoro then
		sock = nil
	elseif #tbl < minYieldArrayLength then
		sock = nil
	elseif sock == nil then
		sock = currentThreadSocket()
	end
	auxSort(tbl, 1, #tbl, compareFunc or defaultCompare, sock)
end
return sorter

--[[
-- this plain table sort version is slower, 1.4 vs 0.95 seconds in stock-value event sort, sometimes this can be almost as fast, but it varies much more than rawget version

local function auxSort(tbl, l, u, compareFunc, sock)
	local a, b, i, j, P, P2
	while l < u do
		if sock then
			coroYield(sock)
			if sock.do_close then
				return
			end
		end
		-- sort elements a[l], a[(l+u)/2] and a[u]
		-- do
		a = tbl[l]
		b = tbl[u]
		if compareFunc(b, a) then
			set2(tbl, l, u, b, a) -- /* swap a[l] - a[u] */
		end
		-- end
		if u - l == 1 then
			break
		end -- only 2 elements
		i = floor((l + u) / 2) -- -- for tail recursion (i).
		-- do
		a = tbl[i]
		b = tbl[l]
		if compareFunc(a, b) then -- a[i] < a[l] ?
			set2(tbl, i, l, b, a)
		else
			-- b = nil -- remove a[l]
			b = tbl[u]
			if compareFunc(b, a) then -- a[u]<a[i] ?
				set2(tbl, i, u, b, a)
			end
		end
		-- end
		if u - l == 2 then
			break
		end -- only 3 elements
		P = tbl[i] -- Pivot.
		P2 = P -- lua_pushvalue(L, -1)
		b = tbl[u - 1]
		set2(tbl, i, u - 1, b, P2)
		-- a[l] <= P == a[u-1] <= a[u], only need to sort from l+1 to u-2 */
		i = l
		j = u - 1 -- for tail recursion (j).
		while true do -- for( ; ; )
			-- invariant: a[l..i] <= P <= a[j..u]
			-- repeat ++i until a[i] >= P
			i = i + 1 -- ++i
			a = tbl[i]
			while compareFunc(a, P) do
				i = i + 1 -- ++i
				a = tbl[i]
			end
			-- repeat --j until a[j] <= P
			j = j - 1 -- --j
			b = tbl[j]
			while compareFunc(P, b) do
				j = j - 1 -- --j
				b = tbl[j]
			end
			if j < i then
				break
			end
			set2(tbl, i, j, b, a)
		end -- end for
		-- can't use rawset() because of swap operation, it would need temporary variables
		tbl[u - 1], tbl[i] = tbl[i], tbl[u - 1]

		-- a[l..i-1] <= a[i] == P <= a[i+1..u] */
		-- adjust so that smaller half is in [j..i] and larger one in [l..u] */
		if i - l < u - i then
			j = l
			i = i - 1
			l = i + 2
		else
			j = i + 1
			i = u
			u = j - 2
		end
		auxSort(tbl, j, i, compareFunc, sock) -- call recursively the smaller one */
	end
	-- end of while
	-- repeat the routine for the larger one
end

]]
