--- lib/table/sort.lua
-- common utility functions for Lua and Coffeescript
-- @module util
local sort = {}
local util = require "util"
local l = require"lang".l
local json = require "json"
local peg = require "peg"
local recData = require"recdata".get
local quickSort = require "table/qsort_op"
local tableSort = table.sort

local function sortFunc(sign, option)
	-- This order function receives two arguments and must return true if the first argument should come first in the sorted array.
	if sign == 1 then
		sign = false -- this is opposite to Javascript sort
	else -- sign == 0
		sign = true
	end
	local lowercase = option and option.lowercase ~= false and true or false

	local function compare(a, b)
		if a == nil or b == nil then -- continue sort
			return nil -- 0
		end
		if lowercase then
			if type(a) == "string" then
				a = a:lower()
			end
			if type(b) == "string" then
				b = b:lower()
			end
		end
		if a > b then -- if type(a) == "string" then a.localeCompare(b)
			return sign
		elseif a < b then
			return not sign
		end
		return 0 -- special case, continue sort
	end
	return compare -- return compare function
end

local sortError = {}
local function dynamicSortMany(propertyArray, option)
	local sortFuncArr = {}
	local propertyName = {}
	local propertyNeedRecData = {}
	if #propertyArray % 2 ~= 0 then
		util.printError(l("sort definition must be even length: %s"), json.toJsonRaw(propertyArray))
	else
		local i = 1
		while i <= #propertyArray do
			local property = propertyArray[i]
			local sortOrder = propertyArray[i + 1]
			if sortOrder == ">" then
				sortOrder = 1
			elseif sortOrder == "<" then
				sortOrder = -1
			else
				util.printError(l("sort order == not '<' or '>': %s"), json.toJsonRaw(propertyArray))
				sortOrder = 1
			end
			sortFuncArr[#sortFuncArr + 1] = sortFunc(sortOrder, option)

			propertyName[#propertyName + 1] = property
			if peg.found(property, ".") then
				propertyNeedRecData[#propertyName] = true
			else
				propertyNeedRecData[#propertyName] = false
			end
			i = i + 2
		end
	end
	local propertyCount = #propertyName
	local ret
	local function compare(a, b)
		for i = 1, propertyCount do
			if propertyNeedRecData[i] then
				ret = sortFuncArr[i](recData(a, propertyName[i]), recData(b, propertyName[i]))
			else
				ret = sortFuncArr[i](a[propertyName[i]], b[propertyName[i]])
			end
			if i == 1 and ret == nil then
				if sortError[propertyName[i]] == nil then
					sortError[propertyName[i]] = util.printRed("sort key '%s' is nil in data", propertyName[i])
				end
				ret = 0
			end
			if ret ~= 0 then
				return ret
			end
		end
	end
	return compare -- return compare function
end

local function dynamicSort(property, orderOrOption, option)
	-- http://stackoverflow.com/questions/1129216/sorting-objects-in-an-array-by-a-field-value-in-javascript
	sortError = {}
	if type(property) == "table" then
		return dynamicSortMany(property, orderOrOption), orderOrOption
	end
	return dynamicSortMany({property, orderOrOption}, option), option
end

function sort.sort(arr, property, orderOrOption, option) -- order == optional if property == array
	local sortFunc2, option2 = dynamicSort(property, orderOrOption, option)
	if option2 and option2.table_sort then
		tableSort(arr, sortFunc2)
	else
		quickSort(arr, sortFunc2) -- table.sort(arr, sortFunc) -- return? , arr clone needed?
	end
end

function sort.sortFunction(property, orderOrOption, option) -- order == optional if property == array
	local sortFunc2, option2 = dynamicSort(property, orderOrOption, option)
	if option2 and option2.table_sort then
		return tableSort, sortFunc2
	end
	return quickSort, sortFunc2
end

return sort
