-- thread.lua
-- local class = require "class"
local thread = {} -- thread = class("thread")
thread.__index = thread

local util = require "util"
local l = require"lang".l
local ffi = require "mffi"
local peg = require "peg"
local C = ffi.C
local print = util.print

-- thread
--[[
https://github.com/hnakamur/luajit-examples/blob/master/pthread/thread1.lua
http://pubs.opengroup.org/onlinepubs/007908799/xsh/pthread.h.html
/System/Library/Frameworks/Kernel.framework/Versions/A/Headers/kern/thread.h
/System/Library/Frameworks/Kernel.framework/Versions/A/Headers/sys/_types.h
/Applications/Xcode.app/Contents/Developer/Platforms/MacOSX.platform/Developer/SDKs/MacOSX10.8.sdk/usr/include/pthread.h
]]

ffi.cdef [[
struct nc_msg // message structure
{
	uint32_t msg_id;
	uint32_t data_len;
	uint32_t data_max_len;
	void *data;
};
]]

local errName = { -- defined in lua.h
	[1] = "LUA_YIELD",
	[2] = "LUA_ERRRUN",
	[3] = "LUA_ERRSYNTAX",
	[4] = "LUA_ERRMEM",
	[5] = "LUA_ERRERR"
}

local threadId = 0

function thread.new(name)
	local self = setmetatable({}, thread) -- setmetatable returns 1. argument
	self:initialize(name)
	return self
end

function thread:initialize(name)
	if not name then
		threadId = threadId + 1
		name = tostring(threadId)
	end
	self._name = tostring(name)
end

function thread:__tostring() -- methamethod
	return "object thread: " .. self._name
end

-- common win+mac
function thread:functionToAddress(thread_entry)
	return tonumber(ffi.cast('intptr_t', ffi.cast('void *(*)(void *)', thread_entry)))
end

-- create a separate Lua state first
-- define a callback function in *that* created state
function thread:createLuaState(lua_code, thread_entry_address_name)
	local threadEntryAddressName = thread_entry_address_name or "thread_entry_address"
	local L = C.luaL_newstate()
	if L == nil then
		return nil, l "error in C.luaL_newstate()"
	end

	-- http://williamaadams.wordpress.com/2012/04/03/step-by-step-inch-by-inch/
	C.lua_gc(L, C.LUA_GCSTOP, 0) -- stop collector during initialization
	C.luaL_openlibs(L)
	C.lua_gc(L, C.LUA_GCRESTART, -1)

	local ret = C.luaL_loadstring(L, lua_code)
	if ret ~= 0 then
		return nil, l("error '%d: %s' in C.luaL_newstate()", ret, errName[ret] or "unknown")
		--[[
		0: no errors;
		LUA_ERRSYNTAX: syntax error during pre-compilation;
		LUA_ERRMEM: memory allocation error.
		--]]
	end
	ret = C.lua_pcall(L, 0, 1, 0) -- runs code
	-- defines functions and variables, we need thread_entry_address -variable
	-- that is the thread_entry() -function memory pointer Lua-number
	if ret ~= 0 then
		return nil, l("error '%d: %s' in C.lua_pcall()", ret, errName[ret] or "unknown")
		--[[
		0: no errors;
		LUA_ERRRUN: a runtime error.
		LUA_ERRMEM: memory allocation error. For such errors, Lua does not call the error handler function.
		LUA_ERRERR: error while running the error handler function.
		--]]
	end

	-- get function thread_entry() address from calling thread
	-- http://pgl.yoyo.org/luai/i/lua_call
	C.lua_getfield(L, C.LUA_GLOBALSINDEX, threadEntryAddressName)
	-- thread_entry_address = function or (usually) global variable to be called
	local func_ptr = C.lua_tointeger(L, -1); -- lua_getfield value
	C.lua_settop(L, -2); -- set stack to correct (prev call params -1?)
	return L, func_ptr
end

-- Destroys all objects in the given Lua state
-- if Lua state is still running you WILL get crash
function thread:deleteLuaState(luaState)
	C.lua_close(luaState)
end

function thread:toIdString(threadNum)
	local addr = tostring(threadNum) -- ffi.cast('void *', threadNum[0])
	addr = peg.parseAfter(addr, ": ")
	-- local addr = ffi.cast('intptr_t', ffi.cast('void *', threadNum[0])) -- addr is now int64_t
	return addr

	-- if util.isMac() then
	-- return tonumber(ffi.cast('intptr_t', ffi.cast('void *', threadNum[0]))) -- is this ok?
	-- end
	-- return tonumber(threadNum)
end

function thread:stringToPointer(str)
	local num = tonumber(str);
	if not num then
		return nil, l "invalid number"
	end
	return ffi.cast("void *", ffi.cast("intptr_t", num));
end

-- This helper routine will take a pointer
-- to cdata, and return a string that contains
-- the memory address
-- tonumber(ffi.cast('intptr_t', ffi.cast('void *', ptr)))
function thread:pointerToString(instance)
	if ffi.abi("64bit") then
		return string.format("0x%016x", tonumber(ffi.cast("int64_t", ffi.cast("void *", instance))))
	elseif ffi.abi("32bit") then
		return string.format("0x%08x", tonumber(ffi.cast("int32_t", ffi.cast("void *", instance))))
	end
	return nil
end

if util.isWin() then

	function thread:yield()
		C.SwitchToThread()
		--[[ Return value:
			If calling the SwitchToThread function causes the operating system to switch execution to another thread, the return value is nonzero.
			If there are no other threads ready to execute, the operating system does not switch execution to another thread, and the return value is zero.
		]]
	end

	function thread:selfIdString()
		local id = C.GetCurrentThreadId()
		return thread:toIdString(id)
	end

	-- then use pthread_create() from the original state, passing the callback address of the other state
	function thread:create(func_ptr, arg, flags)
		flags = flags or 0
		local thread_c = ffi.newNoAnchor("DWORD[1]")
		local arg_c = ffi.cast("void *", arg) -- necessary if arg is not cstr, should we we check arg type?
		local func_ptr_c = ffi.cast("void *", func_ptr)
		local handle = C.CreateThread(nil, 0, func_ptr_c, arg_c, flags, thread_c)
		if not handle then
			return nil, l(" *** ERR: luaThreadCreate() failed.")
		end
		return handle -- handle[0]
	end

	function thread:join(thread)
		if type(thread) == "number" then
			thread = ffi.cast("HANDLE", thread) -- HANDLE, void *
		end
		local ret = C.WaitForSingleObject(thread, C.INFINITE)
		-- If the WaitForSingleObject function succeeds, the return value indicates the event that caused the function to return
		local retVal = ffi.newNoAnchor("DWORD[1]")
		ret = C.GetExitCodeThread(thread, retVal) -- ffi.cast(ffi.newNoAnchor("LPDWORD"), retVal))
		-- print("GetExitCodeThread: ", retVal, retVal[0])
		if ret == 0 then
			return nil, l("GetExitCodeThread failed")
		end
		return retVal[0]
	end

	function thread:detach(thread_id)
		-- TODO: check if code is needed for windows?
		-- http://stackoverflow.com/questions/12744324/how-to-detach-a-thread-on-windows-c
		return 0
	end

	function thread:exit(return_val)
		local val_c = ffi.cast("DWORD", return_val)
		C.ExitThread(val_c)
	end

else
	-- Mac + others
	-- Posix threads

	function thread:yield()
		if C.sched_yield() == -1 then
			util.printError("thread:yield(), sched_yield() failed: " .. ffi.errorText())
		end
		-- The sched_yield() function shall return 0 if it completes successfully; otherwise, it shall return a value of -1 and set errno to indicate the error.
	end

	function thread:selfIdString()
		local id = C.pthread_self()
		-- if ffi.os == "OSX" then
		return thread:toIdString(id)
		-- return tostring(ffi.cast("char *", pid)) --tonumber(ffi.cast("uint32_t", id)) 		--tonumber(pid)
	end

	-- then use pthread_create() from the original state, passing the callback address of the other state
	function thread:create(func_ptr, arg)
		local thread_c = ffi.newNoAnchor("pthread_t[1]")
		local arg_c = ffi.cast("void *", arg) -- necessary if arg is not cstr, should we we check arg type?
		local res = C.pthread_create(thread_c, nil, ffi.cast("thread_func", func_ptr), arg_c)
		if res ~= 0 then
			return nil, " *** ERR: luaThreadCreate() failed."
		end
		return thread_c -- thread_c[0]
	end

	function thread:join(thread_id)
		local return_val = ffi.newNoAnchor("int[1]") -- ffi.cast('intptr_t'
		-- local thread = ffi.cast('intptr_t', thread_id)
		-- fix this call in Linux
		local res = C.pthread_join(thread_id[0], ffi.cast("void *", return_val)) -- and IN thread C.pthread_exit(100)
		return return_val[0]
	end

	--[[
		Either pthread_join(3) or pthread_detach() should be called for each thread that an application creates, so that system resources for the thread can be released. (But note that the resources of all threads are freed when the process terminates.)
		]]
	function thread:detach(thread_id)
		return C.pthread_detach(thread_id[0])
	end

	function thread:exit(return_val)
		if false then
			print(" *** ERR: threadExit(return_val) is not supported, in Linux it will cause 'PANIC: unprotected error in call to Lua API (?)'")
		end
		-- if not util.isLinux() then
		C.pthread_exit(ffi.cast("void *", return_val))
		-- end
	end

	--[[
	NOTES
				 POSIX.1 allows an implementation wide freedom in choosing the type used to
				 represent a thread ID; for example, representation using either an
				 arithmetic type or a structure is permitted.  Therefore, variables of type
				 pthread_t can't portably be compared using the C equality operator (==);
				 use pthread_equal(3) instead.

				 thread identifiers should be considered opaque: any attempt to use a thread
				 ID other than in pthreads calls is nonportable and can lead to unspecified
				 results.

				 thread IDs are only guaranteed to be unique within a process.  A thread ID
				 may be reused after a terminated thread has been joined, or a detached
				 thread has terminated.

				 The thread ID returned by pthread_self() is not the same thing as the
				 kernel thread ID returned by a call to gettid(2).
	]]
end

return thread
