--- time.lua
-- FIX: use: https://github.com/ldrumm/chronos
-- https://luapower.com/time
-- wall-clock time, monotonic time and sleeping for Windows, Linux and OSX.
-- Written by Cosmin Apreutesei. Public Domain.
local time = {}

local ffi = require 'mffi'
local C = ffi.C
local print = print
local util, color
local initDone = false
local minSecondWarning = -(1 / 10 ^ 12)
local toNumber = tonumber

local function loadColor()
	if color == nil then
		color = require "ansicolors"
	end
end

local function loadUtil()
	if not util then
		local ok
		ok, util = pcall(require, "util") -- handle recursive call error here
		if not ok then
			util = nil
			loadColor()
			return false
		end
		print = util.print
	end
	return true
end

local function printError(err, val)
	if false and loadUtil() then
		util.printError(err, val)
	else
		loadColor()
		if val ~= nil then
			print(color("%{bright redbg}ERROR: time function: " .. string.format(err, val)))
		else
			print(color("%{bright redbg}ERROR: time function: " .. tostring(err)))
		end
	end
end

local function printWarning(err, val)
	if loadUtil() then
		if val ~= nil then
			util.printWarning(err .. ", call path: '%s'", val, util.callPath())
		else
			util.printWarning(err .. ", call path: '%s'", util.callPath())
		end
	else
		loadColor()
		if val ~= nil then
			print(color("%{bright magenta}WARNING: time function: " .. string.format(err, val)))
		else
			print(color("%{bright magenta}WARNING: time function: " .. err))
		end
	end
end

local function init()
	if ffi.os == 'Windows' then
		ffi.cdef [[
		void time_GetSystemTimeAsFileTime(uint64_t*) asm("GetSystemTimeAsFileTime");
		int  time_QueryPerformanceCounter(int64_t*) asm("QueryPerformanceCounter");
		int  time_QueryPerformanceFrequency(int64_t*) asm("QueryPerformanceFrequency");
		void time_Sleep(uint32_t ms) asm("Sleep");
		]]

		local t, qpf, DELTA_EPOCH_IN_100NS
		local function initOnce()
			initDone = true
			t = ffi.newAnchor("uint64_t[1]")
			DELTA_EPOCH_IN_100NS = ffi.newNoAnchorBasic("uint64_t", 116444736000000000)
			-- ffi.newAnchor("uint64_t[1]") -- 116444736000000000ULL
			-- DELTA_EPOCH_IN_100NS[0] = 116444736000000000
			-- assert(C.time_QueryPerformanceFrequency(t) ~= 0)
			local ret = C.time_QueryPerformanceFrequency(t)
			if ret == 0 then
				printError("time_QueryPerformanceFrequency failed with error: " .. toNumber(ret))
			end
			qpf = toNumber(t[0])
		end

		local function epoch()
			if not initDone then
				initOnce()
			end
			C.time_GetSystemTimeAsFileTime(t)
			return toNumber(t[0] - DELTA_EPOCH_IN_100NS) / 10 ^ 7 -- t[0] is a bit number of 100-nanosecond intervals since midnight Jan 1, 1601
		end
		time.epoch = epoch

		function time.time()
			return epoch() / 10 ^ 6
		end

		function time.clock()
			if not initDone then
				initOnce()
			end
			-- assert(C.time_QueryPerformanceCounter(t) ~= 0)
			local ret = C.time_QueryPerformanceCounter(t)
			if ret == 0 then
				printError("time_QueryPerformanceCounter failed with error: " .. toNumber(ret))
			end
			return toNumber(t[0]) / qpf
		end

		function time.sleep(s)
			C.time_Sleep(s * 1000)
		end

	elseif ffi.os == 'Linux' or ffi.os == 'OSX' then

		if jit then
			ffi.cdef [[
			typedef struct {
				long s;
				long ns;
			} time_timespec;
			int time_nanosleep(time_timespec*, time_timespec *) asm("nanosleep");
			]]
		end
		local EINTR = 4
		local t, timespec
		initDone = false

		local function initOnce()
			initDone = true
			if jit then -- and not ffi.arch == "arm" then
				timespec = ffi.newAnchor("time_timespec")
				if jit.os == "Linux" then -- jit.arch == "arm" then
					t = timespec
				else
					t = ffi.newAnchor("time_timeval")
				end
			else
				timespec = ffi.newAnchor("timespec")
				t = ffi.newAnchor("timeval")
			end
		end

		if jit then
			function time.sleep(s)
				if not initDone then
					initOnce()
				end
				local int, frac = math.modf(s)
				timespec.s = int
				timespec.ns = frac * 10 ^ 9
				local ret = C.time_nanosleep(timespec, timespec)
				while ret == -1 and ffi.errno() == EINTR do -- interrupted
					ret = C.time_nanosleep(timespec, timespec)
				end
				-- assert(ret == 0)
				if ret ~= 0 then
					printError("time_nanosleep failed with error: " .. toNumber(ret))
				end
			end
		else
			function time.sleep(s)
				if not initDone then
					initOnce()
				end
				local int, frac = math.modf(s)
				timespec.s = int
				timespec.ns = frac * 10 ^ 9
				local ret = C.nanosleep(timespec, timespec)
				while ret == -1 and ffi.errno() == EINTR do -- interrupted
					ret = C.nanosleep(timespec, timespec)
				end
				-- assert(ret == 0)
				if ret ~= 0 then
					printError("nanosleep failed with error: " .. toNumber(ret))
				end
			end
		end

		if ffi.os == 'Linux' then
			local librt = C
			--[[ if not librt then
				librt = ffi.load("rt")
			end ]]
			local clock_gettime
			if ffi.arch == "arm" then
				ffi.cdef [[
				typedef struct {
					long    s;
					int32_t us;
				} time_timeval;
				int clock_gettime(int clock_id, time_timespec *tp);
				]]
				clock_gettime = librt.clock_gettime
			else
				ffi.cdef [[
				typedef struct {
					long    s;
					int32_t us;
				} time_timeval;
				int time_clock_gettime(int clock_id, time_timespec *tp) asm("clock_gettime");
				]]
				clock_gettime = librt.time_clock_gettime
			end

			local CLOCK_REALTIME = 0
			local CLOCK_MONOTONIC = 1

			local function tos(tm)
				return toNumber(tm.s) + toNumber(tm.ns) / 10 ^ 9
			end

			local function epoch()
				if not initDone then
					initOnce()
				end
				-- assert(clock_gettime(CLOCK_REALTIME, t) == 0)
				local ret = clock_gettime(CLOCK_REALTIME, t)
				if ret ~= 0 then
					printError("clock_gettime failed with error: " .. toNumber(ret))
				end
				return tos(t)
			end
			time.epoch = epoch

			function time.time()
				return epoch() / 10 ^ 6
			end

			function time.clock()
				if not initDone then
					initOnce()
				end
				--[[ From man clock_gettime(2)
					 CLOCK_MONOTONIC
							Clock  that  cannot  be  set and represents monotonic time since
							some unspecified starting point.  This clock is not affected by
							discontinuous jumps in the system time (e.g., if the system
							administrator manually changes the clock), but is affected by the
							incremental  adjustments  performed by adjtime(3) and NTP.
					 CLOCK_MONOTONIC_COARSE (since Linux 2.6.32; Linux-specific)
									A faster but less precise version of CLOCK_MONOTONIC.
									Use when you need very fast, but not fine-grained timestamps.
					 CLOCK_MONOTONIC_RAW (since Linux 2.6.28; Linux-specific)
									Similar to CLOCK_MONOTONIC, but provides access to a raw
									hardware-based time that is not subject to NTP adjustments or the
									incremental adjustments performed by adjtime(3).
				]]
				-- assert(clock_gettime(CLOCK_MONOTONIC, t) == 0)
				local ret = clock_gettime(CLOCK_MONOTONIC, t)
				if ret ~= 0 then
					printError("clock_gettime failed with error: " .. toNumber(ret))
				end
				return tos(t)
			end

		elseif ffi.os == 'OSX' then

			if jit then
				ffi.cdef [[
				typedef struct {
					long    s;
					int32_t us;
				} time_timeval;

				typedef struct {
					uint32_t numer;
					uint32_t denom;
				} time_mach_timebase_info_data_t;

				int      time_gettimeofday(time_timeval*, void*) asm("gettimeofday");
				int      time_mach_timebase_info(time_mach_timebase_info_data_t* info) asm("mach_timebase_info");
				uint64_t time_mach_absolute_time(void) asm("mach_absolute_time");
				typedef int clockid_t;
				uint64_t clock_gettime_nsec_np(clockid_t clock_id); // For clock_gettime_nsec_np() a return value of non-0 indicates success.  A 0 return value indicates an error occurred and an error code is stored in errno.
				]]

				local function epoch()
					if not initDone then
						initOnce()
					end
					-- assert(C.time_gettimeofday(t, nil) == 0)
					local ret = C.time_gettimeofday(t, nil)
					if ret ~= 0 then
						printError("time_gettimeofday failed with error: '%s'", tostring(ret))
					end
					return toNumber(t.s) + toNumber(t.us)
				end
				time.epoch = epoch

				function time.time()
					return epoch() / 10 ^ 6
				end

				-- NOTE: this appears to be pointless on Intel Macs. The timebase fraction is always 1/1 and mach_absolute_time() does dynamic scaling internally.
				-- But this is needed on ARM Macs.
				--[[ local timebase = ffi.newAnchor("time_mach_timebase_info_data_t")
				local ret = C.time_mach_timebase_info(timebase)
				if ret ~= 0 then
					printError("time_mach_timebase_info failed with error: " .. toNumber(ret))
				end
				local scale = toNumber(timebase.numer) / toNumber(timebase.denom) / 10 ^ 9
				function time.clock()
					return toNumber(C.time_mach_absolute_time()) * scale
				end ]]
				--[[
				static const int CLOCK_REALTIME			= 0;
				static const int CLOCK_MONOTONIC			= 1;
				static const int CLOCK_PROCESS_CPUTIME_ID	= 2;
				static const int CLOCK_THREAD_CPUTIME_ID	= 3;
				static const int CLOCK_MONOTONIC_RAW		= 4;
				static const int CLOCK_REALTIME_COARSE		= 5;
				static const int CLOCK_MONOTONIC_COARSE	= 6;
				static const int CLOCK_BOOTTIME			= 7;
				static const int CLOCK_REALTIME_ALARM		= 8;
				static const int CLOCK_BOOTTIME_ALARM		= 9;
				]]
				-- http://www.manpagez.com/man/3/clock_gettime_nsec_np/
				local timeRet
				local nsInSec = 10 ^ 9
				function time.clock()
					timeRet = toNumber(C.clock_gettime_nsec_np(4)) -- CLOCK_MONOTONIC_RAW
					if timeRet == 0 then
						printError("OSX clock_gettime_nsec_np(4) failed, return value is 0")
					end
					return timeRet / nsInSec
				end
			else -- plain Lua with ffi lib
				ffi.cdef [[
				typedef struct {
					long    s;
					int32_t us;
				} time_timeval;
				int      gettimeofday(time_timeval*, void*);
				// uint64_t mach_absolute_time(void);
				typedef int clockid_t;
				uint64_t clock_gettime_nsec_np(clockid_t clock_id); // For clock_gettime_nsec_np() a return value of non-0 indicates success.  A 0 return value indicates an error occurred and an error code is stored in errno.
				]]

				function time.time()
					if not initDone then
						initOnce()
					end
					-- assert(C.time_gettimeofday(t, nil) == 0)
					local ret = C.gettimeofday(t, nil)
					if ret ~= 0 then
						printError("OSX gettimeofday failed with error: '%s'", tostring(ret))
					end
					return toNumber(t.s) + toNumber(t.us) / 10 ^ 6
				end

				--[[ local scale = 1 / 10 ^ 9
				function time.clock()
					return toNumber(C.mach_absolute_time()) * scale
				end ]]
				local timeRet
				local nsInSec = 10 ^ 9
				function time.clock()
					timeRet = toNumber(C.clock_gettime_nsec_np(4)) -- CLOCK_MONOTONIC_RAW
					if timeRet == 0 then
						printError("OSX clock_gettime_nsec_np(4) failed, return value is 0")
					end
					return timeRet / nsInSec
				end

			end
		end -- OSX

	end -- Linux or OSX

	--[[
	local tv, tps -- for time.get_seconds
	if ffi.os == 'Windows' then
		tps = ffi.newAnchor("LARGE_INTEGER[1]")
			--  Get the high resolution counter's accuracy.
		C.QueryPerformanceFrequency(tps) -- tv = ticksPerSecond
		-- qpf = toNumber(tv[0])
	else
		tv = ffi.newAnchor("struct timeval")
	end

	function time.get_seconds(multiplier, prevMs)
		local returnValue64_c
		local returnValue = 0 -- lua double

		if ffi.os == 'Windows' then
			--  What time is it?
			C.QueryPerformanceCounter(tv) -- tv2 = tick
			--  Convert the tick number into the number of seconds since the system was started.
			returnValue64_c = (tv[0].QuadPart * 100000) / (tps.QuadPart / 1000)
			-- time in microseconds
		else
			-- OSX, Posix, Linux?
			-- Use POSIX gettimeofday function util.to get precise time.
			local rc = C.gettimeofday(tv, nil)
			if rc ~= 0 then
				returnValue64_c = ffi.newNoAnchor("int64_t", -1) -- error here, we need to have returnValue64_c always ctype<int64_t>
			else
				returnValue64_c = (toNumber(tv.tv_sec) * 1000000) + toNumber(tps.tv_usec)
			end
		end
		--[=[
		 in Lua 0x001fffffffffffff is the (about) biggest value that does not change when
		 converting to Lua double with 'toNumber(returnValue64_c)'
		 returnValue64_c = bit.band(returnValue64_c, 0x00ffffff) -- get rid of highest bits
		 bit.band() does not work before Luajit 2.1 with 64 bit integers
		 or?
			returnValue64_c = bit.band(returnValue64_c, 0x00ffffff) -- get rid of highest bits
			returnValue = toNumber(returnValue64_c)
		]=]
		-- old way to get rid of highest bits, better to have unsigned int in timer:
		-- returnValue = toNumber(ffi.cast("uint32_t", returnValue64_c))
		returnValue = toNumber(ffi.cast("uint64_t", returnValue64_c)) --ffi.cast("intptr_t", retD)) -- for PUC Lua 5.1
		-- returnValue = toNumber(ffi.cast("double", returnValue64_c))


		if ffi.os == 'Windows' then
			if multiplier == 1 then
				returnValue = returnValue / 100000000  -- seconds -> microseconds
			elseif multiplier == 2 then
				returnValue = returnValue / 100000 -- seconds -> milliseconds
			else
				returnValue = returnValue / 100
			end
		else
			-- OSX, Posix, Linux?
			if multiplier == 1 then
				returnValue = returnValue / 1000000 -- microseconds -> second
			elseif multiplier == 2 then
				returnValue = returnValue / 1000 -- microseconds -> milliseconds
			end
		end

		if prevMs then
			if prevMs > returnValue then
				returnValue = prevMs - returnValue
			else
				returnValue = returnValue - prevMs
			end
		end
		return returnValue
	end
	--]]

end
init()

local floor = math.floor
local function round(val, decimal)
	return floor(val * 10 ^ decimal + 0.5) / 10 ^ decimal
end

local clock = time.clock
function time.seconds(prev_sec, decimals)
	-- local ret = get_seconds(1, prev_sec)
	local ret = clock()
	if prev_sec then
		ret = ret - prev_sec
		if ret < minSecondWarning then
			printWarning("seconds duration %.16f is smaller than zero, will return zero", ret)
			ret = 0
		end
	end
	if decimals then
		ret = round(ret, decimals)
	end
	return ret
end

function time.milliSeconds(prev_millisec, decimals)
	local ret = clock() * 1000
	if prev_millisec then
		ret = ret - prev_millisec
		if ret < minSecondWarning then
			printWarning("milliseconds duration %.16f is smaller than zero, will return zero", ret)
			ret = 0
		end
	end
	if decimals then
		ret = round(ret, decimals)
	end
	return ret
end

function time.microSeconds(prev_microsec, decimals)
	local ret = clock() * 1000000
	if prev_microsec then
		ret = ret - prev_microsec
		if ret < minSecondWarning then
			printWarning("microseconds duration %.16f is smaller than zero, will return zero", ret)
			ret = 0
		end
	end
	if decimals then
		ret = round(ret, decimals)
	end
	return ret
end

return time
