-- timer.lua
local util = require "util"
local json = require "json"
local peg = require "peg"
local dt = require "dt" -- add dateNumToText() to dt
-- local js = require "spiderjs"
local l = require"lang".l

--[[ Later.js
Later is a basic constraint solver. Each constraint (such as what hours or minutes a schedule is valid for) is considered independently. In other words, the hour constraint has no idea a minute constraint even exists. Later.schedule simply iterates over all of the constraints until a particular time is found where all of the constraints return true. If the current time (or start time passed in) meets all of the constraints, it will be returned. Since there are no millisecond constraints, the millisecond term of the date objects will never affect the result.

This bug is occurring because the timeout is firing sooner than expected (I ask it to time out after 12579 milliseconds, but it actually fires at 12569ms). I then fire the callback since the timeout expired and then reschedule using Later.schedule. However, the next valid schedule is still the time that the schedule calculated last time but now it is only 10ms away. I set the timeout to fire after 10ms and you get the double execution. I assumed that the timeout would occur >= the value specified but this is incorrect.

The fix is to somehow account for the jitter in the timeout function. This can be accomplished by either ensuring that it doesn't fire again so quickly or by capturing the next execution time and making sure we are greater than or equal to that time before firing again.
]]

local param
local testCount = 1
if not ... and (not util.from4d() or param == "") then
	param =
			'{"count":"3", "rule":["every 1 day at 00:00","every 5 min on the 0 min on the 12 sec except on Thursday and Friday","every 1 minutes on the 0 min on the 7 sec","every 3 min on the 0 min on the 44 sec"]}'
end

local scriptTxt
local path = util.mainPath()
scriptTxt = util.readFile(path .. "javascript/later.v8.js")
if not scriptTxt then
	util.printError(l("file '%s' was not found", "javascript/later.v8.js"))
	return util.getError()
end

-- we need our own code to call standard later.min.js, extend it here
local txt = util.readFile(path .. "javascript/later_nc_extend.js")
if not txt then
	util.printError(l("file '%s' was not found", "javascript/later_nc_extend.js"))
	return util.getError()
end
scriptTxt = scriptTxt .. txt

local function dateNumToText(result)
	local ret = peg.splitToArray(result, ",")
	for i in ipairs(ret) do
		local time = tonumber(ret[i])
		if type(time) ~= "number" then
			return nil, "error: ret[i] = " .. tostring(ret[i])
		else
			ret[i] = dt.toString(time) .. " " .. dt.toWeekdayShortName(time)
		end
	end
	return ret
end

local result, size = v8.compileAndRunScript(scriptTxt)
local function calculateRule(timerParam, isRestCall)
	local prf
	local timeAll = util.seconds()
	if type(timerParam) == "table" then
		prf = timerParam
	else
		prf = json.fromJson(timerParam)
	end
	if not prf.rule then
		local err = l("parameter '%s' was not found from preference %s", ".rule", json.toJson(prf))
		-- util.printError(err)
		return err
	end
	if not prf.count then
		prf.count = 1
	end

	local ret = {}
	local time, script, resultTxt, err
	local count = 0
	for _ = 1, testCount do
		for i, timer in ipairs(prf.rule) do
			count = count + 1
			script = "schedule('" .. timer .. "', " .. prf.count .. ");"
			time = util.seconds()
			result, size = v8.compileAndRunScript(script)
			time = util.seconds(time)
			resultTxt, err = dateNumToText(result)
			if err then
				return err
			end
			ret[i] = {timer = timer, result = result, result_text = resultTxt, time = time}
		end
	end
	timeAll = util.seconds(timeAll)
	ret = {total_time = timeAll, result = ret}

	if isRestCall or util.from4d() then
		local retTbl = {}
		for i, rec in ipairs(ret.result) do
			retTbl[i] = table.concat(rec.result_text, ", ")
		end
		if util.from4d() then
			return table.concat(retTbl, ";")
		end
		ret = {no_auth_return = true, data = retTbl}
	end

	return ret
end

local function cleanup()
	v8.cleanup() -- this is needed!
end

return {calculate = calculateRule, cleanup = cleanup}
