--- lib/ui/libcss.lua
-- https://github.com/Unix4ever/libcss.git
-- https://github.com/Unix4ever/libcss/blob/master/libcss/docs/API
-- /Users/pasi/installed/C/ui/libcss/libcss/docs/API
-- @module lib

local ffi = require "mffi"
local util = require "util"

local css = util.loadDll("ui/css/libcss")

ffi.cdef[[
// C/ui/libcss/libcss/include/libcss/fpmath.h

/* type for fixed point numbers */
typedef int32_t css_fixed;
]]

ffi.cdef[[
// C/ui/libcss/libwapcaplet/include/libwapcaplet/libwapcaplet.h

/**
 * The type of a reference counter used in libwapcaplet.
 */
typedef uint32_t lwc_refcounter;

/**
 * The type of a hash value used in libwapcaplet.
 */
typedef uint32_t lwc_hash;

/**
 * An interned string.
 *
 * NOTE: The contents of this struct are considered *PRIVATE* and may
 * change in future revisions.  Do not rely on them whatsoever.
 * They're only here at all so that the ref, unref and matches etc can
 * use them.
 */
typedef struct lwc_string_s {
        struct lwc_string_s **	prevptr;
        struct lwc_string_s *	next;
        size_t		len;
        lwc_hash	hash;
        lwc_refcounter	refcnt;
        struct lwc_string_s *	insensitive;
} lwc_string;
]]

ffi.cdef[[
// C/ui/libcss/libcss/include/libcss/types.h

/**
 * Source of charset information, in order of importance.
 * A client-dictated charset will override all others.
 * A document-specified charset will override autodetection or the default.
 */
typedef enum css_charset_source {
	CSS_CHARSET_DEFAULT          = 0,	/**< Default setting */
	CSS_CHARSET_REFERRED         = 1,	/**< From referring document */
	CSS_CHARSET_METADATA         = 2,	/**< From linking metadata */
	CSS_CHARSET_DOCUMENT         = 3,	/**< Defined in document */
	CSS_CHARSET_DICTATED         = 4	/**< Dictated by client */
} css_charset_source;

/**
 * Stylesheet language level -- defines parsing rules and supported properties
 */
typedef enum css_language_level {
	CSS_LEVEL_1                 = 0,	/**< CSS 1 */
	CSS_LEVEL_2                 = 1,	/**< CSS 2 */
	CSS_LEVEL_21                = 2,	/**< CSS 2.1 */
	CSS_LEVEL_3                 = 3,	/**< CSS 3 */
	CSS_LEVEL_DEFAULT           = CSS_LEVEL_21	/**< Default level */
} css_language_level;

/**
 * Stylesheet media types
 */
typedef enum css_media_type {
	CSS_MEDIA_AURAL             = (1 << 0),
	CSS_MEDIA_BRAILLE           = (1 << 1),
	CSS_MEDIA_EMBOSSED          = (1 << 2),
	CSS_MEDIA_HANDHELD          = (1 << 3),
	CSS_MEDIA_PRINT             = (1 << 4),
	CSS_MEDIA_PROJECTION        = (1 << 5),
	CSS_MEDIA_SCREEN            = (1 << 6),
	CSS_MEDIA_SPEECH            = (1 << 7),
	CSS_MEDIA_TTY               = (1 << 8),
	CSS_MEDIA_TV                = (1 << 9),
	CSS_MEDIA_ALL               = CSS_MEDIA_AURAL | CSS_MEDIA_BRAILLE |
	                              CSS_MEDIA_EMBOSSED | CSS_MEDIA_HANDHELD |
	                              CSS_MEDIA_PRINT | CSS_MEDIA_PROJECTION |
	                              CSS_MEDIA_SCREEN | CSS_MEDIA_SPEECH |
	                              CSS_MEDIA_TTY | CSS_MEDIA_TV
} css_media_type;

/**
 * Stylesheet origin
 */
typedef enum css_origin {
	CSS_ORIGIN_UA               = 0,	/**< User agent stylesheet */
	CSS_ORIGIN_USER             = 1,	/**< User stylesheet */
	CSS_ORIGIN_AUTHOR           = 2		/**< Author stylesheet */
} css_origin;

/** CSS colour -- AARRGGBB */
typedef uint32_t css_color;

/* CSS unit */
typedef enum css_unit {
	CSS_UNIT_PX                 = 0x00,
	CSS_UNIT_EX                 = 0x01,
	CSS_UNIT_EM                 = 0x02,
	CSS_UNIT_IN                 = 0x03,
	CSS_UNIT_CM                 = 0x04,
	CSS_UNIT_MM                 = 0x05,
	CSS_UNIT_PT                 = 0x06,
	CSS_UNIT_PC                 = 0x07,
	CSS_UNIT_CAP                = 0x08,
	CSS_UNIT_CH                 = 0x09,
	CSS_UNIT_IC                 = 0x0a,
	CSS_UNIT_REM                = 0x0b,
	CSS_UNIT_LH                 = 0x0c,
	CSS_UNIT_RLH                = 0x0d,
	CSS_UNIT_VH                 = 0x0e,
	CSS_UNIT_VW                 = 0x0f,
	CSS_UNIT_VI                 = 0x10,
	CSS_UNIT_VB                 = 0x11,
	CSS_UNIT_VMIN               = 0x12,
	CSS_UNIT_VMAX               = 0x13,
	CSS_UNIT_Q                  = 0x14,

	CSS_UNIT_PCT                = 0x15,	/* Percentage */

	CSS_UNIT_DEG                = 0x16,
	CSS_UNIT_GRAD               = 0x17,
	CSS_UNIT_RAD                = 0x18,

	CSS_UNIT_MS                 = 0x19,
	CSS_UNIT_S                  = 0x1a,

	CSS_UNIT_HZ                 = 0x1b,
	CSS_UNIT_KHZ                = 0x1c
} css_unit;

/**
 * Media orienations
 */
typedef enum css_media_orientation {
	CSS_MEDIA_ORIENTATION_PORTRAIT  = 0,
	CSS_MEDIA_ORIENTATION_LANDSCAPE = 1
} css_media_orientation;

/**
 * Media scans
 */
typedef enum css_media_scan {
	CSS_MEDIA_SCAN_PROGRESSIVE = 0,
	CSS_MEDIA_SCAN_INTERLACE   = 1
} css_media_scan;

/**
 * Media update-frequencies
 */
typedef enum css_media_update_frequency {
	CSS_MEDIA_UPDATE_FREQUENCY_NORMAL = 0,
	CSS_MEDIA_UPDATE_FREQUENCY_SLOW   = 1,
	CSS_MEDIA_UPDATE_FREQUENCY_NONE   = 2
} css_media_update_frequency;

/**
 * Media block overflows
 */
typedef enum css_media_overflow_block {
	CSS_MEDIA_OVERFLOW_BLOCK_NONE           = 0,
	CSS_MEDIA_OVERFLOW_BLOCK_SCROLL         = 1,
	CSS_MEDIA_OVERFLOW_BLOCK_OPTIONAL_PAGED = 2,
	CSS_MEDIA_OVERFLOW_BLOCK_PAGED          = 3
} css_media_overflow_block;

/**
 * Media inline overflows
 */
typedef enum css_media_overflow_inline {
	CSS_MEDIA_OVERFLOW_INLINE_NONE   = 0,
	CSS_MEDIA_OVERFLOW_INLINE_SCROLL = 1
} css_media_overflow_inline;

/**
 * Media pointers
 */
typedef enum css_media_pointer {
	CSS_MEDIA_POINTER_NONE   = 0,
	CSS_MEDIA_POINTER_COARSE = 1,
	CSS_MEDIA_POINTER_FINE   = 2
} css_media_pointer;

/**
 * Media hovers
 */
typedef enum css_media_hover {
	CSS_MEDIA_HOVER_NONE      = 0,
	CSS_MEDIA_HOVER_ON_DEMAND = 1,
	CSS_MEDIA_HOVER_HOVER     = 2
} css_media_hover;

/**
 * Media light-levels
 */
typedef enum css_media_light_level {
	CSS_MEDIA_LIGHT_LEVEL_NORMAL = 0,
	CSS_MEDIA_LIGHT_LEVEL_DIM    = 1,
	CSS_MEDIA_LIGHT_LEVEL_WASHED = 2
} css_media_light_level;

/**
 * Media scriptings
 */
typedef enum css_media_scripting {
	CSS_MEDIA_SCRIPTING_NONE         = 0,
	CSS_MEDIA_SCRIPTING_INITIAL_ONLY = 1,
	CSS_MEDIA_SCRIPTING_ENABLED      = 2
} css_media_scripting;

typedef struct css_media_resolution {
	css_fixed value;
	css_unit unit;
} css_media_resolution;

/**
 * Media specification
 */
typedef struct css_media {
	/* Media type */
	css_media_type        type;

	/* Viewport / page media features */
	css_fixed             width;  /* In css pixels */
	css_fixed             height; /* In css pixels */
	css_fixed             aspect_ratio;
	css_media_orientation orientation;

	/* Display quality media features */
	css_media_resolution       resolution;
	css_media_scan             scan;
	css_fixed                  grid; /** boolean: {0|1} */
	css_media_update_frequency update;
	css_media_overflow_block   overflow_block;
	css_media_overflow_inline  overflow_inline;

	/* Color media features */
	css_fixed color;      /* colour bpp (0 for monochrome) */
	css_fixed color_index;
	css_fixed monochrome; /* monochrome bpp (0 for colour) */
	css_fixed inverted_colors; /** boolean: {0|1} */

	/* Interaction media features */
	css_media_pointer pointer;
	css_media_pointer any_pointer;
	css_media_hover   hover;
	css_media_hover   any_hover;

	/* Environmental media features */
	css_media_light_level light_level;

	/* Scripting media features */
	css_media_scripting scripting;

	/* Client details for length conversion */
	css_fixed client_font_size;   /* In pt */
	css_fixed client_line_height; /* In css pixels */
} css_media;

/**
 * Type of a qualified name
 */
typedef struct css_qname {
	/**
	 * Namespace URI:
	 *
	 * NULL for no namespace
	 * '*' for any namespace (including none)
	 * URI for a specific namespace
	 */
	lwc_string *ns;

	/**
	 * Local part of qualified name
	 */
	lwc_string *name;
} css_qname;


typedef struct css_stylesheet css_stylesheet;

typedef struct css_select_ctx css_select_ctx;

typedef struct css_computed_style css_computed_style;

typedef struct css_font_face css_font_face;

typedef struct css_font_face_src css_font_face_src;

typedef uint32_t css_unicode_range[2];
]]

ffi.cdef[[
// C/ui/libcss/libcss/include/libcss/errors.h
typedef enum css_error {
	CSS_OK               = 0,

	CSS_NOMEM            = 1,
	CSS_BADPARM          = 2,
	CSS_INVALID          = 3,
	CSS_FILENOTFOUND     = 4,
	CSS_NEEDDATA         = 5,
	CSS_BADCHARSET       = 6,
	CSS_EOF              = 7,
	CSS_IMPORTS_PENDING  = 8,
	CSS_PROPERTY_NOT_SET = 9
} css_error;

/* Convert a libcss error value to a string */
const char *css_error_to_string(css_error error);
]]

ffi.cdef[[
// C/ui/libcss/libcss/include/libcss/properties.h

enum css_properties_e {
	CSS_PROP_AZIMUTH			= 0x000,
	CSS_PROP_BACKGROUND_ATTACHMENT		= 0x001,
	CSS_PROP_BACKGROUND_COLOR		= 0x002,
	CSS_PROP_BACKGROUND_IMAGE		= 0x003,
	CSS_PROP_BACKGROUND_POSITION		= 0x004,
	CSS_PROP_BACKGROUND_REPEAT		= 0x005,
	CSS_PROP_BORDER_COLLAPSE		= 0x006,
	CSS_PROP_BORDER_SPACING			= 0x007,
	CSS_PROP_BORDER_TOP_COLOR		= 0x008,
	CSS_PROP_BORDER_RIGHT_COLOR		= 0x009,
	CSS_PROP_BORDER_BOTTOM_COLOR		= 0x00a,
	CSS_PROP_BORDER_LEFT_COLOR		= 0x00b,
	CSS_PROP_BORDER_TOP_STYLE		= 0x00c,
	CSS_PROP_BORDER_RIGHT_STYLE		= 0x00d,
	CSS_PROP_BORDER_BOTTOM_STYLE		= 0x00e,
	CSS_PROP_BORDER_LEFT_STYLE		= 0x00f,
	CSS_PROP_BORDER_TOP_WIDTH		= 0x010,
	CSS_PROP_BORDER_RIGHT_WIDTH		= 0x011,
	CSS_PROP_BORDER_BOTTOM_WIDTH		= 0x012,
	CSS_PROP_BORDER_LEFT_WIDTH		= 0x013,
	CSS_PROP_BOTTOM				= 0x014,
	CSS_PROP_CAPTION_SIDE			= 0x015,
	CSS_PROP_CLEAR				= 0x016,
	CSS_PROP_CLIP				= 0x017,
	CSS_PROP_COLOR				= 0x018,
	CSS_PROP_CONTENT			= 0x019,
	CSS_PROP_COUNTER_INCREMENT		= 0x01a,
	CSS_PROP_COUNTER_RESET			= 0x01b,
	CSS_PROP_CUE_AFTER			= 0x01c,
	CSS_PROP_CUE_BEFORE			= 0x01d,
	CSS_PROP_CURSOR				= 0x01e,
	CSS_PROP_DIRECTION			= 0x01f,
	CSS_PROP_DISPLAY			= 0x020,
	CSS_PROP_ELEVATION			= 0x021,
	CSS_PROP_EMPTY_CELLS			= 0x022,
	CSS_PROP_FLOAT				= 0x023,
	CSS_PROP_FONT_FAMILY			= 0x024,
	CSS_PROP_FONT_SIZE			= 0x025,
	CSS_PROP_FONT_STYLE			= 0x026,
	CSS_PROP_FONT_VARIANT			= 0x027,
	CSS_PROP_FONT_WEIGHT			= 0x028,
	CSS_PROP_HEIGHT				= 0x029,
	CSS_PROP_LEFT				= 0x02a,
	CSS_PROP_LETTER_SPACING			= 0x02b,
	CSS_PROP_LINE_HEIGHT			= 0x02c,
	CSS_PROP_LIST_STYLE_IMAGE		= 0x02d,
	CSS_PROP_LIST_STYLE_POSITION		= 0x02e,
	CSS_PROP_LIST_STYLE_TYPE		= 0x02f,
	CSS_PROP_MARGIN_TOP			= 0x030,
	CSS_PROP_MARGIN_RIGHT			= 0x031,
	CSS_PROP_MARGIN_BOTTOM			= 0x032,
	CSS_PROP_MARGIN_LEFT			= 0x033,
	CSS_PROP_MAX_HEIGHT			= 0x034,
	CSS_PROP_MAX_WIDTH			= 0x035,
	CSS_PROP_MIN_HEIGHT			= 0x036,
	CSS_PROP_MIN_WIDTH			= 0x037,
	CSS_PROP_ORPHANS			= 0x038,
	CSS_PROP_OUTLINE_COLOR			= 0x039,
	CSS_PROP_OUTLINE_STYLE			= 0x03a,
	CSS_PROP_OUTLINE_WIDTH			= 0x03b,
	CSS_PROP_OVERFLOW_X			= 0x03c,
	CSS_PROP_PADDING_TOP			= 0x03d,
	CSS_PROP_PADDING_RIGHT			= 0x03e,
	CSS_PROP_PADDING_BOTTOM			= 0x03f,
	CSS_PROP_PADDING_LEFT			= 0x040,
	CSS_PROP_PAGE_BREAK_AFTER		= 0x041,
	CSS_PROP_PAGE_BREAK_BEFORE		= 0x042,
	CSS_PROP_PAGE_BREAK_INSIDE		= 0x043,
	CSS_PROP_PAUSE_AFTER			= 0x044,
	CSS_PROP_PAUSE_BEFORE			= 0x045,
	CSS_PROP_PITCH_RANGE			= 0x046,
	CSS_PROP_PITCH				= 0x047,
	CSS_PROP_PLAY_DURING			= 0x048,
	CSS_PROP_POSITION			= 0x049,
	CSS_PROP_QUOTES				= 0x04a,
	CSS_PROP_RICHNESS			= 0x04b,
	CSS_PROP_RIGHT				= 0x04c,
	CSS_PROP_SPEAK_HEADER			= 0x04d,
	CSS_PROP_SPEAK_NUMERAL			= 0x04e,
	CSS_PROP_SPEAK_PUNCTUATION		= 0x04f,
	CSS_PROP_SPEAK				= 0x050,
	CSS_PROP_SPEECH_RATE			= 0x051,
	CSS_PROP_STRESS				= 0x052,
	CSS_PROP_TABLE_LAYOUT			= 0x053,
	CSS_PROP_TEXT_ALIGN			= 0x054,
	CSS_PROP_TEXT_DECORATION		= 0x055,
	CSS_PROP_TEXT_INDENT			= 0x056,
	CSS_PROP_TEXT_TRANSFORM			= 0x057,
	CSS_PROP_TOP				= 0x058,
	CSS_PROP_UNICODE_BIDI			= 0x059,
	CSS_PROP_VERTICAL_ALIGN			= 0x05a,
	CSS_PROP_VISIBILITY			= 0x05b,
	CSS_PROP_VOICE_FAMILY			= 0x05c,
	CSS_PROP_VOLUME				= 0x05d,
	CSS_PROP_WHITE_SPACE			= 0x05e,
	CSS_PROP_WIDOWS				= 0x05f,
	CSS_PROP_WIDTH				= 0x060,
	CSS_PROP_WORD_SPACING			= 0x061,
	CSS_PROP_Z_INDEX			= 0x062,
	CSS_PROP_OPACITY			= 0x063,
	CSS_PROP_BREAK_AFTER			= 0x064,
	CSS_PROP_BREAK_BEFORE			= 0x065,
	CSS_PROP_BREAK_INSIDE			= 0x066,
	CSS_PROP_COLUMN_COUNT			= 0x067,
	CSS_PROP_COLUMN_FILL			= 0x068,
	CSS_PROP_COLUMN_GAP			= 0x069,
	CSS_PROP_COLUMN_RULE_COLOR		= 0x06a,
	CSS_PROP_COLUMN_RULE_STYLE		= 0x06b,
	CSS_PROP_COLUMN_RULE_WIDTH		= 0x06c,
	CSS_PROP_COLUMN_SPAN			= 0x06d,
	CSS_PROP_COLUMN_WIDTH			= 0x06e,
	CSS_PROP_WRITING_MODE			= 0x06f,
	CSS_PROP_OVERFLOW_Y			= 0x070,
	CSS_PROP_BOX_SIZING			= 0x071,
	CSS_PROP_ALIGN_CONTENT			= 0x072,
	CSS_PROP_ALIGN_ITEMS			= 0x073,
	CSS_PROP_ALIGN_SELF			= 0x074,
	CSS_PROP_FLEX_BASIS			= 0x075,
	CSS_PROP_FLEX_DIRECTION			= 0x076,
	CSS_PROP_FLEX_GROW			= 0x077,
	CSS_PROP_FLEX_SHRINK			= 0x078,
	CSS_PROP_FLEX_WRAP			= 0x079,
	CSS_PROP_JUSTIFY_CONTENT		= 0x07a,
	CSS_PROP_ORDER				= 0x07b,

	CSS_PROP_BORDER_TOP_LEFT_RADIUS     = 0x07c,
	CSS_PROP_BORDER_TOP_RIGHT_RADIUS    = 0x07d,
	CSS_PROP_BORDER_BOTTOM_RIGHT_RADIUS = 0x07e,
	CSS_PROP_BORDER_BOTTOM_LEFT_RADIUS  = 0x07f,

	CSS_N_PROPERTIES
};

enum css_align_content_e {
	CSS_ALIGN_CONTENT_INHERIT		= 0x0,
	CSS_ALIGN_CONTENT_STRETCH		= 0x1,
	CSS_ALIGN_CONTENT_FLEX_START		= 0x2,
	CSS_ALIGN_CONTENT_FLEX_END		= 0x3,
	CSS_ALIGN_CONTENT_CENTER		= 0x4,
	CSS_ALIGN_CONTENT_SPACE_BETWEEN		= 0x5,
	CSS_ALIGN_CONTENT_SPACE_AROUND		= 0x6,
	CSS_ALIGN_CONTENT_SPACE_EVENLY		= 0x7
};

enum css_align_items_e {
	CSS_ALIGN_ITEMS_INHERIT			= 0x0,
	CSS_ALIGN_ITEMS_STRETCH			= 0x1,
	CSS_ALIGN_ITEMS_FLEX_START		= 0x2,
	CSS_ALIGN_ITEMS_FLEX_END		= 0x3,
	CSS_ALIGN_ITEMS_CENTER			= 0x4,
	CSS_ALIGN_ITEMS_BASELINE		= 0x5
};

enum css_align_self_e {
	CSS_ALIGN_SELF_INHERIT			= 0x0,
	CSS_ALIGN_SELF_STRETCH			= 0x1,
	CSS_ALIGN_SELF_FLEX_START		= 0x2,
	CSS_ALIGN_SELF_FLEX_END			= 0x3,
	CSS_ALIGN_SELF_CENTER			= 0x4,
	CSS_ALIGN_SELF_BASELINE			= 0x5,
	CSS_ALIGN_SELF_AUTO			= 0x6
};

enum css_background_attachment_e {
	CSS_BACKGROUND_ATTACHMENT_INHERIT	= 0x0,
	CSS_BACKGROUND_ATTACHMENT_FIXED		= 0x1,
	CSS_BACKGROUND_ATTACHMENT_SCROLL	= 0x2
};

enum css_background_color_e {
	CSS_BACKGROUND_COLOR_INHERIT		= 0x0,
	CSS_BACKGROUND_COLOR_COLOR		= 0x1,
	CSS_BACKGROUND_COLOR_CURRENT_COLOR	= 0x2
};

enum css_background_image_e {
	CSS_BACKGROUND_IMAGE_INHERIT		= 0x0,
	/* Consult pointer in struct to determine which */
	CSS_BACKGROUND_IMAGE_NONE		= 0x1,
	CSS_BACKGROUND_IMAGE_IMAGE		= 0x1
};

enum css_background_position_e {
	CSS_BACKGROUND_POSITION_INHERIT		= 0x0,
	CSS_BACKGROUND_POSITION_SET		= 0x1
};

enum css_background_repeat_e {
	CSS_BACKGROUND_REPEAT_INHERIT		= 0x0,
	CSS_BACKGROUND_REPEAT_REPEAT_X		= 0x1,
	CSS_BACKGROUND_REPEAT_REPEAT_Y		= 0x2,
	CSS_BACKGROUND_REPEAT_REPEAT		= 0x3,
	CSS_BACKGROUND_REPEAT_NO_REPEAT		= 0x4
};

enum css_border_collapse_e {
	CSS_BORDER_COLLAPSE_INHERIT		= 0x0,
	CSS_BORDER_COLLAPSE_SEPARATE		= 0x1,
	CSS_BORDER_COLLAPSE_COLLAPSE		= 0x2
};

enum css_border_spacing_e {
	CSS_BORDER_SPACING_INHERIT		= 0x0,
	CSS_BORDER_SPACING_SET			= 0x1
};

enum css_border_color_e {
	CSS_BORDER_COLOR_INHERIT		= CSS_BACKGROUND_COLOR_INHERIT,
	CSS_BORDER_COLOR_COLOR			= CSS_BACKGROUND_COLOR_COLOR,
	CSS_BORDER_COLOR_CURRENT_COLOR		= CSS_BACKGROUND_COLOR_CURRENT_COLOR
};

enum css_border_style_e {
	CSS_BORDER_STYLE_INHERIT		= 0x0,
	CSS_BORDER_STYLE_NONE			= 0x1,
	CSS_BORDER_STYLE_HIDDEN			= 0x2,
	CSS_BORDER_STYLE_DOTTED			= 0x3,
	CSS_BORDER_STYLE_DASHED			= 0x4,
	CSS_BORDER_STYLE_SOLID			= 0x5,
	CSS_BORDER_STYLE_DOUBLE			= 0x6,
	CSS_BORDER_STYLE_GROOVE			= 0x7,
	CSS_BORDER_STYLE_RIDGE			= 0x8,
	CSS_BORDER_STYLE_INSET			= 0x9,
	CSS_BORDER_STYLE_OUTSET			= 0xa
};

enum css_border_width_e {
	CSS_BORDER_WIDTH_INHERIT		= 0x0,
	CSS_BORDER_WIDTH_THIN			= 0x1,
	CSS_BORDER_WIDTH_MEDIUM			= 0x2,
	CSS_BORDER_WIDTH_THICK			= 0x3,
	CSS_BORDER_WIDTH_WIDTH			= 0x4
};

enum css_bottom_e {
	CSS_BOTTOM_INHERIT			= 0x0,
	CSS_BOTTOM_SET				= 0x1,
	CSS_BOTTOM_AUTO				= 0x2
};

enum css_box_sizing_e {
	CSS_BOX_SIZING_INHERIT			= 0x0,
	CSS_BOX_SIZING_CONTENT_BOX		= 0x1,
	CSS_BOX_SIZING_BORDER_BOX		= 0x2
};

enum css_break_after_e {
	CSS_BREAK_AFTER_INHERIT			= 0x0,
	CSS_BREAK_AFTER_AUTO			= 0x1,
	CSS_BREAK_AFTER_AVOID			= 0x2,
	CSS_BREAK_AFTER_ALWAYS			= 0x3,
	CSS_BREAK_AFTER_LEFT			= 0x4,
	CSS_BREAK_AFTER_RIGHT			= 0x5,
	CSS_BREAK_AFTER_PAGE			= 0x6,
	CSS_BREAK_AFTER_COLUMN			= 0x7,
	CSS_BREAK_AFTER_AVOID_PAGE		= 0x8,
	CSS_BREAK_AFTER_AVOID_COLUMN		= 0x9
};

enum css_break_before_e {
	CSS_BREAK_BEFORE_INHERIT		= CSS_BREAK_AFTER_INHERIT,
	CSS_BREAK_BEFORE_AUTO			= CSS_BREAK_AFTER_AUTO,
	CSS_BREAK_BEFORE_AVOID			= CSS_BREAK_AFTER_AVOID,
	CSS_BREAK_BEFORE_ALWAYS			= CSS_BREAK_AFTER_ALWAYS,
	CSS_BREAK_BEFORE_LEFT			= CSS_BREAK_AFTER_LEFT,
	CSS_BREAK_BEFORE_RIGHT			= CSS_BREAK_AFTER_RIGHT,
	CSS_BREAK_BEFORE_PAGE			= CSS_BREAK_AFTER_PAGE,
	CSS_BREAK_BEFORE_COLUMN			= CSS_BREAK_AFTER_COLUMN,
	CSS_BREAK_BEFORE_AVOID_PAGE		= CSS_BREAK_AFTER_AVOID_PAGE,
	CSS_BREAK_BEFORE_AVOID_COLUMN		= CSS_BREAK_AFTER_AVOID_COLUMN
};

enum css_break_inside_e {
	CSS_BREAK_INSIDE_INHERIT		= CSS_BREAK_AFTER_INHERIT,
	CSS_BREAK_INSIDE_AUTO			= CSS_BREAK_AFTER_AUTO,
	CSS_BREAK_INSIDE_AVOID			= CSS_BREAK_AFTER_AVOID,
	CSS_BREAK_INSIDE_AVOID_PAGE		= CSS_BREAK_AFTER_AVOID_PAGE,
	CSS_BREAK_INSIDE_AVOID_COLUMN		= CSS_BREAK_AFTER_AVOID_COLUMN
};

enum css_caption_side_e {
	CSS_CAPTION_SIDE_INHERIT		= 0x0,
	CSS_CAPTION_SIDE_TOP			= 0x1,
	CSS_CAPTION_SIDE_BOTTOM			= 0x2
};

enum css_clear_e {
	CSS_CLEAR_INHERIT			= 0x0,
	CSS_CLEAR_NONE				= 0x1,
	CSS_CLEAR_LEFT				= 0x2,
	CSS_CLEAR_RIGHT				= 0x3,
	CSS_CLEAR_BOTH				= 0x4
};

enum css_clip_e {
	CSS_CLIP_INHERIT			= 0x0,
	CSS_CLIP_AUTO				= 0x1,
	CSS_CLIP_RECT				= 0x2
};

enum css_color_e {
	CSS_COLOR_INHERIT			= 0x0,
	CSS_COLOR_COLOR				= 0x1
};

enum css_column_count_e {
	CSS_COLUMN_COUNT_INHERIT		= 0x0,
	CSS_COLUMN_COUNT_AUTO			= 0x1,
	CSS_COLUMN_COUNT_SET			= 0x2
};

enum css_column_fill_e {
	CSS_COLUMN_FILL_INHERIT			= 0x0,
	CSS_COLUMN_FILL_BALANCE			= 0x1,
	CSS_COLUMN_FILL_AUTO			= 0x2
};

enum css_column_gap_e {
	CSS_COLUMN_GAP_INHERIT			= 0x0,
	CSS_COLUMN_GAP_SET			= 0x1,
	CSS_COLUMN_GAP_NORMAL			= 0x2
};

enum css_column_rule_color_e {
	CSS_COLUMN_RULE_COLOR_INHERIT		= CSS_BACKGROUND_COLOR_INHERIT,
	CSS_COLUMN_RULE_COLOR_COLOR		= CSS_BACKGROUND_COLOR_COLOR,
	CSS_COLUMN_RULE_COLOR_CURRENT_COLOR	= CSS_BACKGROUND_COLOR_CURRENT_COLOR
};

enum css_column_rule_style_e {
	CSS_COLUMN_RULE_STYLE_INHERIT		= CSS_BORDER_STYLE_INHERIT,
	CSS_COLUMN_RULE_STYLE_NONE		= CSS_BORDER_STYLE_NONE,
	CSS_COLUMN_RULE_STYLE_HIDDEN		= CSS_BORDER_STYLE_HIDDEN,
	CSS_COLUMN_RULE_STYLE_DOTTED		= CSS_BORDER_STYLE_DOTTED,
	CSS_COLUMN_RULE_STYLE_DASHED		= CSS_BORDER_STYLE_DASHED,
	CSS_COLUMN_RULE_STYLE_SOLID		= CSS_BORDER_STYLE_SOLID,
	CSS_COLUMN_RULE_STYLE_DOUBLE		= CSS_BORDER_STYLE_DOUBLE,
	CSS_COLUMN_RULE_STYLE_GROOVE		= CSS_BORDER_STYLE_GROOVE,
	CSS_COLUMN_RULE_STYLE_RIDGE		= CSS_BORDER_STYLE_RIDGE,
	CSS_COLUMN_RULE_STYLE_INSET		= CSS_BORDER_STYLE_INSET,
	CSS_COLUMN_RULE_STYLE_OUTSET		= CSS_BORDER_STYLE_OUTSET
};

enum css_column_rule_width_e {
	CSS_COLUMN_RULE_WIDTH_INHERIT		= CSS_BORDER_WIDTH_INHERIT,
	CSS_COLUMN_RULE_WIDTH_THIN		= CSS_BORDER_WIDTH_THIN,
	CSS_COLUMN_RULE_WIDTH_MEDIUM		= CSS_BORDER_WIDTH_MEDIUM,
	CSS_COLUMN_RULE_WIDTH_THICK		= CSS_BORDER_WIDTH_THICK,
	CSS_COLUMN_RULE_WIDTH_WIDTH		= CSS_BORDER_WIDTH_WIDTH
};

enum css_column_span_e {
	CSS_COLUMN_SPAN_INHERIT			= 0x0,
	CSS_COLUMN_SPAN_NONE			= 0x1,
	CSS_COLUMN_SPAN_ALL			= 0x2
};

enum css_column_width_e {
	CSS_COLUMN_WIDTH_INHERIT		= 0x0,
	CSS_COLUMN_WIDTH_SET			= 0x1,
	CSS_COLUMN_WIDTH_AUTO			= 0x2
};

enum css_content_e {
	CSS_CONTENT_INHERIT			= 0x0,
	CSS_CONTENT_NONE			= 0x1,
	CSS_CONTENT_NORMAL			= 0x2,
	CSS_CONTENT_SET				= 0x3
};

enum css_counter_increment_e {
	CSS_COUNTER_INCREMENT_INHERIT		= 0x0,
	/* Consult pointer in struct to determine which */
	CSS_COUNTER_INCREMENT_NAMED		= 0x1,
	CSS_COUNTER_INCREMENT_NONE		= 0x1
};

enum css_counter_reset_e {
	CSS_COUNTER_RESET_INHERIT		= 0x0,
	/* Consult pointer in struct to determine which */
	CSS_COUNTER_RESET_NAMED			= 0x1,
	CSS_COUNTER_RESET_NONE			= 0x1
};

enum css_cursor_e {
	CSS_CURSOR_INHERIT			= 0x00,
	/* URLs exist if pointer is non-NULL */
	CSS_CURSOR_AUTO				= 0x01,
	CSS_CURSOR_CROSSHAIR			= 0x02,
	CSS_CURSOR_DEFAULT			= 0x03,
	CSS_CURSOR_POINTER			= 0x04,
	CSS_CURSOR_MOVE				= 0x05,
	CSS_CURSOR_E_RESIZE			= 0x06,
	CSS_CURSOR_NE_RESIZE			= 0x07,
	CSS_CURSOR_NW_RESIZE			= 0x08,
	CSS_CURSOR_N_RESIZE			= 0x09,
	CSS_CURSOR_SE_RESIZE			= 0x0a,
	CSS_CURSOR_SW_RESIZE			= 0x0b,
	CSS_CURSOR_S_RESIZE			= 0x0c,
	CSS_CURSOR_W_RESIZE			= 0x0d,
	CSS_CURSOR_TEXT				= 0x0e,
	CSS_CURSOR_WAIT				= 0x0f,
	CSS_CURSOR_HELP				= 0x10,
	CSS_CURSOR_PROGRESS			= 0x11
};

enum css_direction_e {
	CSS_DIRECTION_INHERIT			= 0x0,
	CSS_DIRECTION_LTR			= 0x1,
	CSS_DIRECTION_RTL			= 0x2
};

enum css_display_e {
	CSS_DISPLAY_INHERIT			= 0x00,
	CSS_DISPLAY_INLINE			= 0x01,
	CSS_DISPLAY_BLOCK			= 0x02,
	CSS_DISPLAY_LIST_ITEM			= 0x03,
	CSS_DISPLAY_RUN_IN			= 0x04,
	CSS_DISPLAY_INLINE_BLOCK		= 0x05,
	CSS_DISPLAY_TABLE			= 0x06,
	CSS_DISPLAY_INLINE_TABLE		= 0x07,
	CSS_DISPLAY_TABLE_ROW_GROUP		= 0x08,
	CSS_DISPLAY_TABLE_HEADER_GROUP		= 0x09,
	CSS_DISPLAY_TABLE_FOOTER_GROUP		= 0x0a,
	CSS_DISPLAY_TABLE_ROW			= 0x0b,
	CSS_DISPLAY_TABLE_COLUMN_GROUP		= 0x0c,
	CSS_DISPLAY_TABLE_COLUMN		= 0x0d,
	CSS_DISPLAY_TABLE_CELL			= 0x0e,
	CSS_DISPLAY_TABLE_CAPTION		= 0x0f,
	CSS_DISPLAY_NONE			= 0x10,
	CSS_DISPLAY_FLEX			= 0x11,
	CSS_DISPLAY_INLINE_FLEX			= 0x12
};

enum css_empty_cells_e {
	CSS_EMPTY_CELLS_INHERIT			= 0x0,
	CSS_EMPTY_CELLS_SHOW			= 0x1,
	CSS_EMPTY_CELLS_HIDE			= 0x2
};

enum css_flex_basis_e {
	CSS_FLEX_BASIS_INHERIT			= 0x0,
	CSS_FLEX_BASIS_SET			= 0x1,
	CSS_FLEX_BASIS_AUTO			= 0x2,
	CSS_FLEX_BASIS_CONTENT			= 0x3
};

enum css_border_radius_e {
	CSS_BORDER_RADIUS_INHERIT   = 0x0,
	CSS_BORDER_RADIUS_SET       = 0x1,
	CSS_BORDER_RADIUS_AUTO      = 0x2
};

enum css_flex_direction_e {
	CSS_FLEX_DIRECTION_INHERIT		= 0x0,
	CSS_FLEX_DIRECTION_ROW			= 0x1,
	CSS_FLEX_DIRECTION_ROW_REVERSE		= 0x2,
	CSS_FLEX_DIRECTION_COLUMN		= 0x3,
	CSS_FLEX_DIRECTION_COLUMN_REVERSE	= 0x4
};

enum css_flex_grow_e {
	CSS_FLEX_GROW_INHERIT			= 0x0,
	CSS_FLEX_GROW_SET			= 0x1
};

enum css_flex_shrink_e {
	CSS_FLEX_SHRINK_INHERIT			= 0x0,
	CSS_FLEX_SHRINK_SET			= 0x1
};

enum css_flex_wrap_e {
	CSS_FLEX_WRAP_INHERIT			= 0x0,
	CSS_FLEX_WRAP_NOWRAP			= 0x1,
	CSS_FLEX_WRAP_WRAP			= 0x2,
	CSS_FLEX_WRAP_WRAP_REVERSE		= 0x3
};

enum css_float_e {
	CSS_FLOAT_INHERIT			= 0x0,
	CSS_FLOAT_LEFT				= 0x1,
	CSS_FLOAT_RIGHT				= 0x2,
	CSS_FLOAT_NONE				= 0x3
};

enum css_font_family_e {
	CSS_FONT_FAMILY_INHERIT			= 0x0,
	/* Named fonts exist if pointer is non-NULL */
	CSS_FONT_FAMILY_SERIF			= 0x1,
	CSS_FONT_FAMILY_SANS_SERIF		= 0x2,
	CSS_FONT_FAMILY_CURSIVE			= 0x3,
	CSS_FONT_FAMILY_FANTASY			= 0x4,
	CSS_FONT_FAMILY_MONOSPACE		= 0x5
};

enum css_font_size_e {
	CSS_FONT_SIZE_INHERIT			= 0x0,
	CSS_FONT_SIZE_XX_SMALL			= 0x1,
	CSS_FONT_SIZE_X_SMALL			= 0x2,
	CSS_FONT_SIZE_SMALL			= 0x3,
	CSS_FONT_SIZE_MEDIUM			= 0x4,
	CSS_FONT_SIZE_LARGE			= 0x5,
	CSS_FONT_SIZE_X_LARGE			= 0x6,
	CSS_FONT_SIZE_XX_LARGE			= 0x7,
	CSS_FONT_SIZE_LARGER			= 0x8,
	CSS_FONT_SIZE_SMALLER			= 0x9,
	CSS_FONT_SIZE_DIMENSION			= 0xa
};

enum css_font_style_e {
	CSS_FONT_STYLE_INHERIT			= 0x0,
	CSS_FONT_STYLE_NORMAL			= 0x1,
	CSS_FONT_STYLE_ITALIC			= 0x2,
	CSS_FONT_STYLE_OBLIQUE			= 0x3
};

enum css_font_variant_e {
	CSS_FONT_VARIANT_INHERIT		= 0x0,
	CSS_FONT_VARIANT_NORMAL			= 0x1,
	CSS_FONT_VARIANT_SMALL_CAPS		= 0x2
};

enum css_font_weight_e {
	CSS_FONT_WEIGHT_INHERIT			= 0x0,
	CSS_FONT_WEIGHT_NORMAL			= 0x1,
	CSS_FONT_WEIGHT_BOLD			= 0x2,
	CSS_FONT_WEIGHT_BOLDER			= 0x3,
	CSS_FONT_WEIGHT_LIGHTER			= 0x4,
	CSS_FONT_WEIGHT_100			= 0x5,
	CSS_FONT_WEIGHT_200			= 0x6,
	CSS_FONT_WEIGHT_300			= 0x7,
	CSS_FONT_WEIGHT_400			= 0x8,
	CSS_FONT_WEIGHT_500			= 0x9,
	CSS_FONT_WEIGHT_600			= 0xa,
	CSS_FONT_WEIGHT_700			= 0xb,
	CSS_FONT_WEIGHT_800			= 0xc,
	CSS_FONT_WEIGHT_900			= 0xd
};

enum css_height_e {
	CSS_HEIGHT_INHERIT			= 0x0,
	CSS_HEIGHT_SET				= 0x1,
	CSS_HEIGHT_AUTO				= 0x2
};

enum css_justify_content_e {
	CSS_JUSTIFY_CONTENT_INHERIT		= 0x0,
	CSS_JUSTIFY_CONTENT_FLEX_START		= 0x1,
	CSS_JUSTIFY_CONTENT_FLEX_END		= 0x2,
	CSS_JUSTIFY_CONTENT_CENTER		= 0x3,
	CSS_JUSTIFY_CONTENT_SPACE_BETWEEN	= 0x4,
	CSS_JUSTIFY_CONTENT_SPACE_AROUND	= 0x5,
	CSS_JUSTIFY_CONTENT_SPACE_EVENLY	= 0x6
};

enum css_left_e {
	CSS_LEFT_INHERIT			= 0x0,
	CSS_LEFT_SET				= 0x1,
	CSS_LEFT_AUTO				= 0x2
};

enum css_letter_spacing_e {
	CSS_LETTER_SPACING_INHERIT		= CSS_COLUMN_GAP_INHERIT,
	CSS_LETTER_SPACING_SET			= CSS_COLUMN_GAP_SET,
	CSS_LETTER_SPACING_NORMAL		= CSS_COLUMN_GAP_NORMAL
};

enum css_line_height_e {
	CSS_LINE_HEIGHT_INHERIT			= 0x0,
	CSS_LINE_HEIGHT_NUMBER			= 0x1,
	CSS_LINE_HEIGHT_DIMENSION		= 0x2,
	CSS_LINE_HEIGHT_NORMAL			= 0x3
};

enum css_list_style_image_e {
	CSS_LIST_STYLE_IMAGE_INHERIT		= 0x0,
	/* Consult pointer in struct to determine which */
	CSS_LIST_STYLE_IMAGE_URI		= 0x1,
	CSS_LIST_STYLE_IMAGE_NONE		= 0x1
};

enum css_list_style_position_e {
	CSS_LIST_STYLE_POSITION_INHERIT		= 0x0,
	CSS_LIST_STYLE_POSITION_INSIDE		= 0x1,
	CSS_LIST_STYLE_POSITION_OUTSIDE		= 0x2
};

enum css_list_style_type_e {
	CSS_LIST_STYLE_TYPE_INHERIT		= 0x0,
	CSS_LIST_STYLE_TYPE_DISC		= 0x1,
	CSS_LIST_STYLE_TYPE_CIRCLE		= 0x2,
	CSS_LIST_STYLE_TYPE_SQUARE		= 0x3,
	CSS_LIST_STYLE_TYPE_DECIMAL		= 0x4,
	CSS_LIST_STYLE_TYPE_DECIMAL_LEADING_ZERO= 0x5,
	CSS_LIST_STYLE_TYPE_LOWER_ROMAN		= 0x6,
	CSS_LIST_STYLE_TYPE_UPPER_ROMAN		= 0x7,
	CSS_LIST_STYLE_TYPE_LOWER_GREEK		= 0x8,
	CSS_LIST_STYLE_TYPE_LOWER_LATIN		= 0x9,
	CSS_LIST_STYLE_TYPE_UPPER_LATIN		= 0xa,
	CSS_LIST_STYLE_TYPE_ARMENIAN		= 0xb,
	CSS_LIST_STYLE_TYPE_GEORGIAN		= 0xc,
	CSS_LIST_STYLE_TYPE_LOWER_ALPHA		= 0xd,
	CSS_LIST_STYLE_TYPE_UPPER_ALPHA		= 0xe,
	CSS_LIST_STYLE_TYPE_NONE		= 0xf
};

enum css_margin_e {
	CSS_MARGIN_INHERIT			= 0x0,
	CSS_MARGIN_SET				= 0x1,
	CSS_MARGIN_AUTO				= 0x2
};

enum css_max_height_e {
	CSS_MAX_HEIGHT_INHERIT			= 0x0,
	CSS_MAX_HEIGHT_SET			= 0x1,
	CSS_MAX_HEIGHT_NONE			= 0x2
};

enum css_max_width_e {
	CSS_MAX_WIDTH_INHERIT			= 0x0,
	CSS_MAX_WIDTH_SET			= 0x1,
	CSS_MAX_WIDTH_NONE			= 0x2
};

enum css_min_height_e {
	CSS_MIN_HEIGHT_INHERIT			= 0x0,
	CSS_MIN_HEIGHT_SET			= 0x1,
	CSS_MIN_HEIGHT_AUTO			= 0x2
};

enum css_min_width_e {
	CSS_MIN_WIDTH_INHERIT			= 0x0,
	CSS_MIN_WIDTH_SET			= 0x1,
	CSS_MIN_WIDTH_AUTO			= 0x2
};

enum css_opacity_e {
	CSS_OPACITY_INHERIT			= 0x0,
	CSS_OPACITY_SET				= 0x1
};

enum css_order_e {
	CSS_ORDER_INHERIT			= 0x0,
	CSS_ORDER_SET				= 0x1
};

enum css_outline_color_e {
	CSS_OUTLINE_COLOR_INHERIT		= CSS_BACKGROUND_COLOR_INHERIT,
	CSS_OUTLINE_COLOR_COLOR			= CSS_BACKGROUND_COLOR_COLOR,
	CSS_OUTLINE_COLOR_CURRENT_COLOR		= CSS_BACKGROUND_COLOR_CURRENT_COLOR,
	CSS_OUTLINE_COLOR_INVERT		= 0x3
};

enum css_outline_style_e {
	CSS_OUTLINE_STYLE_INHERIT		= CSS_BORDER_STYLE_INHERIT,
	CSS_OUTLINE_STYLE_NONE			= CSS_BORDER_STYLE_NONE,
	CSS_OUTLINE_STYLE_DOTTED		= CSS_BORDER_STYLE_DOTTED,
	CSS_OUTLINE_STYLE_DASHED		= CSS_BORDER_STYLE_DASHED,
	CSS_OUTLINE_STYLE_SOLID			= CSS_BORDER_STYLE_SOLID,
	CSS_OUTLINE_STYLE_DOUBLE		= CSS_BORDER_STYLE_DOUBLE,
	CSS_OUTLINE_STYLE_GROOVE		= CSS_BORDER_STYLE_GROOVE,
	CSS_OUTLINE_STYLE_RIDGE			= CSS_BORDER_STYLE_RIDGE,
	CSS_OUTLINE_STYLE_INSET			= CSS_BORDER_STYLE_INSET,
	CSS_OUTLINE_STYLE_OUTSET		= CSS_BORDER_STYLE_OUTSET
};

enum css_outline_width_e {
	CSS_OUTLINE_WIDTH_INHERIT		= CSS_BORDER_WIDTH_INHERIT,
	CSS_OUTLINE_WIDTH_THIN			= CSS_BORDER_WIDTH_THIN,
	CSS_OUTLINE_WIDTH_MEDIUM		= CSS_BORDER_WIDTH_MEDIUM,
	CSS_OUTLINE_WIDTH_THICK			= CSS_BORDER_WIDTH_THICK,
	CSS_OUTLINE_WIDTH_WIDTH			= CSS_BORDER_WIDTH_WIDTH
};

enum css_overflow_e {
	CSS_OVERFLOW_INHERIT			= 0x0,
	CSS_OVERFLOW_VISIBLE			= 0x1,
	CSS_OVERFLOW_HIDDEN			= 0x2,
	CSS_OVERFLOW_SCROLL			= 0x3,
	CSS_OVERFLOW_AUTO			= 0x4
};

enum css_orphans_e {
	CSS_ORPHANS_INHERIT			= 0x0,
	CSS_ORPHANS_SET				= 0x1
};

enum css_padding_e {
	CSS_PADDING_INHERIT			= 0x0,
	CSS_PADDING_SET				= 0x1
};

enum css_page_break_after_e {
	CSS_PAGE_BREAK_AFTER_INHERIT		= CSS_BREAK_AFTER_INHERIT,
	CSS_PAGE_BREAK_AFTER_AUTO		= CSS_BREAK_AFTER_AUTO,
	CSS_PAGE_BREAK_AFTER_AVOID		= CSS_BREAK_AFTER_AVOID,
	CSS_PAGE_BREAK_AFTER_ALWAYS		= CSS_BREAK_AFTER_ALWAYS,
	CSS_PAGE_BREAK_AFTER_LEFT		= CSS_BREAK_AFTER_LEFT,
	CSS_PAGE_BREAK_AFTER_RIGHT		= CSS_BREAK_AFTER_RIGHT
};

enum css_page_break_before_e {
	CSS_PAGE_BREAK_BEFORE_INHERIT		= CSS_BREAK_AFTER_INHERIT,
	CSS_PAGE_BREAK_BEFORE_AUTO		= CSS_BREAK_AFTER_AUTO,
	CSS_PAGE_BREAK_BEFORE_AVOID		= CSS_BREAK_AFTER_AVOID,
	CSS_PAGE_BREAK_BEFORE_ALWAYS		= CSS_BREAK_AFTER_ALWAYS,
	CSS_PAGE_BREAK_BEFORE_LEFT		= CSS_BREAK_AFTER_LEFT,
	CSS_PAGE_BREAK_BEFORE_RIGHT		= CSS_BREAK_AFTER_RIGHT
};

enum css_page_break_inside_e {
	CSS_PAGE_BREAK_INSIDE_INHERIT		= CSS_BREAK_AFTER_INHERIT,
	CSS_PAGE_BREAK_INSIDE_AUTO		= CSS_BREAK_AFTER_AUTO,
	CSS_PAGE_BREAK_INSIDE_AVOID		= CSS_BREAK_AFTER_AVOID
};

enum css_position_e {
	CSS_POSITION_INHERIT			= 0x0,
	CSS_POSITION_STATIC			= 0x1,
	CSS_POSITION_RELATIVE			= 0x2,
	CSS_POSITION_ABSOLUTE			= 0x3,
	CSS_POSITION_FIXED			= 0x4
};

enum css_quotes_e {
	CSS_QUOTES_INHERIT			= 0x0,
	/* Consult pointer in struct to determine which */
	CSS_QUOTES_STRING			= 0x1,
	CSS_QUOTES_NONE				= 0x1
};

enum css_right_e {
	CSS_RIGHT_INHERIT			= 0x0,
	CSS_RIGHT_SET				= 0x1,
	CSS_RIGHT_AUTO				= 0x2
};

enum css_table_layout_e {
	CSS_TABLE_LAYOUT_INHERIT		= 0x0,
	CSS_TABLE_LAYOUT_AUTO			= 0x1,
	CSS_TABLE_LAYOUT_FIXED			= 0x2
};

enum css_text_align_e {
	CSS_TEXT_ALIGN_INHERIT			= 0x0,
	CSS_TEXT_ALIGN_INHERIT_IF_NON_MAGIC	= 0x1,
	CSS_TEXT_ALIGN_LEFT			= 0x2,
	CSS_TEXT_ALIGN_RIGHT			= 0x3,
	CSS_TEXT_ALIGN_CENTER			= 0x4,
	CSS_TEXT_ALIGN_JUSTIFY			= 0x5,
	CSS_TEXT_ALIGN_DEFAULT			= 0x6,
	CSS_TEXT_ALIGN_LIBCSS_LEFT		= 0x7,
	CSS_TEXT_ALIGN_LIBCSS_CENTER		= 0x8,
	CSS_TEXT_ALIGN_LIBCSS_RIGHT		= 0x9
};

enum css_text_decoration_e {
	CSS_TEXT_DECORATION_INHERIT		= 0x00,
	CSS_TEXT_DECORATION_NONE		= 0x10,
	CSS_TEXT_DECORATION_BLINK		= (1<<3),
	CSS_TEXT_DECORATION_LINE_THROUGH	= (1<<2),
	CSS_TEXT_DECORATION_OVERLINE		= (1<<1),
	CSS_TEXT_DECORATION_UNDERLINE		= (1<<0)
};

enum css_text_indent_e {
	CSS_TEXT_INDENT_INHERIT			= 0x0,
	CSS_TEXT_INDENT_SET			= 0x1
};

enum css_text_transform_e {
	CSS_TEXT_TRANSFORM_INHERIT		= 0x0,
	CSS_TEXT_TRANSFORM_CAPITALIZE		= 0x1,
	CSS_TEXT_TRANSFORM_UPPERCASE		= 0x2,
	CSS_TEXT_TRANSFORM_LOWERCASE		= 0x3,
	CSS_TEXT_TRANSFORM_NONE			= 0x4
};

enum css_top_e {
	CSS_TOP_INHERIT				= 0x0,
	CSS_TOP_SET				= 0x1,
	CSS_TOP_AUTO				= 0x2
};

enum css_unicode_bidi_e {
	CSS_UNICODE_BIDI_INHERIT		= 0x0,
	CSS_UNICODE_BIDI_NORMAL			= 0x1,
	CSS_UNICODE_BIDI_EMBED			= 0x2,
	CSS_UNICODE_BIDI_BIDI_OVERRIDE		= 0x3
};

enum css_vertical_align_e {
	CSS_VERTICAL_ALIGN_INHERIT		= 0x0,
	CSS_VERTICAL_ALIGN_BASELINE		= 0x1,
	CSS_VERTICAL_ALIGN_SUB			= 0x2,
	CSS_VERTICAL_ALIGN_SUPER		= 0x3,
	CSS_VERTICAL_ALIGN_TOP			= 0x4,
	CSS_VERTICAL_ALIGN_TEXT_TOP		= 0x5,
	CSS_VERTICAL_ALIGN_MIDDLE		= 0x6,
	CSS_VERTICAL_ALIGN_BOTTOM		= 0x7,
	CSS_VERTICAL_ALIGN_TEXT_BOTTOM		= 0x8,
	CSS_VERTICAL_ALIGN_SET			= 0x9
};

enum css_visibility_e {
	CSS_VISIBILITY_INHERIT			= 0x0,
	CSS_VISIBILITY_VISIBLE			= 0x1,
	CSS_VISIBILITY_HIDDEN			= 0x2,
	CSS_VISIBILITY_COLLAPSE			= 0x3
};

enum css_white_space_e {
	CSS_WHITE_SPACE_INHERIT			= 0x0,
	CSS_WHITE_SPACE_NORMAL			= 0x1,
	CSS_WHITE_SPACE_PRE			= 0x2,
	CSS_WHITE_SPACE_NOWRAP			= 0x3,
	CSS_WHITE_SPACE_PRE_WRAP		= 0x4,
	CSS_WHITE_SPACE_PRE_LINE		= 0x5
};

enum css_widows_e {
	CSS_WIDOWS_INHERIT			= 0x0,
	CSS_WIDOWS_SET				= 0x1
};

enum css_width_e {
	CSS_WIDTH_INHERIT			= 0x0,
	CSS_WIDTH_SET				= 0x1,
	CSS_WIDTH_AUTO				= 0x2
};

enum css_word_spacing_e {
	CSS_WORD_SPACING_INHERIT		= CSS_COLUMN_GAP_INHERIT,
	CSS_WORD_SPACING_SET			= CSS_COLUMN_GAP_SET,
	CSS_WORD_SPACING_NORMAL			= CSS_COLUMN_GAP_NORMAL
};

enum css_writing_mode_e {
	CSS_WRITING_MODE_INHERIT		= 0x0,
	CSS_WRITING_MODE_HORIZONTAL_TB		= 0x1,
	CSS_WRITING_MODE_VERTICAL_RL		= 0x2,
	CSS_WRITING_MODE_VERTICAL_LR		= 0x3
};

enum css_z_index_e {
	CSS_Z_INDEX_INHERIT			= 0x0,
	CSS_Z_INDEX_SET				= 0x1,
	CSS_Z_INDEX_AUTO			= 0x2
};
]]

ffi.cdef[[
// C/ui/libcss/libparserutils/include/parserutils/utils/vector.h
struct parserutils_vector;
typedef struct parserutils_vector parserutils_vector;

]]

ffi.cdef[[
// /C/ui/libcss/libparserutils/include/parserutils/utils/buffer.h
struct parserutils_buffer
{
	uint8_t *data;
	size_t length;
	size_t allocated;
};
typedef struct parserutils_buffer parserutils_buffer;
]]

ffi.cdef[[
// C/ui/libcss/libcss/src/libparserutils/include/parserutils/input/inputstream.h
/**
 * Input stream object
 */
typedef struct parserutils_inputstream
{
	parserutils_buffer *utf8;	/**< Buffer containing UTF-8 data */

	uint32_t cursor;		/**< Byte offset of current position */

	bool had_eof;			/**< Whether EOF has been reached */
} parserutils_inputstream;

]]

ffi.cdef[[
// C/ui/libcss/libcss/src/bytecode/bytecode.h
typedef uint32_t css_code_t;
]]

ffi.cdef[[
// C/ui/libcss/libcss/src/select/bloom.h
typedef uint32_t css_bloom;
]]

ffi.cdef[[
// C/ui/libcss/libcss/src/bytecode/bytecode.h
typedef uint32_t css_code_t;
typedef enum css_properties_e opcode_t;

]]

ffi.cdef[[
// C/ui/libcss/libcss/src/lex/lex.h

typedef struct css_lexer css_lexer;
/**
 * Lexer option types
 */
typedef enum css_lexer_opttype {
	CSS_LEXER_EMIT_COMMENTS
} css_lexer_opttype;

/**
 * Lexer option parameters
 */
typedef union css_lexer_optparams {
	bool emit_comments;
} css_lexer_optparams;

/**
 * Token type
 */
typedef enum css_token_type {
	CSS_TOKEN_IDENT, CSS_TOKEN_ATKEYWORD, CSS_TOKEN_HASH,
	CSS_TOKEN_FUNCTION, CSS_TOKEN_STRING, CSS_TOKEN_INVALID_STRING,
	CSS_TOKEN_URI, CSS_TOKEN_UNICODE_RANGE, CSS_TOKEN_CHAR,
	CSS_TOKEN_NUMBER, CSS_TOKEN_PERCENTAGE, CSS_TOKEN_DIMENSION,

	/* Those tokens that want strings interned appear above */
	CSS_TOKEN_LAST_INTERN,

	CSS_TOKEN_CDO, CSS_TOKEN_CDC, CSS_TOKEN_S, CSS_TOKEN_COMMENT,
	CSS_TOKEN_INCLUDES, CSS_TOKEN_DASHMATCH, CSS_TOKEN_PREFIXMATCH,
	CSS_TOKEN_SUFFIXMATCH, CSS_TOKEN_SUBSTRINGMATCH, CSS_TOKEN_EOF
} css_token_type;

/**
 * Token object
 */
typedef struct css_token {
	css_token_type type;

	struct {
		uint8_t *data;
		size_t len;
	} data;

	lwc_string *idata;

	uint32_t col;
	uint32_t line;
} css_token;

css_error css__lexer_create(parserutils_inputstream *input, css_lexer **lexer);
css_error css__lexer_destroy(css_lexer *lexer);

css_error css__lexer_setopt(css_lexer *lexer, css_lexer_opttype type,
		css_lexer_optparams *params);

css_error css__lexer_get_token(css_lexer *lexer, css_token **token);
]]

ffi.cdef[[
// C/ui/libcss/libcss/src/lex/lex.c

/**
 * CSS lexer object
 */
struct css_lexer
{
	parserutils_inputstream *input;	/**< Inputstream containing CSS */

	size_t bytesReadForToken;	/**< Total bytes read from the
					 * inputstream for the current token */

	css_token token;		/**< The current token */

	bool escapeSeen;		/**< Whether an escape sequence has
					 * been seen while processing the input
					 * for the current token */
	parserutils_buffer *unescapedTokenData;	/**< Buffer containing
						 * unescaped token data
						 * (used iff escapeSeen == true)
						 */

	unsigned int state    : 4,	/**< Current state */
		     substate : 4;	/**< Current substate */

	struct {
		uint8_t first;		/**< First character read for token */
		size_t origBytes;	/**< Storage of current number of
					 * bytes read, for rewinding */
		bool lastWasStar;	/**< Whether the previous character
					 * was an asterisk */
		bool lastWasCR;		/**< Whether the previous character
					 * was CR */
		size_t bytesForURL;	/**< Input bytes read for "url(", for
					 * rewinding */
		size_t dataLenForURL;	/**< Output length for "url(", for
					 * rewinding */
		int hexCount;		/**< Counter for reading hex digits */
	} context;			/**< Context for the current state */

	bool emit_comments;		/**< Whether to emit comment tokens */

	uint32_t currentCol;		/**< Current column in source */
	uint32_t currentLine;		/**< Current line in source */
};
]]

ffi.cdef[[
// /Users/pasi/installed/C/ui/libcss/libparserutils/include/parserutils/utils/stack.h

typedef struct parserutils_stack parserutils_stack;
///Users/pasi/installed/C/ui/libcss/libparserutils/src/utils/stack.c
struct parserutils_stack
{
	size_t item_size;		/**< Size of an item in the stack */
	size_t chunk_size;		/**< Size of a stack chunk */
	size_t items_allocated;		/**< Number of slots allocated */
	int32_t current_item;		/**< Index of current item */
	void *items;			/**< Items in stack */
};
]]


ffi.cdef[[
// C/ui/libcss/libcss/src/parse/parse.c
typedef struct css_parser css_parser;

struct css_parser
{
	parserutils_inputstream *stream;	/**< The inputstream */
	css_lexer *lexer;		/**< The lexer to use */

	bool quirks;			/**< Whether to enable parsing quirks */

	parserutils_stack *states;	/**< Stack of states */

	parserutils_vector *tokens;	/**< Vector of pending tokens */

	const css_token *pushback;	/**< Push back buffer */

	bool parseError;		/**< A parse error has occurred */
	parserutils_stack *open_items;	/**< Stack of open brackets */

	bool last_was_ws;		/**< Last token was whitespace */

	// css_parser_event_handler event;	/**< Client's event handler */
	void *event_pw;			/**< Client data for event handler */
};
]]

ffi.cdef[[
// C/ui/libcss/libcss/src/parse/parse.h

// typedef struct css_parser css_parser;

/**
 * Parser event types
 */
typedef enum css_parser_event {
	CSS_PARSER_START_STYLESHEET,
	CSS_PARSER_END_STYLESHEET,
	CSS_PARSER_START_RULESET,
	CSS_PARSER_END_RULESET,
	CSS_PARSER_START_ATRULE,
	CSS_PARSER_END_ATRULE,
	CSS_PARSER_START_BLOCK,
	CSS_PARSER_END_BLOCK,
	CSS_PARSER_BLOCK_CONTENT,
	CSS_PARSER_END_BLOCK_CONTENT,
	CSS_PARSER_DECLARATION
} css_parser_event;

typedef css_error (*css_parser_event_handler)(css_parser_event type,
		const parserutils_vector *tokens, void *pw);

/**
 * Parser option types
 */
typedef enum css_parser_opttype {
	CSS_PARSER_QUIRKS,
	CSS_PARSER_EVENT_HANDLER
} css_parser_opttype;

/**
 * Parser option parameters
 */
typedef union css_parser_optparams {
	bool quirks;

	struct {
		css_parser_event_handler handler;
		void *pw;
	} event_handler;
} css_parser_optparams;

css_error css__parser_create(const char *charset, css_charset_source cs_source,
		css_parser **parser);
css_error css__parser_create_for_inline_style(const char *charset,
		css_charset_source cs_source, css_parser **parser);
css_error css__parser_create_for_media_query(const char *charset,
		css_charset_source cs_source, css_parser **parser);
css_error css__parser_destroy(css_parser *parser);

css_error css__parser_setopt(css_parser *parser, css_parser_opttype type,
		css_parser_optparams *params);

css_error css__parser_parse_chunk(css_parser *parser, const uint8_t *data,
		size_t len);
css_error css__parser_completed(css_parser *parser);

const char *css__parser_read_charset(css_parser *parser,
		css_charset_source *source);
bool css__parser_quirks_permitted(css_parser *parser);
]]

ffi.cdef[[
// /C/ui/libcss/libcss/include/libcss/stylesheet.h

/**
 * Callback to resolve an URL
 *
 * \param pw    Client data
 * \param dict  String internment context
 * \param base  Base URI (absolute)
 * \param rel   URL to resolve, either absolute or relative to base
 * \param abs   Pointer to location to receive result
 * \return CSS_OK on success, appropriate error otherwise.
 */
typedef css_error (*css_url_resolution_fn)(void *pw,
		const char *base, lwc_string *rel, lwc_string **abs);

/**
 * Callback to be notified of the need for an imported stylesheet
 *
 * \param pw      Client data
 * \param parent  Stylesheet requesting the import
 * \param url     URL of the imported sheet
 * \return CSS_OK on success, appropriate error otherwise
 *
 * \note This function will be invoked for notification purposes
 *       only. The client may use this to trigger a parallel fetch
 *       of the imported stylesheet. The imported sheet must be
 *       registered with its parent using the post-parse import
 *       registration API.
 */
typedef css_error (*css_import_notification_fn)(void *pw,
		css_stylesheet *parent, lwc_string *url);

/**
 * Callback use to resolve system colour names to RGB values
 *
 * \param pw     Client data
 * \param name   System colour name
 * \param color  Pointer to location to receive color value
 * \return CSS_OK       on success,
 *         CSS_INVALID  if the name is unknown.
 */
typedef css_error (*css_color_resolution_fn)(void *pw,
		lwc_string *name, css_color *color);

/** System font callback result data. */
typedef struct css_system_font {
	enum css_font_style_e style;
	enum css_font_variant_e variant;
	enum css_font_weight_e weight;
	struct {
		css_fixed size;
		css_unit unit;
	} size;
	struct {
		css_fixed size;
		css_unit unit;
	} line_height;
	/* Note: must be a single family name only */
	lwc_string *family;
} css_system_font;

/**
 * Callback use to resolve system font names to font values
 *
 * \param pw           Client data
 * \param name         System font identifier
 * \param system_font  Pointer to system font descriptor to be filled
 * \return CSS_OK       on success,
 *         CSS_INVALID  if the name is unknown.
 */
typedef css_error (*css_font_resolution_fn)(void *pw,
		lwc_string *name, css_system_font *system_font);

typedef enum css_stylesheet_params_version {
	CSS_STYLESHEET_PARAMS_VERSION_1 = 1
} css_stylesheet_params_version;

/**
 * Parameter block for css_stylesheet_create()
 */
typedef struct css_stylesheet_params {
	/** ABI version of this structure */
	uint32_t params_version;

	/** The language level of the stylesheet */
	css_language_level level;

	/** The charset of the stylesheet data, or NULL to detect */
	const char *charset;
	/** URL of stylesheet */
	const char *url;
	/** Title of stylesheet */
	const char *title;

	/** Permit quirky parsing of stylesheet */
	bool allow_quirks;
	/** This stylesheet is an inline style */
	bool inline_style;

	/** URL resolution function */
	css_url_resolution_fn resolve;
	/** Client private data for resolve */
	void *resolve_pw;

	/** Import notification function */
	css_import_notification_fn import;
	/** Client private data for import */
	void *import_pw;

	/** Colour resolution function */
	css_color_resolution_fn color;
	/** Client private data for color */
	void *color_pw;

	/** Font resolution function */
	css_font_resolution_fn font;
	/** Client private data for font */
	void *font_pw;
} css_stylesheet_params;

css_error css_stylesheet_create(const css_stylesheet_params *params,
		css_stylesheet **stylesheet);
css_error css_stylesheet_destroy(css_stylesheet *sheet);

css_error css_stylesheet_append_data(css_stylesheet *sheet,
		const uint8_t *data, size_t len);
css_error css_stylesheet_data_done(css_stylesheet *sheet);

css_error css_stylesheet_next_pending_import(css_stylesheet *parent,
		lwc_string **url);
css_error css_stylesheet_register_import(css_stylesheet *parent,
		css_stylesheet *child);

css_error css_stylesheet_get_language_level(css_stylesheet *sheet,
		css_language_level *level);
css_error css_stylesheet_get_url(css_stylesheet *sheet, const char **url);
css_error css_stylesheet_get_title(css_stylesheet *sheet, const char **title);
css_error css_stylesheet_quirks_allowed(css_stylesheet *sheet, bool *allowed);
css_error css_stylesheet_used_quirks(css_stylesheet *sheet, bool *quirks);

css_error css_stylesheet_get_disabled(css_stylesheet *sheet, bool *disabled);
css_error css_stylesheet_set_disabled(css_stylesheet *sheet, bool disabled);

css_error css_stylesheet_size(css_stylesheet *sheet, size_t *size);
]]

ffi.cdef[[
// C/ui/libcss/libcss/src/parse/language.h

/**
 * CSS namespace mapping
 */
typedef struct css_namespace {
	lwc_string *prefix;		/**< Namespace prefix */
	lwc_string *uri;		/**< Namespace URI */
} css_namespace;

/**
 * Context for a CSS language parser
 */
typedef struct css_language {
	css_stylesheet *sheet;		/**< The stylesheet to parse for */

	parserutils_stack *context;	/**< Context stack */

	enum {
		CHARSET_PERMITTED,
		IMPORT_PERMITTED,
		NAMESPACE_PERMITTED,
		HAD_RULE
	} state;			/**< State flag, for at-rule handling */

	/** Interned strings */
	lwc_string **strings;

	lwc_string *default_namespace;	/**< Default namespace URI */
	css_namespace *namespaces;	/**< Array of namespace mappings */
	uint32_t num_namespaces;	/**< Number of namespace mappings */
} css_language;

css_error css__language_create(css_stylesheet *sheet, css_parser *parser,
		void **language);
css_error css__language_destroy(css_language *language);
]]

ffi.cdef[[
// C/ui/libcss/libcss/src/select/hash.h

struct css_selector;

typedef struct css_selector_hash css_selector_hash;

struct css_hash_selection_requirments {
	css_qname qname;		/* Element name, or universal "*" */
	lwc_string *class;		/* Name of class, or NULL */
	lwc_string *id;			/* Name of id, or NULL */
	lwc_string *uni;		/* Universal element string "*" */
	const css_media *media;		/* Media spec we're selecting for */
	const css_bloom *node_bloom;	/* Node's bloom filter */
};

typedef css_error (*css_selector_hash_iterator)(
		const struct css_hash_selection_requirments *req,
		const struct css_selector **current,
		const struct css_selector ***next);

css_error css__selector_hash_create(css_selector_hash **hash);
css_error css__selector_hash_destroy(css_selector_hash *hash);

css_error css__selector_hash_insert(css_selector_hash *hash,
		const struct css_selector *selector);
css_error css__selector_hash_remove(css_selector_hash *hash,
		const struct css_selector *selector);

css_error css__selector_hash_find(css_selector_hash *hash,
		const struct css_hash_selection_requirments *req,
		css_selector_hash_iterator *iterator,
		const struct css_selector ***matched);
css_error css__selector_hash_find_by_class(css_selector_hash *hash,
		const struct css_hash_selection_requirments *req,
		css_selector_hash_iterator *iterator,
		const struct css_selector ***matched);
css_error css__selector_hash_find_by_id(css_selector_hash *hash,
		const struct css_hash_selection_requirments *req,
		css_selector_hash_iterator *iterator,
		const struct css_selector ***matched);
css_error css__selector_hash_find_universal(css_selector_hash *hash,
		const struct css_hash_selection_requirments *req,
		css_selector_hash_iterator *iterator,
		const struct css_selector ***matched);

css_error css__selector_hash_size(css_selector_hash *hash, size_t *size);
]]

ffi.cdef[[
	//C/ui/libcss/libcss/src/parse/mq.h

typedef struct {
	enum {
		CSS_MQ_VALUE_TYPE_NUM,
		CSS_MQ_VALUE_TYPE_DIM,
		CSS_MQ_VALUE_TYPE_IDENT,
		CSS_MQ_VALUE_TYPE_RATIO
	} type;
	union {
		css_fixed num_or_ratio; /* Where ratio is the result of a/b */
		struct {
			css_fixed len;
			uint32_t unit;
		} dim;
		lwc_string *ident;
	} data;
} css_mq_value;

/*
 * "name : value" is encoded as "name = value"
 * "name" is encoded by setting the operator to "bool"
 * "value op name" is encoded verbatim (with op2 set to "unused")
 * "name op value" inverts the operator to encode (i.e < becomes >=) (and sets op2 to "unused")
 * "value op name op value" is encoded using op2 and value2
 */
typedef enum {
	CSS_MQ_FEATURE_OP_BOOL, /* op only */
	CSS_MQ_FEATURE_OP_UNUSED = CSS_MQ_FEATURE_OP_BOOL, /* op2 only */

	CSS_MQ_FEATURE_OP_LT,
	CSS_MQ_FEATURE_OP_LTE,
	CSS_MQ_FEATURE_OP_EQ, /* op only */
	CSS_MQ_FEATURE_OP_GTE,
	CSS_MQ_FEATURE_OP_GT
} css_mq_feature_op;

typedef struct {
	lwc_string *name;
	css_mq_feature_op op;
	css_mq_feature_op op2;
	css_mq_value value;
	css_mq_value value2;
} css_mq_feature;

typedef struct css_mq_cond_or_feature css_mq_cond_or_feature;

typedef struct {
	uint32_t negate : 1, /* set if "not" */
		 op     : 1; /* clear if "and", set if "or" */
	uint32_t nparts;
	css_mq_cond_or_feature **parts;
} css_mq_cond;

struct css_mq_cond_or_feature {
	enum {
		CSS_MQ_FEATURE,
		CSS_MQ_COND
	} type;
	union {
		css_mq_cond *cond;
		css_mq_feature *feat;
	} data;
};

typedef struct css_mq_query {
	struct css_mq_query *next;

	uint32_t negate_type : 1; /* set if "not type" */
	uint64_t type; /* or NULL */

	css_mq_cond *cond;
} css_mq_query;

css_error css_parse_media_query(lwc_string **strings,
		const uint8_t *mq, size_t len,
		css_mq_query **media_out);

css_error css__mq_parse_media_list(lwc_string **strings,
		const parserutils_vector *vector, int *ctx,
		css_mq_query **media);

void css__mq_query_destroy(css_mq_query *media);
]]


ffi.cdef[[
// C/ui/libcss/libcss/src/stylesheet.h

typedef struct css_rule css_rule;
typedef struct css_selector css_selector;

typedef struct css_style {
	css_code_t *bytecode;	      /**< Pointer to bytecode */
	uint32_t used;		      /**< number of code entries used */
	uint32_t allocated;	      /**< number of allocated code entries */
	struct css_stylesheet *sheet; /**< containing sheet */
} css_style;

typedef enum css_selector_type {
	CSS_SELECTOR_ELEMENT,
	CSS_SELECTOR_CLASS,
	CSS_SELECTOR_ID,
	CSS_SELECTOR_PSEUDO_CLASS,
	CSS_SELECTOR_PSEUDO_ELEMENT,
	CSS_SELECTOR_ATTRIBUTE,
	CSS_SELECTOR_ATTRIBUTE_EQUAL,
	CSS_SELECTOR_ATTRIBUTE_DASHMATCH,
	CSS_SELECTOR_ATTRIBUTE_INCLUDES,
	CSS_SELECTOR_ATTRIBUTE_PREFIX,
	CSS_SELECTOR_ATTRIBUTE_SUFFIX,
	CSS_SELECTOR_ATTRIBUTE_SUBSTRING
} css_selector_type;

typedef enum css_combinator {
	CSS_COMBINATOR_NONE,
	CSS_COMBINATOR_ANCESTOR,
	CSS_COMBINATOR_PARENT,
	CSS_COMBINATOR_SIBLING,
	CSS_COMBINATOR_GENERIC_SIBLING
} css_combinator;

typedef enum css_selector_detail_value_type {
	CSS_SELECTOR_DETAIL_VALUE_STRING,
	CSS_SELECTOR_DETAIL_VALUE_NTH
} css_selector_detail_value_type;

typedef union css_selector_detail_value {
	lwc_string *string;		/**< Interned string, or NULL */
	struct {
		int32_t a;
		int32_t b;
	} nth;				/**< Data for x = an + b */
} css_selector_detail_value;

typedef struct css_selector_detail {
	css_qname qname;			/**< Interned name */
	css_selector_detail_value value;	/**< Detail value */

	unsigned int type       : 4,		/**< Type of selector */
		     comb       : 3,		/**< Type of combinator */
		     next       : 1,		/**< Another selector detail
						 * follows */
		     value_type : 1,		/**< Type of value field */
		     negate     : 1;		/**< Detail match is inverted */
} css_selector_detail;

struct css_selector {
	css_selector *combinator;		/**< Combining selector */

	css_rule *rule;				/**< Owning rule */

// #define CSS_SPECIFICITY_A 0x01000000
// #define CSS_SPECIFICITY_B 0x00010000
// #define CSS_SPECIFICITY_C 0x00000100
// #define CSS_SPECIFICITY_D 0x00000001
	uint32_t specificity;			/**< Specificity of selector */

	css_selector_detail data;		/**< Selector data */
};

typedef enum css_rule_type {
	CSS_RULE_UNKNOWN,
	CSS_RULE_SELECTOR,
	CSS_RULE_CHARSET,
	CSS_RULE_IMPORT,
	CSS_RULE_MEDIA,
	CSS_RULE_FONT_FACE,
	CSS_RULE_PAGE
} css_rule_type;

typedef enum css_rule_parent_type {
	CSS_RULE_PARENT_STYLESHEET,
	CSS_RULE_PARENT_RULE
} css_rule_parent_type;

struct css_rule {
	void *parent;				/**< containing rule or owning
						 * stylesheet (defined by ptype)
						 */
	css_rule *next;				/**< next in list */
	css_rule *prev;				/**< previous in list */

	uint32_t index; /**< index in sheet */
	uint16_t items; /**< number of items (selectors) in rule */
	uint8_t type;   /**< css_rule_type */
	uint8_t ptype;  /**< css_rule_parent_type */
} _ALIGNED;

typedef struct css_rule_selector {
	css_rule base;

	css_selector **selectors;
	css_style *style;
} css_rule_selector;

typedef struct css_rule_media {
	css_rule base;

	css_mq_query *media;

	css_rule *first_child;
	css_rule *last_child;
} css_rule_media;

typedef struct css_rule_font_face {
	css_rule base;

	css_font_face *font_face;
} css_rule_font_face;

typedef struct css_rule_page {
	css_rule base;

	css_selector *selector;
	css_style *style;
} css_rule_page;

typedef struct css_rule_import {
	css_rule base;

	lwc_string *url;
	css_mq_query *media;

	css_stylesheet *sheet;
} css_rule_import;

typedef struct css_rule_charset {
	css_rule base;

	lwc_string *encoding;	/** \todo use MIB enum? */
} css_rule_charset;

struct css_stylesheet {
	css_selector_hash *selectors;		/**< Hashtable of selectors */

	uint32_t rule_count;			/**< Number of rules in sheet */
	css_rule *rule_list;			/**< List of rules in sheet */
	css_rule *last_rule;			/**< Last rule in list */

	bool disabled;				/**< Whether this sheet is
						 * disabled */

	char *url;				/**< URL of this sheet */
	char *title;				/**< Title of this sheet */

	css_language_level level;		/**< Language level of sheet */
	css_parser *parser;			/**< Core parser for sheet */
	void *parser_frontend;			/**< Frontend parser */
	lwc_string **propstrings;		/**< Property strings, for parser */

	bool quirks_allowed;			/**< Quirks permitted */
	bool quirks_used;			/**< Quirks actually used */

	bool inline_style;			/**< Is an inline style */

	size_t size;				/**< Size, in bytes */

	css_import_notification_fn import;	/**< Import notification function */
	void *import_pw;			/**< Private word */

	css_url_resolution_fn resolve;		/**< URL resolution function */
	void *resolve_pw;			/**< Private word */

	css_color_resolution_fn color;		/**< Colour resolution function */
	void *color_pw;				/**< Private word */

	/** System font resolution function */
	css_font_resolution_fn font;
	void *font_pw;				/**< Private word */

	css_style *cached_style;		/**< Cache for style parsing */

	lwc_string **string_vector;             /**< Bytecode string vector */
	uint32_t string_vector_l;               /**< The string vector allocated
						 * length in entries */
	uint32_t string_vector_c;               /**< The number of string
						 * vector entries used */
};

css_error css__stylesheet_style_create(css_stylesheet *sheet,
		css_style **style);
css_error css__stylesheet_style_append(css_style *style, css_code_t code);
css_error css__stylesheet_style_vappend(css_style *style, uint32_t style_count, ...);
css_error css__stylesheet_style_destroy(css_style *style);
css_error css__stylesheet_merge_style(css_style *target, css_style *style);

/** Helper function to avoid distinct buildOPV call */
css_error css__stylesheet_style_appendOPV(css_style *style,
		opcode_t opcode, uint8_t flags, uint16_t value);

/** Helper function to set inherit flag */
css_error css_stylesheet_style_inherit(css_style *style,
		opcode_t opcode);

css_error css__stylesheet_selector_create(css_stylesheet *sheet,
		css_qname *qname, css_selector **selector);
css_error css__stylesheet_selector_destroy(css_stylesheet *sheet,
		css_selector *selector);

css_error css__stylesheet_selector_detail_init(css_stylesheet *sheet,
		css_selector_type type, css_qname *qname,
		css_selector_detail_value value,
		css_selector_detail_value_type value_type,
		bool negate, css_selector_detail *detail);

css_error css__stylesheet_selector_append_specific(css_stylesheet *sheet,
		css_selector **parent, const css_selector_detail *specific);

css_error css__stylesheet_selector_combine(css_stylesheet *sheet,
		css_combinator type, css_selector *a, css_selector *b);

css_error css__stylesheet_rule_create(css_stylesheet *sheet, css_rule_type type,
		css_rule **rule);
css_error css__stylesheet_rule_destroy(css_stylesheet *sheet, css_rule *rule);

css_error css__stylesheet_rule_add_selector(css_stylesheet *sheet,
		css_rule *rule, css_selector *selector);

css_error css__stylesheet_rule_append_style(css_stylesheet *sheet,
		css_rule *rule, css_style *style);

css_error css__stylesheet_rule_set_charset(css_stylesheet *sheet,
		css_rule *rule, lwc_string *charset);

css_error css__stylesheet_rule_set_nascent_import(css_stylesheet *sheet,
		css_rule *rule, lwc_string *url, css_mq_query *media);

css_error css__stylesheet_rule_set_media(css_stylesheet *sheet,
		css_rule *rule, css_mq_query *media);

css_error css__stylesheet_rule_set_page_selector(css_stylesheet *sheet,
		css_rule *rule, css_selector *sel);

css_error css__stylesheet_add_rule(css_stylesheet *sheet, css_rule *rule,
		css_rule *parent);
css_error css__stylesheet_remove_rule(css_stylesheet *sheet, css_rule *rule);

css_error css__stylesheet_string_get(css_stylesheet *sheet,
		uint32_t string_number, lwc_string **string);

css_error css__stylesheet_string_add(css_stylesheet *sheet,
		lwc_string *string, uint32_t *string_number);
]]

ffi.cdef[[
// C/ui/libcss/libcss/src/parse/font_face.h
css_error css__parse_font_descriptor(css_language *c,
		const css_token *descriptor, const parserutils_vector *vector,
		int *ctx, struct css_rule_font_face *rule);
]]

ffi.cdef[[
// C/ui/libcss/libcss/src/parse/important.h
css_error css__parse_important(css_language *c,
		const parserutils_vector *vector, int *ctx,
		uint8_t *result);

void css__make_style_important(css_style *style);
]]

ffi.cdef[[
// C/ui/libcss/libcss/src/parse/propstrings.h
enum {
	/* Universal selector */
	UNIVERSAL,

	/* Common tokens */
	COLON, COMMA, SEMICOLON, OPEN_CURLY_BRACE, CLOSE_CURLY_BRACE,
	ZERO_VALUE,

	/* At-rules */
	CHARSET, LIBCSS_IMPORT, MEDIA, NAMESPACE, FONT_FACE, PAGE,

	/* Media types */
	AURAL, BRAILLE, EMBOSSED, HANDHELD, PRINT, PROJECTION,
	SCREEN, SPEECH, TTY, TV, ALL,

	/* Pseudo classes */
	FIRST_CHILD, LINK, VISITED, HOVER, ACTIVE, FOCUS, LANG,
	/* LEFT, RIGHT, -- already in properties */ FIRST,
	ROOT, NTH_CHILD, NTH_LAST_CHILD, NTH_OF_TYPE, NTH_LAST_OF_TYPE,
	LAST_CHILD, FIRST_OF_TYPE, LAST_OF_TYPE, ONLY_CHILD,
	ONLY_OF_TYPE, EMPTY, TARGET, ENABLED, DISABLED, CHECKED, NOT,

	/* Pseudo elements */
	FIRST_LINE, FIRST_LETTER, BEFORE, AFTER,

	/* Properties */
	FIRST_PROP,

	ALIGN_CONTENT = FIRST_PROP, ALIGN_ITEMS, ALIGN_SELF, AZIMUTH,
	BACKGROUND, BACKGROUND_ATTACHMENT, BACKGROUND_COLOR, BACKGROUND_IMAGE,
	BACKGROUND_POSITION, BACKGROUND_REPEAT, BORDER, BORDER_BOTTOM,
	BORDER_BOTTOM_COLOR, BORDER_BOTTOM_STYLE, BORDER_BOTTOM_WIDTH,
	BORDER_COLLAPSE, BORDER_COLOR, BORDER_LEFT, BORDER_LEFT_COLOR,
	BORDER_LEFT_STYLE, BORDER_LEFT_WIDTH, BORDER_RIGHT, BORDER_RIGHT_COLOR,
	BORDER_RIGHT_STYLE, BORDER_RIGHT_WIDTH, BORDER_SPACING,
	BORDER_STYLE, BORDER_TOP, BORDER_TOP_COLOR, BORDER_TOP_STYLE,
	BORDER_TOP_WIDTH, BORDER_WIDTH, BORDER_RADIUS, BORDER_TOP_LEFT_RADIUS,
	BORDER_TOP_RIGHT_RADIUS, BORDER_BOTTOM_RIGHT_RADIUS, BORDER_BOTTOM_LEFT_RADIUS,
	BOTTOM, BOX_SIZING, BREAK_AFTER, BREAK_BEFORE, BREAK_INSIDE, CAPTION_SIDE, CLEAR,
	CLIP, COLOR, COLUMNS, COLUMN_COUNT, COLUMN_FILL, COLUMN_GAP, COLUMN_RULE,
	COLUMN_RULE_COLOR, COLUMN_RULE_STYLE, COLUMN_RULE_WIDTH, COLUMN_SPAN, COLUMN_WIDTH,
	CONTENT, COUNTER_INCREMENT, COUNTER_RESET, CUE, CUE_AFTER, CUE_BEFORE,
	CURSOR, DIRECTION, DISPLAY, ELEVATION, EMPTY_CELLS, FLEX, FLEX_BASIS,
	FLEX_DIRECTION, FLEX_FLOW, FLEX_GROW, FLEX_SHRINK, FLEX_WRAP,
	LIBCSS_FLOAT, FONT, FONT_FAMILY, FONT_SIZE, FONT_STYLE, FONT_VARIANT,
	FONT_WEIGHT, HEIGHT, JUSTIFY_CONTENT, LEFT, LETTER_SPACING, LINE_HEIGHT,
	LIST_STYLE, LIST_STYLE_IMAGE, LIST_STYLE_POSITION, LIST_STYLE_TYPE,
	MARGIN, MARGIN_BOTTOM, MARGIN_LEFT, MARGIN_RIGHT, MARGIN_TOP,
	MAX_HEIGHT, MAX_WIDTH, MIN_HEIGHT, MIN_WIDTH, OPACITY, ORDER, ORPHANS,
	OUTLINE, OUTLINE_COLOR, OUTLINE_STYLE, OUTLINE_WIDTH, OVERFLOW,
	OVERFLOW_X, OVERFLOW_Y, PADDING, PADDING_BOTTOM, PADDING_LEFT,
	PADDING_RIGHT, PADDING_TOP, PAGE_BREAK_AFTER, PAGE_BREAK_BEFORE,
	PAGE_BREAK_INSIDE, PAUSE, PAUSE_AFTER, PAUSE_BEFORE, PITCH_RANGE, PITCH,
	PLAY_DURING, POSITION, QUOTES, RICHNESS, RIGHT, SPEAK_HEADER,
	SPEAK_NUMERAL, SPEAK_PUNCTUATION, SPEAK, SPEECH_RATE, STRESS,
	TABLE_LAYOUT, TEXT_ALIGN, TEXT_DECORATION, TEXT_INDENT, TEXT_TRANSFORM,
	TOP, UNICODE_BIDI, VERTICAL_ALIGN, VISIBILITY, VOICE_FAMILY, VOLUME,
	WHITE_SPACE, WIDOWS, WIDTH, WORD_SPACING, WRITING_MODE, Z_INDEX,

	LAST_PROP = Z_INDEX,

	/* Other keywords */
	INHERIT, IMPORTANT, NONE, BOTH, FIXED, SCROLL, TRANSPARENT,
	NO_REPEAT, REPEAT_X, REPEAT_Y, REPEAT, HIDDEN, DOTTED, DASHED,
	SOLID, LIBCSS_DOUBLE, GROOVE, RIDGE, INSET, OUTSET, THIN, MEDIUM, THICK,
	COLLAPSE, SEPARATE, AUTO, LTR, RTL, INLINE, BLOCK, LIST_ITEM, RUN_IN,
	INLINE_BLOCK, TABLE, INLINE_TABLE, TABLE_ROW_GROUP, TABLE_HEADER_GROUP,
	TABLE_FOOTER_GROUP, TABLE_ROW, TABLE_COLUMN_GROUP, TABLE_COLUMN,
	TABLE_CELL, TABLE_CAPTION, BELOW, LEVEL, ABOVE, HIGHER, LOWER,
	SHOW, HIDE, XX_SMALL, X_SMALL, SMALL, LARGE, X_LARGE, XX_LARGE,
	LARGER, SMALLER, NORMAL, ITALIC, OBLIQUE, SMALL_CAPS, BOLD, BOLDER,
	LIGHTER, INSIDE, OUTSIDE, DISC, CIRCLE, SQUARE, DECIMAL,
	DECIMAL_LEADING_ZERO, LOWER_ROMAN, UPPER_ROMAN, LOWER_GREEK,
	LOWER_LATIN, UPPER_LATIN, ARMENIAN, GEORGIAN, LOWER_ALPHA, UPPER_ALPHA,
	INVERT, VISIBLE, ALWAYS, AVOID, X_LOW, LOW, HIGH, X_HIGH, LIBCSS_STATIC,
	RELATIVE, ABSOLUTE, ONCE, DIGITS, CONTINUOUS, CODE, SPELL_OUT, X_SLOW,
	SLOW, FAST, X_FAST, FASTER, SLOWER, CENTER, JUSTIFY, CAPITALIZE,
	UPPERCASE, LOWERCASE, EMBED, BIDI_OVERRIDE, BASELINE, SUB, SUPER,
	TEXT_TOP, MIDDLE, TEXT_BOTTOM, SILENT, X_SOFT, SOFT, LOUD, X_LOUD,
	PRE, NOWRAP, PRE_WRAP, PRE_LINE, LEFTWARDS, RIGHTWARDS, LEFT_SIDE,
	FAR_LEFT, CENTER_LEFT, CENTER_RIGHT, FAR_RIGHT, RIGHT_SIDE, BEHIND,
	RECT, OPEN_QUOTE, CLOSE_QUOTE, NO_OPEN_QUOTE, NO_CLOSE_QUOTE, ATTR,
	COUNTER, COUNTERS, CROSSHAIR, DEFAULT, POINTER, MOVE, E_RESIZE,
	NE_RESIZE, NW_RESIZE, N_RESIZE, SE_RESIZE, SW_RESIZE, S_RESIZE,
	W_RESIZE, LIBCSS_TEXT, WAIT, HELP, PROGRESS, SERIF, SANS_SERIF, CURSIVE,
	FANTASY, MONOSPACE, MALE, FEMALE, CHILD, MIX, UNDERLINE, OVERLINE,
	LINE_THROUGH, BLINK, RGB, RGBA, HSL, HSLA, LIBCSS_LEFT, LIBCSS_CENTER,
	LIBCSS_RIGHT, CURRENTCOLOR, ODD, EVEN, SRC, LOCAL, INITIAL,
	FORMAT, WOFF, TRUETYPE, OPENTYPE, EMBEDDED_OPENTYPE, SVG, COLUMN,
	AVOID_PAGE, AVOID_COLUMN, BALANCE, HORIZONTAL_TB, VERTICAL_RL,
	VERTICAL_LR, CONTENT_BOX, BORDER_BOX, STRETCH, INLINE_FLEX, FLEX_START,
	FLEX_END, SPACE_BETWEEN, SPACE_AROUND, SPACE_EVENLY, ROW, ROW_REVERSE,
	COLUMN_REVERSE, WRAP_STRING, WRAP_REVERSE, AND, OR, ONLY, INFINITE, UNICODE_RANGE,

	/* Named colours */
	FIRST_COLOUR,

	ALICEBLUE = FIRST_COLOUR, ANTIQUEWHITE, AQUA, AQUAMARINE, AZURE,
	BEIGE, BISQUE, BLACK, BLANCHEDALMOND, BLUE, BLUEVIOLET, BROWN,
	BURLYWOOD, CADETBLUE, CHARTREUSE, CHOCOLATE, CORAL, CORNFLOWERBLUE,
	CORNSILK, CRIMSON, CYAN, DARKBLUE, DARKCYAN, DARKGOLDENROD, DARKGRAY,
	DARKGREEN, DARKGREY, DARKKHAKI, DARKMAGENTA, DARKOLIVEGREEN, DARKORANGE,
	DARKORCHID, DARKRED, DARKSALMON, DARKSEAGREEN, DARKSLATEBLUE,
	DARKSLATEGRAY, DARKSLATEGREY, DARKTURQUOISE, DARKVIOLET, DEEPPINK,
	DEEPSKYBLUE, DIMGRAY, DIMGREY, DODGERBLUE, FELDSPAR, FIREBRICK,
	FLORALWHITE, FORESTGREEN, FUCHSIA, GAINSBORO, GHOSTWHITE, GOLD,
	GOLDENROD, GRAY, GREEN, GREENYELLOW, GREY, HONEYDEW, HOTPINK,
	INDIANRED, INDIGO, IVORY, KHAKI, LAVENDER, LAVENDERBLUSH, LAWNGREEN,
	LEMONCHIFFON, LIGHTBLUE, LIGHTCORAL, LIGHTCYAN, LIGHTGOLDENRODYELLOW,
	LIGHTGRAY, LIGHTGREEN, LIGHTGREY, LIGHTPINK, LIGHTSALMON, LIGHTSEAGREEN,
	LIGHTSKYBLUE, LIGHTSLATEBLUE, LIGHTSLATEGRAY, LIGHTSLATEGREY,
	LIGHTSTEELBLUE, LIGHTYELLOW, LIME, LIMEGREEN, LINEN, MAGENTA, MAROON,
	MEDIUMAQUAMARINE, MEDIUMBLUE, MEDIUMORCHID, MEDIUMPURPLE,
	MEDIUMSEAGREEN, MEDIUMSLATEBLUE, MEDIUMSPRINGGREEN, MEDIUMTURQUOISE,
	MEDIUMVIOLETRED, MIDNIGHTBLUE, MINTCREAM, MISTYROSE, MOCCASIN,
	NAVAJOWHITE, NAVY, OLDLACE, OLIVE, OLIVEDRAB, ORANGE, ORANGERED,
	ORCHID, PALEGOLDENROD, PALEGREEN, PALETURQUOISE, PALEVIOLETRED,
	PAPAYAWHIP, PEACHPUFF, PERU, PINK, PLUM, POWDERBLUE, PURPLE, RED,
	ROSYBROWN, ROYALBLUE, SADDLEBROWN, SALMON, SANDYBROWN, SEAGREEN,
	SEASHELL, SIENNA, SILVER, SKYBLUE, SLATEBLUE, SLATEGRAY, SLATEGREY,
	SNOW, SPRINGGREEN, STEELBLUE, TAN, TEAL, THISTLE, TOMATO, TURQUOISE,
	VIOLET, VIOLETRED, WHEAT, WHITE, WHITESMOKE, YELLOW, YELLOWGREEN,

	LAST_COLOUR = YELLOWGREEN,

	LAST_KNOWN
};

css_error css__propstrings_get(lwc_string ***strings);
void css__propstrings_unref(void);
]]

ffi.cdef[[
// C/ui/libcss/libcss/src/select/properties/properties.h
// these are generated using gcc -E properties.h
typedef css_error (*css_prop_handler)(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);

const css_prop_handler property_handlers[LAST_PROP + 1 - FIRST_PROP];

css_error css__parse_align_content(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_align_items(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_align_self(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_azimuth(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_background(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_background_attachment(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_background_color(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_background_image(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_background_position(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_background_repeat(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_bottom(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_bottom_color(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_bottom_style(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_bottom_width(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_color(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_collapse(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_left(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_left_color(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_left_style(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_left_width(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_right(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_right_color(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_right_style(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_right_width(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_radius(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_radius_corner(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result, enum css_properties_e op);
css_error css__parse_border_radius_top_left(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_radius_top_right(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_radius_bottom_right(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_radius_bottom_left(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_spacing(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_style(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_top(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_top_color(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_top_style(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_top_width(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_border_width(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_bottom(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_box_sizing(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_break_after(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_break_before(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_break_inside(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_caption_side(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_clear(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_clip(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_color(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_columns(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_column_count(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_column_fill(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_column_gap(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_column_rule(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_column_rule_color(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_column_rule_style(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_column_rule_width(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_column_span(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_column_width(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_content(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_counter_increment(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_counter_reset(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_cue(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_cue_after(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_cue_before(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_cursor(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_direction(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_display(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_elevation(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_empty_cells(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_flex(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_flex_basis(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_flex_direction(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_flex_flow(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_flex_grow(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_flex_shrink(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_flex_wrap(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_float(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_font(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_font_family(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_font_size(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_font_style(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_font_variant(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_font_weight(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_height(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_justify_content(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_left(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_letter_spacing(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_line_height(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_list_style(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_list_style_image(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_list_style_position(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_list_style_type(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_margin(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_margin_bottom(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_margin_left(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_margin_right(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_margin_top(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_max_height(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_max_width(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_min_height(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_min_width(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_opacity(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_order(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_orphans(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_outline(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_outline_color(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_outline_style(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_outline_width(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_overflow(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_overflow_x(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_overflow_y(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_padding(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_padding_bottom(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_padding_left(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_padding_right(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_padding_top(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_page_break_after(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_page_break_before(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_page_break_inside(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_pause(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_pause_after(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_pause_before(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_pitch_range(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_pitch(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_play_during(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_position(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_quotes(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_richness(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_right(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_speak_header(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_speak_numeral(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_speak_punctuation(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_speak(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_speech_rate(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_stress(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_table_layout(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_text_align(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_text_decoration(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_text_indent(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_text_transform(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_top(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_unicode_bidi(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_vertical_align(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_visibility(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_voice_family(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_volume(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_white_space(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_widows(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_width(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_word_spacing(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_writing_mode(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
css_error css__parse_z_index(css_language *c,
  const parserutils_vector *vector, int *ctx,
  css_style *result);
]]

return css
