--- lib/ui/yoga-style.lua
local yogastyle = {}

local yoga = require "ui/yoga"
local util = require "util"
local peg = require "peg"
local availableWidth, availableHeight, pagePadding
local pielxsInMillimeter = 3.7795275591
local pielxsInEm = 14 -- * 0.52 -- https://www.lifewire.com/aspect-ratio-table-common-fonts-3467385 - Arial = Helvetica
-- local keyAll, keyStyle

local noYogaWarn = util.invertTable({"YGNodeStyleSetTransform"})
local notYogaKey = util.invertTable({
	"id",
	"element",
	"tag",
	"parent",
	"parent_node" -- these exist with Vue production build
	,
	"model",
	"idx",
	"ref",
	"info",
	"fit-text",
	"object-fit",
	"code-width",
	"code-height" -- barcode,
	,
	"src",
	"format",
	"rows",
	"font-weight",
	"font-size",
	"text-align",
	"word-wrap",
	"autocomplete",
	"overflow" -- text
	,
	"preserveAspectRatio",
	"aria-hidden",
	"role",
	"xmlns",
	"viewBox",
	"innerHTML",
	"xmlns:xlink" -- svg
	,
	"font-family",
	"name",
	"page-width",
	"page-height",
	"column-width",
	"column",
	"left",
	"top" -- report
	,
	"line-height" -- in draw.lua
}) -- TODO: check line-height, YGNodeSetBaselineFunc()

local function setDefaultProps(yg)
	yoga.YGNodeStyleSetFlexDirection(yg, yoga.YGFlexDirectionRow)
	yoga.YGNodeStyleSetFlexWrap(yg, yoga.YGWrapWrap)
end

local function styleToNumber(val)
	local unit
	local value = tonumber(val)
	if value == nil then
		val = peg.split(val, " ")
		val = peg.replace(val, ",", ".") -- number commas to dots
		if peg.foundNumber(val) then
			if val:sub(-2) == "px" then
				value = tonumber(val:sub(1, -3)) or 0
				unit = "px"
			elseif val:sub(-1) == "%" then
				value = tonumber(val:sub(1, -2)) or 0
				unit = "%"
			elseif val:sub(-2) == "em" then
				value = (tonumber(val:sub(1, -3)) or 0) * pielxsInEm
				unit = "em"
			elseif val:sub(-2) == "mm" then
				value = (tonumber(val:sub(1, -3)) or 0) * pielxsInMillimeter
				unit = "mm"
			elseif val:sub(-2) == "vh" then
				value = tonumber(val:sub(1, -3)) or 0
				value = value * availableHeight / 100
				unit = "vh"
			elseif val:sub(-2) == "vw" then
				value = tonumber(val:sub(1, -3)) or 0
				value = value * availableWidth / 100
				unit = "vw"
			else
				value = nil
				unit = "error"
			end
		else
			value = nil
			unit = nil
		end
	end
	return value, unit
end
yogastyle.styleToNumber = styleToNumber

local edgeName = util.invertTable({YGEdgeLeft = 0, YGEdgeTop = 1, YGEdgeRight = 2, YGEdgeBottom = 3, YGEdgeStart = 4, YGEdgeEnd = 5, YGEdgeHorizontal = 6, YGEdgeVertical = 7, YGEdgeAll = 8})
local function edgeValue(key, val, value)
	---@diagnostic disable-next-line
	if value == nil then
		local unit
		value, unit = styleToNumber(val)
		if unit == "error" then
			util.printWarning("Yoga key '%s', value '%s', has digit but does not parse to number", key, val)
		end
	end
	if value then
		local edge = yoga.YGEdgeAll
		if peg.found(key, "left") then
			edge = yoga.YGEdgeLeft
		elseif peg.found(key, "top") then
			edge = yoga.YGEdgeTop
		elseif peg.found(key, "right") then
			edge = yoga.YGEdgeRight
		elseif peg.found(key, "bottom") then
			edge = yoga.YGEdgeBottom
		elseif peg.found(key, "start") then
			edge = yoga.YGEdgeStart
		elseif peg.found(key, "end") then
			edge = yoga.YGEdgeEnd
		elseif peg.found(key, "horizontal") then
			edge = yoga.YGEdgeHorizontal
		elseif peg.found(key, "vertical") then
			edge = yoga.YGEdgeVertical
		end
		return true, value, edge, edgeName[edge]
	end
end

local setPropCount = 0
function yogastyle.setPropsFunc(renderPrf)
	local copyParentStyle = renderPrf.copy_parent_style
	local funcName, func, arg1, val2, unit, value, numValue, valName, valueName, valueName2, nameStart, ok, ok2, arg1Name -- arg1Name will be used in debug print
	local debug = false
	return function(node)
		local style = node.style
		setDefaultProps(node.yg)
		local style2 = {}
		for key, val in pairs(style) do
			if notYogaKey[key] then
				style2[key] = val
			elseif copyParentStyle[key] and node.parent_node and node.parent_node.style[key] == val then
				style2[key] = val -- do not set same values as parent
			elseif type(val) == "string" and (peg.startsWith(val, "flex-start") or peg.startsWith(val, "wrap")) then -- can be: flex-start !important
				style2[key] = peg.parseBefore(val, " ") -- skip, flex-start and wrap are default props, no need to set them
			else
				value = nil
				numValue = nil
				arg1 = nil
				ok = false
				--[[ if peg.found(key, "padding") then
					key = key..""
				end ]]
				funcName = "YGNodeStyleSet" .. peg.toPascalCase(key) -- align-self -> YGNodeStyleSetAlignSelf
				if key ~= "overflow" and val == "auto" then
					ok, func = pcall(function()
						return yoga[funcName .. "Auto"]
					end)
					if ok then
						funcName = funcName .. "Auto"
					end
				elseif type(val) == "string" and peg.found(val, "%") then
					ok, func = pcall(function()
						return yoga[funcName .. "Percent"]
					end)
					if ok then
						funcName = funcName .. "Percent"
						value, unit = yogastyle.styleToNumber(val)
						if unit == "error" then
							util.printWarning("Yoga key '%s', value '%s', percent function '%s', has digit but does not parse to number", key, val, funcName)
						end
					end
				end
				if not ok then
					if type(val) == "string" then -- and (peg.found(key, "size") or peg.found(key, "weight") or peg.found(key, "width") or peg.found(key, "height")) then
						numValue, unit = yogastyle.styleToNumber(val)
						value = numValue
						if unit == "error" and not noYogaWarn[funcName] then
							util.printWarning("Yoga key '%s', value '%s', function '%s', has digit but does not parse to number", key, val, funcName)
						end
					end
					ok, func = pcall(function()
						return yoga[funcName]
					end)
					if not ok then
						funcName = peg.parseBefore(key, "-") -- margin-top -> margin
						funcName = "YGNodeStyleSet" .. peg.toPascalCase(funcName)
						ok, func = pcall(function()
							return yoga[funcName]
						end)
					end
					if not ok and (peg.found(key, "min-") or peg.found(key, "max-")) then
						funcName = "YGNodeStyleSet" .. peg.toPascalCase(peg.parseAfter(key, "-")) -- min-width -> width
						ok, func = pcall(function()
							return yoga[funcName]
						end)
					end
				end
				if not ok then
					-- if key ~= "idx" then
					-- keyAll[key] = keyAll[key] or {}
					-- keyAll[key][val] = node.type
					if not noYogaWarn[funcName] then
						util.printWarning("Yoga key '%s', value '%s', function '%s' does not exist", key, val, funcName)
					end
				else
					ok2 = false
					if peg.found(key, "margin") then
						ok2, value, arg1, arg1Name = edgeValue(key, val, value)
					elseif peg.found(key, "padding") then
						ok2, value, arg1, arg1Name = edgeValue(key, val, value)
					elseif peg.found(key, "border") then
						ok2, value, arg1, arg1Name = edgeValue(key, val, value)
					end
					if ok2 then
						valueName2 = nil -- function takes 3 params: margin, padding, border
					elseif value ~= nil then
						ok2 = true
						valueName2 = nil
					elseif val == "auto" and (key == "width" or key == "height" or key == "basis") then
						value = nil -- these do not have 2. argument
						ok2 = true
						valueName2 = nil
					else
						valName = peg.toPascalCase(peg.parseBefore(tostring(val), " "))
						val2 = val
						if key == "overflow" and val2 == "auto" then
							val2 = "visible" -- or "scroll"
						end
						nameStart = peg.toPascalCase(peg.parseBefore(key, "-"))
						valueName = "YG" .. nameStart .. valName
						valueName2 = valueName -- for printing
						ok2, value = pcall(function()
							return yoga[valueName]
						end)
						arg1 = nil
						if key == "flex" then
							ok2, value = true, tonumber(val2)
						elseif not ok2 then
							nameStart = peg.toPascalCase(peg.parseAfter(key, "-"))
							valueName2 = "YG" .. nameStart .. valName
							ok2, value = pcall(function()
								return yoga[valueName2]
							end)
							if not ok2 then
								value = nil
							end
						end
					end
					if not ok2 then
						-- keyStyle[key] = keyStyle[key] or {}
						-- keyStyle[key][val] = node.type.." - failed"
						if not notYogaKey[key] then -- and not noYogaWarn[funcName] then
							util.printWarning("Yoga key '%s', value '%s', function '%s' value '%s' does not exist", key, val, funcName, valueName2 or value)
						end
					else
						setPropCount = setPropCount + 1
						if arg1 then
							debug = debug and util.print(setPropCount .. ". node: " .. node.idx .. ", " .. node.type .. " " .. funcName .. "(node, " .. arg1Name .. ", styleToNumber('" .. val .. "') = " .. value .. ") - " .. key .. ": " .. val)
							func(node.yg, arg1, value)
						elseif value ~= nil then
							debug = debug and util.print(setPropCount .. ". node: " .. node.idx .. ", " .. node.type .. " " .. funcName .. "(node, " .. (valueName2 or value) .. ")" .. " - " .. key .. ": " .. val)
							func(node.yg, value)
						else
							debug = debug and util.print(setPropCount .. ". node: " .. node.idx .. ", " .. node.type .. " " .. funcName .. "(node)" .. " - " .. key .. ": " .. val)
							func(node.yg)
						end
						-- keyStyle[key] = keyStyle[key] or {}
						-- keyStyle[key][val] = node.type
					end
				end
				if numValue then
					style2[key] = numValue
				else
					style2[key] = val
				end
			end
		end
		-- util.printOk("node idx: %d, style 1: %s", node.idx, json.toJson(style))
		-- util.printInfo("node idx: %d. %s, style 2: %s, value: '%s'", node.idx, node.type, json.toJson(node.style), tostring(node.value))
		node.style = style2
	end
end

function yogastyle.setPage(availableWidth_, availableHeight_, pagePadding_)
	availableWidth, availableHeight, pagePadding = availableWidth_, availableHeight_, pagePadding_
	-- keyAll = {}
	-- keyStyle = {}
	local config = yoga.YGConfigNew()
	local yg = yoga.YGNodeNewWithConfig(config) -- YGJustifyFlexStart, YGJustifyCenter, YGJustifyFlexEnd, YGJustifySpaceBetween, YGJustifySpaceAround
	yoga.YGNodeStyleSetPadding(yg, yoga.YGEdgeAll, pagePadding)
	yoga.YGNodeStyleSetWidth(yg, availableWidth)
	yoga.YGNodeStyleSetHeight(yg, availableHeight)
	availableWidth = availableWidth - 2 * pagePadding
	availableHeight = availableHeight - 2 * pagePadding
	return yg
end

--[[ function yogastyle.printKeys()
	print("\nkeyStyle: ", keyStyle)
	print("\nkeyAll: ", keyAll)
end ]]

return yogastyle
