--- util.lua
--
-- @module util
local util = require "util-env"
local coroThread = require "coro-thread"
local useCoro = coroThread.useCoro()
local threadNumber = coroThread.threadNumber

local start = require "start"
local loc = start.getLocal()

local ffi = require "mffi"
local C = ffi.C
local bit = require "bit"
local cast, band, bor, rshift, lshift = ffi.cast, bit.band, bit.bor, bit.rshift, bit.lshift
local floor = math.floor
local abs = math.abs
local unpack = table.unpack
local concat = table.concat
-- local l = require "lang".l -- will have circular load
local ioWrite = io.write
local ioFlush = io.flush
local printOrig = print
local stringFormat = string.format
local l = stringFormat -- fix this later
local isWin_l = ffi.os == "Windows"
local isMac_l = ffi.os == "OSX"
local isLinux_l = ffi.os == "Linux"
local isCloud_l
require "ffi_def"
-- debug.traceback = require "StackTracePlus".stacktrace
local utf = require "utf"
local peg = require "peg"
local time = require "time"
local color = require "ansicolors"
local fn, dprf, lfs, fs, json, recdata, recData, recDataSet
local shell32 -- dll loaded later
local printTextRecordLength = 1000

local function loadRecData()
	if recdata == nil then
		recdata = require "recdata"
		recData, recDataSet = recdata.get, recdata.set
	end
end

local function loadJson() -- to prevent circular errors
	if not json then
		json = require "json"
	end
end

local function loadFn() -- to prevent circular errors
	if not fn then
		fn = require "fn"
	end
end

local function loadDprf() -- to prevent circular errors
	if not dprf then
		dprf = require "dprf"
	end
end

local function loadFs() -- to prevent circular errors
	if not lfs then
		lfs = require "lfs_load"
	end
	if not fs then
		fs = require "fs"
	end
end

function util.showWindowsErrorDialog(show)
	if isWin_l and not loc.from4d and show == false then
		print("Do not show windows error dialog")
		-- https://blogs.msdn.microsoft.com/oldnewthing/20040727-00/?p=38323
		ffi.C.SetErrorMode(bor(ffi.C.SEM_FAILCRITICALERRORS, ffi.C.SEM_NOGPFAULTERRORBOX))
	end
end

-- old globals set in start.lua
--[[ function util.debugger()
	if util.fromEditor() then
		local ok, mobdebug = pcall(require, 'mobdebug')
		if ok then
			mobdebug.pause()
		end
	end
end ]]

local pathMain
function util.mainPath()
	if pathMain then
		return pathMain
	end
	if util.from4d() then
		pathMain = loc.pathFile -- must NOT be used anywhere else but here
	elseif loc.pathFile == "" then
		pathMain = loc.startPath
		if pathMain == "" then
			loadFs()
			pathMain = fs.currentPath()
		end
	else
		pathMain = loc.pathFile -- fs.currentPath()
	end
	-- print("pathMain: "..pathMain)
	isCloud_l = false
	if loc.prefixPath ~= "" then
		pathMain = loc.prefixPath
	elseif pathMain == "/var/task/" then -- real AWS Lambda
		isCloud_l = true
		pathMain = pathMain .. "aws-nc/"
	elseif peg.found(pathMain, "/aws") then
		pathMain = pathMain .. "" -- .."aws-nc/" -- code here to prevent lint errors
	elseif peg.endsWith(pathMain, "/run/") then
		pathMain = peg.parseBefore(pathMain, "/run/") .. "/"
	elseif peg.endsWith(pathMain, "/bin/build/") then
		pathMain = peg.replace(pathMain, "/bin/build/", "/nc-server/") -- we need pathMain to end with /nc-server/
	elseif peg.found(pathMain, "/test/") then
		pathMain = peg.parseBefore(pathMain, "/test/") .. "/"
	elseif peg.found(pathMain, "/custom/") then
		pathMain = peg.parseBefore(pathMain, "/custom/") .. "/nc-server/" -- we need pathMain to end with /nc-server/
		start.setFilePath(pathMain)
	elseif peg.found(pathMain, "/nc-server/") and not peg.found(pathMain, "/nc-server/nc-server/") then
		pathMain = peg.parseBeforeWithDivider(pathMain, "/nc-server/")
	elseif peg.found(pathMain, "/nc-server-dist/") then
		pathMain = peg.parseBeforeWithDivider(pathMain, "/nc-server-dist/")
	elseif peg.found(pathMain, "/nc-plugin/") then
		pathMain = peg.parseBefore(pathMain, "/nc-plugin/") .. "/nc-server/"
	end
	if peg.endsWith(pathMain, "/nc-server/") == false and peg.endsWith(pathMain, "/nc-server-dist/") == false then
		util.printWarning(l("main path '%s' could not be resolved", pathMain))
	end
	return pathMain
end

function util.pluginPath()
	loadFs()
	return fs.filePathFix(util.mainPath() .. "../nc-plugin/")
end

function util.backendPath()
	loadFs()
	return fs.filePathFix(util.mainPath() .. "../nc-backend/")
end

function util.preferencePath()
	return util.mainPath() .. "preference/"
end

function util.pathLuaFolder()
	local dir = loc.pathFile -- util.runPath() --
	if peg.endsWith(dir, "/manager/4d/") then
		-- return peg.replace(loc.pathFile, "/manager/4d/", "/")
		return peg.replace(loc.pathFile, "/nc-plugin/plugin/external/manager/4d/", "/nc-server/")
	end
	if not util.from4d() then
		dir = util.mainPath()
	end
	return dir
end

-- end old globals

function util.writePrfFile(name, data)
	local path = util.runPath() .. "preference/" .. name
	return util.writeFile(path, data)
end

function util.prf(name, option)
	loadDprf()
	if not peg.found(name, "common/preference/") then
		name = peg.parseAfter(name, "preference/")
	end
	local prfName
	local dotPos = peg.find(name, ".")
	if dotPos == 0 then
		prfName = name .. ".json"
		return dprf.prf(prfName, option)
	elseif peg.endsWith(name, ".json") then
		return dprf.prf(name, option)
		--[[ elseif name:sub(dotPos) == ".html" then
		return util.readPreferenceFile(name)
	elseif name:sub(dotPos) == ".js" then
		return util.readPreferenceFile(name)
	elseif name:sub(dotPos) == ".xml" then
		return util.readPreferenceFile(name) ]]
	end
	local ret = dprf.prf(name, option)
	if type(ret) == "string" then -- .html, .xml, .css and so on
		return ret
	end
	prfName = name:sub(1, dotPos - 1) .. ".json"
	local err
	ret, err = dprf.prf(prfName, option)
	local tagPart = name:sub(dotPos + 1)
	peg.iterateSeparators(tagPart, ".", function(tag)
		if ret then
			local ret2 = ret[tag]
			if ret2 then
				ret = ret2
			elseif tostring(tonumber(tag)) == tag then -- test ret[1] in addition to ret["1"]
				ret = ret[tonumber(tag)]
			end
		end
	end)
	return ret, err
end

function util.prfTag(name, tagName, option)
	local ret
	if type(name) == "string" then
		ret = util.prf(name, option)
	else
		ret = name
	end
	if type(ret) == "table" and ret[tagName] ~= nil then
		return ret[tagName]
	else
		return nil
	end
end

function util.isCloud()
	if isCloud_l ~= nil then
		return isCloud_l
	end
	util.mainPath() -- sets isCloud_l
	return isCloud_l
end

function util.setCloud(val)
	util.mainPath() -- sets isCloud_l
	isCloud_l = val
end

function util.isWin()
	return isWin_l
end

function util.os()
	return ffi.os:lower()
end

local isWine_l
function util.isWine(option)
	if isWine_l == nil then
		if isWin_l == false then
			isWine_l = false
		end
	end
	if isWine_l == false then
		return isWine_l
	end
	if isWine_l ~= nil and not option then
		return isWine_l
	end
	local hntdll = C.GetModuleHandleA("ntdll.dll")
	local pwine_get_version = C.GetProcAddress(hntdll, "wine_get_version")
	if pwine_get_version ~= nil then
		isWine_l = true
	else
		isWine_l = false
	end
	if isWine_l and option then -- == "get-version" then
		local ntdll = ffi.load("ntdll")
		local vrs = ffi.string(ntdll.wine_get_version())
		return isWine_l, vrs -- " Wine version: '"..vrs.."'"
	end
	return isWine_l
end

function util.isMac()
	return isMac_l
end

function util.isLinux()
	return isLinux_l
end

if util.from4d() then
	color.setSupported(false)
elseif util.isWine() then
	color.setSupported(false)
	--[=[
	ffi.cdef [[
		int printf (__const char *__restrict __format, ...);
	]]
	print = ffi.C.printf
	color.setSupported(true) ]=]
	-- elseif util.fromEditor() then
	-- 	color.setSupported(true)
end

-- http://www.freelists.org/post/luajit/fficast-weird-behaviour,6
local lohi_p = ffi.typeof("struct { int32_t " .. (ffi.abi("le") and "lo, hi" or "hi, lo") .. "; } *")
function util.doubleIsNan(p)
	local q = cast(lohi_p, p)
	return band(q.hi, 0x7ff00000) == 0x7ff00000 and bor(q.lo, band(q.hi, 0xfffff)) ~= 0
end

function util.isNan(x) -- https://luapower.com/luajit-notes
	return x ~= x -- ignore this warning: comparison NaN vs NaN is false
end

function util.isInf(x)
	return x <= -math.huge or x >= math.huge
end

local function fmod(a, b)
	if a < b then
		return a
	end
	if a - b < b then
		return a - b
	end
	local r = fmod(a, b + b)
	if r < b then
		return r
	end
	return r - b
end
util.fmod = fmod

--[[ function util.tableRemoveCount(list, pos, count)
  -- from Lua mailing list <lua-l@lists.lua.org>: Egor Skriptunoff, Subject: Re: Proposal to enhance table.remove, Date: 11. September 2022 at 0.09.06 EEST
	pos = pos or #list
	count = count or 1
	return (function(...)
		table.move(list, pos + count, #list + count, pos)
		return ...
	end)(unpack(list, pos, pos + count - 1))
end ]]

function util.tableDottedNameValue(tbl, name)
	local pos = peg.find(name, ".")
	local elem = tbl
	while pos > 0 do
		elem = elem[name:sub(1, pos - 1)]
		if elem == nil then
			break
		end
		name = name:sub(pos + 1)
		-- elem = elem[name] or elem
		pos = peg.find(name, ".")
	end
	return elem
end

function util.tableKeyCount(tbl)
	local count = 0
	for _ in pairs(tbl) do
		count = count + 1
	end
	return count
end

function util.pairsByKeys(tbl, f)
	-- http://www.lua.org/pil/19.3.html
	local a = {}
	for n in pairs(tbl) do
		table.insert(a, n)
	end
	table.sort(a, f)
	local i = 0 -- iterator variable
	local iter = function() -- iterator function
		i = i + 1
		if a[i] == nil then
			return nil
		else
			return a[i], tbl[a[i]]
		end
	end
	return iter
end

function util.tableDifference(a, b)
	-- https://stackoverflow.com/questions/24621346/differences-between-two-tables-in-lua
	local aa = {}
	for _, v in pairs(a) do
		aa[v] = true
	end
	for _, v in pairs(b) do
		aa[v] = nil
	end
	local ret = {}
	local n = 0
	for _, v in pairs(a) do
		if aa[v] then
			n = n + 1
			ret[n] = v
		end
	end
	return ret
end

local compareAmount = 1e-9
local maxCompareAmount = 5.55e19
local function tableEqual(t1, t2, deep, compareMetatable)
	-- deep nil == true
	local ty1 = type(t1)
	local ty2 = type(t2)
	if ty1 ~= ty2 then
		return false
	end
	-- non-table types can be directly compared
	if ty1 == 'number' and ty2 == 'number' then
		if abs(t1 - t2) <= compareAmount then
			return true
		elseif t1 > maxCompareAmount and t2 > maxCompareAmount then
			return tostring(t1) == tostring(t2)
		end
		return false
	elseif ty1 ~= 'table' and ty2 ~= 'table' then
		return t1 == t2
	end
	-- as well as tables which have the metamethod __eq
	if compareMetatable then
		local mt = getmetatable(t1)
		if mt and mt.__eq then
			return t1 == t2
		end
	end
	for k1, v1 in pairs(t1) do
		local v2 = t2[k1]
		if deep ~= false or not ((v2 == nil or type(v2) == "table") and type(v1) == "table") then
			-- todo: should we add keyChecked to speed up next loop? is allocating memory fo keyChecked[] slower than function call?
			if v2 == nil or not tableEqual(v1, v2, deep, compareMetatable) then
				return false
			end
		end
	end
	for k2, v2 in pairs(t2) do
		local v1 = t1[k2]
		if deep ~= false or not ((v1 == nil or type(v1) == "table") and type(v2) == "table") then
			if v1 == nil or not tableEqual(v1, v2, deep, compareMetatable) then
				return false
			end
		end
	end
	return true
end
util.tableEqual = tableEqual

function util.tableKeyEqual(tbl, tbl2)
	for key, val in pairs(tbl) do
		if not tbl2[key] or tbl2[key] ~= val then
			return false
		end
	end
	for key, val in pairs(tbl2) do
		if not tbl[key] or tbl[key] ~= val then
			return false
		end
	end
	return true
end

do
	local _, tableNew = pcall(require, "table.new")
	if type(tableNew) == "function" then
		function util.newTable(arrayPartSize, hashPartSize)
			if type(arrayPartSize) == "table" then
				arrayPartSize = #arrayPartSize
			end
			if type(hashPartSize) == "table" then
				hashPartSize = util.tableKeyCount(hashPartSize)
			end
			return tableNew(arrayPartSize, hashPartSize or 0)
		end
	else
		function util.newTable()
			return {}
		end
	end
end

function util.tableToSortedJson(tbl, depth)
	if not depth then
		depth = 1
	else
		depth = depth + 1
	end
	local ret = util.newTable(util.tableKeyCount(tbl))
	ret[#ret + 1] = "{"
	for key, val in util.pairsByKeys(tbl) do
		if type(val) == "table" then
			val = util.tableToSortedJson(val, depth)
		elseif type(val) == "string" then
			val = '"' .. val .. '"'
		else
			val = tostring(val)
		end
		ret[#ret + 1] = string.rep("\t", depth) .. '"' .. key .. '": ' .. val .. ','
	end
	if #ret > 1 then
		ret[#ret] = ret[#ret]:sub(1, -2)
	end
	depth = depth - 1
	ret[#ret + 1] = string.rep("\t", depth) .. "}"
	return concat(ret, "\n")
end

function util.callPath()
	--[[if util.from4d() then
		return ""
	end]]
	local trace = debug.traceback()
	local path = peg.parseBetweenWithoutDelimiter(trace, "/Users/", "/nc-server/")
	if path then
		trace = peg.replace(trace, "/Users/" .. path, "")
	end
	return trace
	--[[

		local current_func = debug.getinfo(1)
    local calling_func = debug.getinfo(2)
    print(current_func.name.. " was called by ".. calling_func.name.. "!")
	]]
end

---@return string
function util.parameterError(param, paramType)
	if paramType then
		return "parameter '" .. param .. "' type is not '" .. paramType .. "'"
	end
	return "parameter '" .. param .. "' is missing"
end

function util.tableToText(tbl, indet)
	indet = indet or 0
	local txt = {}
	for key, val in pairs(tbl) do
		if type(val) == "table" then
			txt[#txt + 1] = string.rep("  ", indet) .. key .. ":\n" .. util.tableToText(val, indet + 1)
		else
			txt[#txt + 1] = string.rep("  ", indet) .. key .. ": " .. tostring(val)
		end
	end
	return concat(txt, "\n")
end

-- print

local printLfIdx
local function formatPrintTxtLf(errTxt)
	printLfIdx = 1
	repeat
		ioWrite("\n")
		printLfIdx = printLfIdx + 1
	until errTxt:sub(printLfIdx, printLfIdx) ~= "\n"
	return errTxt:sub(printLfIdx)
end

-- local threadColor = color("%{bright black}%s%{reset} ") -- or %03d for 001
local threadColor = color("%{dim white}%s%{reset} ") -- or %03d for 001
if util.from4d() then
	threadColor = "%s"
	--[[ elseif util.fromEditor() then
	threadColor = color("%{dim white}%s%{reset} ") -- black background in vscode ]]
end

local function printThread(errTxt, ...)
	if errTxt == nil then
		errTxt = "errTxt is nil when calling print(errTxt)"
	end
	if useCoro then
		if errTxt:sub(1, 1) == "\n" then
			errTxt = formatPrintTxtLf(errTxt)
		end
		ioWrite(stringFormat(threadColor, threadNumber()))
	end
	printOrig(errTxt, ...)
end
local print = printThread

local function arrayConcat(arr, arr2, count)
	local pos = #arr
	for i, rec in ipairs(arr2) do
		if count and i > count then
			break
		end
		pos = pos + 1
		arr[pos] = rec
	end
end
util.arrayConcat = arrayConcat

-- errors
do
	local errorCache = {}
	local testModeOn = false

	function util.closeProgram(showCallPath)
		if showCallPath ~= false then
			util.printInfo(util.callPath())
		end
		repeat
			util.printInfo(l("please close the program with ctrl-C"))
			io.read()
		until false
	end

	function util.testMode(mode)
		testModeOn = mode
	end

	local function formatPrintTxt(printTxt, ...)
		if printTxt == nil then
			printTxt = "printTxt is nil when calling util.printXxx(printTxt)"
		end
		if type(printTxt) == "table" then
			printTxt = util.tableToText(printTxt)
		end
		printTxt = tostring(printTxt)
		if util.isWin() then
			printTxt = utf.latin9ToUtf8(printTxt)
		end
		local arg = {...}
		if arg == nil or #arg == 0 then
			return printTxt
		end
		if useCoro and printTxt:sub(1, 1) == "\n" then
			printTxt = formatPrintTxtLf(printTxt)
		end
		-- todo: count % params from string and try to fin what arguments are nil and push "nil" to arguments -array
		for i = 1, #arg do
			if arg[i] == nil then
				util.printError("print argument is nil at index: " .. i .. ", error text: '" .. printTxt .. "'")
				arg[i] = "nil"
			elseif type(arg[i]) == "table" then
				if json == nil then
					loadJson()
				end
				arg[i] = json.toJson(arg[i]):sub(1, printTextRecordLength)
			elseif type(arg[i]) ~= "number" and type(arg[i]) ~= "string" then
				arg[i] = tostring(arg[i])
			end
		end
		local ok, ret = pcall(l, printTxt, unpack(arg))
		return ok and ret or printTxt
	end

	function util.print(errText, ...)
		local errTxt = formatPrintTxt(errText, ...)
		print(errTxt) -- color("%{bright black}"..errTxt))
		return errTxt
	end

	function util.print(errText, ...)
		local errTxt = formatPrintTxt(errText, ...)
		print(errTxt) -- color("%{bright black}"..errTxt))
		return errTxt
	end

	function util.color(colorName, message, ...)
		local msgText = formatPrintTxt(message, ...)
		msgText = msgText:gsub("_{", "%%{") -- % needs to be escaped by antoher %
		--[[ if util.isWin() and not util.isWine() then
			print(color("%{bright " .. colorName .. "bg}" .. msgText))
		else ]]
		if type(colorName) ~= "string" then
			util.printError("color first parameter 'color name' type '%s' is not string, error text: '%s'", type(colorName), tostring(message))
		else
			msgText = color("%{" .. colorName .. "}" .. msgText)
		end
		-- end
		return msgText
	end

	function util.printColor(colorName, errText, ...)
		local errTxt = formatPrintTxt(errText, ...)
		errTxt = errTxt:gsub("_{", "%%{") -- % needs to be escaped by antoher %
		--[[ if util.isWin() and not util.isWine() then
			print(color("%{bright " .. colorName .. "bg}" .. errTxt))
		else ]]
		if type(colorName) ~= "string" then
			util.printError("print color first parameter 'color name' type '%s' is not string, error text: '%s'", type(colorName), tostring(errText))
		else
			print(color("%{" .. colorName .. "}" .. errTxt))
		end
		-- end
		return errTxt
	end

	function util.ioWrite(printTxt, ...)
		if printTxt == nil then
			printTxt = "printTxt is nil when calling util.ioWrite(printTxt)"
		end
		if useCoro then
			if printTxt:sub(1, 1) == "\n" then
				printTxt = formatPrintTxtLf(printTxt)
			end
			ioWrite(stringFormat(threadColor, threadNumber()))
		end
		printTxt = formatPrintTxt(printTxt, ...)
		ioWrite(printTxt)
		ioFlush()
	end

	function util.printRed(errText, ...)
		local errTxt = formatPrintTxt(errText, ...)
		if testModeOn then
			print(color("%{bright magenta} - test-mode printRed: " .. errTxt))
		elseif util.isWin() and not util.isWine() and not util.fromEditor() then
			print(color("%{bright redbg}" .. errTxt))
		else
			print(color("%{bright red}" .. errTxt))
		end
		return errTxt
	end

	function util.printError(errText, ...)
		local errTxt = formatPrintTxt(errText, ...)
		if testModeOn then
			print(color("%{bright magenta} - test-mode error: " .. errTxt))
			return errTxt
		else
			errorCache[#errorCache + 1] = "error: " .. errTxt .. "\n  [" .. util.callPath() .. "]"
			if util.fromEditor() then
				util.printRed(errorCache[#errorCache])
			else
				util.printRed("\n" .. errorCache[#errorCache] .. "\n")
			end
		end
		return errorCache[#errorCache]
	end

	function util.printWarning(errText, ...)
		local errTxt = formatPrintTxt(errText, ...)
		if testModeOn then
			print(color("%{yellow} - test-mode warning: " .. errTxt))
		else
			print(color("%{yellow}warning: " .. errTxt .. "\n"))
		end
		return "warning: " .. errTxt
	end

	function util.printWarningWithCallPath(errText, ...)
		local errTxt = formatPrintTxt(errText, ...)
		if testModeOn then
			print(color("%{bright magenta} - test-mode warning: " .. errTxt))
		else
			print(color("\n%{bright magenta}warning: " .. errTxt .. "\n  [" .. util.callPath() .. "]%{reset}"))
		end
		return "warning: " .. errTxt
	end

	function util.printInfo(errText, ...)
		local errTxt = formatPrintTxt(errText, ...)
		print(color("%{bright cyan}" .. errTxt)) -- .."\n"
		return errTxt
	end

	function util.printDebug(errText, ...)
		local errTxt = formatPrintTxt(errText, ...)
		print(color("%{bright blue}" .. errTxt .. "\n"))
		return errTxt
	end

	function util.printOk(errText, ...)
		local errTxt = formatPrintTxt(errText, ...)
		print(color("%{bright green}" .. errTxt))
		return errTxt
	end

	function util.unknownOs()
		util.printError("unknown operating system")
	end

	local function clearError()
		errorCache = {}
	end
	util.clearError = clearError

	function util.getError(defaultError, ...)
		if #errorCache > 0 then
			return concat(errorCache, "\n\n")
		end
		if defaultError and ... then
			defaultError = stringFormat(defaultError, ...)
		end
		clearError()
		return defaultError or nil
	end

	local maxErrorCount = 30
	function util.addError(err, newErr, ...) -- adds max maxErrorCount errors
		if not err then
			err = errorCache
			-- must NOT set errorCache = err or errors will transfer between threads
		end
		if type(newErr) == "table" then
			if err ~= newErr then
				arrayConcat(err, newErr, maxErrorCount)
			end
		else
			if newErr and ... then
				newErr = l(newErr, ...)
				util.printRed(newErr)
			end
			if err[#err] ~= newErr then -- skip duplicate error
				err[#err + 1] = newErr
			end
		end
		return err
	end

	function util.combineError(err, newErr)
		if err == nil then
			return newErr
		end
		if newErr then
			if type(err) == "table" then
				if type(newErr) == "table" then
					arrayConcat(err, newErr)
				else
					err[#err + 1] = newErr
				end
			elseif type(newErr) == "table" then
				err = err .. "\n" .. concat(newErr, "\n")
			else
				err = err .. "\n" .. newErr
			end
		end
		return err
	end

	function util.errorIfFalse(condition, errTxt)
		if not condition then
			util.printError(errTxt)
		end
	end
end

local kernel32
function util.winErrorText(err)
	if not err then
		return ("ERR: winErrorText() called with nil value")
	end
	-- sock.WSAGetLastError() --err --ffi.string(ffi.C.gai_strerror(err))
	local buffer = ffi.newNoAnchor("char[512]")
	if not kernel32 then
		kernel32 = ffi.load("kernel32")
	end
	local flags = bor(C.FORMAT_MESSAGE_IGNORE_INSERTS, C.FORMAT_MESSAGE_FROM_SYSTEM)
	local err_c = cast("int", err)
	--[[ use ffi.C.FormatMessageA ???

		They ought to be in ffi.C. It's even in the docs:

		 On Windows systems, this binds to symbols exported from the *.exe,
		 the lua51.dll (i.e. the Lua/C API provided by LuaJIT itself), the C
		 runtime library LuaJIT was linked with (msvcrt*.dll), kernel32.dll,
		 user32.dll and gdi32.dll.

		[On a tangent and not directed to you: considering this has been in
		the docs since forever, I never cease to be amazed by how many people
		ffi.load("kernel32.dll") in dozens of places for no good reason.]

		--Mike

		]]
	kernel32.FormatMessageA(flags, nil, err_c, 0, buffer, ffi.sizeof(buffer), nil)
	local errTxt = string.sub(ffi.string(buffer), 1, -3) .. " (" .. err .. ")" -- remove last crlf
	errTxt = utf.latin9ToUtf8(errTxt)
	return errTxt
end

-- [[ -- not in use / does not work in win7
function util.setWindowsColorOutput()
	-- If we're on Windows (10), enable the VT100 terminal mode
	-- to enable colorized ANSI output
	-- https://github.com/LuaJIT/LuaJIT/pull/342/files#diff-0
	-- https://docs.microsoft.com/en-us/windows/console/setconsolemode
	if isWin_l and not loc.from4d then
		local hardware = require "hardware"
		if not hardware.windowsMinVersion(10) or util.fromEditor() then
			return
		end
		local console = C.GetStdHandle(C.STD_OUTPUT_HANDLE)
		if ffi.isNull(console) then
			util.printWarning("Failed to get windows console handle (GetStdHandle): " .. util.winErrorText(ffi.errno()))
			return
		end
		local mode = ffi.newNoAnchor("int[1]")
		local ret = C.GetConsoleMode(console, mode)
		if ret == 0 then
			util.printWarning("Failed to get windows console mode (GetConsoleMode): " .. util.winErrorText(ffi.errno()))
			return
		end
		local newMode = bor(mode[0], C.ENABLE_VIRTUAL_TERMINAL_PROCESSING)
		ret = C.SetConsoleMode(console, newMode)
		-- C.ENABLE_PROCESSED_OUTPUT +
		-- C.ENABLE_WRAP_AT_EOL_OUTPUT +
		-- C.ENABLE_VIRTUAL_TERMINAL_PROCESSING +
		-- C.ENABLE_EXTENDED_FLAGS -- no ENABLE_QUICK_EDIT_MODE and no ENABLE_INSERT_MODE
		-- )
		if ret == 0 then
			util.printWarning("Unable to enable virtual terminal processing (SetConsoleMode): " + util.winErrorText(ffi.errno()))
			return
		end
		util.print("Windows 10+, set ANSI color prosessing ok.")
	end
end
util.setWindowsColorOutput()
-- ]]

function util.beep()
	ioWrite("\a") -- bell char "\007"
end

function util.loadDll(fileName, option)
	if util.from4d() and util.isWin() then
		-- do not load ssl libraries, they have been already loaded when 4D starts
		if fileName:find("libeay32", 1, true) or fileName:find("ssleay32", 1, true) then
			return C
		end
	end
	local path
	path = util.pathBin()
	if peg.found(fileName, "/") then
		path = path .. peg.parseBeforeLast(fileName, "/") .. "/"
		fileName = peg.parseLast(fileName, "/")
	end
	--[[ -- disable this loop, it causes "loop or previous error loading module" -errors with loadFs()
	-- print("path: '"..path, "', fileName: "..fileName)
	loadFs()
	local pathLoop = 0
	while pathLoop < 5 and #path > 0 and not fs.dirExists(path) do -- util.checkPath(path) do
		local path2 = path:gsub("(.*)/.*/(bin/bin_.*)", "%1/%2") -- remove folder names before /bin
		if path == path2 then
			pathLoop = pathLoop + 1 -- perevent infinite loop
			path = "../"..path -- if we got to uppermost level in gsub() then we need to start finding upper level
		else
			path = path2
		end
	end
	]]
	local ret, err = ffi.loadDll(fileName, path, option)
	if type(ret) == "userdata" then
		return ret
	end
	if err and (option == nil or not peg.found(option, "no-error")) then
		util.printError(err)
	end
	return ret, err
end

function util.fileSize(bytes, decimals)
	decimals = decimals or 2
	local ret
	if bytes == 0 then
		ret = "0 Bytes"
	else
		local filesizename = {" Bytes", " KB", " MB", " GB", " TB", " PB", " EB", " ZB", " YB"}
		local factor = floor((string.len(tostring(bytes)) - 1) / 3)
		ret = util.formatNum(bytes / 1024 ^ factor, decimals, "")
		-- no space or comma between int numbers
		-- wo don't want "1 024" - we want "1024" because int part is never more than 4 numbers
		ret = ret .. filesizename[factor + 1]
	end
	return ret
end

local filePathCache
function util.getFilePath(mainFolderName, fileName)
	if mainFolderName ~= "" then
		if fileName then
			if peg.endsWith(mainFolderName, "/") then
				fileName = mainFolderName .. fileName
			else
				fileName = mainFolderName .. "/" .. fileName
			end
		else
			fileName = mainFolderName
		end
	end
	if filePathCache == nil then
		filePathCache = {}
	end
	if filePathCache[fileName] then
		return filePathCache[fileName]
	end
	fileName = util.filePathFix(fileName)
	local path = fileName
	if not util.fileExists(fileName) then
		local mainPath = util.mainPath()
		if peg.startsWith(fileName, mainPath) then
			path = fileName
			if util.fileExists(path) == false then
				path = ""
			end
		else
			path = mainPath .. fileName
			if util.fileExists(path) == false then
				path = mainPath .. "../" .. fileName
				if util.fileExists(path) == false then
					path = mainPath .. "../../" .. fileName
					if util.fileExists(path) == false then
						path = ""
					end
				end
			end
			if path == "" then
				path = peg.replace(mainPath, "nc-server", "nc-backend") .. fileName
				if util.fileExists(path) == false then
					path = mainPath .. "../" .. fileName
					if util.fileExists(path) == false then
						path = mainPath .. "../../" .. fileName
						if util.fileExists(path) == false then
							path = ""
						end
					end
				end
			end
		end
	end
	if filePathCache[fileName] == nil and path ~= "" then
		filePathCache[fileName] = path
	end
	return path
end

local function readFile(fileName, option)
	loadFs()
	return fs.readFile(fileName, option)
end

local hash
function util.getFile(mainFolderName, fileName, option, hashTbl)
	local path, err, ret
	if not fileName then
		fileName = mainFolderName
		mainFolderName = ""
	end
	path = util.getFilePath(mainFolderName, fileName)
	if path ~= nil then
		ret = readFile(path)
		if ret ~= nil then
			if string.find(fileName, ".json", 1, true) then
				if hash == nil then
					hash = require "hash"
				end
				ret, err = hash.setHash(ret, fileName, option, hashTbl)
			end
		end
	end
	return ret, err
end

function util.filePathFix(fileName, option)
	loadFs()
	return fs.filePathFix(fileName, option)
end

function util.filePath(fileName)
	loadFs()
	return fs.filePath(fileName)
end

do
	function util.createFilePath(path)
		if not lfs then
			lfs = require "lfs_load"
		end
		local attr = lfs.attributes(path)
		if not attr then -- path does not exist
			lfs.mkdir(path)
			attr = lfs.attributes(path)
			if not attr then -- path does not exist -> create lower paths
				local pathTbl = peg.splitToArray(path, "/")
				local makePath = ""
				for i, folder in ipairs(pathTbl) do
					if i > 1 then
						makePath = makePath .. "/"
					end
					makePath = makePath .. folder
					if makePath ~= "" and i > 1 then -- do not make root-folder: "c:"
						attr = lfs.attributes(makePath)
						if not attr then -- path does not exist
							lfs.mkdir(makePath)
						end
					end
				end
			end
		end
		--[[
		local attr = lfs.attributes(path)
		if not attr then  -- path does not exist
			lfs.mkdir(path)
		end
		]]
	end

	function util.runPath()
		if not lfs then
			lfs = require "lfs_load"
		end
		-- print("lfs.currentdir: '"..lfs.currentdir().."'")
		local path = lfs.currentdir()
		if util.isWin() then
			path = peg.replace(path, "\\", "/")
		end
		if path:sub(-4) == "/run" or path:sub(-5) == "/tool" then
			return "../" -- return path:sub(1, -4)
		else
			return ""
		end
	end

	function util.fileExists(path)
		if not lfs then
			lfs = require "lfs_load"
		end
		local attr = lfs.attributes(path)
		if attr then -- path exists
			return true
		end
		return false -- not nil
	end

	function util.tempDirPath()
		local path = util.filePathFix("~/MA/")
		util.createFilePath(path)
		return path
	end

	function util.deletePathContentRecursive(path)
		if not lfs then
			lfs = require "lfs_load"
		end
		local attr = lfs.attributes(path)
		if attr then -- path exists
			for fileName in lfs.dir(path) do
				if fileName:sub(1, 1) ~= "." then
					local f = path .. '/' .. fileName
					attr = lfs.attributes(f)
					if attr.mode == "directory" then
						util.deletePathContentRecursive(f)
						lfs.rmdir(f)
					else
						os.remove(f)
					end
				end
			end
		end
	end
end

function util.openApplication(app, args, opt)
	-- print("util.openApplication: ", app, args, opt)
	app = util.filePathFix(app)
	-- print("util.openApplication2: ", app, args, opt)
	if isWin_l then
		app = app:gsub("/", "\\")
		local cmd
		if opt == "wait" then
			cmd = 'cmd /c start "MA" /WAIT' -- http://www.computerhope.com/starthlp.htm
			-- must NOT contain /B or will not wait
			-- else -- use ShellExecuteA
		end
		if cmd and args then
			if args:sub(-1) == '"' then -- lasu char is " -> already quoted many args
				cmd = cmd .. ' "' .. app .. '" ' .. args -- "" is app name
			else
				cmd = cmd .. ' "' .. app .. '" "' .. args .. '"' -- "" is app name
			end
		elseif cmd then
			cmd = cmd .. ' "' .. app .. '"'
		end
		if cmd then
			print("util.openApplication: ", cmd)
			os.execute(cmd)
		else
			if not shell32 then
				shell32 = util.loadDll("shell32")
			end
			local hwnd = nil
			local directory = nil
			-- https://msdn.microsoft.com/en-us/library/windows/desktop/bb762153(v=vs.85).aspx
			local ret
			if util.from4d() then
				util.print('ShellExecute command: "%s" "%s"', app, args)
				if peg.endsWith(app, ".exe") then
					ret = shell32.ShellExecuteA(hwnd, "open", app, args, directory, C.SW_SHOWNORMAL)
				else
					ret = shell32.ShellExecuteA(hwnd, "open", app, args, directory, C.SW_SHOWMINNOACTIVE)
				end
			else
				ret = shell32.ShellExecuteA(hwnd, "open", app, args, directory, C.SW_SHOWNORMAL)
			end
			if ret <= 32 then
				util.printError('ShellExecute failed with error %s, command: "%s" "%s"', tostring(ret), app, args)
			end
		end
	else
		local cmd
		if args then
			cmd = "open -a " .. app .. " --args " .. args
		else
			cmd = "open -a " .. app
		end
		print("util.openApplication: ", cmd)
		os.execute(cmd)
	end
end

function util.openUrl(url, browser)
	-- url = util.filePathFix(url)
	local browserPath
	if type(browser) == "string" and browser ~= "" then
		if isWin_l then
			-- browser: ﻿"Mozilla Firefox/firefox", "Google/Chrome/Application/chrome"
			local exePath = {}
			exePath[1] = "C:/Program Files/" .. browser .. ".exe"
			exePath[2] = "C:/Program Files (x86)/" .. browser .. ".exe"
			exePath[3] = "%USERPROFILE%/Local Settings/Application Data/" .. browser .. ".exe"
			exePath[4] = "%USERPROFILE%/AppData/Local/" .. browser .. ".exe"
			for _, path in ipairs(exePath) do
				if util.fileExists(path) then
					browserPath = path
					break
				end
			end
		else
			if util.fileExists(browser) then
				browserPath = browser
			end
		end
	end
	if browserPath then
		-- print(browserPath, url)
		util.openApplication(browserPath, url)
	elseif isWin_l then
		url = url:gsub("/", "\\")
		-- print('start "" "'..url..'"')
		os.execute('start "" "' .. url .. '"')
	else
		-- print("open "..url)
		if not peg.found(url, '"') then
			os.execute('open "' .. url .. '"')
		else
			os.execute("open " .. url)
		end
	end
end

function util.openFile(fileName, delay)
	loadFs()
	return fs.openFile(fileName, delay)
end

function util.uncompress(fileName, destPath)
	-- http://www.dotnetperls.com/7-zip-examples
	fileName = util.filePathFix(fileName)
	destPath = util.filePathFix(destPath) -- .."/WebManager"
	local pathExe = util.pathBin() .. "7z"
	if isWin_l then
		pathExe = pathExe:gsub("/", "\\") -- ..".exe"
		destPath = destPath:gsub("/", "\\") -- AK 2015-03-05
		fileName = fileName:gsub("/", "\\")
	end
	print(l("uncompress 7z path: %s, destPath: %s, fileName: %s", pathExe, destPath, fileName))
	-- print("destPath: "..destPath)
	-- print("fileName: "..fileName)

	-- compress: os.execute("~/svnroot/cpp/MA_Lua/bin/OSX/7z a source"..suffix..".7z source"..suffix.." -m0=LZMA2 -mmt=on -mx=3 -pwebmanageri3+")
	-- x = extract with paths, -y = overwrite without prompt, -p = psswrd, -o = outputPath
	local param = ' x -y -pwebmanageri3+ -o"' .. destPath .. '" "' .. fileName .. '"'
	return util.openApplication(pathExe, param, "wait")
end

function util.tableKeyIndex(key, tbl)
	local i = 1
	for k, _ in pairs(tbl) do
		if k == key then
			return i
		end
		i = i + 1
	end
	return -1
end

function util.tableKeyToArray(tbl)
	local arr = {}
	for key in pairs(tbl) do
		arr[#arr + 1] = key
	end
	return arr
end

local function flattenARecord(item, ret)
	for key, val in pairs(item) do
		if type(val) == "table" then
			flattenARecord(val, ret)
		elseif ret[key] == nil then
			ret[key] = val
		elseif ret[key] ~= val then
			util.printWarning("flattenARecord, key '%s' value '%s' exists, new value: '%s'", tostring(key), tostring(ret[key]), tostring(val))
			ret[key] = val
		end
	end
end
util.flattenARecord = flattenARecord

function util.flattenArray(arr)
	local retArr = {}
	for i, item in ipairs(arr) do
		retArr[i] = {}
		flattenARecord(item, retArr[i])
	end
	return retArr
end

function util.setArrayNullValue(arr, value)
	local maxIndex = 0
	for key in pairs(arr) do
		if key > maxIndex then
			maxIndex = key
		end
	end
	local i = 0
	for _ in pairs(arr) do
		i = i + 1
		if arr[i] == nil then
			arr[i] = value
		end
	end
	return arr
end

function util.copyArray(arr, starPos, endPos)
	local ret = {}
	if not starPos then
		starPos = 1
	end
	if not endPos or endPos > #arr then
		endPos = #arr
	end
	for i = starPos, endPos do
		ret[#ret + 1] = arr[i]
	end
	return ret
end

function util.arraySize(arr)
	return #arr
end

function util.arrayItemIndex(key, tbl)
	for i = 1, #tbl do
		local v = tbl[i]
		if v == key then
			return i
		end
	end
	return -1
end

function util.arrayItemFound(key, tbl)
	return util.arrayItemIndex(key, tbl) ~= -1
end

function util.arrayRecordField(array, fieldName, fieldValue, tagName)
	for _, rec in ipairs(array) do
		if rec[fieldName] == fieldValue then
			return rec[tagName]
		end
	end
	return nil
end

-- http://lua-users.org/wiki/CopyTable
function util.cloneTableClean(orig, level) -- shallowcopy, clean cdata
	local copy
	if type(orig) == "table" then
		copy = {}
		for orig_key, orig_value in pairs(orig) do
			if level ~= 0 or type(orig_value) ~= "table" then
				if type(orig_value) ~= "cdata" and type(orig_value) ~= "userdata" and type(orig_value) ~= "function" and type(orig_value) ~= "thread" then
					copy[orig_key] = orig_value
				end
			end
		end
	else -- number, string, boolean, etc
		copy = orig
	end
	return copy
end

--[[ -- remove comment when there is need for this
local tableNumKeyse = require "table.nkeys"
function util.numberOfTableKeys(tbl) -- number of non-nil array elements plus the number of non-nil key-value pairs
	return tableNumKeyse(tbl)
end
]]

do
	local _, isArray = pcall(require, "table.isarray") -- https://github.com/openresty/luajit2#tableisarray
	if type(isArray) == "function" then
		function util.isArray(array)
			if type(array) ~= "table" then
				return false
			end
			return isArray(array) -- print(isArray{}) --> true
		end
	else
		function util.isArray(array)
			if type(array) ~= "table" then
				return false
			end
			local len = #array
			if len > 0 and (array[1] == nil or array[len] == nil) then
				return false
			end
			local n = 1
			for k in pairs(array) do
				if k ~= n then
					return false
				end
				n = n + 1
			end
			return true
		end
	end
end
-- local isArray = util.isArray

local _, tableClone = pcall(require, "table.clone")
if type(tableClone) == "function" then
	function util.cloneShallow(tbl)
		return tableClone(tbl)
	end
else
	function util.cloneShallow(tbl)
		if type(tbl) == "table" then
			-- shallow copy, no metatables
			local copy = {}
			for tblKey, tblValue in pairs(tbl) do
				copy[tblKey] = tblValue
			end
			return copy
		end -- else: number, string, boolean, etc
		return tbl
	end
end
local cloneShallow = util.cloneShallow

local function cloneDeep(tbl, skipKey) -- deepcopy
	if type(tbl) == "table" then
		-- deepcopy, no metatables
		local copy = {}
		if skipKey and skipKey[1] then -- isArray(skipKey)
			skipKey = util.invertTable(skipKey)
		end
		for tblKey, tblValue in pairs(tbl) do
			if skipKey == nil or not skipKey[tblKey] then
				if type(tblKey) == "table" then
					tblKey = cloneShallow(tblKey) -- ,skipKey
				end
				if type(tblValue) == "table" then
					copy[tblKey] = cloneDeep(tblValue, skipKey)
				else
					copy[tblKey] = tblValue
				end
			end
		end
		return copy
	end -- else: number, string, boolean, etc
	return tbl
end
util.clone = cloneDeep

local function cloneMeta(orig) -- deepcopy with metatables
	local copy
	if type(orig) == "table" then
		copy = {}
		for orig_key, orig_value in pairs(orig) do -- next, orig, nil do
			copy[util.cloneMeta(orig_key)] = cloneMeta(orig_value)
		end
		local metatbl = getmetatable(orig)
		if metatbl then
			setmetatable(copy, cloneMeta(metatbl))
		end
	else -- number, string, boolean, etc
		copy = orig
	end
	return copy
end
util.cloneMeta = cloneMeta

function util.printf(txt, ...)
	ioWrite(stringFormat(txt, ...))
end

function util.printfln(txt, ...)
	util.printf(txt .. "\n", ...)
end

function util.bufferToHex(buffer, bufLen, separator)
	-- TODO: test bit.tohex()
	local ret = {}
	for i = 1, bufLen do
		ret[i] = stringFormat("%02x", buffer[i - 1]) -- "%02X"
	end
	return concat(ret, separator)
end

function util.isNumber(field)
	return tostring(tonumber(field)) == field
end

function util.numToHex(num)
	return stringFormat("%x", tonumber(num))
end

--[[
function util.hexToNum(s)
	-- or Call strtoull() via the FFI
	s = s:sub(3)
	local n = #s
	local s2 = s:sub(n - 7)
	local s1 = s:sub(1, n - #s2)
	return tonumber("0x" .. s1) * 0x100000000LL + tonumber("0x" .. s2)
end
function hex2num(hex) -- see: https://github.com/LuaJIT/LuaJIT/pull/894
	return tonumber(hex, 16)
end
]]

function util.stringToHex(str, spacer)
	return (string.gsub(str, "(.)", function(c)
		return stringFormat("%02x%s", string.byte(c), spacer or "")
	end))
end

function util.hexToString(str)
	local ret = ""
	for i = 1, #str - 1, 2 do
		local hex = str:sub(i, i + 1)
		local dec = tonumber(hex, 16)
		local ascii = string.char(dec)
		ret = ret .. ascii
	end
	return ret
end

function util.printHex(str, spacer)
	local ret = util.stringToHex(str, spacer)
	print(ret)
end

function util.skip(idx, skip)
	if skip <= 0 then
		return skip
	end
	return idx - floor(idx / skip) * skip
end

function util.skipFunction(skip)
	if skip <= 0 then
		return function()
			return skip
		end
	end
	return function(idx)
		return idx - floor(idx / skip) * skip
	end
end

-- local bufferArray
--[[
ffi.cdef [-[
	typedef struct {
			void* p1;
			// void* p2;
	} finalizer_st;
]-]

local mt = {
  __gc = function(self)
      if self.p1 ~= nil then
				print("finalizer: free p1", self.p1)
        ffi.C.free(self.p1)
      end
			--[-[
      if self.p2 ~= nil then
				print("finalizer: free p2", self.p2)
        ffi.C.free(self.p2)
      end
			]-]
  end
}
local finalizer = ffi.metatype("finalizer_st", mt)
]]

function util.getOffsetPointer(cdata, offset)
	local address_as_number
	if ffi.abi("64bit") then
		address_as_number = cast("int64_t", cdata)
		-- return cast("int64_t *", address_as_number + offset)
	else -- if is32bit then
		address_as_number = cast("int32_t", cdata)
		-- return cast("int32_t *", address_as_number + offset)
	end -- is there 16 bit luajit systems?
	return cast("int8_t *", tonumber(address_as_number) + offset)
end

if util.from4d() then
	local plg4d
	function util.sleep(millisec)
		if not plg4d then
			plg4d = require "plg4d"
		end
		local delayInTicks = util.round(millisec / 1000 * 60, 0)
		plg4d.sleep(delayInTicks)
	end
else
	if isWin_l then
		function util.sleep(millisec)
			C.Sleep(millisec)
		end
	else
		function util.sleep(millisec)
			-- C.poll(nil, 0, millisec)
			local microseconds = millisec * 1000
			C.usleep(microseconds)
		end
	end
end

function util.sleepCount(millisec, n)
	loadFn()
	fn.range(n):each(function(i)
		print("sleep", i, millisec .. " millisec")
		util.sleep(millisec)
	end)
end

--- Files

function util.runCommandLineRaw(command, option)
	local ret = ""
	-- local cmdFile, err = io.popen
	-- if util.from4d() then
	if option ~= "no-message" then
		util.printColor("magenta", "* run in command line: '%s'", command)
	end
	local cmdFile = io.popen(command .. " 2>&1", "r") --  2>&1 returns also stderr -- use in windows: ShellExecute()
	if cmdFile then
		ret = cmdFile:read("*all")
		cmdFile:close()
	end
	return ret
end

function util.runCommandLine(command, option)
	local ret = util.runCommandLineRaw(command, option)
	ret = ret:gsub("\r", "")
	return ret:gsub("\n", "")
end

local homeDir
function util.homeDirectory()
	if homeDir then
		return homeDir
	end
	if isWin_l then
		homeDir = os.getenv("USERPROFILE") -- C:\users\me
	else
		homeDir = os.getenv("HOME") -- /Users/me, same as "echo ~""
	end
	return homeDir
end

function util.systemRelease()
	if util.isWin() then
		return "not done yet for windows"
	end
	local name = ffi.newNoAnchor("struct utsname[1]")
	local ret = C.uname(name)
	if ret ~= 0 then
		return "error: " .. tostring(ret)
	end
	return ffi.string(name[0].release)
end

function util.osxVersion()
	if not util.isMac() then
		return "only for osx"
	end
	local release = util.systemRelease()
	if release:sub(1, 3) == "15." then
		release = "10.11"
	elseif release:sub(1, 3) == "14." then
		release = "10.10"
	elseif release:sub(1, 3) == "13." then
		release = "10.9"
	elseif release:sub(1, 3) == "12." then
		release = "10.8"
	elseif release:sub(1, 3) == "11." then
		release = "10.8"
	elseif release:sub(1, 3) == "10." then
		release = "10.8"
	elseif release:sub(1, 2) == "9." then
		release = "10.8"
	elseif release:sub(1, 2) == "8." then
		release = "10.8"
	elseif release:sub(1, 2) == "7." then
		release = "10.8"
	elseif release:sub(1, 2) == "6." then
		release = "10.8"
	elseif release:sub(1, 2) == "5." then
		release = "10.8"
	else
		release = "unknown"
	end
	return release
	--[[
	Darwin version to OS X release:
	15.x.x  OS X 10.11.x El Capitan
	14.x.x  OS X 10.10.x Yosemite
	13.x.x  OS X 10.9.x Mavericks
	12.x.x  OS X 10.8.x Mountain Lion
	11.x.x  OS X 10.7.x Lion
	10.x.x  OS X 10.6.x Snow Leopard
	 9.x.x  OS X 10.5.x Leopard
	 8.x.x  OS X 10.4.x Tiger
	 7.x.x  OS X 10.3.x Panther
	 6.x.x  OS X 10.2.x Jaguar
	 5.x    OS X 10.1.x Puma
	]]
end

local systemUserName
function util.systemUserName(force)
	if systemUserName and not force then
		return systemUserName
	end
	if util.isWin() then
		local advapi = ffi.load("Advapi32.dll") -- could cache if needed more than one time
		local size = 256
		local retVal = ffi.newNoAnchor("DWORD[1]", size)
		local buffer = ffi.newNoAnchor("CHAR[?]", size)
		local ok = advapi.GetUserNameA(buffer, retVal)
		if ok == 0 then
			systemUserName = util.winErrorText(ffi.errno())
		else
			systemUserName = ffi.string(buffer, retVal[0] - 1) -- characters will hold the maximum length user name including the terminating null character
		end
	else -- osx, linux
		--[[
		local uid = C.geteuid()
		local pw = C.getpwuid(uid)
		return pw.pw_name
		]]
		systemUserName = util.runCommandLine("whoami")
	end
	return systemUserName
end

function util.readPreferenceFile(fileName, readPath)
	loadFs()
	local pathFile = readPath or util.mainPath()
	local path = pathFile .. "preference/" .. fileName
	local data = readFile(path)
	if not data then -- only 1 upper level is allowed, for example started in /run or /test
		path = pathFile .. "../preference/" .. fileName
		data = readFile(path)
	end
	return data
end

local separatePluginPath = 1 -- 1 only on first call, then true or false
function util.readUpperLevelPreferenceFile(fileName, option)
	loadFs()
	local pathFile = util.mainPath()
	-- util.print("* upper level preference start path: '%s', finding preference '%s'", pathFile, fileName)
	if fileName:sub(1, 6) == "table/" then -- and util.from4d() then
		if separatePluginPath == 1 then
			separatePluginPath = peg.replace(pathFile, "/nc-server/", "/nc-plugin/")
			if not fs.exists(separatePluginPath) then
				separatePluginPath = peg.replace(pathFile, "/nc-server/", "/nc-server-dist/")
				if not fs.exists(separatePluginPath) then
					separatePluginPath = false
				end
			end
		end
		if separatePluginPath then
			pathFile = separatePluginPath
		end
	end
	local path = fs.filePathFix(pathFile .. "../nc-preference/" .. fileName)
	local path1 = path
	local option2 = option
	if option2 == nil then
		option2 = "no-error"
	elseif not peg.found(option2, "no-error") then
		option2 = option2 .. " no-error"
	end
	local data = fs.readFile(path, nil, option2)
	if not data then
		path = pathFile .. "../../nc-preference/" .. fileName
		data = fs.readFile(path, nil, option2)
		if not data then
			path = pathFile .. "../nc-preference/" .. fileName
			data = fs.readFile(path, nil, option2)
			if not data and (peg.found(option2, "use-default")) then
				path = fileName
				data = fs.readFile(path, nil, option2)
				if not data then
					path = "preference/" .. fileName
					data = fs.readFile(path, nil, option2)
				end
			end
		end
	end
	if data then
		loadJson()
		data = json.fromJson(data)
	end
	local printNotFound
	if data then
		if not util.from4d() then
			util.print("* load upper level nc-preference file: '%s'", path)
		end
	elseif option == nil or not peg.found(option, "no-error") then
		printNotFound = true
		util.printInfo("* upper level preference start path: '%s', finding preference '%s'", pathFile, fileName)
	end
	if data == nil then
		loadDprf()
		data = dprf.prf(fileName, option) -- dprf handles "no-db"
	end
	if printNotFound then
		util.printInfo("  - load upper level nc-preference file '%s' was not found, using default preference", path1)
	end
	return data or {}, path
end

function util.writeFile(fileName, data)
	loadFs()
	return fs.writeFile(fileName, data)
end

function util.appendFile(fileName, data)
	loadFs()
	return fs.appendFile(fileName, data)
end

--- Tables
--- Find distinct values by field name
-- return (possibly sorted) table and it's inverted table
-- for ex.: {"a","c","c","a","b"} -> {"a","b","c"}, {"b"=2,"a"=1,"c"=3}
function util.distinctValues(tbl, fldName, doSort)
	-- http://stackoverflow.com/questions/9754285/in-lua-how-do-you-find-out-the-key-an-object-is-stored-in
	local ret = {}
	local count = 0
	for _, rec in ipairs(tbl) do
		local id = rec[fldName]
		if not ret[id] then
			count = count + 1
			ret[id] = count
		end
	end
	local retInvert = util.invertTable(ret)
	if doSort then
		table.sort(retInvert)
		ret = util.invertTable(retInvert)
	end
	return retInvert, ret
end

function util.arrDistinctValues(arr, doSort)
	-- http://stackoverflow.com/questions/9754285/in-lua-how-do-you-find-out-the-key-an-object-is-stored-in
	local ret = {}
	local notDistinct = {}
	local count = 0
	for _, value in ipairs(arr) do
		if not ret[value] then
			count = count + 1
			ret[value] = count
		else
			notDistinct[#notDistinct + 1] = value
		end
	end
	local retInvert = util.invertTable(ret)
	if doSort then
		table.sort(retInvert)
		-- ret = util.invertTable(retInvert)
	end
	return retInvert, notDistinct -- ,ret
end

function util.notInBothArray(arr1, arr2)
	local tbl = util.invertTable(arr1)
	local ret = {}
	for _, val in ipairs(arr2) do
		if not tbl[val] then
			ret[#ret + 1] = val
		end
	end
	return ret
end

function util.clearRecord(rec)
	for key in pairs(rec) do -- clear record without changing it's address
		rec[key] = nil
	end
end

function util.recordValue(tbl, findFieldName, findValue, returnFieldName)
	for _, rec in ipairs(tbl) do
		if rec[findFieldName] == findValue then
			return rec[returnFieldName]
		end
	end
	return nil
end

--- Invert table keys and values
---@return table
function util.invertTable(tbl)
	if tbl == nil then
		return {} -- must return table, not nil
	end
	-- http://stackoverflow.com/questions/9754285/in-lua-how-do-you-find-out-the-key-an-object-is-stored-in
	local ret = {}
	for key, val in pairs(tbl) do
		ret[val] = key
	end
	return ret
end

--- Time

function util.seconds_to_clock(sSeconds, decimals)
	local nSeconds = tonumber(sSeconds) -- you can give string or number
	if nSeconds == 0 then
		-- return nil;
		return "00:00:00"
	else
		local nHours = stringFormat("%02.f", floor(nSeconds / 3600))
		local nMins = stringFormat("%02.f", floor(nSeconds / 60 - nHours * 60))
		local nSecs = stringFormat("%02.f", floor(nSeconds - nHours * 3600 - nMins * 60))
		if decimals and decimals > 0 then
			local _, fractional = math.modf(nSeconds)
			local frac = stringFormat("%15.15f", fractional) -- tostring(util.round(fractional, decimals))
			return nHours .. ":" .. nMins .. ":" .. nSecs .. frac:sub(2, 2 + decimals) -- 0.xx -> .xx
		else
			return nHours .. ":" .. nMins .. ":" .. nSecs
		end
	end
end

function util.seconds(prev_sec, decimals)
	return time.seconds(prev_sec, decimals)
end

function util.milliSeconds(prev_millisec, decimals)
	return time.milliSeconds(prev_millisec, decimals)
end

function util.microSeconds(prev_microsec, decimals)
	return time.microSeconds(prev_microsec, decimals)
end

--- old

-- add comma to separate thousands
function util.comma_value(amount, comma)
	comma = comma or " " -- in us comma = ","
	if comma == "" then
		comma = " "
	end -- must be something or will not work
	local formatted = amount

	local k
	while true do
		formatted, k = string.gsub(formatted, "^(-?%d+)(%d%d%d)", "%1" .. comma .. "%2")
		if k == 0 then
			break
		end
	end
	return formatted
end

-- rounds a number to the nearest decimal places
function util.round(val, decimal)
	if decimal then
		return floor(val * 10 ^ decimal + 0.5) / 10 ^ decimal
	else
		return floor(val + 0.5)
	end
	-- return x>=0 and floor(x+0.5) or math.ceil(x-0.5)
end

function util.roundFunction(decimal)
	return function(val)
		return floor(val * 10 ^ decimal + 0.5) / 10 ^ decimal
		-- return x>=0 and floor(x+0.5) or math.ceil(x-0.5)
	end
end

function util.string_starts(String, Start)
	return string.sub(String, 1, string.len(Start)) == Start
end

-- given a numeric value formats output with comma to separate thousands
-- and rounded to given decimal places
local function formatNum(amount, decimal, comma, prefix, neg_prefix)
	local formatted, famount, remain
	decimal = decimal or 2 -- default 2 decimal places
	comma = comma or " "
	neg_prefix = neg_prefix or "-" -- default negative sign
	famount = math.abs(util.round(amount, decimal))
	famount = floor(famount)
	remain = util.round(math.abs(amount) - famount, decimal)
	-- comma to separate the thousands
	if comma ~= "" then
		formatted = util.comma_value(famount, comma)
	else
		formatted = tostring(famount)
	end
	-- attach the decimal portion
	if decimal > 0 then
		remain = string.sub(tostring(remain), 3)
		formatted = formatted .. "." .. remain .. string.rep("0", decimal - string.len(remain))
	end
	-- attach prefix string e.g "$"
	formatted = (prefix or "") .. formatted
	-- if value is negative then format accordingly
	if amount < 0 then
		if neg_prefix == "()" then
			formatted = "(" .. formatted .. ")"
		else
			formatted = neg_prefix .. formatted
		end
	end
	return formatted
end
util.formatNum = formatNum

function util.formatNumFunction(decimal, comma, prefix, neg_prefix)
	return function(amount)
		return formatNum(amount, decimal, comma, prefix, neg_prefix)
	end
end

function util.formatInt(amount)
	return formatNum(amount, 0)
end

function util.fileSize(bytes, decimals)
	decimals = decimals or 2
	local ret
	if bytes == 0 then
		ret = "0 Bytes"
	else
		-- local filesizename = {" Bytes", " KB", " MB", " GB", " TB", " PB", " EB", " ZB", " YB"}
		local filesizename = {" bytes", " kb", " mb", " GB", " TB", " PB", " EB", " ZB", " YB"}
		local factor = floor((string.len(tostring(bytes)) - 1) / 3)
		ret = formatNum(bytes / 1024 ^ factor, decimals, "")
		-- no space or comma between int numbers
		-- wo don't want "1 024" - we want "1024" because int part is never more than 4 numbers
		ret = ret .. filesizename[factor + 1]
	end
	return ret
end

function util.err(txt)
	--[[
	http://luajit.org/extensions.html
	debug.* functions identify metamethods

	debug.getinfo() and lua_getinfo() also return information about invoked metamethods. The namewhat field is set to "metamethod" and the name field has the name of the corresponding metamethod (e.g. "__index").
	--]]

	local funcName = debug.getinfo(2, "n").name
	funcName = funcName .. ", line: #" .. debug.getinfo(2, "l").currentline
	funcName = funcName .. ", file: " .. string.sub(string.gsub(debug.getinfo(2, "S").source, "\\\\", "\\"), 2)
	--[[ if (...) then
		returnTxt = returnTxt.."err: "..txt
		if funcName then
			returnTxt = returnTxt.." ("..funcName..")"
		end
		returnTxt = returnTxt.."\n"
	else]]
	if funcName then
		print("  *** ERROR: " .. txt .. " (" .. funcName .. ") ***")
	else
		print("  *** ERROR: " .. txt .. " ***")
	end
	-- end
end

function util.distinctIndexArray(arr)
	local keyArr = {}
	local idx = 0
	for k, _ in pairs(arr) do
		if not keyArr[k] then
			idx = idx + 1
			keyArr[idx] = k
		end
	end
	return keyArr
end

function util.distinctKeyArray(arr, key)
	local keyArr = {}
	local idx = 0
	for i = 1, #arr do
		-- local rec = arr[i][key]
		-- local value = rec[key]
		local value = arr[i][key]
		if not keyArr[value] then
			idx = idx + 1
			keyArr[value] = idx
		end
	end
	return keyArr
end

function util.distinctValueArray(recArr, key)
	local keyArr = {}
	local valueArr = {}
	local idx = 0
	for i = 1, #recArr do
		local value = recArr[i][key]
		if not keyArr[value] then
			idx = idx + 1
			keyArr[value] = idx
			valueArr[idx] = value
		end
	end
	return valueArr
end

function util.table_key_to_index_invert(tbl)
	local u = {}
	local i = 1
	for k, _ in pairs(tbl) do
		u[k] = i
		i = i + 1
	end
	return u
end

function util.arrayFoundRemove(arr1, arr2)
	-- remove found elements from arr1
	local removeIdx = {}
	if #arr1 < 1 or #arr2 < 1 then
		return arr1
	end
	for idx, val1 in ipairs(arr1) do
		for _, val2 in ipairs(arr2) do
			if val2 == val1 then
				removeIdx[#removeIdx + 1] = idx
				break
			end
		end
	end
	if #removeIdx < 1 then
		return arr1
	end
	for i = #removeIdx, 1, -1 do -- must remove from end to start
		table.remove(arr1, removeIdx[i])
	end
	return arr1
end

function util.table_not_found_remove(tbl, findTbl)
	-- remove not-found elements from tbl
	local findIdTbl = util.invertTable(findTbl)
	for idx, _ in pairs(tbl) do
		if not findIdTbl[idx] then
			tbl[idx] = nil
		end
	end
end

function util.arrayIntersection(tbl, tbl2)
	-- remove not-found keys from tbl
	local ret = {}
	local findIdTbl = util.invertTable(tbl2)
	for _, key in pairs(tbl) do
		if findIdTbl[key] then
			ret[#ret + 1] = key
		end
	end
	return ret
end

function util.tableIntersection(tbl, tbl2)
	-- remove not-found keys from tbl
	local ret = {}
	for key, value in pairs(tbl) do
		if tbl2[key] then
			ret[key] = value
		end
	end
	return ret
end

function util.table_key_index(key, tbl)
	local i = 1
	for k, _ in pairs(tbl) do
		if k == key then
			return i
		end
		i = i + 1
	end
	return -1
end

function util.table_item_index(value, tbl, option)
	for i = 1, #tbl do
		if tbl[i] == value then
			return i
		end
	end
	if option == "lowercase" and type(value) == "string" then
		local lowercaseValue = peg.lower(value)
		for i = 1, #tbl do
			if tbl[i] and peg.lower(tbl[i]) == lowercaseValue then
				return i
			end
		end
	end
	return nil
end

function util.table_field_value_index(tbl, field, value, startIndex, lowerCase)
	startIndex = startIndex or 1
	if not field:find(".", 1, true) then
		for i = startIndex, #tbl do
			if lowerCase then
				if tbl[i][field]:lower() == value:lower() then
					return i
				end
			else
				if tbl[i][field] == value then
					return i
				end
			end
		end
	end
	if recdata == nil then
		loadRecData()
	end
	for i = startIndex, #tbl do
		if lowerCase then
			if recData(tbl[i], field):lower() == value:lower() then
				return i
			end
		else
			if recData(tbl[i], field) == value then
				return i
			end
		end
	end

	if type(value) == "string" then
		local lowercaseValue = peg.lower(value)
		local val
		for i = startIndex, #tbl do
			val = recData(tbl[i], field)
			if val and peg.lower(val) == lowercaseValue then
				return i
			end
		end
	end

	return -1
end

function util.arrayRecord(valueToFind, recordArray, fieldName, startIndex)
	local idx = util.table_field_value_index(recordArray, fieldName, valueToFind, startIndex)
	if idx > 0 then
		return recordArray[idx], idx
	end
	return nil
end

function util.arrayRecordLowerCase(valueToFind, recordArray, fieldName, startIndex)
	local idx = util.table_field_value_index(recordArray, fieldName, valueToFind, startIndex, "lower")
	if idx > 0 then
		return recordArray[idx], idx
	end
	return nil
end

function util.table_field_value_record(tbl, field, value)
	for _, v in ipairs(tbl) do
		if v[field] == value then
			return v
		end
	end

	if type(value) == "string" then
		local lowercaseValue = peg.lower(value)
		for _, v in ipairs(tbl) do
			if v[field] and peg.lower(v[field]) == lowercaseValue then
				return v
			end
		end
	end

	return nil
end

function util.objectFieldToArr(data, fldName)
	local arr = {}
	for _, rec in pairs(data) do
		arr[#arr + 1] = rec[fldName]
	end
	return arr
end

function util.stringToNumberArray(soureString, separator, defaultValue)
	if separator == nil then
		local arr = {}
		for str in string.gmatch(soureString, "(%d+)") do
			arr[#arr + 1] = tonumber(str)
		end
		return arr
	end
	local arr = peg.splitToArray(soureString, separator)
	for i, str in ipairs(arr) do
		arr[i] = tonumber(str) or defaultValue
	end
	return arr
end

function util.table_value_remove(tbl, val)
	for i = #tbl, 1, -1 do -- must loop backwards
		if tbl[i] == val then
			table.remove(tbl, i)
		end
	end
end

function util.arrayAnd(arr1, arr2)
	local ret = {}
	local j = 0
	for _, val in ipairs(arr1) do
		if util.table_item_index(val, arr2) then
			j = j + 1
			ret[j] = val
		end
	end
	return ret
end

function util.arrayOr(arr1, arr2)
	-- this function adds all elements that are not found from arr1 to the end of arr1
	for _, val in ipairs(arr2) do
		if util.table_item_index(val, arr1) == nil then
			arr1[#arr1 + 1] = val
		end
	end
end

--[[
function util.arrayCombine(arr1, arr2)
	local combinedArr = util.clone(arr1)
	for _,val in ipairs(arr2) do
		combinedArr[#combinedArr+1] = val
	end
	return combinedArr
end
]]

function util.arrayCombine(arrTbl)
	loadFn()
	local combinedArr = {}
	local i = 0
	for _, tbl in ipairs(arrTbl) do
		-- if tbl ~= nil and #tbl > 0 then
		for _, rec in ipairs(tbl) do
			i = i + 1
			combinedArr[i] = rec
		end
		-- end
	end
	return combinedArr
end

function util.tableCombine(tbl1, tbl2, reportError, option)
	local combinedTbl
	if type(tbl1) ~= "table" then
		if reportError == "warning" then
			util.printWarningWithCallPath("table combine, table 1 type '%s' is not a table", type(tbl1))
		elseif reportError ~= "no-error" then
			util.printError("table combine, table 1 type '%s' is not a table", type(tbl1))
		end
		return tbl2 or {}
	end
	if type(tbl2) ~= "table" then
		if reportError == "warning" then
			util.printWarningWithCallPath("table combine, table 2 type '%s' is not a table", type(tbl2))
		elseif reportError ~= "no-error" then
			util.printError("table combine, table 2 type '%s' is not a table", type(tbl2))
		end
		return tbl1 or {}
	end
	if option and peg.found(option, "no-clone") then
		combinedTbl = tbl1
	else
		combinedTbl = util.clone(tbl1)
	end
	for key, val in pairs(tbl2) do
		if combinedTbl[key] then
			if option and peg.found(option, "deep") and type(val) == "table" and type(combinedTbl[key]) == "table" then -- or reportError == true then
				combinedTbl[key] = util.tableCombine(combinedTbl[key], val, reportError, option)
			elseif reportError == "warning" then -- or reportError == true then
				util.printWarning("table 1 key will be overriden by table 2 key: '" .. tostring(key) .. "', value: '" .. tostring(val) .. "'")
			elseif reportError == nil then -- or reportError == true then
				util.printError("table 1 key will be overriden by table 2 key: '" .. tostring(key) .. "', value: '" .. tostring(val) .. "'")
			end
		end
		combinedTbl[key] = val
	end
	return combinedTbl
end

function util.arrayToSubsetArray(arr, subsetArrSize)
	if subsetArrSize == nil or tonumber(subsetArrSize) <= 0 then
		return arr
	end
	local subsetArr = {}
	local subsetRec
	for _, rec in ipairs(arr) do
		if #subsetArr == 0 or #subsetArr >= subsetArrSize then
			subsetArr[#subsetArr + 1] = {}
			subsetRec = subsetArr[#subsetArr]
		end
		subsetRec[#subsetRec + 1] = rec
	end
	return subsetArr
end

-- http://lua-users.org/wiki/TableSerialization
--[[
   Author: Julio Manuel Fernandez-Diaz
   Date:   January 12, 2007
   (For Lua 5.1)

   Modified slightly by RiciLake to avoid the unnecessary table traversal in tablecount()

   Formats tables with cycles recursively to any depth.
   The output is returned as a string.
   References to other tables are shown as values.
   Self references are indicated.

   The string returned is "Lua code", which can be procesed
   (in the case in which indent is composed by spaces or "--").
   Userdata and function util.keys and values are shown as strings,
   which logically are exactly not equivalent to the original code.

   This routine can serve for pretty formating tables with
   proper indentations, apart from printing them:

      print(util.table_show(tbl, "tbl"))   -- a typical use

   Heavily based on "Saving tables with cycles", PIL2, p. 113.

   Arguments:
      tbl is the table.
      name is the name of the table (optional)
      indent is a first indentation (optional).
--]]

local _, tableIsEmpty = pcall(require, "table.isempty")
if type(tableIsEmpty) == "function" then
	function util.tableIsEmpty(tbl)
		if tbl == nil then
			util.printError(l("util.tableIsEmpty parameter table is nil"))
			return true
		end
		return tableIsEmpty(tbl)
	end
else
	function util.tableIsEmpty(tbl)
		if tbl == nil then
			util.printError(l("util.tableIsEmpty parameter table is nil"))
			return true
		end
		if type(tbl) ~= "table" then
			return true
		end
		return next(tbl) == nil
	end
	tableIsEmpty = util.tableIsEmpty
end

function util.table_show(tbl, name, indent)
	local cart -- a container
	local autoref -- for self references

	--[[ counts the number of elements in a table
   local function tablecount(tbl)
      local n = 0
      for _, _ in pairs(tbl) do n = n+1 end
      return n
   end
	]]
	-- (RiciLake) returns true if the table is empty

	local function basicSerialize(o)
		local so = tostring(o)
		if type(o) == "function" then
			local info = debug.getinfo(o, "S")
			-- info.name is nil because o is not a calling level
			if info.what == "C" then
				return stringFormat("%q", so .. ", C function")
			else
				-- the information is defined through lines
				return stringFormat("%q", so .. ", defined in (" .. info.linedefined .. "-" .. info.lastlinedefined .. ")" .. info.source)
			end
		elseif type(o) == "number" or type(o) == "boolean" then
			return so
		else
			return stringFormat("%q", so)
		end
	end

	local function addtocart(value, name2, indent2, saved, field)
		indent2 = indent2 or ""
		saved = saved or {}
		field = field or name2
		cart = cart .. indent2 .. field
		if type(value) ~= "table" then
			cart = cart .. " = " .. basicSerialize(value) .. ";\n"
		else
			if saved[value] then
				cart = cart .. " = {}; -- " .. saved[value] .. " (self reference)\n"
				autoref = autoref .. name2 .. " = " .. saved[value] .. ";\n"
			else
				saved[value] = name2
				-- if tablecount(value) == 0 then
				if tableIsEmpty(value) then
					cart = cart .. " = {};\n"
				else
					cart = cart .. " = {\n"
					for k, v in pairs(value) do
						k = basicSerialize(k)
						local fname = stringFormat("%s[%s]", name2, k)
						field = stringFormat("[%s]", k)
						-- three spaces between levels
						addtocart(v, fname, indent2 .. " ", saved, field)
					end
					cart = cart .. indent2 .. "};\n"
				end
			end
		end
	end

	name = name or "__unnamed__"
	if type(tbl) ~= "table" then
		return name .. " = " .. basicSerialize(tbl)
	end
	cart, autoref = "", ""
	addtocart(tbl, name, indent)
	return cart:sub(1, -2) .. autoref
end

function util.printToPreviousLine(txt)
	if util.from4d() then
		print(txt)
	else
		ioWrite(string.char(27) .. "[1A" .. "\r" .. txt) -- "\r" goes to start of current console line
		ioFlush()
		-- print(string.char(27).."[1A"..txt) -- goes up 1 line, but erases needed info - not good
	end
end

function util.printToSameLine(txt)
	if util.from4d() then
		print(txt)
	else
		ioWrite("\r" .. txt) -- "\r" goes to start of current console line
		ioFlush()
		-- print(string.char(27).."[1A"..txt) -- goes up 1 line, but erases needed info - not good
	end
end

function util.printTable(tbl, name, indent)
	print(util.table_show(tbl, name, indent))
end

function util.moveFile(fromFilePath, toFilePath)
	local command
	if ffi.os == "Windows" then
		command = 'MOVE "' .. fromFilePath .. '" "' .. toFilePath .. '"'
		command = command:gsub("/", "\\")
	else
		command = 'mv "' .. fromFilePath .. '" "' .. toFilePath .. '"'
	end
	return os.execute(command)
end

function util.stringParse(tbl, div, num)
	local ret = peg.splitToArray(tbl, div)
	if #ret >= num then
		return ret[num]
	end
	return nil
end

function util.parseKey(data, key, separator)
	separator = separator or "."
	local object = data
	local partTbl = peg.splitToArray(key, separator)
	for _, key2 in ipairs(partTbl) do
		object = object[key2]
		if object == nil then
			break
		end
	end
	return object
end

function util.endParse(tbl, div, num)
	local ret = peg.splitToArray(tbl, div)
	local ret_text = ""
	if #ret >= num then
		for i, val in ipairs(ret) do
			if i > num then
				if ret_text ~= "" then
					ret_text = ret_text .. div
				end
				ret_text = ret_text .. val
			end
		end
		return ret_text
	end
	return ""
end

function util.removeLastChar(txt, char)
	if txt:sub(-1) == char then
		return txt:sub(1, -1)
	end
	return txt
end

function util.removeEndPart(txt, divider)
	local tbl = peg.splitToArray(txt, divider)
	table.remove(tbl, #tbl)
	return concat(tbl, divider)
end

function util.removeEndText(txt, removeTxt)
	if txt:find(removeTxt, 1, true) then
		local reverseTxt = txt:reverse()
		local reverseRemoveTxt = removeTxt:reverse()
		if reverseTxt:find(reverseRemoveTxt, 1, true) == 1 then
			local charLength = removeTxt:len()
			reverseTxt = reverseTxt:sub(charLength + 1)
			txt = reverseTxt:reverse()
		end
		--[[
		txt = txt:sub(1, -#removeTxt)
		print("removeEndText: "..txt)
		]]
	end
	return txt
end

function util.removeStartText(txt, removeTxt)
	if txt:find(removeTxt, 1, true) then
		if txt:find(removeTxt, 1, true) == 1 then
			txt = txt:sub(removeTxt:len() + 1)
		end
	end
	return txt
end

function util.upperPath(path)
	if path:sub(-1) == "/" then
		path = path:sub(1, -2)
	end
	return util.removeEndPart(path, "/")
end

function util.checkPath(path)
	local p = util.removeEndText(path, "/")
	if not p then
		return nil, "path is not defined"
	else
		if not lfs then
			lfs = require "lfs_load"
		end
		local attr = lfs.attributes(p)
		if not attr or attr.mode ~= "directory" then
			return nil, "wrong path" .. " (" .. path .. ")"
		end
	end
	return p
end

function util.keyArrayFromStructure(source, ret, key)
	-- ret is a table passed by reference, it will be changed
	--[[
		Without proper tail calls, each user move would create a new stack level. After some number of moves, there would be a stack overflow. With proper tail calls, there is no limit to the number of moves that a user can make, because each move actually performs a goto to another function, not a conventional call.

	WARN: this is not proper tail-call version because last recursive line is not return!
	]]
	local function parseStructure(source2, ret2, keyId)
		for key2, val in pairs(source2) do
			if key2 == keyId then
				if util.isArray(val) == false then
					loadJson()
					util.printError(stringFormat("key '%s' value is not an array, json: %s", key2, json.toJson(val)))
					return
				end
				if not ret2[key2] then
					ret2[key2] = {}
				end
				for _, fld in ipairs(val) do
					local len = #ret2[key2] + 1
					-- print(stringFormat("table[%s][%d] = %s", keyId, len, tostring(fld)))
					ret2[key2][len] = fld
					if type(fld) == "table" then
						parseStructure(fld, ret2, keyId)
					end
				end
			elseif type(val) == "table" then
				parseStructure(val, ret2, keyId)
			end
		end
	end

	if type(key) == "table" then
		for _, keyId in ipairs(key) do
			parseStructure(source, ret, keyId)
		end
	else
		parseStructure(source, ret, key)
	end
end

function util.recordArrayToIdArray(array, idFieldName, deleteIdField)
	local ret = {}
	if #array > 0 then
		local rec = array[1]
		if rec[idFieldName] == nil then
			loadJson()
			util.printWarning(l("util.recordArrayToIdArray, field '%s' was not found from array[1]: %s", idFieldName, json.toJson(rec)))
		end
	end
	for _, rec in ipairs(array) do
		local id = rec[idFieldName]
		if id then
			if not ret[id] then
				ret[id] = {}
			end
			ret[id][#ret[id] + 1] = rec
			if deleteIdField and deleteIdField == true then
				ret[id][#ret[id]][idFieldName] = nil
			end
		end
	end
	return ret
end

function util.arrayTableToRecordTable(sel)
	local ret = {}
	if type(sel) == "table" then
		for key, arr in pairs(sel) do
			for i, rec in ipairs(arr) do
				if ret[i] == nil then
					ret[i] = {}
				end
				ret[i][key] = rec
			end
		end
	end
	return ret
end

--[[
function util.arrayReverse(arr)
	local count = #arr
	local half = math.floor(count / 2)
	for i = 1, half do
		arr[i], arr[count - i + 1] = arr[count - i + 1], arr[i]
	end
end
]]

function util.clearTableKeys(rec)
	for key in pairs(rec) do
		rec[key] = nil -- clear first all values
	end
end

function util.copyTableKeys(mainRec, table2, option)
	if option and option.clear == true then
		util.clearTableKeys(mainRec)
	end
	for key, val in pairs(table2) do
		mainRec[key] = val
	end
end

function util.mergeTables(table1, table2)
	if table2 == nil then
		return table1
	end
	local ret = util.clone(table1)
	for key, val in pairs(table2) do
		ret[key] = val
	end
	return ret
end

function util.recToRec(mainRec, rec, option)
	if recdata == nil then
		loadRecData()
	end
	if mainRec == nil then
		util.printError("destination record is nil")
	elseif rec == nil then
		util.printError("source record is nil")
	end
	local copyKey = option and option.copy_key
	local replace = true
	if option and option.replace == false then
		replace = false
	end
	for key, val in pairs(rec) do
		if replace then
			if option and option.deep and type(mainRec[key]) == "table" and type(val) == "table" then
				util.recToRec(mainRec[key], val, option)
			elseif peg.found(key, ".") then
				recDataSet(mainRec, key, val) -- , "use-tag"
			elseif key == "name" and copyKey == nil and mainRec[key] ~= val then
				util.printWarning("util.recToRec, key 'name' was copied from preference '%s' to '%s'", tostring(val), tostring(mainRec.name))
				mainRec[key] = val
			elseif copyKey then
				if copyKey[key] ~= false then
					mainRec[key] = val
				end
			elseif type(key) == "string" and key:sub(-1) ~= "-" then -- do not copy keys ending with '-'
				mainRec[key] = val
			end
		elseif mainRec[key] == nil then
			mainRec[key] = val
		elseif option.warn and mainRec[key] ~= val then
			util.printWarning("util.recToRec, key '%s', value '%s' was not copied over value '%s'", tostring(key), tostring(rec[key]), tostring(mainRec[key]))
			return true
		end
	end
end

--[[
function util.mergeStructureTables(table1, table2)
	local ret = util.clone(table1)
	for key,val in pairs(table2) do
		if type(val) == "table" then
			if type(ret[key]) == "table" then
				local lowerTbl = util.mergeStructureTables(ret[key], val)
				if type(lowerTbl) == "table" then
					ret[key] = lowerTbl
				end
			end
		else
			ret[key] = val
		end
	end
	return ret
end
]]

function util.concatenate(divider, ...)
	local arg = {...}
	local ret = ""
	for i = 1, #arg do
		if arg[i] ~= "" then
			if #ret > 0 then
				ret = ret .. divider
			end
			ret = ret .. arg[i]
		end
	end
	return ret
end

-- Checks if a specific bit in a value is set.
--- @param value number The value to check.
--- @param bitToCheck number The bit position to check.
--- @return boolean, true if the bit is set, false otherwise.
function util.checkBit(value, bitToCheck)
	local ret = band(value, lshift(1, bitToCheck)) > 0
	return ret
end

--- Calculates the remainder when `number` is divided by `mod`.
--- @param number number The number to be divided.
--- @param mod number The divisor.
--- @return number The remainder.
function util.modZero(number, mod)
	local a = number % mod
	return rshift(a - 1, 31)
end

function util.setRecordArrConcatField(arr, field, concatArr, fieldFormat)
	-- fieldFormat: {ordr.row_number = "%03d"}: 2 -> 002
	loadFn()
	-- fn.iter(arr):each(function(rec)
	for _, rec in ipairs(arr) do
		rec[field] = concat(fn.iter(concatArr):map(function(cval)
			if fieldFormat and fieldFormat[cval] then
				return stringFormat(fieldFormat[cval], rec[cval] or cval)
			end
			return rec[cval] or cval
		end):totable())
	end -- )
end

function util.parameterToTable(param)
	local paramTbl
	if type(param) == "table" then
		paramTbl = param
	elseif param ~= "" then
		loadJson()
		paramTbl = json.fromJson(param)
	end
	return paramTbl
end

function util.modulePrefixToFilename(prefix)
	local prf = util.prf("module_prefix.json")
	if prf ~= nil and prf[prefix] ~= nil then
		return prf[prefix]
	end
	return prefix
end

function util.confirmJson(param)
	if param == nil then
		return nil -- , l"parameter is missing"
	elseif param.type == nil then
		return nil
	end
	local jsonData = param
	if param.type == "info" then
		jsonData.window_title = l("Information")
	elseif param.type == "confirm" then
		jsonData.window_title = l("Confirm")
	elseif param.type == "warning" then
		jsonData.window_title = l("Warning")
	elseif param.type == "error" then
		jsonData.window_title = l("Error")
	elseif param.type == "fatal_error" then
		jsonData.window_title = l("Fatal error")
	else
		return nil
	end
	--[[
	jsonData.yes =
	jsonData.no =
	jsonData.cancel =
	]]

	return jsonData
end

function util.selectFieldsFromArray(data, fldNameArr)
	local ret = {}
	if fldNameArr == nil or #fldNameArr <= 0 then
		ret = data
	elseif #data > 0 then
		local findIdTbl = util.invertTable(fldNameArr)
		for _, rec in ipairs(data) do
			ret[#ret + 1] = {}
			for key, value in pairs(rec) do
				if findIdTbl[key] then
					ret[#ret][key] = value
				end
			end
		end
	end
	return ret
end

function util.arrayFieldMaxValue(data, fldName) -- if fldName = nil then data is single array
	local maxValue
	if data ~= nil and (fldName == nil or type(fldName) == "string") then
		for _, rec in ipairs(data) do
			if fldName == nil then
				if maxValue == nil then
					maxValue = rec
				elseif maxValue < rec then
					maxValue = rec
				end
			else
				if rec[fldName] ~= nil then
					if maxValue == nil then
						maxValue = rec[fldName]
					elseif maxValue < rec[fldName] then
						maxValue = rec[fldName]
					end
				end
			end
		end
	end
	return maxValue
end

function util.fieldValueToArray(data, fldName) -- was: util.recordArrayFieldValueToDistinctArray(data, fldName)
	loadFn()
	return fn.util.mapFieldValue(data, fldName):toDistinctArr()
end

function util.removeChildStructurefromRecordTable(data)
	for _, rec in ipairs(data) do
		for key, _ in pairs(rec) do
			if type(rec[key]) == "table" then
				rec[key] = nil
			end
		end
	end
end

--[=[function util.folderToStructureTable(path, mainFolder, findArray)
	-- for filename, attr in fs.dirTreeIter(path, {suffix = ".json"}) do
		--[[
		if peg.found(filename, ".json")
			local path, name = peg.splitLast(filename, "/")
			path =peg.removeFromStart(path, pathPrefix)
			filename =peg.removeFromStart(filename, pathPrefix)
			name = peg.replace(name, ".json", "")
			if fileTbl[path] == nil then
				fileTbl[path] = {}
			end
			fileTbl[path][#fileTbl[path]+1] = {value = filename, show = name}
		end
		]]
	-- end
end
--]=]

function util.createFileStructure(prfTagName)
	local errTbl
	loadFn()
	loadFs()
	local fileTbl = {}
	if type(prfTagName) ~= "string" then
		return nil, nil, l("parameter must be 'create_file_structure.json' tag name")
	end
	local prf = util.prf("create_file_structure.json", "no-cache")
	if prf[prfTagName] == nil then
		return nil, nil, l("'%s' tag does not exist in 'create_file_structure.json' preference", prfTagName)
	elseif type(prf[prfTagName].path) ~= "table" then
		return nil, nil, l("'path' tag must be array in 'create_file_structure.json' preference")
	end
	local pathPrefix = util.runPath() .. "preference/"
	fn.iter(prf[prfTagName].path):each(function(folderPath)
		local dirExist, err = fs.dirExists(pathPrefix .. folderPath)
		if not dirExist then
			if errTbl == nil then
				errTbl = {}
			end
			errTbl[#errTbl + 1] = err
		else
			-- fileTbl[folderPath] = {}
			-- local pathTbl = fileTbl[folderPath]
			for filename in fs.dirTreeIter(pathPrefix .. folderPath, {suffix = ".json"}) do
				local path, name = peg.splitLast(filename, "/")
				-- path = peg.removeFromStart(path, pathPrefix)
				-- filename = peg.removeFromStart(filename, pathPrefix)
				name = peg.replace(name, ".json", "")
				if fileTbl[path] == nil then
					fileTbl[path] = {}
				end
				fileTbl[path][#fileTbl[path] + 1] = {value = filename, show = name}
			end
		end
	end)

	--[[
	local structTbl = {}
	for folder, fileArr pairs(fileTbl) do
		local upperFolder, name = peg.splitLast(folder, "/")
		if name == nil or name == "" then

		end

		fn.iter(fileTbl):each(function(fileArr)
			local upperFolder, name = peg.splitLast(folder, "/")

		end)

	end
	]]

	if errTbl then
		return fileTbl, prf[prfTagName].preference_name, concat(errTbl, "\n")
	else
		return fileTbl, prf[prfTagName].preference_name
	end
end

function util.checkOption(option, optionText)
	if option and peg.find(option, optionText) > 0 then
		return true
	end
	return false
end

function util.escapeSeparator(txt)
	return peg.replace(peg.replace(txt, "/", "|"), "-", "–")
end

function util.oppositeNumber(num)
	if num and type(num) == "number" then
		return num * -1
	end
	return num
end

-- time.init(util)
for key, val in pairs(time) do
	if key ~= "init" and key ~= "sleep" then
		util[key] = val -- copy time functions to util
	end
end

local rsihft = bit.rshift
local hexTable = {"0", "1", "2", "3", "4", "5", "6", "7", "8", "9", "a", "b", "c", "d", "e", "f"}
function util.numberToHex(num)
	-- see: https://www.rapidtables.com/convert/number/decimal-to-hex.html, it has correct results and explanation how to calculate
	if num == 0 then
		return '0'
	end
	local neg = false
	if num < 0 then
		neg = true
		num = num * -1
	end
	local result = ""
	local n
	repeat
		n = tonumber(num % 16)
		result = hexTable[n + 1] .. result -- we can't use table here because result is build on reverse order and we dont know loop count
		num = rsihft(num, 4) -- floor(tonumber(num / 16)), but floor() does not work always with big ULL numbers
	until num <= 0
	if neg then
		return '-' .. result
	end
	return result
end

--[=[ -- this version is slower than previous one: 0.69881312500002 sec vs 0.9418683340009 sec
function util.numberToHex2(num)
	if num == 0 then
		return '0'
	end
	local neg = false
	if num < 0 then
		neg = true
		num = num * -1
	end
	local i = 0
	local index = {}
	local n
	while num > 0 do
		i = i + 1
		n = tonumber(num % 16)
		-- result = hexTable[n + 1] .. result -- we can't use table here because result is build on reverse order
		index[i] = n + 1
		num = rsihft(num, 4) -- floor(tonumber(num / 16)), but this dows not work always with big ULL numbers
	end
	local result = {}
	for j = 1, i  do
		result[j] = hexTable[index[i - j + 1]]
	end
	if neg then
		return '-' .. concat(result)
	end
	return concat(result)
end ]=]

--[=[
ffi.cdef [[
// int sprintf(char *str, const char *format, ...);
// int sprintf(char *__restrict __s, __const char *__restrict __format, ...) __attribute__ ((__nothrow__));
int snprintf(char * s, size_t n, const char * format, ...);
int _snprintf(char * s, size_t n, const char * format, ...);
]]
local snprintf
if util.isWin() then
	snprintf = C._snprintf
else
	snprintf = C.snprintf
end
local ffiString = ffi.string
local numberToHexBuf = ffi.newAnchor("char[?]", 17)
function util.numberULLToHex(num)
	snprintf(numberToHexBuf, 17, "%016llx", num) -- this is much faster than util.numberToHex(), but does not work all the time correctly
	return ffiString(numberToHexBuf, 16)
end
-- ]=]

return util
