--- lib/uuid.lua
---------------------------------------------------------------------------------------
-- Copyright 2012 Rackspace (original), 2013 Thijs Schreijer (modifications)
--
-- Licensed under the Apache License, Version 2.0 (the "License");
-- you may not use this file except in compliance with the License.
-- You may obtain a copy of the License at
--
--     http://www.apache.org/licenses/LICENSE-2.0
--
-- Unless required by applicable law or agreed to in writing, software
-- distributed under the License is distributed on an "AS-IS" BASIS,
-- WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
-- See the License for the specific language governing permissions and
-- limitations under the License.
--
-- see http://www.ietf.org/rfc/rfc4122.txt
--
-- Note that this is not a true version 4 (random) UUID.  Since `os.time()` precision is only 1 second, it would be hard
-- to guarantee spacial uniqueness when two hosts generate a uuid after being seeded during the same second.  This
-- is solved by using the node field from a version 1 UUID.  It represents the mac address.
--
-- 28-apr-2013 modified by Thijs Schreijer from the original [Rackspace code](https://github.com/kans/zirgo/blob/807250b1af6725bad4776c931c89a784c1e34db2/util/uuid.lua) as a generic Lua module.
-- Regarding the above mention on `os.time()`; the modifications use the `socket.gettime()` function from LuaSocket
-- if available and hence reduce that problem (provided LuaSocket has been loaded before uuid).
-- local ffi = require "mffi"
local util = require "util"
local epoch = require"time".epoch
local randomBuf, bin2hex -- delay load
local fmod, floor, random = util.fmod, math.floor, math.random

local seed = util.seconds()
seed = floor(seed * 10000) -- used later for randomseed

local mac
local function macAddress()
	mac = require"system/net".getMacAddress()
	mac = mac:gsub(":", "")
	return mac
end

local uuid = {}
-- local math = require('math')
-- local os = require('os')
-- local string = require('string')

-- local randomBuffer, hexBuffer
function uuid.cryptoRandomUuid() -- crypto strength random uuid
	if randomBuf == nil then
		local sodium = require "luasodium/sodium"
		randomBuf = sodium.randombytes_buf
		bin2hex = sodium.sodium_bin2hex
	end
	local bufStr = bin2hex(randomBuf(16))
	return bufStr:sub(1, 8) .. "-" .. bufStr:sub(9, 12) .. "-" .. bufStr:sub(13, 16) .. "-" .. bufStr:sub(17, 20) .. "-" .. bufStr:sub(21) -- format like uuid
end

local bitsize = 32 -- bitsize assumed for Lua VM. See randomseed function below.
local lua_version = tonumber(_VERSION:match("%d%.*%d*")) -- grab Lua version used

local MATRIX_AND = {{0, 0}, {0, 1}}
local MATRIX_OR = {{0, 1}, {1, 1}}
local HEXES = '0123456789abcdef'

-- performs the bitwise operation specified by truth matrix on two numbers.
local function BITWISE(x, y, matrix)
	local z = 0
	local pow = 1
	while x > 0 or y > 0 do
		z = z + (matrix[x % 2 + 1][y % 2 + 1] * pow)
		pow = pow * 2
		x = floor(x / 2)
		y = floor(y / 2)
	end
	return z
end

local function INT2HEX(x)
	local s, base = '', 16
	local d
	while x > 0 do
		d = fmod(x, base) + 1
		x = floor(x / base)
		s = HEXES:sub(d, d) .. s
	end
	while #s < 2 do
		s = "0" .. s
	end
	return s
end

----------------------------------------------------------------------------
-- Creates a new uuid. Either provide a unique hex string, or make sure the
-- random seed is properly set. The module table itself is a shortcut to this
-- function, so `my_uuid = uuid.new()` equals `my_uuid = uuid()`.
--
-- For proper use there are 3 options;
--
-- 1. first require `luasocket`, then call `uuid.seed()`, and request a uuid using no
-- parameter, eg. `my_uuid = uuid()`
-- 2. use `uuid` without `luasocket`, set a random seed using `uuid.randomseed(some_good_seed)`,
-- and request a uuid using no parameter, eg. `my_uuid = uuid()`
-- 3. use `uuid` without `luasocket`, and request a uuid using an unique hex string,
-- eg. `my_uuid = uuid(my_networkcard_macaddress)`
--
-- @return a properly formatted uuid string
-- @param hwaddr (optional) string containing a unique hex value (e.g.: `00:0c:29:69:41:c6`), to be used to compensate for the lesser `random()` function. Use a mac address for solid results. If omitted, a fully randomized uuid will be generated, but then you must ensure that the random seed is set properly!
-- @usage
-- local uuid = require("uuid")
-- print("here's a new uuid: ",uuid())
function uuid.new() -- hwaddr)
	-- bytes are treated as 8bit unsigned bytes.
	local epochTime = epoch()
	local bytes = {
		0, -- random(0, 255),
		0, -- random(0, 255),
		0, -- random(0, 255),
		0, -- random(0, 255),
		random(0, 255),
		random(0, 255),
		random(0, 255),
		random(0, 255),
		random(0, 255),
		random(0, 255)
		-- random(0, 255),
		-- random(0, 255),
		-- random(0, 255),
		-- random(0, 255),
		-- random(0, 255),
		-- random(0, 255),
	}
	--[[
  if false and hwaddr then
    assert(type(hwaddr)=="string", "Expected hex string, got "..type(hwaddr))
    -- Cleanup provided string, assume mac address, so start from back and cleanup until we've got 12 characters
    local i, str, hwaddr = #hwaddr, hwaddr, ""
    while i>0 and #hwaddr<12 do
      local c = str:sub(i,i):lower()
      if HEXES:find(c, 1, true) then
        -- valid HEX character, so append it
        hwaddr = c..hwaddr
      end
      i = i - 1
    end
    if #hwaddr ~= 12 then
			util.printError("Provided string did not contain at least 12 hex characters, retrieved '"..hwaddr.."' from '"..str.."'")
		end

    -- no split() in lua. :(
    bytes[11] = tonumber(hwaddr:sub(1, 2), 16)
    bytes[12] = tonumber(hwaddr:sub(3, 4), 16)
    bytes[13] = tonumber(hwaddr:sub(5, 6), 16)
    bytes[14] = tonumber(hwaddr:sub(7, 8), 16)
    bytes[15] = tonumber(hwaddr:sub(9, 10), 16)
    bytes[16] = tonumber(hwaddr:sub(11, 12), 16)
  end
	]]

	-- set the version
	bytes[7] = BITWISE(bytes[7], 0x0f, MATRIX_AND)
	bytes[7] = BITWISE(bytes[7], 0x40, MATRIX_OR)
	-- set the variant
	bytes[9] = BITWISE(bytes[7], 0x3f, MATRIX_AND)
	bytes[9] = BITWISE(bytes[7], 0x80, MATRIX_OR)
	local start = string.format("%x", epochTime) -- INT2HEX(bytes[1])..INT2HEX(bytes[2])..INT2HEX(bytes[3])..INT2HEX(bytes[4])
	if #start ~= 8 then
		util.printError("epoch time string '%s' length is not 8 hex characters", start)
		-- else
		-- util.printInfo("epoch time string '%s' - '%d'", start, epochTime)
	end
	return (start .. "-" .. INT2HEX(bytes[5]) .. INT2HEX(bytes[6]) .. "-" .. INT2HEX(bytes[7]) .. INT2HEX(bytes[8]) .. "-" .. INT2HEX(bytes[9]) .. INT2HEX(bytes[10]) .. "-" .. (mac or macAddress()))
	--[[		INT2HEX(bytes[11])..INT2HEX(bytes[12])..INT2HEX(bytes[13])..INT2HEX(bytes[14])..INT2HEX(bytes[15])..INT2HEX(bytes[16])
				 ]]
end

----------------------------------------------------------------------------
-- Improved randomseed function.
-- Lua 5.1 and 5.2 both truncate the seed given if it exceeds the integer
-- range. If this happens, the seed will be 0 or 1 and all randomness will
-- be gone (each application run will generate the same sequence of random
-- numbers in that case). This improved version drops the most significant
-- bits in those cases to get the seed within the proper range again.
-- @param seed the random seed to set (integer from 0 - 2^32, negative values will be made positive)
-- @return the (potentially modified) seed used
-- @usage
-- local socket = require("socket")  -- gettime() has higher precision than os.time()
-- local uuid = require("uuid")
-- -- see also example at uuid.seed()
-- uuid.randomseed(socket.gettime()*10000)
-- print("here's a new uuid: ",uuid())
local function randomseed()
	seed = floor(math.abs(seed))
	if seed >= (2 ^ bitsize) then
		-- integer overflow, so reduce to prevent a bad seed
		seed = seed - floor(seed / 2 ^ bitsize) * (2 ^ bitsize)
	end
	if lua_version < 5.2 then
		-- 5.1 uses (incorrect) signed int
		math.randomseed(seed - 2 ^ (bitsize - 1))
	else
		-- 5.2 uses (correct) unsigned int
		math.randomseed(seed)
	end
	return seed
end

----------------------------------------------------------------------------
-- Seeds the random generator.
-- It does so in 2 possible ways;
--
-- 1. use `os.time()`: this only offers resolution to one second (used when
-- LuaSocket hasn't been loaded yet
-- 2. use luasocket `gettime()` function, but it only does so when LuaSocket
-- has been required already.
-- @usage
-- local socket = require("socket")  -- gettime() has higher precision than os.time()
-- -- LuaSocket loaded, so below line does the same as the example from randomseed()
-- uuid.seed()
-- print("here's a new uuid: ",uuid())

--[[
function uuid.seed()
  if package.loaded["socket"] and package.loaded["socket"].gettime then
    return uuid.randomseed(package.loaded["socket"].gettime()*10000)
  else
    return uuid.randomseed(os.time())
  end
end
]]

randomseed() -- 2 ** 32
-- print(uuid.new(macAddress))
-- print(uuid.new())
-- print(uuid.new(macAddress))

return uuid -- setmetatable(uuid, { __call = function(self, hwaddr) return self.new(hwaddr) end})
