--- web.lua
-- E-mail, ftp and other web protocol functions.
-- @module web

local util = require "util"
local dt = require "dt"
local peg = require "peg"
local uuid = require "uuid"
local curl = require "curl"

local defaultEmailParam

local function setEmailParam(param)
	--[[
	param = {
		from = string,
		to = string or string-table,
		cc = string or string-table,
		bcc = string or string-table,
		body = string,
		[username = string,]
		[password = string,]
		[server = string,]
		[domain = string,]
	}
	]]
	if not defaultEmailParam then
		defaultEmailParam = util.prf("auth/preference.json", "no-cache")
		if util.tableIsEmpty(defaultEmailParam) or not defaultEmailParam.email then
			return
		end
		defaultEmailParam = defaultEmailParam.email
	end
	return util.tableCombine(defaultEmailParam, param, "no-error")
end

local function cleanEmailAddress(addr)

	local function clean(addr)
		local pos1 = peg.find(addr, "<")
		if pos1 < 1 then
			return addr
		end
		local pos2 = peg.find(addr, ">")
		if pos2 < 1 then
			return addr
		end
		return addr:sub(pos1+1, pos2-1)
	end

	if not addr or addr == "" then
		return addr
	elseif type(addr) == "table" then
		local ret = {}
		for i,txt in ipairs(addr) do
			ret[i] = clean(txt)
		end
		return ret
	else -- is text
		return clean(addr)
	end
end

local function sendEmail(param_)
	local param = setEmailParam(param_)
	local now = dt.toRfc2822(dt.currentDateTime())
	local id = "<"..uuid.new().."@"..param.domain..">"
	param.body = param.body:gsub("\n", "\r\n") or ""

	local from, to, cc, subject
	from = param.from or ""
	if type(param.to) == "table" then
		to = table.concat(param.to, ", ")
	else
		to = param.to or ""
	end
	if type(param.cc) == "table" then
		cc = table.concat(param.cc, ", ")
	else
		cc = param.cc or ""
	end
	subject = param.subject or ""

	local data = {
		"Date: "..now,
		"From: "..from,
		"To: "..to,
		"Cc: "..cc, -- no bcc here, please
		"Message-ID: "..id,
		"Subject: "..subject,
		"", -- empty line to divide headers from body, see RFC5322
		param.body,
	}
	if not param.cc or param.cc == "" then
		table.remove(data, 4)
	end
	if not param.to or param.to == "" then
		table.remove(data, 3)
	end
	if not param.from or param.from == "" then
		table.remove(data, 2)
	end
	param.body = table.concat(data, "\r\n")

	param.from = cleanEmailAddress(param.from)
	param.to = cleanEmailAddress(param.to)
	param.cc = cleanEmailAddress(param.cc)
	param.bcc = cleanEmailAddress(param.bcc)

	local ret, err, retHeader = curl.sendEmail(param)
	if err then
		util.printError(err)
	end
	return ret, err, retHeader
end

return {
	sendEmail = sendEmail,

	escape = curl.escape,
	getUrl = curl.getUrl,
	ftpFileList = curl.ftpFileList,
	deleteFtpFile = curl.deleteFtpFile,
	moveFtpFile = curl.moveFtpFile,
	sendFtpFile = curl.sendFtpFile,
	receiveFtpFile = curl.receiveFtpFile,
	receiveFtpList = curl.receiveFtpList,
}
