--- lib/xml/docx.lua
-- copied and adapted from https://github.com/paragasu/lua-docx
-- Replace docx tag with string value
-- Author: Jeffry L <paragasu@gmail.com>
local zip = require "compress/minizip2"
local util = require "util"
local peg = require "peg"
local fs = require "fs"

local docx = {}
docx.__index = docx

local function escapeXml(str)
	local xml_chars = {
		['<'] = "&#60;",
		['>'] = "&#62;",
		['&'] = "&#38;",
		["'"] = "&#39;", -- single quote
		['"'] = "&#34;" -- double quote
	}
	return string.gsub(str, '[<>&\'"]', xml_chars)
end

local function escapeXmlChars(tags)
	local data = {}
	for tag, value in pairs(tags) do
		data[tag] = escapeXml(value)
	end
	return data or ""
end

-- @param string filePath full filename path to the docx template
-- @param string tmp directory to process the file
--        using /tmp end up with Renaming temporary file failed: Operation not permitted
--        util.printError. But using other directory is fine.
function docx.new(destinationDir)
	if not destinationDir then
		util.printError("Writable destination directory was not provided")
		return
	end
	if destinationDir == "" then
		util.printError("Writable destination directory path is empty")
		return
	end
	if not fs.dirExists(destinationDir) then
		-- return util.printError(destinationDir .. " does not exist")
		fs.createPath(destinationDir)
	end
	local self = setmetatable({}, docx)
	self.dest_dir = peg.addToEnd(destinationDir, "/")
	util.print("  destination directory: " .. self.dest_dir)
	return self
end

function docx.replaceTags(self, filePath, tagsOnce, tagsAll)
	local time = util.seconds()
	local err = docx.validateFile(filePath)
	if err then
		return nil, err
	end
	-- file must be writeable
	self.docx = filePath
	if not fs.fileExists(self.docx) then
		return nil, util.printError(self.docx .. " does not exist")
	end
	local newFolderPath = self.dest_dir .. peg.parseAfterLast(filePath, "/")
	newFolderPath = peg.replace(newFolderPath, ".docx", "-replaced")
	-- fs.copyFile(filePath, newFolderPath)
	if fs.exists(newFolderPath .. ".docx") then
		fs.deletePath(newFolderPath)
		-- return nil, util.printError(newFolderPath .. ".docx already exists")
	end
	if fs.exists(newFolderPath) then
		fs.deletePath(newFolderPath)
		-- return nil, util.printError(newFolderPath .. " already exists")
	end
	util.print("  opening original document '%s' as zip file...\n", self.docx)
	self.ar = zip.open({file = self.docx})
	self.ar:extract_all(newFolderPath)
	self.new_content = {}
	self.replaced = {}
	self.escaped_tags_once = escapeXmlChars(tagsOnce)
	self.escaped_tags_all = escapeXmlChars(tagsAll)
	docx.replaceDocxDocument(self)
	docx.replaceDocxHeader(self)
	docx.replaceDocxFooter(self)
	self.ar:close()
	-- for entry in self.ar:entries() do
	local arr = {}
	for key in pairs(tagsOnce) do
		if self.replaced[key] == nil and not peg.found(key, "_2") then
			arr[#arr + 1] = key:sub(3, -3)
		end
	end
	for key in pairs(tagsAll) do
		if self.replaced[key] == nil and not peg.found(key, "_2") then
			arr[#arr + 1] = key:sub(3, -3)
		end
	end
	if #arr > 0 then
		table.sort(arr)
		util.print("  not replaced tags: %s", table.concat(arr, ", "))
	end
	local notReplacedCount
	for path, content in pairs(self.new_content) do
		notReplacedCount = peg.countOccurrenceFromTable(self.new_content, path, "{{")
		if notReplacedCount > 0 then
			util.printRed("  %d tags were not replaced in document '%s'", notReplacedCount, path)
		else
			util.printOk("  all tags were replaced in document '%s'", path)
		end
		fs.writeFile(newFolderPath .. "/" .. path, content)
	end
	-- write new content to the zip
	self.docx = newFolderPath .. ".docx"
	self.ar = zip.open({file = self.docx, mode = "w"})
	-- self.ar.aes = false
	self.ar:add_all(newFolderPath)
	self.ar:close()
	self.ar = nil
	fs.deletePath(newFolderPath)
	util.printOk("  replaced document '%s' tags in %.3f seconds", self.docx, util.seconds(time))
	return newFolderPath
end

function docx.replaceDocxDocument(self)
	local documentIdx = self.ar:find("word/document.xml", true)
	self.new_content["word/document.xml"] = docx.replaceDocxXmlContent(self, documentIdx)
end

function docx.replaceDocxHeader(self)
	local zipElementFound
	local num = 0
	repeat
		num = num + 1
		zipElementFound = self.ar:find("word/header" .. num .. ".xml", true)
		if zipElementFound then
			self.new_content["word/header" .. num .. ".xml"] = docx.replaceDocxXmlContent(self, zipElementFound)
		end
	until zipElementFound == false
end

function docx.replaceDocxFooter(self)
	local zipElementFound
	local num = 0
	repeat
		num = num + 1
		zipElementFound = self.ar:find("word/footer" .. num .. ".xml", true)
		if zipElementFound then
			self.new_content["word/footer" .. num .. ".xml"] = docx.replaceDocxXmlContent(self, zipElementFound)
		end
	until zipElementFound == false
end

local function replaceXmlTags(self, tpl)
	local orig
	for key, val in pairs(self.escaped_tags_all) do
		--[[
		if peg.found(key, "profit_loss_for_the_financial_year") then
			print(key, val)
		end -- ]]
		orig = tpl
		tpl = peg.replace(tpl, key, val)
		if orig ~= tpl then
			self.replaced[key] = true
		end
	end
	for key, val in pairs(self.escaped_tags_once) do
		--[[ if peg.found(key, "total_intangible_assets") then
			print("total_intangible_assets", val)
		end ]]
		orig = tpl
		tpl = peg.replaceOnce(tpl, key, val)
		if orig ~= tpl then
			self.replaced[key] = true
		end
	end
	return tpl
	-- return string.gsub(tpl, "#%a+%.[%a%s%d]+#", self.escaped_tags_once) or ""
end

-- get the content of xml file inside the zip
function docx.replaceDocxXmlContent(self, idx)
	if idx == false then
		util.printError("file not found in docx archive")
		return
	end
	--[[ local file = self.ar:open_entry(idx)
	local stat = self.ar:stat(idx)
	local tpl = file:read(stat.size) ]]
	-- print(self.ar.entry.uncompressed_size)
	local tpl = self.ar:read("*a")
	local replaced = replaceXmlTags(self, tpl)
	return replaced ~= tpl and replaced or nil
	-- self.ar:replace(idx, "string", replaced)
	-- self.ar:close_entry() -- file:close()
end

function docx.validateFile(filePath)
	if type(filePath) ~= "string" then
		return util.printError("Invalid docx file")
	end
	if not string.match(filePath, "%.docx") then
		return util.printError("Only docx file is supported, file '%s'", filePath)
	end
	if string.find(filePath, "%.%/") then
		return util.printError("Relative path using ./ is not supported, file '%s'", filePath)
	end
	if string.find(filePath, "%~%/") then
		return util.printError("Relative path using ~/ is not supported, file '%s'", filePath)
	end
	if not fs.fileExists(filePath) then
		return util.printError("File does not exist, file '%s' ", filePath)
	end
end

return docx
