--- xml.lua
-- xml helper functions
local xmlWrite = require "xml/xml_write"
local expat = require "xml/expat"
local peg = require "peg"

local P, C, Ct = peg.pattern, peg.capture, peg.Ct

local possibleStartMarks = P "\n\r" + P "\r\n" + P "\n" + P "\t"
local apostrophe = P "\""
local notApostrophe = (1 - apostrophe)

local function findEnd(fileTxt, string)
	string = P(string)
	local notString = (1 - string)
	local patt = possibleStartMarks ^ 0 * notString ^ 0 * string ^ -1
	return patt:match(fileTxt) - 1
end

local function findXmlTagWithAttribute(fullText, tag, attribute)
	local nameTag = P "<" * P(tag)
	local elementEndTag = P "</" * P(tag) * P ">"
	local anotherElementEnd = P "/>"
	local elementEnd = elementEndTag + anotherElementEnd
	local notElementEnd = (1 - elementEnd)
	local elementWithoutAttribute = nameTag * notElementEnd ^ 0 * elementEnd ^ 1
	local notElementWithoutAttribute = (1 - elementWithoutAttribute)
	local elem = possibleStartMarks ^ 0 * notElementWithoutAttribute ^ 0 * C(elementWithoutAttribute)
	local patt = Ct(elem ^ 0)
	local tags = patt:match(fullText)
	local elementTxt
	for _, val in ipairs(tags) do
		local startPlace = string.find(val, " " .. attribute .. "=")
		if startPlace then
			elementTxt = val
			break
		end
	end

	local startPos, endPos
	if elementTxt then
		endPos = findEnd(fullText, elementTxt)
		startPos = endPos - elementTxt:len()
		-- print("element: ", elementTxt)
		-- print("endPos: ", endPos)
	end

	return elementTxt, startPos, endPos
end

local function captureXmlAttribute(fullText, attr)
	local nameAttribute = P(attr .. "=\"")

	local fullNameAttribute = nameAttribute ^ 1 * notApostrophe ^ 0 * apostrophe ^ 1
	local notFullNameAttribute = (1 - fullNameAttribute)

	local patt = possibleStartMarks ^ 0 * notFullNameAttribute ^ 0 * C(fullNameAttribute ^ 1) -- C(nameAttribute^1)--
	return patt:match(fullText)
end

local function xmlFileToTable(pathToXml)
	return expat.treeParse({path = pathToXml})
end

local function xmlStringToTable(xmlString)
	return expat.treeParse({string = xmlString})
end

local function escapeXml(txt)
	-- http://stackoverflow.com/questions/1091945/what-characters-do-i-need-to-escape-in-xml-documents
	txt = peg.replace(txt, "&", '&amp;') -- MUST be first to replace, following replaces contain &
	txt = peg.replace(txt, '"', '&quot;')
	txt = peg.replace(txt, "'", '&apos;')
	txt = peg.replace(txt, "<", '&lt;')
	return peg.replace(txt, ">", '&gt;')
end

return {
	read = expat,
	write = xmlWrite,
	xmlStringToTable = xmlStringToTable,
	xmlFileToTable = xmlFileToTable,
	captureXmlAttribute = captureXmlAttribute,
	-- captureIdAttribute = captureIdAttribute,
	findXmlTagWithAttribute = findXmlTagWithAttribute,
	findEnd = findEnd,
	escapeXml = escapeXml
}
