--- xmlschema.lua
-- see: https://github.com/lsh123/xmlschema/blob/master/src/soap.c

local xmlschema = {}

local ffi = require "mffi"
local util = require "util"
local xml = require "xmlsec".xml -- definitions and libs here

ffi.cdef[[
typedef struct _xmlSchema xmlSchema;
typedef xmlSchema *xmlSchemaPtr;

typedef struct _xmlSchemaParserCtxt xmlSchemaParserCtxt;
typedef xmlSchemaParserCtxt *xmlSchemaParserCtxtPtr;

typedef struct _xmlSchemaValidCtxt xmlSchemaValidCtxt;
typedef xmlSchemaValidCtxt *xmlSchemaValidCtxtPtr;

/**
 * xmlSchemaValidityErrorFunc:
 * @ctx: the validation context
 * @msg: the message
 * @...: extra arguments
 *
 * Signature of an error callback from an XSD validation
 */
// typedef void (*xmlSchemaValidityErrorFunc) (void *ctx, const char *msg, ...) ; // LIBXML_ATTR_FORMAT(2,3);
typedef void (*xmlSchemaValidityErrorFunc) (void *ctx, const char *msg, const char *msg2, const char *msg3, const char *msg4) ; // LIBXML_ATTR_FORMAT(2,3);
// typedef void (*xmlSchemavalidityWarningFunc) (void *ctx, const char *msg, ...); // LIBXML_ATTR_FORMAT(2,3);
typedef void (*xmlSchemavalidityWarningFunc) (void *ctx, const char *msg, const char *msg2, const char *msg3, const char *msg4); // LIBXML_ATTR_FORMAT(2,3);
void xmlSchemaSetParserErrors	(xmlSchemaParserCtxtPtr ctxt,
					 xmlSchemaValidityErrorFunc err,
					 xmlSchemavalidityWarningFunc warn,
					 void *ctx);
void xmlSchemaSetValidErrors	(xmlSchemaValidCtxtPtr ctxt,
					 xmlSchemaValidityErrorFunc err,
					 xmlSchemavalidityWarningFunc warn,
					 void *ctx);

xmlSchemaParserCtxtPtr xmlSchemaNewParserCtxt(const char *URL);
xmlSchemaPtr xmlSchemaParse(xmlSchemaParserCtxtPtr ctxt);
void xmlSchemaFreeParserCtxt(xmlSchemaParserCtxtPtr ctxt);
xmlSchemaValidCtxtPtr xmlSchemaNewValidCtxt(xmlSchemaPtr schema);
int xmlSchemaValidateDoc(xmlSchemaValidCtxtPtr ctxt, xmlDocPtr instance);
void xmlSchemaFreeValidCtxt(xmlSchemaValidCtxtPtr ctxt);
void xmlSchemaFree(xmlSchemaPtr schema);
void xmlSchemaCleanupTypes(void);
]]

-- local callbackCtxPtr = ffi.cast("void *", callbackCtx)
local validityError = nil
local validityWarning = nil

local function formatMsg(msg, msg2, msg3, msg4)
	local formatStr = ffi.string(msg)
	local _, argCount = string.gsub(formatStr, "%%", "") -- count occurences of '%', usually formatStr is "%s"
	local msg
	if argCount == 1 then
		msg = string.format(formatStr, ffi.string(msg2))
	elseif argCount == 2 then
		msg = string.format(formatStr, ffi.string(msg2), ffi.string(msg3))
	else
		if argCount > 3 then
			util.printError("more than 3 parameters in xmlschema callback")
		end
		msg = string.format(formatStr, ffi.string(msg2), ffi.string(msg3), ffi.string(msg4))
	end
	if msg:sub(-1) == "\n" then -- remove trailing \n
		msg = msg:sub(1, -3)
	end
	return msg
end

local function xmlSchemavalidityWarningFunc(_, msg, msg2, msg3, msg4)
	validityWarning = formatMsg(msg, msg2, msg3, msg4)
	-- util.printWarning(validityWarning)
end
local xmlSchemavalidityWarningFuncPointer = ffi.cast("xmlSchemavalidityWarningFunc (*)(void *ctx, const char *msg, const char *msg2, const char *msg3, const char *msg4)", xmlSchemavalidityWarningFunc)

local function xmlSchemaValidityErrorFunc(_, msg, msg2, msg3, msg4)
	validityError = formatMsg(msg, msg2, msg3, msg4)
	-- util.printError(validityError)
end
local xmlSchemaValidityErrorFuncPointer = ffi.cast("xmlSchemaValidityErrorFunc (*)(void *ctx, const char *msg, const char *msg2, const char *msg3, const char *msg4)", xmlSchemaValidityErrorFunc)


function xmlschema.validate(XMLFileName, XSDFileName)
	xml.xmlLineNumbersDefault(1)
	local ctxt = xml.xmlSchemaNewParserCtxt(XSDFileName)
	if ffi.isNull(ctxt) then
		return false, string.format("could not create schema parser context, file '%s'", XMLFileName)
	end
	-- xml.xmlSchemaSetParserErrors(ctxt, (xmlSchemaValidityErrorFunc) fprintf, (xmlSchemavalidityWarningFunc) fprintf, stderr)
	-- callbackCtx[0] = 1
	xml.xmlSchemaSetParserErrors(ctxt, xmlSchemaValidityErrorFuncPointer, xmlSchemavalidityWarningFuncPointer, nil)
  local schema = xml.xmlSchemaParse(ctxt)
  xml.xmlSchemaFreeParserCtxt(ctxt)
	if ffi.isNull(schema) then
		-- util.printError("could not parse schema, file '%s'", XMLFileName)
		return false, validityError, validityWarning
	end
	local doc = xml.xmlReadFile(XSDFileName, nil, 0)
	if ffi.isNull(doc) then
		-- util.printError("could not parse xml, file '%s'", XMLFileName)
		return false, validityError, validityWarning
	end
	ctxt = xml.xmlSchemaNewValidCtxt(schema)
	if ffi.isNull(ctxt) then
		return false, string.format("could not create schema valid context, file '%s'", XMLFileName)
	end
	-- xmlSchemaSetValidErrors(ctxt, (xmlSchemaValidityErrorFunc) fprintf, (xmlSchemavalidityWarningFunc) fprintf, stderr)
	xml.xmlSchemaSetValidErrors(ctxt, xmlSchemaValidityErrorFuncPointer, xmlSchemavalidityWarningFuncPointer, nil)
	local ret = xml.xmlSchemaValidateDoc(ctxt, doc)
	xml.xmlSchemaFreeValidCtxt(ctxt)
	xml.xmlSchemaFree(schema)
	if ret == 0 then
		return true
	elseif ret < 0 then
		validityError = (validityError or "")..string.format(" - validation generated an internal error, file '%s'", XMLFileName)
	end
	return false, validityError, validityWarning
end

function xmlschema.shutdown()
	xml.xmlSchemaCleanupTypes()
  xml.xmlCleanupParser()
  -- xml.xmlMemoryDump()
end

return xmlschema