--- xmlsec.lua
-- see: https://github.com/lsh123/xmlsec/blob/master/src/soap.c
local xmlsec = {}

local ffi = require "mffi"
local bor = require"bit".bor
local C = ffi.C
require "xmlsec_ffi"
local util = require "util"
local l = require"lang".l
local peg = require "peg"

-- load libs
if util.isWin() then
	util.loadDll("graphics/libwinpthread-1")
	if ffi.arch == "x64" then
		util.loadDll("graphics/libgcc_s_seh-1")
	else
		util.loadDll("graphics/libgcc_s_dw2-1")
	end
	util.loadDll("graphics/zlib1.dll")
	util.loadDll("graphics/libiconv-2")

	util.loadDll("libltdl-7")
	util.loadDll("libeay32")
	util.loadDll("graphics/zlib1.dll")
	util.loadDll("graphics/liblzma-5.dll")
	if ffi.arch == "x64" then
		xmlsec.xml = util.loadDll("graphics/libxml2-2")
	else
		xmlsec.xml = util.loadDll("libxml2-2") -- different from vips version
	end
	xmlsec.xslt = util.loadDll("libxslt-1")
	xmlsec.sec = util.loadDll("libxmlsec1")
	xmlsec.secSsl = util.loadDll("libxmlsec1-openssl")
elseif util.isMac() then
	xmlsec.secSsl = util.loadDll("libxmlsec1-OpenSSL.1.dylib") -- "libxmlsec1-OpenSSL.1.dylib") -- this is linked to libxmlsec1.1, can use both commands
	xmlsec.sec = util.loadDll("libxmlsec1.1.dylib") -- this is linked to libxml2.2.dylib, can use both commands
	xmlsec.xml = xmlsec.sec
	xmlsec.xslt = xmlsec.sec
elseif util.isLinux() then
	-- sudo apt install libxmlsec1-OpenSSL libxmlsec1
	xmlsec.secSsl = util.loadDll("/usr/lib/libxmlsec1-OpenSSL.so.1")
	xmlsec.sec = util.loadDll("/usr/lib/libxmlsec1.so.1")
	xmlsec.xml = xmlsec.sec
	xmlsec.xslt = xmlsec.sec
end

xmlsec.xmlutil = {}
-- xml.xmlFree() is a pointer to free function, calling it will crash, we need to get pointer to function
local xmlFreeFuncPtr = ffi.newNoAnchor("xmlFreeFunc[1]")
xmlsec.xml.xmlMemGet(xmlFreeFuncPtr, nil, nil, nil)
-- xml.xmlMemGet(  &xmlMem.xmlFree, &xmlMem.xmlMalloc, &xmlMem.xmlRealloc, &xmlMem.xmlMemStrdup);
xmlsec.xmlutil.xmlFree = xmlFreeFuncPtr[0]

local sec, secSsl = xmlsec.sec, xmlsec.secSsl
local xml, xmlutil = xmlsec.xml, xmlsec.xmlutil

function xmlsec.outText(doc, outFile)
	local mem = ffi.newNoAnchor("xmlChar*[1]")
	local size = ffi.newNoAnchor("int[1]")
	xml.xmlDocDumpMemory(doc, mem, size)
	local out = ffi.string(mem[0], size[0])
	if outFile then
		util.writeFile(outFile, out)
	end
	-- xml.xmlFree() is a pointer to free function, calling it will crash, must use xmlutil.xmlFree()
	xmlutil.xmlFree(mem[0])
	return out
end

local function verifyResult(dsigCtx)
	local status = tonumber(dsigCtx.status)
	local retTxt
	if status == C.xmlSecDSigStatusUnknown then
		retTxt = "ERROR"
	elseif status == C.xmlSecDSigStatusSucceeded then
		retTxt = "OK"
	elseif status == C.xmlSecDSigStatusInvalid then
		retTxt = "FAIL"
	else
		retTxt = "UNKNOWN STATUS: " .. status
	end

	-- print stats about # of good/bad references/manifests
	local size1 = tonumber(sec.xmlSecPtrListGetSize(dsigCtx.signedInfoReferences))
	local good1 = 0
	if size1 < 100 then
		for i = 0, size1 - 1 do
			local secPtr = sec.xmlSecPtrListGetItem(dsigCtx.signedInfoReferences, i)
			local dsigRefCtx = ffi.cast("xmlSecDSigReferenceCtxPtr", secPtr)
			if ffi.isNull(dsigRefCtx) then
				retTxt = retTxt .. l("\n  error: signedInfo reference ctx is null in position %d", i + 1)
			elseif tonumber(dsigRefCtx[0].status) == C.xmlSecDSigStatusSucceeded then
				good1 = good1 + 1
			end
		end
	end

	-- print stats about # of good/bad manifests
	local size2 = tonumber(sec.xmlSecPtrListGetSize(dsigCtx.manifestReferences))
	local good2 = 0
	if size2 < 100 then
		for i = 0, size2 - 1 do
			local secPtr = sec.xmlSecPtrListGetItem(dsigCtx.manifestReferences, i)
			local dsigRefCtx = ffi.cast("xmlSecDSigReferenceCtxPtr", secPtr)
			if ffi.isNull(dsigRefCtx) then
				retTxt = retTxt .. l("\n  error: manifest reference ctx is null in position %d", i + 1)
			elseif tonumber(dsigRefCtx[0].status) == C.xmlSecDSigStatusSucceeded then
				good2 = good2 + 1
			end
		end
	end

	if retTxt == "OK" then
		util.print("xml verify result: " .. retTxt) -- util.printInfo("Verify result: "..retTxt)
	else
		util.printWarning("xml verify result: " .. retTxt)
	end
	if good1 == size1 then
		util.printOk(l("SignedInfo References (ok/all): %d/%d", good1, size1))
	else
		util.printOk(l("SignedInfo References (ok/all): %d/%d", good1, size1))
	end
	if good2 == size2 then
		util.printOk(l("Manifests References (ok/all): %d/%d", good2, size2))
	else
		util.printOk(l("Manifests References (ok/all): %d/%d", good2, size2))
	end
	return good1 == size1 and good2 == size2
end

function xmlsec.action(action, inFile, keyFile, pwd, outFile)
	local ret, mngr, doc, startNode
	local signTime = 0
	local dsigCtx -- = sec.xmlSecDSigCtxCreate(nil) -- ffi.newNoAnchor("xmlSecDSigCtx[1]")

	local function done()
		if ffi.isNotNull(dsigCtx) then
			-- sec.xmlSecDSigCtxFinalize(dsigCtx)
			-- https://www.aleksey.com/xmlsec/api/xmlsec-notes-verify.html: Destroy signature context xmlSecDSigCtx using xmlSecDSigCtxDestroy or xmlSecDSigCtxFinalize functions.
			sec.xmlSecDSigCtxDestroy(dsigCtx)
		end
		if ffi.isNotNull(mngr) then
			sec.xmlSecKeysMngrDestroy(mngr)
		end
		if ffi.isNotNull(doc) then
			xml.xmlFreeDoc(doc)
		end
		return -1
	end

	local function loadKey()
		local fileData = util.readFile(keyFile, "binary")
		if fileData == nil then
			util.printError(l("failed to load key file '%s'", keyFile))
			return
		end
		-- local key = secSsl.xmlSecOpenSSLAppKeyLoad(file, sec.xmlSecKeyDataFormatPkcs12, pwd, nil, nil)
		local key = secSsl.xmlSecOpenSSLAppPkcs12LoadMemory(fileData, #fileData, pwd, nil, nil)
		-- local key = secSsl.xmlSecOpenSSLAppKeyLoadMemory(fileData, #fileData, sec.xmlSecKeyDataFormatPkcs12, pwd, nil, nil)
		-- print("signKey: "..tostring(key))
		if ffi.isNull(key) then
			util.printError("failed to create a key")
			return
		end
		-- ret = sec.xmlSecKeySetName (key, name)
		ret = secSsl.xmlSecOpenSSLAppDefaultKeysMngrAdoptKey(mngr, key)
		if ret < 0 then
			util.printError("failed to adopt the key")
			return
		end
		return key
	end

	mngr = sec.xmlSecKeysMngrCreate()
	if ffi.isNull(mngr) then
		util.printError("failed to create keys manager");
		return done()
	end
	ret = secSsl.xmlSecOpenSSLAppDefaultKeysMngrInit(mngr)
	if ret < 0 then
		util.printError("failed to initialize keys manager");
		return done()
	end
	local key = loadKey()
	if ffi.isNull(key) then
		util.printError("failed to create key")
		return done()
	end

	dsigCtx = sec.xmlSecDSigCtxCreate(mngr)
	ret = sec.xmlSecDSigCtxInitialize(dsigCtx, mngr)
	dsigCtx.keyInfoReadCtx.flags = bor(dsigCtx.keyInfoReadCtx.flags, C.XMLSEC_KEYINFO_FLAGS_X509DATA_DONT_VERIFY_CERTS)
	-- , C.XMLSEC_KEYINFO_FLAGS_DONT_STOP_ON_KEY_FOUND
	if ret < 0 then
		util.printError("failed to initialize DSig context");
		return done()
	end
	-- ret = sec.xmlSecAppPrepareDSigCtx(&dsigCtx, mngr)
	local function xmlSecAppdataCreate(file)
		if peg.endsWith(file, ".xml") then
			doc = sec.xmlSecParseFile(file)
		else
			local recovery = 1
			doc = sec.xmlSecParseMemory(file, #file, recovery)
		end
		if ffi.isNull(doc) then
			util.printError(l("failed to parse xml file '%s'", file))
			return -1
		end
		local root = xml.xmlDocGetRootElement(doc)
		if ffi.isNull(root) then
			util.printError(l("failed to get root element if file '%s'", inFile))
			return -1
		end

		-- local xmlSecNodeSignature = ffi.string(C.xmlSecNodeSignature) --does not work in win,  const xmlChar xmlSecNodeSignature[];
		-- win error: cannot resolve symbol 'xmlSecNodeSignature'
		-- local xmlSecDSigNs = ffi.string(C.xmlSecDSigNs)
		local xmlSecNodeSignature = "Signature"
		local xmlSecDSigNs = "http://www.w3.org/2000/09/xmldsig#"
		startNode = sec.xmlSecFindNode(root, xmlSecNodeSignature, xmlSecDSigNs)
		if ffi.isNull(startNode) then
			util.printError(l("failed to find default start node with name '%s'", xmlSecNodeSignature))
			return -1
		end
		return 0
	end
	ret = xmlSecAppdataCreate(inFile)
	if ret < 0 then
		return done()
	end

	local function sign()
		signTime = util.seconds()
		ret = sec.xmlSecDSigCtxSign(dsigCtx, startNode)
		signTime = util.seconds(signTime)
		if ret >= 0 then
			return xmlsec.outText(doc, outFile)
		end
		return nil
	end

	local function verify()
		signTime = util.seconds()
		ret = sec.xmlSecDSigCtxVerify(dsigCtx, startNode)
		signTime = util.seconds(signTime)
		if ret < 0 then
			util.printError("verify call failed, xml 'Signature' -tag was not found or was not valid")
			return nil
		end
		if verifyResult(dsigCtx) == false then
			return nil
		end
		return true
	end

	if action == "sign" then
		ret = sign()
		if ret == nil then
			util.printError("sign failed")
		end
	elseif action == "verify" then
		ret = verify()
		if ret == nil then
			util.printError("verify failed")
		else
			ret = xmlsec.outText(doc, outFile)
		end
	else
		util.printError("actuion is nit 'sign' or 'verify'")
	end
	done()

	--[[ dsigCtx = sec.xmlSecDSigCtxCreate(nil)
	-- https://github.com/lsh123/xmlsec/blob/master/apps/crypto.c

	-- we need to declare dsigCtx
	dsigCtx.signKey = key
	local keyName = sec.xmlSecKeySetName(dsigCtx.signKey, key_file)
	local xx = sec.xmlSecDSigCtxSign(dsigCtx.signKey, key_file)
	xml.xmlDocDump(io.stdout, data[0].doc) -- xmlDocDumpMemory
	]]

	--[[
    /* find start node */
    startNode = xmlSecFindNode(xmlDocGetRootElement(doc), xmlSecNodeSignature, xmlSecDSigNs);
    if(startNode == NULL) {
        fprintf(stderr, "Error: start start node not found in \"%s\"\n", tmpl_file);
        goto done;
    }

    /* create signature context, we don't need keys manager in this example */
    dsigCtx = xmlSecDSigCtxCreate(NULL);
    if(dsigCtx == NULL) {
        fprintf(stderr,"Error: failed to create signature context\n");
        goto done;
    }

    /* load private key, assuming that there is not password */
    dsigCtx->signKey = xmlSecOpenSSLAppKeyLoad(key_file, xmlSecKeyDataFormatPem, NULL, NULL, NULL);
    if(dsigCtx->signKey == NULL) {
        fprintf(stderr,"Error: failed to load private pem key from \"%s\"\n", key_file);
        goto done;
    }

    /* set key name to the file name, this is just an example! */
    if(xmlSecKeySetName(dsigCtx->signKey, key_file) < 0) {
        fprintf(stderr,"Error: failed to set key name for key from \"%s\"\n", key_file);
        goto done;
    }

    /* sign the template */
    if(xmlSecDSigCtxSign(dsigCtx, startNode) < 0) {
        fprintf(stderr,"Error: signature failed\n");
        goto done;
    }

    /* print signed document to stdout */
    xmlDocDump(stdout, doc);
	--]]
	return ret, signTime
end

function xmlsec.init()
	xml.xmlInitParser()
	local ret = sec.xmlSecInit()
	ret = sec.xmlSecCheckVersionExt(C.XMLSEC_VERSION_MAJOR, C.XMLSEC_VERSION_MINOR, C.XMLSEC_VERSION_SUBMINOR, C.xmlSecCheckVersionABICompatible)
	if ret ~= 1 then
		util.printError("loaded xmlsec library version is not compatible");
		return
	end
	ret = secSsl.xmlSecOpenSSLAppInit(nil) -- xmlSecOpenSSLAppInit(nil)
	if ret < 0 then
		util.printError("xmlSecOpenSSLAppInit initialization failed");
		return
	end
	ret = secSsl.xmlSecOpenSSLInit()
	if ret < 0 then
		util.printError("xmlSecOpenSSLInit initialization failed");
		return
	end
	return true
end

function xmlsec.shutdown()
	local ret = secSsl.xmlSecOpenSSLShutdown()
	ret = secSsl.xmlSecOpenSSLAppShutdown()
	ret = sec.xmlSecShutdown()
	xml.xmlCleanupParser()

	--[[ Shutdown libxslt/libxml
			xsltFreeSecurityPrefs(xsltSecPrefs);
			xsltCleanupGlobals();
			xmlCleanupParser();
	]]
end

return xmlsec
