--- lib/xslt.lua
-- see: https://github.com/lsh123/xslt/blob/master/src/soap.c

local xslt = {}

local ffi = require "mffi"
local fs = require "fs"
local peg = require "peg"
local utf = require "utf"
local util = require "util"
local l = require "lang".l
local xmlua = require "xmlua"
local xmlsec = require "xmlsec"
local xsltLib = xmlsec.xslt
local initDone = false


-- load libs
function xslt.init()
	initDone = true
	xsltLib.xsltInit()
end

function xslt.convert(fileName, xlsFile, outFileName, parseAfter)
	if not initDone then
		util.printWarning("xslt.init() is not done, do also xslt.cleanup() when done")
		xslt.init()
	end
	local err
	if not fs.fileExists(xlsFile) then
		err = l("xls file '%s' does not exist")
		util.printError(err)
		return nil, err
	end
	local txt = fs.readFile(fileName) -- ExportData_112_989_tekpal_testi.xml
	txt = peg.replace(txt, "\r\n", "\n")
	if parseAfter then
		txt = peg.parseAfter(txt, parseAfter.."\n") -- "</SOAP-ENV:Envelope>\n"
	end
	txt = peg.replace(txt, 'encoding="ISO-8859-15"', 'encoding="UTF-8"')
	txt = utf.latin9ToUtf8(txt)
	local doc = xmlua.XML.parse(txt)
	if not util.tableIsEmpty(doc.errors) then
		err = l("xml parse for file '%s' failed with errors '%s'", table.concat(doc.errors, "\n"))
		util.printError(err)
		return nil, err
	end
	--[[
	xml.xmlInitParser()
	local root = xml.xmlDocGetRootElement(doc)
	if ffi.isNull(root) then
		util.printError(l("failed to get root element if file '%s'", inFile))
		return -1
	end
	xml.xmlFreeDoc(doc)
	xml.xmlCleanupParser()
	]]
	local filePtr, res, cur, ret
	local function cleanup()
		if filePtr then
			filePtr:close()
		end
		if ffi.isNotNull(res) then
			xsltLib.xmlFreeDoc(res)
		end
		-- xmlFreeDoc(doc) -- is handled by xmlua
		if ffi.isNotNull(cur) then
			xsltLib.xsltFreeStylesheet(cur)
		end
		xsltLib.xsltCleanupGlobals()
		-- xmlCleanupParser()
	end

	cur = xsltLib.xsltParseStylesheetFile(ffi.cast("xmlChar *", xlsFile)) -- need to open new cur for each file or will crash
	if ffi.isNull(cur) then
		err = l("xml stylesheet load from file '%s' failed", xlsFile)
		util.printError(err)
		cleanup()
		return ret, err
	end
	res = xsltLib.xsltApplyStylesheet(cur, doc.document, nil)
	if ffi.isNull(res) then
		err = l("xml stylesheet apply for file '%s', stylesheet '%s' failed", fileName, xlsFile)
		util.printError(err)
		cleanup()
		return ret, err
	end
	filePtr = io.open(outFileName, "w")
	if filePtr == nil then
		err = l("xml stylesheet output file file '%s', could not be opened", outFileName)
		util.printError(err)
		cleanup()
		return ret, err
	end
	ret = xsltLib.xsltSaveResultToFile(filePtr, res, cur)
	if ffi.isNull(ret) then
		ret = nil
		err = l("xml stylesheet save result to file file '%s', failed", fileName)
		util.printError(err)
	end
	cleanup()
	return ret, err -- ret is count of chars in output html
end

function xslt.cleanup()
	initDone = false
	xsltLib.xsltCleanupGlobals()
end

function xslt.shutdown()
	initDone = false
	xmlsec.shutdown()
end

return xslt