--- xxhash
-- https://github.com/sjnam/luajit-xxHash/blob/master/lib/resty/xxhash.lua
-- libs taken from
-- see also pure-luajit: https://github.com/szensk/luaxxhash
-- @module crypto
local ffi = require 'mffi'
local util = require 'util'
local xxhash = util.loadDll('xxhash')

-- local C = ffi.C
local tostring = tostring
local format = string.format
local rshift = bit.rshift
local band = bit.band
local setmetatable = setmetatable

--[[ if ffi.os == "Windows" then
	C = ffi.loadMsvcr()
end ]]

ffi.cdef [[
unsigned XXH_versionNumber (void);
typedef enum { XXH_OK=0, XXH_ERROR } XXH_errorcode;

/* 32-bit hash */
typedef unsigned int XXH32_hash_t;
XXH32_hash_t XXH32 (const void* input, size_t length, unsigned int seed);

/*======   Streaming   ======*/
struct XXH32_state_s;
typedef struct XXH32_state_s XXH32_state_t;   /* incomplete type */
XXH32_state_t* XXH32_createState(void);
XXH_errorcode  XXH32_freeState(XXH32_state_t* statePtr);
XXH_errorcode XXH32_reset(XXH32_state_t* statePtr, XXH32_hash_t seed);
XXH_errorcode XXH32_update(XXH32_state_t* statePtr, const void* input, size_t length);
XXH32_hash_t  XXH32_digest(const XXH32_state_t* state);

/* 64-bit hash */
typedef uint64_t XXH64_hash_t;
static const int  XXH3_SECRET_DEFAULT_SIZE = 192;
static const int  XXH3_INTERNALBUFFER_SIZE = 256;

/*======   Streaming   ======*/
struct XXH3_state_s;
typedef struct XXH3_state_s XXH3_state_t;   /* incomplete type */
XXH3_state_t* XXH3_createState(void);
XXH_errorcode XXH3_freeState(XXH3_state_t* statePtr);

/* 64-bit hash */
XXH64_hash_t XXH3_64bits (const void* data, size_t len);
/* XXH64_hash_t XXH3_64bits_withSeed (const void* data, size_t len, XXH64_hash_t seed); */

XXH_errorcode XXH3_64bits_reset(XXH3_state_t* statePtr);
/* XXH_errorcode XXH3_64bits_reset_withSeed(XXH3_state_t* statePtr, XXH64_hash_t seed); */
XXH_errorcode XXH3_64bits_update(XXH3_state_t* statePtr, const void* input, size_t length);
XXH64_hash_t  XXH3_64bits_digest(const XXH3_state_t* statePtr);

/* 128-bit hash */
typedef struct {
    XXH64_hash_t low64;   /*!< `value & 0xFFFFFFFFFFFFFFFF` */
    XXH64_hash_t high64;  /*!< `value >> 64` */
} XXH128_hash_t;
XXH128_hash_t XXH3_128bits(const void* input, size_t len);
XXH_errorcode XXH3_128bits_reset(XXH3_state_t* statePtr);
XXH_errorcode XXH3_128bits_update (XXH3_state_t* statePtr, const void* input, size_t length);
XXH128_hash_t XXH3_128bits_digest (const XXH3_state_t* statePtr);

/* add 128-bit hash when needed */
]]

-- 32-bit hash

local _xxhash32 = {}

local function reset(self)
	return xxhash.XXH32_reset(self._state, self.seed)
end

_xxhash32.reset = reset

function _xxhash32.new(self, seed)
	local obj = {seed = seed or 0, _state = xxhash.XXH32_createState()}
	reset(obj)
	setmetatable(obj, self)
	self.__index = self
	return obj
end

function _xxhash32.update(self, input)
	return xxhash.XXH32_update(self._state, input, #input)
end

function _xxhash32.digest(self)
	self._hash = xxhash.XXH32_digest(self._state)
	return tostring(self._hash)
end

function _xxhash32.free(self)
	xxhash.XXH32_freeState(self._state)
end

-- 64-bit hash

local _xxhash64 = {}

local _ull = util.numberToHex -- util.numberULLToHex,  util.numberToHex

local function reset64(self)
	return _ull(xxhash.XXH3_64bits_reset(self._state, self.seed))
end
_xxhash64.reset = reset64

function _xxhash64.new(self, seed)
	local obj = {seed = seed or 0, _state = xxhash.XXH3_createState()}
	reset64(obj)
	setmetatable(obj, self)
	self.__index = self
	return obj
end

function _xxhash64.update(self, input)
	return xxhash.XXH3_64bits_update(self._state, input, #input)
end

function _xxhash64.digest(self)
	self._hash = xxhash.XXH3_64bits_digest(self._state)
	return _ull(self._hash)
end

-- 128-bit hash

local _xxhash128 = {}

local function reset128(self)
	return _ull(xxhash.XXH3_128bits_reset(self._state, self.seed))
end
_xxhash128.reset = reset128

function _xxhash128.new(self, seed)
	local obj = {seed = seed or 0, _state = xxhash.XXH3_createState()}
	reset128(obj)
	setmetatable(obj, self)
	self.__index = self
	return obj
end

function _xxhash128.update(self, input)
	return xxhash.XXH3_128bits_update(self._state, input, #input)
end

function _xxhash128.digest(self)
	self._hash = xxhash.XXH3_128bits_digest(self._state)
	return _ull(self._hash)
end

function _xxhash128.free(self)
	xxhash.XXH3_freeState(self._state)
end

-- xxhash
local xx = {_VERSION = '0.8.1', [32] = _xxhash32, [64] = _xxhash64, [128] = _xxhash128}

function xx.version()
	return xxhash.XXH_versionNumber()
end

function xx.hash32string(input)
	return tostring(xxhash.XXH32(input, #input, 0))
end

function xx.hash32num(input)
	return xxhash.XXH32(input, #input, 0)
end

function xx.hash32(input)
	return format("%x", xxhash.XXH32(input, #input, 0))
end

function xx.hash64string(input)
	return _ull(xxhash.XXH3_64bits(input, #input))
	-- return tostring(xxhash.XXH3_64bits(input, #input)):sub(1, -4) -- much faster, but wrong result
end

function xx.hash64num(input)
	return xxhash.XXH3_64bits(input, #input)
end

xx.hash64 = xx.hash64string
--[[ function xx.hash64(input)
	local out = xxhash.XXH3_64bits(input, #input)
	return format("%x%x", tonumber(rshift(out, 32)), tonumber(band(out, 0xFFFFFFFF)))
end ]]

function xx.hash128string(input)
	local ret = xxhash.XXH3_128bits(input, #input)
	return _ull(ret.high64) .. _ull(ret.low64)
	-- return tostring(xxhash.XXH3_128bits(input, #input)):sub(1, -4) -- much faster, but wrong result
end

--[[ function xx.hash128num(input)
	return xxhash.XXH3_128bits(input, #input -- this returns struct
end ]]

function xx.hash128(input)
	local out = xxhash.XXH3_128bits(input, #input)
	return format("%x%x%x%x", tonumber(rshift(out, 96)), tonumber(rshift(out, 64)), tonumber(rshift(out, 32)), tonumber(band(out, 0xFFFFFFFF)))
end

function xx.new(bits)
	local seed = 0
	if bits ~= 32 and bits ~= 64 and bits ~= 128 then
		seed = bits
		bits = 32
	end
	return xx[bits]:new(seed)
end

return xx
