# Database Synchronization Function Naming Conventions

## Core Naming Principles

### 1. SINGULAR Names Only

- **USE**: `deleteTargetRecord` (not `deleteTargetRecords`)
- **WHY**: Consistency, simplicity, avoids singular/plural confusion
- **AVOID**: Plural forms like `Records`, `Arrays`, `Ranges`

### 2. Target vs Source is CRITICAL

- **USE**: `queryTargetRecord`, `deleteSourceData`
- **WHY**: Database sync safety - must know which database is being modified
- **FORMAT**: Action + Target/Source + What
- **EXAMPLE**: `deleteTargetOutOfRange` (delete FROM target, the out-of-range records)

### 3. Technical Verbs Only

- **USE**: `query`, `delete`, `count`, `find`, `merge`
- **AVOID**: `identify`, `mark`, `apply`, `discover`
- **WHY**: Use programming operations, not cognitive terms
- **EXAMPLE**: `queryTargetRecord` (not `identifyTargetRecord`)

### 4. No Redundant Suffixes

- **USE**: `deleteTargetRecord` when obvious
- **USE**: `countTargetAll` (not `countTargetRecord` - "Record" is redundant)
- **WHY**: Don't state the obvious
- **EXCEPTION**: Use suffix when needed for clarity (`getIdArrayInRange`)

### 5. Action + Target + What Word Order

- **GOOD**: `deleteTargetOutOfRange`
- **BAD**: `deleteOutOfRangeTarget` (confusing)
- **WHY**: Clear what action, where, what

## Function Name Examples

| Original Name | Corrected Name | Why |
|---------------|----------------|-----|
| `markOutOfRangeForExclusion` | `queryTargetOutOfRange` | "mark" vague, "ForExclusion" jargon |
| `deleteMarkedBoundaryRecords` | `deleteTargetOutOfRange` | "MarkedBoundary" vague, "Records" redundant |
| `queryTargetCountAll` | `countTargetAll` | "queryCount" redundant, "Record" confusing |
| `findSourceMinMaxRecordId` | `findMinMaxRecordId` | Gets from BOTH databases, not just source |

## Good Names to Keep

These are already clear and accurate:

- **`countInRange`** - "count" is action, "InRange" is scope
- **`getIdArrayInRange`** - "IdArray" specifies return type clearly
- **`verifyDeletionSafety`** - descriptive and accurate
- **`findMinMaxRecordId`** - specific and accurate

## Anti-Patterns to Avoid

### Corporate/Abstract Terms

- **AVOID**: `identify`, `mark`, `apply`, `discover`, `exclusion`
- **USE**: `query`, `find`, `delete`, `count`, `merge`

### Redundant Information

- **AVOID**: `deleteRecordArray`, `countRecordCount`
- **USE**: `deleteRecord`, `countAll`

### Confusing Word Order

- **AVOID**: `deleteOutOfRangeTarget`
- **USE**: `deleteTargetOutOfRange`

### Assumptions

- **AVOID**: `mergeOptimizedRange` (who optimized?)
- **USE**: `mergeRange` (simple, accurate)

## Database Sync Specific Rules

### Target/Source Distinction

- **Target**: Database being synchronized TO
- **Source**: Authoritative database being synchronized FROM
- **NEVER mix them up** - could cause data corruption

### Range Operations

- **"InRange"** - clear scope specification
- **"OutOfRange"** - clear scope specification
- **"MinMax"** - clear boundary operation

### Return Type Clarity

- **Add suffix only when needed**: `getIdArrayInRange`
- **Skip when obvious**: `deleteTargetRecord`
- **Count needs object**: `countTargetAll` (what are we counting?)

## Testing Your Names

Ask these questions:

1. **Is it singular?** (no plurals)
2. **Is Target/Source clear?** (database safety)
3. **Is the verb technical?** (programming operation)
4. **Is anything redundant?** (remove obvious words)
5. **Is word order clear?** (Action + Target + What)
6. **Does it make assumptions?** (remove "optimized", etc.)

Good names = Clear, specific, technical, and expressive about what they do or are. NOT vague, redundant, or confusing.

See: /Volumes/nc/nc-devops/conf/vscode/AGENTS.md
