# PE Dependency Checker

A Lua program to analyze PE (Portable Executable) files for DLL dependencies and check for missing libraries in the target directory `/Volumes/nc/bin/bin_win_x64`.

## Features

- Extracts DLL dependencies from PE files using `objdump`
- Checks if required DLLs exist in the target directory
- Reports found and missing DLLs
- Interactive mode to copy missing DLLs
- Batch processing for multiple files

## Usage

### Command Line

```bash
# Basic usage
lua dependency-check.lua /path/to/program.exe

# With custom target directory
lua dependency-check.lua /path/to/program.exe /custom/target/dir
```

### Lua Module

```lua
local peChecker = require "dependency-check"

-- Check single file
local results = peChecker.checkDependency("/path/to/program.exe")

-- Check multiple files
local peFileArray = {"/path/to/file1.exe", "/path/to/file2.dll"}
local allResults = peChecker.checkMultipleFile(peFileArray)

-- Interactive copy of missing DLLs
peChecker.promptCopyMissing(results)
```

## API Reference

### `checkDependency(peFilePath, targetDir)`

Analyzes a PE file for DLL dependencies.

**Parameters:**

- `peFilePath` (string): Path to the PE file
- `targetDir` (string, optional): Target directory (default: `/Volumes/nc/bin/bin_win_x64`)

**Returns:**

- `results` (table): Object with dependency analysis
  - `dependencyArray`: All DLL dependencies found
  - `foundArray`: DLLs that exist in target directory
  - `missingArray`: DLLs that need to be copied
  - `targetDir`: The target directory used

### `checkMultipleFile(peFileArray, targetDir)`

Analyzes multiple PE files and combines missing DLL results.

**Parameters:**

- `peFileArray` (table): Array of PE file paths
- `targetDir` (string, optional): Target directory

**Returns:**

- `resultsByFile` (table): Results keyed by file path

### `promptCopyMissing(results)`

Interactive prompt to copy missing DLLs to target directory.

**Parameters:**

- `results` (table): Results from `checkDependency()`

## Requirements

- Lua interpreter
- `objdump` utility (part of GNU binutils)
- Access to the NC codebase modules: `util`, `fs`, `peg`

## Example Output

```
Analyzing PE file: /path/to/program.exe
Target directory: /Volumes/nc/bin/bin_win_x64

Found 7 DLL dependency:
✓ ADVAPI32.dll
✓ GDI32.dll
✓ KERNEL32.dll
✗ msvcrt.dll - MISSING
✓ SHELL32.DLL
✗ USER32.dll - MISSING
✓ WS2_32.dll

Summary:
- Total dependency: 7
- Found: 4
- Missing: 2

Missing DLL that need to be copied to /Volumes/nc/bin/bin_win_x64:
  - msvcrt.dll
  - USER32.dll

To copy missing DLL, you can use commands like:
  cp msvcrt.dll /Volumes/nc/bin/bin_win_x64/
  cp USER32.dll /Volumes/nc/bin/bin_win_x64/
```

## Implementation Details

The program uses `objdump -p` to parse PE headers and extract DLL dependencies. It then checks the target directory for each required DLL and reports the status.

Key functions:

- `extractDllNameArray()`: Parses objdump output to get DLL names
- `getPeDependencyArray()`: Executes objdump and processes output
- `dllExists()`: Checks if DLL exists in target directory
- `checkDependency()`: Main analysis function
