// form/nc/nc-help/nc-help.js
// NC Help Component Business Logic
// Follows NC coding standards with proper state management

// module-level variables to hold help data
let helpContent = null
let editableContent = null

// Toggle help display
function toggleHelp() {
	state.rec.show_help = !state.rec.show_help
	if (state.rec.show_help) {
		loadHelpContent(state.rec.help_topic)
	}
}

// Toggle edit mode
function toggleEditMode() {
	state.rec.edit_mode = !state.rec.edit_mode
	if (state.rec.edit_mode) {
		loadHelpContentForEditing(state.rec.help_topic)
	}
}

// Close help
function closeHelp() {
	state.rec.show_help = false
	state.rec.current_step = 0
}

// Load help content for display
function loadHelpContent(topic) {
	const contentPath = state.arr[topic]
	if (!contentPath) {
		console.warn(state.hdr.nc_help.missing_help_topic + topic)
		return
	}
	nc.callServer(
		'help.load_content',
		{
			topic: topic,
			language: state.rec.language || 'en'
		},
		response => {
			if (response && response.content) {
				helpContent = response.content
				state.rec.current_step = 0
			}
		}
	)
}

// Load help content for editing
function loadHelpContentForEditing(topic) {
	const contentPath = state.arr[topic]
	if (!contentPath) {
		console.warn(state.hdr.nc_help.missing_help_topic + topic)
		return
	}
	nc.callServer(
		'help.load_content_for_editing',
		{
			topic: topic,
			language: state.rec.language || 'en'
		},
		response => {
			if (response && response.content) {
				editableContent = response.content
			}
		}
	)
}

// Navigate to next step
function nextStep() {
	if (helpContent && helpContent.steps) {
		if (state.rec.current_step < helpContent.steps.length - 1) {
			state.rec.current_step++
		} else {
			closeHelp()
		}
	}
}

// Navigate to previous step
function previousStep() {
	if (state.rec.current_step > 0) {
		state.rec.current_step--
	}
}

// Go to specific step
function goToStep(stepIndex) {
	if (helpContent && helpContent.steps && stepIndex >= 0 && stepIndex < helpContent.steps.length) {
		state.rec.current_step = stepIndex
	}
}

// Save help content
function saveHelp() {
	if (!editableContent) {
		return
	}
	nc.callServer(
		'help.save_content',
		{
			topic: state.rec.help_topic,
			language: state.rec.language || 'en',
			content: editableContent
		},
		response => {
			if (response && response.success) {
				state.rec.edit_mode = false
				loadHelpContent(state.rec.help_topic)
			}
		}
	)
}

// Cancel editing
function cancelEdit() {
	state.rec.edit_mode = false
	editableContent = null
}

// Add new step to help content
function addStep() {
	if (!editableContent) {
		editableContent = { steps: [] }
	}
	const newStep = {
		title: state.rec.step_title || '',
		content: state.rec.step_content || '',
		type: state.rec.step_type || 'info',
		action: state.rec.step_action || 'none',
		selector: state.rec.step_selector || '',
		delay: state.rec.step_delay || 1000,
		position: state.rec.step_position || 'center'
	}
	editableContent.steps.push(newStep)
}

// Delete step from help content
function deleteStep(stepIndex) {
	if (editableContent && editableContent.steps && stepIndex >= 0 && stepIndex < editableContent.steps.length) {
		editableContent.steps.splice(stepIndex, 1)
	}
}

// Move step up in order
function moveStepUp(stepIndex) {
	if (editableContent && editableContent.steps && stepIndex > 0) {
		const step = editableContent.steps.splice(stepIndex, 1)[0]
		editableContent.steps.splice(stepIndex - 1, 0, step)
	}
}

// Move step down in order
function moveStepDown(stepIndex) {
	if (editableContent && editableContent.steps && stepIndex < editableContent.steps.length - 1) {
		const step = editableContent.steps.splice(stepIndex, 1)[0]
		editableContent.steps.splice(stepIndex + 1, 0, step)
	}
}

// Execute step action
function executeStepAction(step) {
	if (!step || !step.action) {
		return
	}
	switch (step.action) {
		case 'click':
			clickElement(step.selector)
			break
		case 'focus':
			focusElement(step.selector)
			break
		case 'highlight':
			highlightElement(step.selector)
			break
		case 'scroll':
			scrollToElement(step.selector)
			break
		case 'wait':
			wait(step.delay)
			break
		default:
			console.log('Unknown action:', step.action)
	}
}

// Click element by selector
function clickElement(selector) {
	const element = document.querySelector(selector)
	if (element) {
		element.click()
	}
}

// Focus element by selector
function focusElement(selector) {
	const element = document.querySelector(selector)
	if (element) {
		element.focus()
	}
}

// Highlight element by selector
function highlightElement(selector) {
	const element = document.querySelector(selector)
	if (element) {
		element.classList.add('nc-help-highlight')
		setTimeout(() => {
			element.classList.remove('nc-help-highlight')
		}, 2000)
	}
}

// Scroll to element by selector
function scrollToElement(selector) {
	const element = document.querySelector(selector)
	if (element) {
		element.scrollIntoView({ behavior: 'smooth' })
	}
}

// Wait for specified milliseconds
function wait(delay) {
	return new Promise(resolve => setTimeout(resolve, delay))
}

// Check if step is the last step
function isLastStep() {
	return helpContent && helpContent.steps && state.rec.current_step >= helpContent.steps.length - 1
}

// Check if step is the first step
function isFirstStep() {
	return state.rec.current_step <= 0
}

// Get current step
function getCurrentStep() {
	if (!helpContent || !helpContent.steps) {
		return null
	}
	return helpContent.steps[state.rec.current_step]
}

// Get total steps count
function getTotalSteps() {
	if (!helpContent || !helpContent.steps) {
		return 0
	}
	return helpContent.steps.length
}

// Export for use in Vue components
if (typeof module !== 'undefined' && module.exports) {
	module.exports = {
		initialize,
		toggleHelp,
		toggleEditMode,
		closeHelp,
		loadHelpContent,
		loadHelpContentForEditing,
		nextStep,
		previousStep,
		goToStep,
		saveHelp,
		cancelEdit,
		addStep,
		deleteStep,
		moveStepUp,
		moveStepDown,
		executeStepAction,
		clickElement,
		focusElement,
		highlightElement,
		scrollToElement,
		wait,
		isLastStep,
		isFirstStep,
		getCurrentStep,
		getTotalSteps
	}
}
