// /public/sw.js
// https://developers.google.com/web/ilt/pwa/caching-files-with-service-worker

const CACHE_NAME = 'v001'
const DEV_MODE = true
let openedCache

function isSafari() {
	return navigator.userAgent.indexOf('Safari') > -1 && navigator.userAgent.indexOf('Chrome') === -1
}

function isFirefox() {
	return navigator.userAgent.indexOf('Firefox') > -1
}

function postMessage(msg) {
	self.clients.matchAll({ includeUncontrolled: true, type: 'window' }).then(clients => {
		if (clients && clients.length > 0) {
			// you need to decide which clients you want to send the message to, we use only first
			const client = clients[0]
			const msg2 = JSON.parse(JSON.stringify(msg)) // prevent DOMException: Failed to execute 'postMessage' on 'Client': Response object could not be cloned.
			client.postMessage(msg2)
		}
	})
}

if (isSafari() || isFirefox()) {
	const debugOrig = console.debug
	const warnOrig = console.warn
	console.debug = (...args) => {
		debugOrig.apply(null, args)
		postMessage({ type: 'debug', args })
	}
	console.warn = (...args) => {
		warnOrig.apply(null, args)
		postMessage({ type: 'warn', args })
	}
}

self.addEventListener('activate', event => {
	return event.waitUntil(
		caches.keys().then(keyList => {
			console.debug(
				// `*** service-worker: activate, self.clients.claim() and delete old cache keys not equal to '${CACHE_NAME}'`
				`*** service-worker activate: delete all cache keys, keyList`,
				keyList
			)
			const ret = Promise.all(
				keyList.map(key => {
					// if (key !== CACHE_NAME) {
					console.debug(`*** service-worker activate: deleted cache key '${key}'`)
					return caches.delete(key)
					// }
				})
			)
			/* self.registration
				.unregister()
				.then(function () {
					return self.clients.matchAll()
				})
				.then(function (clients) {
					clients.forEach(client => {
						console.debug(`*** service-worker activate: unregister, reload client.url '${client.url}'`)
						client.navigate(client.url)
					})
				}) */
			self.clients.claim()
			return ret
		})
	)
})

self.addEventListener('install', event => {
	console.debug('*** service-worker: install start, event', event)
	self.skipWaiting()
	return event.waitUntil(
		caches.open(CACHE_NAME).then(cache => {
			openedCache = cache
			openedCache.keys().then(keys => {
				console.debug(`*** service-worker: install done, caches.open '${CACHE_NAME}', keys`, keys)
			})
			return
			/* 	return cache.addAll([
				'/src/cache-test/import.js'
			]) */
		})
	)
})

self.addEventListener('fetch', event => {
	if (openedCache) {
		const match = openedCache.match(event.request)
		return event.respondWith(
			match.then(response => {
				if (response) {
					// https://stackoverflow.com/questions/56654119/how-do-you-set-the-url-when-creating-a-new-response
					Object.defineProperty(response, 'url', { value: event.request.url })
					console.debug(`*** service-worker: SERVED FROM CACHE ${event.request.url}, response`, response)
					return response
				} else {
					const match2 = fetch(event.request) // must not be event.request.url because request may be PUSH, not GET
					return match2
						.then(response => {
							if (response.ok === false && response.status !== 304) {
								// 304 = Not Modified
								console.warn(`*** service-worker: match failed, fetch FAILED ${event.request.url}, response`, response)
								return response
							}
							if (!DEV_MODE || event.request.url.indexOf('.woff') > 0) {
								openedCache.put(event.request, response.clone())
								console.debug(`*** service-worker: match failed, fetch ok, cached ${event.request.url}`)
							} /* else {
								console.debug(
									`*** service-worker: match failed, fetch ok, not cached ${event.request.url}, status: ${response.status} ${response.statusText}`
								)
							} */
							return response
						})
						.catch(error => {
							console.warn(`*** service-worker: match failed, fetch ERROR ${event.request.url}, error`, error)
							return
						})
				}
			})
		)
	}
	// fallback method 'Cache falling back to the network'
	console.warn(`*** service-worker: XXX XXX, fetch ${event.request.url}`)
	return event.respondWith(
		caches.match(event.request).then(response => {
			if (response) {
				console.debug(`*** service-worker: XXX XXX, SERVED FROM CACHE ${event.request.url}`)
				return response
			}
			return fetch(event.request)
		})
	)
})
