// /public/sw.js
// see: https://stackoverflow.com/questions/40100922/activate-updated-service-worker-on-refresh
// https://developers.google.com/web/ilt/pwa/caching-files-with-service-worker

const CACHE_NAME = 'v001'
const DEV_MODE = 1
const componentCache = {}
const fileTypeToCache = {}
// const fileTypeToCacheArr = ['js', 'css', 'png', 'jpg', 'jpeg', 'gif', 'svg', 'woff', 'woff2', 'ttf', 'eot', 'otf', 'ico']
const fileTypeToCacheArr = ['css', 'png', 'jpg', 'jpeg', 'gif', 'svg', 'woff', 'woff2', 'ttf', 'eot', 'otf', 'ico']
for (const key of fileTypeToCacheArr) {
	fileTypeToCache[key] = true
}
let openedCache
let cacheOpenState = false

function parseAfter(subjectString, separator) {
	const pos = subjectString.indexOf(separator)
	if (pos >= 0) {
		return subjectString.substring(pos + separator.length)
	}
	return subjectString
}
function endsWith(subjectString, endString) {
	return subjectString.substring(subjectString.length - endString.length) === endString
}
function isSafari() {
	return navigator.userAgent.indexOf('Safari') > -1 && navigator.userAgent.indexOf('Chrome') === -1
}

/* function isFirefox() {
	return navigator.userAgent.indexOf('Firefox') > -1
} */

function postMessage(msg) {
	self.clients.matchAll({ includeUncontrolled: true, type: 'window' }).then(clients => {
		if (clients && clients.length > 0) {
			// you need to decide which clients you want to send the message to, we use only first
			const client = clients[0]
			const msg2 = JSON.parse(JSON.stringify(msg)) // prevent DOMException: Failed to execute 'postMessage' on 'Client': Response object could not be cloned.
			client.postMessage(msg2)
		}
	})
}

if (isSafari()) {
	// || isFirefox()
	const debugOrig = console.debug
	const warnOrig = console.warn
	console.debug = (...args) => {
		debugOrig.apply(null, args)
		postMessage({ type: 'debug', args })
	}
	console.warn = (...args) => {
		warnOrig.apply(null, args)
		postMessage({ type: 'warn', args })
	}
}

async function deleteKey(keyList) {
	const ret = await Promise.all(
		keyList.map(key => {
			// if (key !== CACHE_NAME) {
			console.debug(`•• service-worker activate: deleted cache key '${key}'`)
			return caches.delete(key)
			// }
		})
	)
	console.debug(`•• service-worker activate: delete all cache keys, keyList`, keyList)
	return ret
}
async function deleteKeyList() {
	// `•• service-worker: activate, self.clients.claim() and delete old cache keys not equal to '${CACHE_NAME}'`
	await caches.keys().then(keyList => {
		deleteKey(keyList)
	})
}

async function activate(event) {
	await event.waitUntil(self.clients.claim())
	/* self.registration
	.unregister()
	.then(function () {
		return self.clients.matchAll()
	})
	.then(function (clients) {
		clients.forEach(client => {
			console.debug(`•• service-worker activate: unregister, reload client.url '${client.url}'`)
			client.navigate(client.url)
		})
	}) */
	if (navigator.onLine) {
		await event.waitUntil(deleteKeyList())
	}
	await event.waitUntil(openCache())
}

self.addEventListener('activate', event => {
	console.debug(`•• service-worker activate start`)
	event.waitUntil(activate(event))
	console.debug(`•• service-worker activate done`)
})

async function openCache() {
	const ret = await caches
		.open(CACHE_NAME)
		.then(cache => {
			openedCache = cache
			return openedCache
				.keys()
				.then(keys => {
					console.debug(`•• service-worker: caches.open '${CACHE_NAME}', keys`, keys)
				})
				.catch(error => {
					console.warn(`•• service-worker: caches.open '${CACHE_NAME}' keys error`, error)
				})

			/* 	return cache.addAll([
				'/src/cache-test/import.js'
			]) */
		})
		.catch(error => {
			console.warn(`•• service-worker: caches.open error`, error)
		})
	return ret
}

self.addEventListener('install', event => {
	event.waitUntil(self.skipWaiting())
	console.debug(`•• service-worker: install done`)
})

async function componentRespose(event, response) {
	const ret = await response.clone().json()
	// console.debug(`🚀 ~ file: sw.js ~ line 93 ~ componentRespose ~ ret`, ret)
	if (ret.include) {
		const baseUrl = event.request.url.substring(0, event.request.url.indexOf('/rest/nc/')) + '/plugin/'
		for (const item of ret.include) {
			const url = baseUrl + item.path
			componentCache[url] = item.path
			const newRequest = new Request(url)
			let js
			if (event.request.url.search('localhost:8080') > -1 || event.request.url.search('dev.nadacode.com') > -1) {
				js = item.js.replaceAll('/assets/', '/src/core/')
			}
			const newResponse = new Response(js || item.js, { headers: { 'Content-Type': 'text/javascript' } })
			openedCache.put(newRequest, newResponse)
			if (DEV_MODE > 0) {
				console.debug(`•••• service-worker: ADD TO CACHE ${url}, ${Math.floor((item.js.length / 1024) * 10) / 10} Kb`)
			}
		}
	}
	return response
}

function fetchRespose(event, response) {
	if (response.ok === false && response.status !== 304) {
		// 304 = Not Modified
		console.warn(`•• service-worker: fetch FAILED ${event.request.url}, response`, response)
		return response
	}
	// console.log(`🚀 ~ file: sw.js ~ line 153 ~ fetchRespose ~ fileType`, fileType, event.request.url)
	if (endsWith(event.request.url, '/rest/nc/import/component') || endsWith(event.request.url, '/rest/nc/input')) {
		return componentRespose(event, response)
	}
	const fileType = parseAfter(event.request.url, '.')
	if (
		event.request.method == 'GET' &&
		event.request.url.indexOf('?') === -1 &&
		fileTypeToCache[fileType] &&
		event.request.url.substring(0, 4) === 'http'
	) {
		// skip chrome extensions do not start with 'http'
		if (DEV_MODE > 2) {
			console.debug(`••• service-worker: fetch ok, cached ${event.request.url}`)
		}
		openedCache.put(event.request, response.clone())
	} else {
		if (DEV_MODE > 3) {
			console.debug(
				`•• service-worker: fetch ok, not cached ${event.request.url}, status: ${response.status} ${response.statusText}` // example: status: 200 OK
			)
		}
	}
	return response
}

self.addEventListener('fetch', event => {
	if (!openedCache && cacheOpenState === false) {
		console.debug(`••• service-worker: fetch -event, openCache`)
		cacheOpenState = true
		openCache()
	}
	if (openedCache) {
		const match = openedCache.match(event.request)
		return event.respondWith(
			match.then(response => {
				if (response) {
					// https://stackoverflow.com/questions/56654119/how-do-you-set-the-url-when-creating-a-new-response
					// Object.defineProperty(response, 'url', { value: event.request.url })
					if (DEV_MODE > 0 && componentCache[event.request.url]) {
						console.debug(`••• service-worker: SERVED COMPONENT FROM CACHE ${event.request.url}`)
					} else if (DEV_MODE > 1) {
						console.debug(`••• service-worker: SERVED FROM CACHE ${event.request.url}`)
					}
					return response
				} else {
					return fetch(event.request) // must not be event.request.url because request may be PUSH, not GET
						.then(fetchResponse => {
							return fetchRespose(event, fetchResponse)
						})
						.catch(error => {
							console.warn(`•• service-worker: fetch ERROR ${event.request.url}, error`, error)
							return
						})
				}
			})
		)
	}
	// fallback method 'Cache falling back to the network'
	console.warn(`•• XXX XXX service-worker: XXX XXX, fetch ${event.request.url}`)
	return event.respondWith(
		caches.match(event.request).then(response => {
			if (response) {
				console.debug(`•• XXX XXX service-worker: XXX XXX, SERVED FROM CACHE ${event.request.url}`)
				return response
			}
			return fetch(event.request)
				.then(fetchResponse => {
					return fetchRespose(event, fetchResponse)
				})
				.catch(error => {
					console.warn(`•• XXX XXX service-worker: fetch ERROR ${event.request.url}, error`, error)
					return
				})
		})
	)
})
